/**
 * @file    CorTst_Shell.c
 * @version V2.0.0
 *
 * @brief   AUTOSAR CorTst module shell
 * @details API implementation for CORTST shell
 *
 * @addtogroup CORTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Core
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "CorTst_Compiler.h"
#include "CorTst_Shell.h"
#include "CorTst_Data.h"

#ifdef CORTST_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CORTST_SHELL_VENDOR_ID_C                      (180)
#define CORTST_SHELL_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CORTST_SHELL_AR_RELEASE_MINOR_VERSION_C       (4)
#define CORTST_SHELL_AR_RELEASE_REVISION_VERSION_C    (0)
#define CORTST_SHELL_SW_MAJOR_VERSION_C               (2)
#define CORTST_SHELL_SW_MINOR_VERSION_C               (0)
#define CORTST_SHELL_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* CorTst_Shell.h version check */
#if (CORTST_SHELL_VENDOR_ID_C != CORTST_SHELL_VENDOR_ID)
#error "CorTst_Shell.c and CorTst_Shell.h have different vendor ids"
#endif
#if ((CORTST_SHELL_AR_RELEASE_MAJOR_VERSION_C != CORTST_SHELL_AR_RELEASE_MAJOR_VERSION) || \
     (CORTST_SHELL_AR_RELEASE_MINOR_VERSION_C != CORTST_SHELL_AR_RELEASE_MINOR_VERSION) || \
     (CORTST_SHELL_AR_RELEASE_REVISION_VERSION_C != CORTST_SHELL_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CorTst_Shell.c and CorTst_Shell.h are different"
#endif
#if ((CORTST_SHELL_SW_MAJOR_VERSION_C != CORTST_SHELL_SW_MAJOR_VERSION) || \
     (CORTST_SHELL_SW_MINOR_VERSION_C != CORTST_SHELL_SW_MINOR_VERSION) || \
     (CORTST_SHELL_SW_PATCH_VERSION_C != CORTST_SHELL_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CorTst_Shell.c and CorTst_Shell.h are different"
#endif

/* CorTst_Data.h version check */
#if (CORTST_SHELL_VENDOR_ID_C != CORTST_DATA_VENDOR_ID)
#error "CorTst_Shell.c and CorTst_Data.h have different vendor ids"
#endif
#if ((CORTST_SHELL_AR_RELEASE_MAJOR_VERSION_C != CORTST_DATA_AR_RELEASE_MAJOR_VERSION) || \
     (CORTST_SHELL_AR_RELEASE_MINOR_VERSION_C != CORTST_DATA_AR_RELEASE_MINOR_VERSION) || \
     (CORTST_SHELL_AR_RELEASE_REVISION_VERSION_C != CORTST_DATA_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CorTst_Shell.c and CorTst_Data.h are different"
#endif
#if ((CORTST_SHELL_SW_MAJOR_VERSION_C != CORTST_DATA_SW_MAJOR_VERSION) || \
     (CORTST_SHELL_SW_MINOR_VERSION_C != CORTST_DATA_SW_MINOR_VERSION) || \
     (CORTST_SHELL_SW_PATCH_VERSION_C != CORTST_DATA_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CorTst_Shell.c and CorTst_Data.h are different"
#endif

/* CorTst_Compiler.h version check */
#if (CORTST_SHELL_VENDOR_ID_C != CORTST_COMPILER_VENDOR_ID )
#error "CorTst_Shell.c and CorTst_Compiler.h have different vendor ids"
#endif
#if ((CORTST_SHELL_AR_RELEASE_MAJOR_VERSION_C != CORTST_COMPILER_AR_RELEASE_MAJOR_VERSION ) || \
     (CORTST_SHELL_AR_RELEASE_MINOR_VERSION_C != CORTST_COMPILER_AR_RELEASE_MINOR_VERSION ) || \
     (CORTST_SHELL_AR_RELEASE_REVISION_VERSION_C != CORTST_COMPILER_AR_RELEASE_REVISION_VERSION ) \
    )
#error "AutoSar Version Numbers of CorTst_Shell.c and CorTst_Compiler.h are different"
#endif
#if ((CORTST_SHELL_SW_MAJOR_VERSION_C != CORTST_COMPILER_SW_MAJOR_VERSION ) || \
     (CORTST_SHELL_SW_MINOR_VERSION_C != CORTST_COMPILER_SW_MINOR_VERSION ) || \
     (CORTST_SHELL_SW_PATCH_VERSION_C != CORTST_COMPILER_SW_PATCH_VERSION ) \
    )
#error "Software Version Numbers of CorTst_Shell.c and CorTst_Compiler.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
#define CORTST_START_SEC_VAR_INIT_32
#include "CorTst_MemMap.h"
/* This variable contains the number of the most recently executed atomic test */
CORTST_VAR static uint32 CorTst_LastExecutedTestNumber = 0xFFFFFFFFUL;

/** MR12 1514 8.9 VIOLATION: This symbol is used in asm lib too */
CORTST_VAR uint32 CorTst_FaultInjectValue = 0x00000000UL; /* PRQA S 1514 */

#define CORTST_STOP_SEC_VAR_INIT_32
#include "CorTst_MemMap.h"
/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
#define CORTST_START_SEC_CODE
#include "CorTst_MemMap.h"
CORTST_FUNC Std_ReturnType CorTst_Execute(uint32 AtomicTestId, uint32 *Signature)
{
    uint32 AtomicTestResult;
    uint32 ConvertedAtomicTestResult;
    uint32(* AtomicTestStartAddress)(void);
    Std_ReturnType Ret = E_NOT_OK;
    CorTst_FaultInjectValue = 0U;
    CorTst_AccumulatedSignature = 0U;
    if ((AtomicTestId < CORTST_ATOMIC_TEST_NUM) && (Signature != NULL_PTR))
    {
        CorTst_LastExecutedTestNumber = AtomicTestId;
        AtomicTestStartAddress = CorTst_AtomicTestDesc[AtomicTestId].AtomicTestStartAddress;
        /* Execute the current core test and retrieve its result. */
#ifdef CORTST_ENABLE_USER_MODE_SUPPORT
        AtomicTestResult = (uint32)OsIf_Trusted_Call_Return(AtomicTestStartAddress);
#else
        AtomicTestResult = AtomicTestStartAddress();
#endif
        /* Check whether the result of the executed test is correct or not. */
        ConvertedAtomicTestResult = ((AtomicTestResult << 1U) | (AtomicTestResult >> 31U));
        ConvertedAtomicTestResult ^= AtomicTestResult;
        *Signature = ConvertedAtomicTestResult;
        Ret = E_OK;
    }
    return Ret;
}
#define CORTST_STOP_SEC_CODE
#include "CorTst_MemMap.h"
#ifdef __cplusplus
}
#endif

/** @} */

