/**
 * @file    CorTst.c
 * @version V2.0.0
 *
 * @brief   AUTOSAR CorTst module interface
 * @details API implementation for CorTst driver
 *
 * @addtogroup CORTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Core
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#include "Platform_Types.h"
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 0759 Rule 19.2: An object of union type has been defined.
 */

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "CorTst.h"
#include "CorTst_Data.h"
#include "CorTst_Shell.h"
#include "SchM_CorTst.h"
#include "OsIf.h"
#if (CORTST_M33_FPU_ENABLE == 1U)
#include "Platform.h"
#endif
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif
#if (CORTST_DEM_EVENT_REPORT == STD_ON)
#include "Dem.h"
#endif

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CORTST_VENDOR_ID_C                      (180)
#define CORTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CORTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define CORTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define CORTST_SW_MAJOR_VERSION_C               (2)
#define CORTST_SW_MINOR_VERSION_C               (0)
#define CORTST_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* CorTst.h version check */
#if (CORTST_VENDOR_ID_C != CORTST_VENDOR_ID)
#error "CorTst.c and CorTst.h have different vendor ids"
#endif
#if ((CORTST_AR_RELEASE_MAJOR_VERSION_C != CORTST_AR_RELEASE_MAJOR_VERSION) || \
     (CORTST_AR_RELEASE_MINOR_VERSION_C != CORTST_AR_RELEASE_MINOR_VERSION) || \
     (CORTST_AR_RELEASE_REVISION_VERSION_C != CORTST_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CorTst.c and CorTst.h are different"
#endif
#if ((CORTST_SW_MAJOR_VERSION_C != CORTST_SW_MAJOR_VERSION) || \
     (CORTST_SW_MINOR_VERSION_C != CORTST_SW_MINOR_VERSION) || \
     (CORTST_SW_PATCH_VERSION_C != CORTST_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CorTst.c and CorTst.h are different"
#endif

/* CorTst_Shell.h version check */
#if (CORTST_VENDOR_ID_C != CORTST_SHELL_VENDOR_ID)
#error "CorTst.c and CorTst_Shell.h have different vendor ids"
#endif
#if ((CORTST_AR_RELEASE_MAJOR_VERSION_C != CORTST_SHELL_AR_RELEASE_MAJOR_VERSION) || \
     (CORTST_AR_RELEASE_MINOR_VERSION_C != CORTST_SHELL_AR_RELEASE_MINOR_VERSION) || \
     (CORTST_AR_RELEASE_REVISION_VERSION_C != CORTST_SHELL_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CorTst.c and CorTst_Shell.h are different"
#endif
#if ((CORTST_SW_MAJOR_VERSION_C != CORTST_SHELL_SW_MAJOR_VERSION) || \
     (CORTST_SW_MINOR_VERSION_C != CORTST_SHELL_SW_MINOR_VERSION) || \
     (CORTST_SW_PATCH_VERSION_C != CORTST_SHELL_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CorTst.c and CorTst_Shell.h are different"
#endif

/* CorTst_Data.h version check */
#if (CORTST_VENDOR_ID_C != CORTST_DATA_VENDOR_ID)
#error "CorTst.c and CorTst_Data.h have different vendor ids"
#endif
#if ((CORTST_AR_RELEASE_MAJOR_VERSION_C != CORTST_DATA_AR_RELEASE_MAJOR_VERSION) || \
     (CORTST_AR_RELEASE_MINOR_VERSION_C != CORTST_DATA_AR_RELEASE_MINOR_VERSION) || \
     (CORTST_AR_RELEASE_REVISION_VERSION_C != CORTST_DATA_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CorTst.c and CorTst_Data.h are different"
#endif
#if ((CORTST_SW_MAJOR_VERSION_C != CORTST_DATA_SW_MAJOR_VERSION) || \
     (CORTST_SW_MINOR_VERSION_C != CORTST_DATA_SW_MINOR_VERSION) || \
     (CORTST_SW_PATCH_VERSION_C != CORTST_DATA_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CorTst.c and CorTst_Data.h are different"
#endif

/* SchM_CorTst.h version check */
#if (CORTST_VENDOR_ID_C != SCHM_CORTST_VENDOR_ID)
#error "CorTst.c and CorTst_Data.h have different vendor ids"
#endif
#if ((CORTST_AR_RELEASE_MAJOR_VERSION_C != SCHM_CORTST_AR_RELEASE_MAJOR_VERSION) || \
     (CORTST_AR_RELEASE_MINOR_VERSION_C != SCHM_CORTST_AR_RELEASE_MINOR_VERSION) || \
     (CORTST_AR_RELEASE_REVISION_VERSION_C != SCHM_CORTST_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CorTst.c and CorTst_Data.h are different"
#endif
#if ((CORTST_SW_MAJOR_VERSION_C != SCHM_CORTST_SW_MAJOR_VERSION) || \
     (CORTST_SW_MINOR_VERSION_C != SCHM_CORTST_SW_MINOR_VERSION) || \
     (CORTST_SW_PATCH_VERSION_C != SCHM_CORTST_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CorTst.c and CorTst_Data.h are different"
#endif

/*==================================================================================================
 *                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

/*==================================================================================================
 *                                        GLOBAL CONSTANTS
==================================================================================================*/
/** @brief   CorTst driver configuration structure */
extern const CorTst_ConfigType CorTst_Config;


/*==================================================================================================
 *                                        LOCAL VARIABLES
==================================================================================================*/
#define CORTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "CorTst_MemMap.h"

/**
 * @brief       CorTst driver state structure
 * @private
 */
CORTST_VAR static CorTst_DriverStateType CorTst_DriverState =
{
    .CorTstConfig = NULL_PTR,
    .State = CORTST_UNINIT,
    .TestId = 0U,
    .AtomicTestId = 0U,
    .TestIntervalId = 0U,
    .ErrOk =
    {
        .TestIntervalId = 0U,
        .Result = CORTST_E_NOT_TESTED,
    },
    .FgndSignErrCnt = 0U,
    .FgndAccumSign = 0U,
    .BgndResult = CORTST_E_NOT_TESTED,
    .BgndSignErrCnt = 0U,
    .BgndAccumSign = 0U,
    .BgndSign =
    {
        .TestIntervalId = 0U,
        .Signature = 0U,
    }
};

/**
 * @brief       CorTst FgndResult
 * @private
 */
CORTST_VAR static CorTst_ResultType  CorTst_FgndResult = CORTST_E_NOT_TESTED;

#define CORTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "CorTst_MemMap.h"

#define CORTST_START_SEC_CODE
#include "CorTst_MemMap.h"
/*==================================================================================================
 *                                         LOCAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       Reset the common global variables of CorTst.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 */
CORTST_FUNC static void CorTst_ResetGlobalVariables(void)
{
    CorTst_DriverState.TestId = 0U;
    CorTst_DriverState.AtomicTestId = 0U;
    CorTst_DriverState.TestIntervalId = 0U;
    CorTst_DriverState.ErrOk.TestIntervalId = 0U;
    CorTst_DriverState.ErrOk.Result = CORTST_E_NOT_TESTED;
    for (CorTst_TestIdAtomicType AtomicTestId = 0; AtomicTestId < CORTST_ATOMIC_TEST_NUM; ++AtomicTestId)
    {
        CorTst_DriverState.AtomicResults[AtomicTestId] = CORTST_E_NOT_TESTED;
    }
    CorTst_FgndResult = CORTST_E_NOT_TESTED;
    CorTst_DriverState.FgndSignErrCnt = 0U;
    CorTst_DriverState.FgndAccumSign = 0U;
    CorTst_DriverState.BgndResult = CORTST_E_NOT_TESTED;
    CorTst_DriverState.BgndSignErrCnt = 0U;
    CorTst_DriverState.BgndAccumSign = 0U;
    CorTst_DriverState.BgndSign.TestIntervalId = 0U;
    CorTst_DriverState.BgndSign.Signature = 0U;
}

#if (CORTST_M33_FPU_ENABLE == 1U)
/**
 * @brief       Setup FPCA bit in CONTROL register to 1.
 * @return      void
 * @sw_type     sw_detail
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
CORTST_FUNC static void CorTst_SetupFPCA(void)
{
    /** MR12 0759 19.2 VIOLATION: Union type is allowed */
    CONTROL_Type Control; /* PRQA S 0759 */
    /* Enalbe CONTROL.FPCA.*/
    Control.w = __get_CONTROL();
    Control.b.FPCA = 1;
    __set_CONTROL(Control.w);
}
#endif


/**
 * @brief       Change CorTst from CORTST_UNINIT to CORTST_INIT state.
 * @return      void
 * @sw_type     sw_detail
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
CORTST_FUNC static void CorTst_Uninit2Init(void)
{
#if (CORTST_M33_FPU_ENABLE == 1U)
#ifdef CORTST_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call(CorTst_SetupFPCA);
#else
    CorTst_SetupFPCA();
#endif
#endif
    /* SWS_CorTst_00041, SWS_CorTst_00179 */
    CorTst_ResetGlobalVariables();
    CorTst_DriverState.CorTstConfig = &CorTst_Config;
    /* SWS_CorTst_00042 */
    CorTst_DriverState.State = CORTST_INIT;
}

/**
 * @brief       Change CorTst from CORTST_INIT to CORTST_UNINIT state.
 * @return      void
 * @sw_type     sw_detail
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
CORTST_FUNC static void CorTst_Init2Uninit(void)
{
    /* SWS_CorTst_00046 */
    CorTst_ResetGlobalVariables();
    CorTst_DriverState.CorTstConfig = NULL_PTR;
    /* SWS_CorTst_00047 */
    CorTst_DriverState.State = CORTST_UNINIT;
}

/**
 * @brief       Change CorTst from CORTST_INIT to CORTST_RUNNING_BGND state.
 * @return      void
 * @sw_type     sw_detail
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
CORTST_FUNC static void CorTst_Init2RunningBgnd(void)
{
    CorTst_DriverState.TestId = 0U;
    CorTst_DriverState.AtomicTestId = 0U;
    SchM_Enter_CorTst_CORTST_EXCLUSIVE_AREA_02();
    CorTst_DriverState.ErrOk.TestIntervalId = CorTst_DriverState.TestIntervalId;
    CorTst_DriverState.ErrOk.Result = CORTST_E_NOT_TESTED;
    CorTst_DriverState.BgndSign.TestIntervalId = CorTst_DriverState.TestIntervalId;
    CorTst_DriverState.BgndSign.Signature = 0U;
    SchM_Exit_CorTst_CORTST_EXCLUSIVE_AREA_02();
    for (CorTst_TestIdFgndType TestId = 0; TestId < CORTST_ATOMIC_TEST_NUM; ++TestId)
    {
        CorTst_DriverState.AtomicResults[TestId] = CORTST_E_NOT_TESTED;
    }
    CorTst_DriverState.BgndResult = CORTST_E_NOT_TESTED;
    CorTst_DriverState.BgndSignErrCnt = 0U;
    CorTst_DriverState.BgndAccumSign = 0U;
    /* SWS_CorTst_00070 */
    CorTst_DriverState.State = CORTST_RUNNING_BGND;
}

#if ((CORTST_START_API == STD_ON))
/**
 * @brief       Service for executing all the enabled atomic tests in foreground mode.
 * @param[in]   TestId Id of the foreground test configuration to be executed
 * @return      E_OK Executed successfully or E_NOT_OK Executed failed
 * @sw_type     sw_detail
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
CORTST_FUNC static Std_ReturnType CorTst_ExecuteFgndTest(CorTst_TestIdFgndType TestId)
{
    Std_ReturnType ReturnStatus = E_OK;
    CorTst_CsumSignatureType AtomicTestSignature = 0U;
    CorTst_FgndResult = CORTST_E_NOT_TESTED;
    CorTst_DriverState.FgndSignErrCnt = 0U;
    CorTst_DriverState.FgndAccumSign = 0U;
    for (CorTst_TestIdAtomicType AtomicTestId = 0U; AtomicTestId < CORTST_ATOMIC_TEST_NUM; ++AtomicTestId)
    {
        if (TRUE == CorTst_DriverState.CorTstConfig->FgndConfig[TestId].AtomicTestEn[AtomicTestId])
        {
            /* Execute an atomic test here. */
            ReturnStatus |= CorTst_Execute(AtomicTestId, &AtomicTestSignature);
            if (E_OK == ReturnStatus)
            {
                /* Calculate the foreground accumulated signature. */
                CorTst_DriverState.FgndAccumSign ^= AtomicTestSignature;
                if ((CorTst_AtomicTestDesc[AtomicTestId].OrigSign != AtomicTestSignature) &&
                        (CorTst_AtomicTestDesc[AtomicTestId].InjSign != AtomicTestSignature))
                {
                    /* Store atomic test results */
                    CorTst_DriverState.AtomicResults[AtomicTestId] = CORTST_E_NOT_OK;
                    ++CorTst_DriverState.FgndSignErrCnt;
                    CorTst_FgndResult = CORTST_E_NOT_OK;
#if (CORTST_DEM_EVENT_REPORT == STD_ON)
                    /* SWS_CorTst_01000, SWS_CorTst_00154 */
                    (void)Dem_SetEventStatus
                    (
                        CORTST_E_CORE_FAILURE_DEM_EVENT_ID,
                        DEM_EVENT_STATUS_PREFAILED
                    );
#endif
                }
                else
                {
                    /* Store atomic test results */
                    CorTst_DriverState.AtomicResults[AtomicTestId] = CORTST_E_OKAY;
                }
            }
        }
    }
    /* Completed all the atomic test sequences. */
    if (0U == CorTst_DriverState.FgndSignErrCnt)
    {
        CorTst_FgndResult = CORTST_E_OKAY;
#if (CORTST_DEM_EVENT_REPORT == STD_ON)
        /* SWS_CorTst_01000, SWS_CorTst_01001 */
        (void)Dem_SetEventStatus
        (
            CORTST_E_CORE_FAILURE_DEM_EVENT_ID,
            DEM_EVENT_STATUS_PREPASSED
        );
#endif
    }
    return ReturnStatus;
}
#endif /* ((CORTST_START_API == STD_ON) */

/**
 * @brief       Service for executing all the enabled atomic tests one by one in background mode.
 * @return      void
 * @sw_type     sw_detail
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
CORTST_FUNC static void CorTst_ExecuteBgndTest(void)
{
    Std_ReturnType ReturnStatus;
    CorTst_CsumSignatureType AtomicTestSignature = 0U;
    if (TRUE == CorTst_DriverState.CorTstConfig->BgndConfig[CorTst_DriverState.TestId].AtomicTestEn[CorTst_DriverState.AtomicTestId])
    {
        /* SWS_CorTst_00071 */
        /* Execute an atomic test here, test all selected core hardware entities. */
        ReturnStatus = CorTst_Execute(CorTst_DriverState.AtomicTestId, &AtomicTestSignature);
        /* After the atomic test. */
        if (E_OK == ReturnStatus)
        {
            /* Calculate the background accumulated signature. */
            CorTst_DriverState.BgndAccumSign ^= AtomicTestSignature;
            if ((CorTst_AtomicTestDesc[CorTst_DriverState.AtomicTestId].OrigSign != AtomicTestSignature) &&
                    (CorTst_AtomicTestDesc[CorTst_DriverState.AtomicTestId].InjSign != AtomicTestSignature))
            {
                /* Store atomic test results */
                CorTst_DriverState.AtomicResults[CorTst_DriverState.AtomicTestId] = CORTST_E_NOT_OK;
                ++CorTst_DriverState.BgndSignErrCnt;
#if (CORTST_DEM_EVENT_REPORT == STD_ON)
                /* SWS_CorTst_01000, SWS_CorTst_00155 */
                (void)Dem_SetEventStatus
                (
                    CORTST_E_CORE_FAILURE_DEM_EVENT_ID,
                    DEM_EVENT_STATUS_PREFAILED
                );
#endif
                /* SWS_CorTst_00072, SWS_CorTst_00074 */
                CorTst_DriverState.BgndResult = CORTST_E_NOT_OK;
                SchM_Enter_CorTst_CORTST_EXCLUSIVE_AREA_00();
                CorTst_DriverState.ErrOk.TestIntervalId = CorTst_DriverState.TestIntervalId;
                CorTst_DriverState.ErrOk.Result = CorTst_DriverState.BgndResult;
                SchM_Exit_CorTst_CORTST_EXCLUSIVE_AREA_00();
            }
            else
            {
                /* Store atomic test results */
                CorTst_DriverState.AtomicResults[CorTst_DriverState.AtomicTestId] = CORTST_E_OKAY;
            }
        }
    }
    /* Start a new atomic test.  */
    ++CorTst_DriverState.AtomicTestId;
    /* Completed all the tests of the current background configuration. */
    if (CorTst_DriverState.AtomicTestId >= CORTST_ATOMIC_TEST_NUM)
    {
        CorTst_DriverState.AtomicTestId = 0U;
        /* Start a new background configuration. */
        ++CorTst_DriverState.TestId;
        /* Completed all the tests of all the background configurations. */
        if (CorTst_DriverState.TestId >= CORTST_BGND_TEST_NUMBER)
        {
            CorTst_DriverState.TestId = 0U;
            if (0U == CorTst_DriverState.BgndSignErrCnt)
            {
                /* SWS_CorTst_00072, SWS_CorTst_00073 */
                CorTst_DriverState.BgndResult = CORTST_E_OKAY;
#if (CORTST_DEM_EVENT_REPORT == STD_ON)
                /* SWS_CorTst_01000, SWS_CorTst_01002 */
                (void)Dem_SetEventStatus
                (
                    CORTST_E_CORE_FAILURE_DEM_EVENT_ID,
                    DEM_EVENT_STATUS_PREPASSED
                );
#endif
            }
            /* SWS_CorTst_00056, SWS_CorTst_00058 */
            SchM_Enter_CorTst_CORTST_EXCLUSIVE_AREA_01();
            CorTst_DriverState.ErrOk.TestIntervalId = CorTst_DriverState.TestIntervalId;
            CorTst_DriverState.ErrOk.Result = CorTst_DriverState.BgndResult;
            CorTst_DriverState.BgndSign.TestIntervalId = CorTst_DriverState.TestIntervalId;
            CorTst_DriverState.BgndSign.Signature = CorTst_DriverState.BgndAccumSign;
            SchM_Exit_CorTst_CORTST_EXCLUSIVE_AREA_01();
            /* SWS_CorTst_00077, SWS_CorTst_00140 */
#if (CORTST_NOTIFICATION_SUPPORT== STD_ON)
            CorTst_TestCompletedNotification(CorTst_DriverState.ErrOk);
#endif
            /* Start a new test interval. */
            /* SWS_CorTst_00139 */
            ++CorTst_DriverState.TestIntervalId;
            if (CorTst_DriverState.TestIntervalId >= CORTST_TEST_INTERVAL_ID_END_VALUE)
            {
                CorTst_DriverState.TestIntervalId = 0U;
            }
            /* SWS_CorTst_00068, SWS_CorTst_00069 */
            CorTst_DriverState.State = CORTST_INIT;
        }
    }
}

/*==================================================================================================
 *                                         GLOBAL FUNCTIONS
==================================================================================================*/
CORTST_FUNC void CorTst_Init(const CorTst_ConfigType *ConfigPtr)
{
    /* SWS_CorTst_01004 */
    (void)ConfigPtr;
    switch (CorTst_DriverState.State)
    {
        case CORTST_ABORT:
        case CORTST_INIT:
        case CORTST_RUNNING_BGND:
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
            /* SWS_CorTst_00178 */
            (void)Det_ReportError
            (
                CORTST_MODULE_ID,
                CORTST_INDEX,
                CORTST_SID_INIT,
                CORTST_E_ALREADY_INITIALIZED
            );
#endif
            break;
        case CORTST_UNINIT:
            CorTst_Uninit2Init();
            break;
        default:
            /* Do nothing */
            break;
    }
}

CORTST_FUNC void CorTst_DeInit(void)
{
    switch (CorTst_DriverState.State)
    {
        case CORTST_RUNNING_BGND:
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
            /* SWS_CorTst_00149 */
            (void)Det_ReportError
            (
                CORTST_MODULE_ID,
                CORTST_INDEX,
                CORTST_SID_DEINIT,
                CORTST_E_STATUS_FAILURE
            );
#endif
            break;
        case CORTST_ABORT:
        /* SWS_CorTst_00136 */
        case CORTST_INIT:
            CorTst_Init2Uninit();
            break;
        case CORTST_UNINIT:
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
            /* SWS_CorTst_00044 */
            (void)Det_ReportError
            (
                CORTST_MODULE_ID,
                CORTST_INDEX,
                CORTST_SID_DEINIT,
                CORTST_E_UNINIT
            );
#endif
            break;
        default:
            /* Do nothing */
            break;
    }
}

#if ((CORTST_ABORT_API == STD_ON))
CORTST_FUNC void CorTst_Abort(void)
{
    switch (CorTst_DriverState.State)
    {
        /* SWS_CorTst_00105 */
        case CORTST_RUNNING_BGND:
        /* SWS_CorTst_00049 */
        case CORTST_INIT:
            CorTst_DriverState.State = CORTST_ABORT;
            break;
        case CORTST_UNINIT:
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
            /* SWS_CorTst_00044 */
            (void)Det_ReportError
            (
                CORTST_MODULE_ID,
                CORTST_INDEX,
                CORTST_SID_ABORT,
                CORTST_E_UNINIT
            );
#endif
            break;
        /* SWS_CorTst_00052 */
        default:
            /* Do nothing */
            break;
    }
}
#endif

#if (CORTST_GET_STATE_API == STD_ON)
CORTST_FUNC CorTst_StateType CorTst_GetState(void)
{
    /* SWS_CorTst_00054 */
    return CorTst_DriverState.State;
}
#endif

#if (CORTST_GET_CURRENT_STATUS == STD_ON)
CORTST_FUNC void CorTst_GetCurrentStatus(CorTst_ErrOkType *ErrOk)
{
    if (ErrOk == NULL_PTR)
    {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
        /* SWS_CorTst_00180 */
        (void)Det_ReportError
        (
            CORTST_MODULE_ID,
            CORTST_INDEX,
            CORTST_SID_GET_CURRENT_STATUS,
            CORTST_E_PARAM_POINTER
        );
#endif
    }
    else
    {
        if (CORTST_UNINIT == CorTst_DriverState.State)
        {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
            /* SWS_CorTst_00044 */
            (void)Det_ReportError
            (
                CORTST_MODULE_ID,
                CORTST_INDEX,
                CORTST_SID_GET_CURRENT_STATUS,
                CORTST_E_UNINIT
            );
#endif
        }
        else
        {
            /* SWS_CorTst_00056 */
            *ErrOk = CorTst_DriverState.ErrOk;
        }
    }
}
#endif

#if (CORTST_GET_FGND_RESULT == STD_ON)
CORTST_FUNC CorTst_ResultType CorTst_GetFgndResult(void)
{
    CorTst_ResultType RetResult = CORTST_E_NOT_TESTED;
    if (CORTST_UNINIT == CorTst_DriverState.State)
    {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
        /* SWS_CorTst_00044 */
        (void)Det_ReportError
        (
            CORTST_MODULE_ID,
            CORTST_INDEX,
            CORTST_SID_GET_FGND_RESULT,
            CORTST_E_UNINIT
        );
#endif
    }
    else
    {
        /* SWS_CorTst_00061 */
        RetResult = CorTst_FgndResult;
    }
    return RetResult;
}
#endif

#if (CORTST_GET_SIGNATURE == STD_ON)
CORTST_FUNC CorTst_CsumSignatureBgndType CorTst_GetSignature(void)
{
    /* SWS_CorTst_00121 */
    CorTst_CsumSignatureBgndType RetSign =
    {
        .TestIntervalId = 0U,
        .Signature = 0U
    };
    if (CORTST_UNINIT == CorTst_DriverState.State)
    {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
        /* SWS_CorTst_00044 */
        (void)Det_ReportError
        (
            CORTST_MODULE_ID,
            CORTST_INDEX,
            CORTST_SID_GET_SIGNATURE,
            CORTST_E_UNINIT
        );
#endif
    }
    else
    {
        /* SWS_CorTst_00058 */
        RetSign = CorTst_DriverState.BgndSign;
    }
    return RetSign;
}
#endif

#if (CORTST_GET_FGND_SIGNATURE == STD_ON)
CORTST_FUNC CorTst_CsumSignatureType CorTst_GetFgndSignature(void)
{
    /* SWS_CorTst_00122 */
    CorTst_CsumSignatureType RetSign = 0U;
    if (CORTST_UNINIT == CorTst_DriverState.State)
    {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
        /* SWS_CorTst_00044 */
        (void)Det_ReportError
        (
            CORTST_MODULE_ID,
            CORTST_INDEX,
            CORTST_SID_GET_FGND_SIGNATURE,
            CORTST_E_UNINIT
        );
#endif
    }
    else
    {
        /* SWS_CorTst_00061 */
        RetSign = CorTst_DriverState.FgndAccumSign;
    }
    return RetSign;
}
#endif


#if ((CORTST_START_API == STD_ON))
CORTST_FUNC Std_ReturnType CorTst_Start(CorTst_TestIdFgndType TestId)
{
    Std_ReturnType ReturnStatus = E_NOT_OK;
    if (TestId >= CORTST_FGND_TEST_NUMBER)
    {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
        /* SWS_CorTst_00161 */
        (void)Det_ReportError
        (
            CORTST_MODULE_ID,
            CORTST_INDEX,
            CORTST_SID_START,
            CORTST_E_PARAM_INVALID
        );
#endif
    }
    else
    {
        switch (CorTst_DriverState.State)
        {
            case CORTST_RUNNING_BGND:
                /* SWS_CorTst_00065, SWS_CorTst_00109 */
                ReturnStatus = E_OK;
                break;
            case CORTST_ABORT:
            case CORTST_INIT:
                ReturnStatus = CorTst_ExecuteFgndTest(TestId);
                break;
            case CORTST_UNINIT:
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
                /* SWS_CorTst_00044 */
                (void)Det_ReportError
                (
                    CORTST_MODULE_ID,
                    CORTST_INDEX,
                    CORTST_SID_START,
                    CORTST_E_UNINIT
                );
#endif
                break;
            default:
                /* Do nothing */
                break;
        }
    }
    return ReturnStatus;
}
#endif

#if (CORTST_VERSION_INFO_API == STD_ON)
CORTST_FUNC void CorTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    if (VersionInfo == NULL_PTR)
    {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
        /* SWS_CorTst_00180, SWS_CorTst_00118 */
        (void)Det_ReportError
        (
            CORTST_MODULE_ID,
            CORTST_INDEX,
            CORTST_SID_GET_VERSION_INFO,
            CORTST_E_PARAM_POINTER
        );
#endif
    }
    else
    {
        VersionInfo->vendorID = (uint16)CORTST_VENDOR_ID;
        VersionInfo->moduleID = (uint16)CORTST_MODULE_ID;
        VersionInfo->sw_major_version = (uint8)CORTST_SW_MAJOR_VERSION;
        VersionInfo->sw_minor_version = (uint8)CORTST_SW_MINOR_VERSION;
        VersionInfo->sw_patch_version = (uint8)CORTST_SW_PATCH_VERSION;
    }
}
#endif

#if (CORTST_GET_DETAIL_RESULT_API == STD_ON)
CORTST_FUNC void CorTst_GetDetailsResult(CorTst_ResultType *CorTstDetailsPtr)
{
    if (CorTstDetailsPtr == NULL_PTR)
    {
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
        /* SWS_CorTst_00180, SWS_CorTst_00118 */
        (void)Det_ReportError
        (
            CORTST_MODULE_ID,
            CORTST_INDEX,
            CORTST_SID_GET_VERSION_INFO,
            CORTST_E_PARAM_POINTER
        );
#endif
    }
    else
    {
        for (CorTst_TestIdAtomicType AtomicTestId = 0; AtomicTestId < CORTST_ATOMIC_TEST_NUM; ++AtomicTestId)
        {
            CorTstDetailsPtr[AtomicTestId] = CorTst_DriverState.AtomicResults[AtomicTestId];
        }
    }
}
#endif /* (CORTST_GET_DETAIL_RESULT_API == STD_ON) */

CORTST_FUNC void CorTst_MainFunction(void)
{
    switch (CorTst_DriverState.State)
    {
        case CORTST_RUNNING_BGND:
            CorTst_ExecuteBgndTest();
            break;
        case CORTST_INIT:
            CorTst_Init2RunningBgnd();
            break;
        case CORTST_UNINIT:
#if (CORTST_DEV_ERROR_DETECT == STD_ON)
            /* SWS_CorTst_00044 */
            (void)Det_ReportError
            (
                CORTST_MODULE_ID,
                CORTST_INDEX,
                CORTST_SID_MAIN_FUNCTION,
                CORTST_E_UNINIT
            );
#endif
            break;
        /* SWS_CorTst_00051 */
        default:
            /* Do nothing */
            break;
    }
}

#define CORTST_STOP_SEC_CODE
#include "CorTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

