/**
 * @file    CorTst.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR CorTst module interface
 * @details API implementation for CorTst driver
 *
 * @addtogroup CORTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Core
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifndef CORTST_H
#define CORTST_H

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "CorTst_Types.h"
/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CORTST_VENDOR_ID                      (180)
#define CORTST_MODULE_ID                      (103)
#define CORTST_AR_RELEASE_MAJOR_VERSION       (4)
#define CORTST_AR_RELEASE_MINOR_VERSION       (4)
#define CORTST_AR_RELEASE_REVISION_VERSION    (0)
#define CORTST_SW_MAJOR_VERSION               (2)
#define CORTST_SW_MINOR_VERSION               (0)
#define CORTST_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* CorTst_Types.h version check */
#if (CORTST_VENDOR_ID != CORTST_TYPES_VENDOR_ID)
#error "CorTst.h and CorTst_Types.h have different vendor ids"
#endif
#if ((CORTST_AR_RELEASE_MAJOR_VERSION != CORTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (CORTST_AR_RELEASE_MINOR_VERSION != CORTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (CORTST_AR_RELEASE_REVISION_VERSION != CORTST_TYPES_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CorTst.h and CorTst_Types.h are different"
#endif
#if ((CORTST_SW_MAJOR_VERSION != CORTST_TYPES_SW_MAJOR_VERSION) || \
     (CORTST_SW_MINOR_VERSION != CORTST_TYPES_SW_MINOR_VERSION) || \
     (CORTST_SW_PATCH_VERSION != CORTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of CorTst.h and CorTst_Types.h are different"
#endif

/*==================================================================================================
 *                                  GLOBAL CONSTANT DECLARATIONS
==================================================================================================*/

/*==================================================================================================
 *                                       FUNCTION PROTOTYPES
==================================================================================================*/
/**
 * @brief       Service for initialization and change of state of the Core Test.
 * @param[in]   ConfigPtr Pointer to the selected configuration set.
 * @return      void
 * @table       @service_id:       0x00        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
void CorTst_Init(const CorTst_ConfigType *ConfigPtr);

/**
 * @brief       Service to change from CORTST_ABORT or CORTST_INIT to CORTST_UNINIT state.
 * @return      void
 * @table       @service_id:       0x01        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
void CorTst_DeInit(void);

#if (CORTST_ABORT_API == STD_ON)
/**
 * @brief       Service to change from CORTST_INIT to CORTST_ABORT state.
 * @return      void
 * @table       @service_id:       0x02        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
void CorTst_Abort(void);
#endif

#if (CORTST_GET_STATE_API == STD_ON)
/**
 * @brief       Service for Core Test to immediately return status on currently executed Core Test.
 * @return      CorTst driver state machine value.
 * @table       @service_id:       0x03        \n
 *              @is_reentrant:     true        \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
CorTst_StateType CorTst_GetState(void);
#endif

#if (CORTST_GET_CURRENT_STATUS == STD_ON)
/**
 * @brief       Service for Core Test to get indicator of the last executed Core Test result in
 *              background mode.
 * @param[out]  ErrOk Pointer to the current status.
 * @return      void
 * @table       @service_id:       0x04        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
void CorTst_GetCurrentStatus(CorTst_ErrOkType *ErrOk);
#endif

#if (CORTST_GET_FGND_SIGNATURE == STD_ON)
/**
 * @brief       Service to get signature of the last executed Core Test in foreground mode.
 * @return      The last executed Core Test in foreground mode.
 * @table       @service_id:       0x05        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
CorTst_CsumSignatureType CorTst_GetFgndSignature(void);
#endif

#if (CORTST_GET_SIGNATURE == STD_ON)
/**
 * @brief       Service to get signature of the last executed Core Test in background mode.
 * @return      The signature and test interval ID of last executed CorTst atomic test.
 * @table       @service_id:       0x06        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
CorTst_CsumSignatureBgndType CorTst_GetSignature(void);
#endif

#if ((CORTST_START_API == STD_ON))
/**
 * @brief       Service for executing foreground Core Test.
 * @param[in]   TestId Id of the foreground test configuration to be executed.
 * @return      Standard return type.
 * @table       @service_id:       0x07        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
Std_ReturnType CorTst_Start(CorTst_TestIdFgndType TestId);
#endif

#if (CORTST_VERSION_INFO_API == STD_ON)
/**
 * @brief       Service returns the version information of this module.
 * @param[out]  VersionInfo Pointer to where to store the version information of this module.
 * @return      void
 * @table       @service_id:       0x08        \n
 *              @is_reentrant:     true        \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
void CorTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif

/**
 * @brief       Cyclically called by scheduler to perform processing of Core Test.
 * @return      void
 * @table       @service_id:       0x0B        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
void CorTst_MainFunction(void);

#if (CORTST_NOTIFICATION_SUPPORT== STD_ON)
/**
 * @brief       The function CorTst_TestCompletedNotification shall be called every time when
 *              a complete test cycle has been executed.
 * @param[in]   ResultOfLastCorTstRun Result of the last Core Test run.
 * @return      void
 * @table       @service_id:       0x0C        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      true        \n
 * @sw_type     sw_arch
 */
void CorTst_TestCompletedNotification(CorTst_ErrOkType ResultOfLastCorTstRun);
#endif

#if (CORTST_GET_FGND_RESULT == STD_ON)
/**
 * @brief       Service to get result of the last executed Core Test in foreground mode.
 * @return      Result of the foreground test.
 * @table       @service_id:       0x0D        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @sw_type     sw_arch
 */
CorTst_ResultType CorTst_GetFgndResult(void);
#endif

#if (CORTST_GET_DETAIL_RESULT_API == STD_ON)
/**
 * @brief       Service for getting Core Test result details.
 *
 * @param[out]  CorTstDetailsPtr Pointer to where to store the test details.
 * @return      void
 * @sw_type     sw_arch
 * @table       @service_id:       0x0E        \n
 *              @is_reentrant:     true        \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 */
void CorTst_GetDetailsResult(CorTst_ResultType *CorTstDetailsPtr);
#endif /* (CORTST_GET_DETAIL_RESULT_API == STD_ON) */


#ifdef __cplusplus
}
#endif

#endif /* CORTST_H */

/** @} */

