/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
* Checks the floating-point minimum/maximum selection functionality provided by
* the cm33fpu_sp_add.v module.
*
* The following functionality is tested:
*   Case1: Handling of NaN operands
*   Case2: Handling of Zero operands
*   Case3: Handling of Infinite operands
*   Case4: Handling of operand signs
*   Case5: Check exponent comparators
*   Case6: Check fraction comparators
*   Case7: Check result selection logic
*
* SP FPU Decoder coverage:
* ------------------------
*  - VMAXNM.F32,VMINNM.F32
*  - VNEG.F32
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FPU_ENABLE==1)

    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_SpfpuMinMaxTest

    /* Symbols defined outside but used within current module */
    CST_EXTERN m33_cst_test_tail_fpu_end
    CST_EXTERN m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    /* Pre-signature */
    CST_SET(PRESIGNATURE_SEED_0, 0xEE8A)
    CST_SET(PRESIGNATURE_SEED_1, 0x97C6)
    
    /* Values stored in registers:
       - R1  ... accumulated signature
       - R12 ... FPSCR
    */
    
    /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
    /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_SpfpuMinMaxTest" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_TYPE(M33_Cst_SpfpuMinMaxTest, function)
    CST_THUMB2
M33_Cst_SpfpuMinMaxTest:

    PUSH    {R4-R12,R14}
    MRS     R1,CONTROL  /* Store CONTROL prior first FPU instruction */
    PUSH    {R1}
    VPUSH   {S16-S31}

    /*-----------------------------------------------------------------------*/
    /* Test - preparation                                                    */
    /*-----------------------------------------------------------------------*/
    MOV     R1,#PRESIGNATURE_SEED_0
    MOVT    R1,#PRESIGNATURE_SEED_1
        
    /*-----------------------------------------------------------------------*/
    /* Test - start                                                          */
    /*-----------------------------------------------------------------------*/
    VMRS    R12,FPSCR       /* Store FPSCR register */

    /* Prepare test vectors */
    MOV     R0,#0
    MOV     R3,#(255<<23)       /* +Inf  -> S0 */
    ORR     R5,R3,#(1<<22)      /* +qNaN -> S2 */
    ORR     R7,R3,#(1<<21)      /* +sNaN -> S4 */
    MOV     R10,#(1<<31)        /* -0    -> S7 */
    
    /* Write floating-point registers and test VNEG instruction
       for both positive-to-negative and negative-to-positive */
    VMOV    S0,R3               /*             +Inf  -> S0  */
    VMOV    S2,R5               /*             +qNaN -> S2  */
    VMOV    S4,R7               /*             +sNaN -> S4  */
    VMOV    S7,R10              /*             -0    -> S7  */
    VNEG.F32    S1,S0           /* VNEG(+Inf)  -Inf  -> S1  */
    VNEG.F32    S3,S2           /* VNEG(+qNaN) -qNaN -> S3  */
    VNEG.F32    S5,S4           /* VNEG(+sNaN) -sNaN -> S5  */
    VNEG.F32    S6,S7           /* VNEG(-0)    +0    -> S6  */
    VMOV.F32    S9,#-1.1875     /*             -NumA -> S9  */
    VNEG.F32    S8,S9           /* VNEG(-Num)  +NumA -> S8  */
    VMOV.F32    S11,#-3.5       /*             -NumB -> S11 */
    VNEG.F32    S10,S11         /* VNEG(-NumB) +NumB -> S10 */


    /**************************************************************************/
    /* Case1: Check handling of NaN operands                                  */
    /**************************************************************************/
    /* Case1a: quite NaN 
       Case1b: signaling NaN (sets FPSCR.IOC flag)
     */
    VMSR    FPSCR,R0    /* Clear FPSCR */
    
    /* Case1a: quite NaN */
    VMAXNM.F32  S29,S2,S2       /* MAX(qNaN,qNaN)   = +qNaN */
    VMINNM.F32  S28,S3,S3       /* MIN(-qNaN,-qNaN) = -qNaN */
    /* Check FPSCR.IOC is not set */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Case1b: signaling NaN */
    VMAXNM.F32  S31,S5,S5       /* MAX(-sNaN,-sNaN) = -qNaN */
    VMINNM.F32  S30,S4,S4       /* MIN(sNaN,sNaN)   = +qNaN */
    VMAXNM.F32  S27,S5,S8       /* MAX(-sNaN,+Num)  = -qNaN */
    VMINNM.F32  S26,S9,S4       /* MIN(-Num,+sNaN)  = +qNaN */
    VMAXNM.F32  S25,S8,S5       /* MAX(+Num,-sNaN)  = -qNaN */
    VMINNM.F32  S24,S4,S9       /* MIN(+sNaN,-Num,) = +qNaN */
    /* Check FPSCR.IOC is set */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* !! Udate Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    
    /**************************************************************************/
    /* Case2: Check handling of Zero operands                                 */
    /**************************************************************************/
    VMAXNM.F32  S31,S6,S7       /* MAX(+0,-0)   = +0 */
    VMINNM.F32  S30,S7,S6       /* MIN(-0,+0)   = -0 */
    VMAXNM.F32  S29,S6,S7       /* MAX(-0,+0)   = +0 */
    VMINNM.F32  S28,S7,S6       /* MIN(+0,-0)   = -0 */
    VMAXNM.F32  S27,S7,S9       /* MAX(-0,-Num) = -0 */
    VMINNM.F32  S26,S8,S6       /* MIN(+Num,+0) = +0 */
    VMAXNM.F32  S25,S9,S6       /* MAX(-Num,+0) = +0 */
    VMINNM.F32  S24,S7,S8       /* MAX(-0,+Num) = -0 */
    
    /* !! Udate Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
     
     
    /**************************************************************************/
    /* Case3: Check handling of Infinite operands                             */
    /**************************************************************************/
    VMAXNM.F32  S31,S0,S1       /* MAX(+Inf,-Inf) = +Inf */
    VMINNM.F32  S30,S1,S0       /* MIN(-Inf,+Inf) = -Inf */
    VMAXNM.F32  S29,S1,S0       /* MAX(-Inf,+Inf) = +Inf */
    VMINNM.F32  S28,S0,S1       /* MIN(+Inf,-Inf) = -Inf */
    VMAXNM.F32  S27,S0,S8       /* MAX(+Inf,+Num) = +Inf */
    VMINNM.F32  S26,S9,S1       /* MIN(-Num,-Inf) = -Inf */
    VMAXNM.F32  S25,S8,S0       /* MAX(+Num,+Inf) = +Inf */
    VMINNM.F32  S24,S1,S9       /* MIN(-Inf,-Num) = -Inf */
    
    /* !! Udate Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    
    /**************************************************************************/
    /* Case4: Handling of operands signs                                      */
    /**************************************************************************/
    /*  Test if expected operand is selected depending on signs comparison.
    
       Let's assume:
       ------------
       1) NumA!={NaN,Inf,0}, NumB!={NaN,Inf,0}
       2) Compared operands have:
          a)the same exponents and fractions (Case4a)
          b)different exps and fracs, where (expA<expB)&&(fracA<fracB) (Case4b)
       3) Compared operands have different signs
     */
    /* Case4a: the same exps and fracs  */ 
    VMAXNM.F32  S31,S8,S9       /* MAX(+NumA,-NumA) = +NumA */
    VMINNM.F32  S30,S9,S8       /* MIN(-NumA,+NumA) = -NumA */
    VMAXNM.F32  S29,S11,S10     /* MAX(-NumB,+NumB) = +NumB */
    VMINNM.F32  S28,S10,S11     /* MIN(+NumB,-NumB) = -NumB */
    
    /* Case4b: different exps and fracs */
    VMAXNM.F32  S27,S8,S11      /* MAX(+NumA,-NumB) = +NumA */
    VMINNM.F32  S26,S11,S8      /* MIN(-NumB,+NumA) = -NumB */
    VMAXNM.F32  S25,S11,S8      /* MAX(-NumB,+NumA) = +NumA */
    VMINNM.F32  S24,S8,S11      /* MIN(+NumA,-NumB) = -NumB */
    
    /* !! Udate Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    
    /**************************************************************************/
    /* Case5: Check Operand Selection logic - compare exponents               */
    /**************************************************************************/  
    /* Test if expected operand is selected depending on exponents comparison.
        
       Let's assume:
       ------------
       1) NumA!={NaN,Inf,0}, NumB!={NaN,Inf,0}
       2) Compared operands have the same signs, Case5a(+), Case5b(-)
       
       Proposed test vectors:
       ---------------------
       expA: 0x80 0x40 0x20 ... 0x01
       expB: 0x40 0x20 0x10 ... 0x00
            
       Note, that exponents are being primarily compared in the 
       M33_Cst_SpfpuCmpTest test.
    *************************************************************************/
    MOV     R7,#(1<<29)
 m33_cst_spfpu_minmax_test_test_loop1:
    
    /* Prepare next expA and expB */
    ORR     R3,R0,R7            /* expB */
    ADD     R4,R3,R7            /* expA */
    
    /* Write floating-point registers*/
    VMOV    S12,R4              /* (+) expA -> S12 */
    VMOV    S13,R3              /* (+) expB -> S13 */
    VNEG.F32    S14,S12         /* (-) expA -> S14 */
    VNEG.F32    S15,S13         /* (-) expB -> S15 */
    
    /* Case5a: (positive numbers) */
    VMAXNM.F32  S31,S12,S13     /* MAX selects NumA; (expA > expB) */     
    VMAXNM.F32  S30,S13,S12     /* MAX selects NumA; (expA > expB) */
    VMINNM.F32  S29,S12,S13     /* MIN selects NumB; (expB < expA) */     
    VMINNM.F32  S28,S13,S12     /* MIN selects NumB; (expB < expA) */
    
    /* Case5b: (negative numbers) */
    VMAXNM.F32  S27,S14,S15     /* MAX selects NumB; (expA > expB) */
    VMAXNM.F32  S26,S15,S14     /* MAX selects NumB; (expA > expB) */
    VMINNM.F32  S25,S14,S15     /* MIN selects NumA; (expB < expA) */
    VMINNM.F32  S24,S15,S14     /* MIN selects NumA; (expB < expA) */
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    LSR     R7,R7,#1
    CMP     R7,#(1<<21)
    BNE     m33_cst_spfpu_minmax_test_test_loop1    /* Exponent loop 8x */
    
    
    /**************************************************************************/
    /* Case6: Check Operand Selection logic - compare fractions               */
    /**************************************************************************/
    /* Test if expected operand is selected depending on fractions comparison.
       
       Let's assume:
       ------------
       1) NumA!={NaN,Inf,0}, NumB!={NaN,Inf,0}
       2) Compared operands have the same signs, Case6a(+), Case6b(-)
       3) Compared operands have the same exponents, Case6a(Exp1=0b10101010), 
          Case6b(Exp2=0b01010101)
       
       Proposed test vectors:
       ---------------------
          fracA: 0x400000 0x200000 ... 0x000001
          fracB: 0x200000 0x100000 ... 0x000000
        
       Note, that fractions are being primarily compared in the 
       M33_Cst_SpfpuCmpTest test.
     **************************************************************************/
    MOV     R7,#(1<<22)
m33_cst_spfpu_minmax_test_test_loop2:
    
    /* Prepare next fracA and fracB */
    LSR     R8,R7,#1
    ORR     R3,R7,#(0xAA<<23)   /* Exp1 + fracA */   
    ADD     R4,R8,#(0xAA<<23)   /* Exp1 + fracB */
    ORR     R5,R7,#(0x55<<23)   /* Exp2 + fracA */   
    ADD     R6,R8,#(0x55<<23)   /* Exp2 + fracB */
    
    /* Write floating-point registers*/
    VMOV    S16,R3              /* (+) Exp1 + fracA -> S16 */
    VMOV    S17,R4              /* (+) Exp1 + fracB -> S17 */
    VMOV    D9,R5,R6   
    VNEG.F32    S18,S18         /* (-) Exp2 + fracA -> S18 */    
    VNEG.F32    S19,S19         /* (-) Exp2 + fracB -> S19 */    
    
    /* Case6a: (positive numbers) */
    VMAXNM.F32  S31,S16,S17     /* MAX selects NumA; (fracA > fracB) */
    VMAXNM.F32  S30,S17,S16     /* MAX selects NumA; (fracA > fracB) */
    VMINNM.F32  S29,S16,S17     /* MIN selects NumB; (fracB < fracA) */
    VMINNM.F32  S28,S17,S16     /* MIN selects NumB; (fracB < fracA) */
    
    /* Case6b: (negative numbers) */
    VMAXNM.F32  S27,S18,S19     /* MAX selects NumB; (fracB < fracA) */
    VMAXNM.F32  S26,S19,S18     /* MAX selects NumB; (fracB < fracA) */
    VMINNM.F32  S25,S18,S19     /* MIN selects NumA; (fracA > fracB) */
    VMINNM.F32  S24,S19,S18     /* MIN selects NumA; (fracA > fracB) */
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
  
    LSRS    R7,R7,#1            
    BNE     m33_cst_spfpu_minmax_test_test_loop2    /* Fraction loop 23x */
     
     
    /**************************************************************************/
    /* Case7: Check denormalized operands selection                           */
    /**************************************************************************/
    /* Let's assume:
       ------------
       1) NumA!={NaN,Inf,0}, NumB!={NaN,Inf,0}
       2) Compared operands have the same signs, Case7a(+), Case7b(-)
       3) Compared operands are denormalized
       
        Proposed test vectors:
       ---------------------
        fracA= 0x400000
        fracB= 0x3FFFFF 
       
       Test both Flush-to-zero Disabled/Enabled modes.
     **************************************************************************/
    MOV     R3,#(1<<22)     
    SUB     R4,R3,#1        
        
    /* Write floating-point registers*/
    VMOV    S20,R3              /* +NumA -> S20 */
    VMOV    S21,R4              /* +NumB -> S21 */
    VNEG.F32    S22,S18         /* -NumA -> S22 */
    VNEG.F32    S23,S19         /* -NumB -> S23 */
    
    /* Clear FPSCR & Disable Flush-to-zero */
    VMSR    FPSCR,R0        
    /* Case7a: (positive numbers) */
    VMAXNM.F32  S31,S20,S21     /* MAX selects NumA; (fracA > fracB) */
    VMINNM.F32  S30,S21,S20     /* MIN selects NumB; (fracB < fracA) */
    /* Case7b: (negative numbers) */
    VMAXNM.F32  S29,S22,S23     /* MAX selects NumB; (fracB < fracA) */
    VMINNM.F32  S28,S23,S22     /* MIN selects NumA; (fracA > fracB) */
    /* Check FPSCR.IDC is not set */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Clear FPSCR & Enable Flush-to-zero */
    EOR     R0,R0,#(1<<24)     /* Enable Flush-to-zero */
    VMSR    FPSCR,R0
    /* Case7a: (positive numbers) */
    VMAXNM.F32  S27,S21,S20     /* MAX selects +0 */
    VMINNM.F32  S26,S20,S21     /* MIN selects +0 */
    /* Case7b: (negative numbers) */
    VMAXNM.F32  S25,S23,S22     /* MAX selects -0 */
    VMINNM.F32  S24,S22,S23     /* MIN selects -0 */
    /* Check FPSCR.IDC is set */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
m33_cst_spfpu_minmax_test_end:
    VMSR    FPSCR,R12   /* Restore FPSCR register */
    /* Test result is returned in R0, according to the conventions */
    MOV     R0,R1 
    B       m33_cst_test_tail_fpu_end
    
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_FPU_ENABLE */

    CST_FILE_END

