/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
* Checks conversions between floating-point and integer representations of
* numbers provided by cm33fpu_sp_cvt.v and cm33fpu_sp_round.v.v modules.
*
* The following conversions are tested:
* -------------------------------------
*  Case1: Floating-point to Integer
*       Case1a: Exact conversions 
*       Case1b: Inexact conversions
*       Case1c: Inexact conversions (with direct rounding) 
*  Case2: Integer to Floating-point
*       Case3a: Exact conversions
*       Case3b: Inexact conversions
*  Case3: Half-precision to/from single-precision
*
*  Case4: In addition, the following functionality is tested:
*       - handling special values (+-sNaN, +-qNaN, +-Inf, +-0)
*       - handling signed values
*       - producing the FPSCR[IOC, IDC, IXC, OFC, UFC] flags
*       - flush-to-zero functionality
*
* SP FPU Decoder coverage:
* ------------------------
*   - VCVT{R}.U32.F32
*   - VCVT{R}.S32.F32
*   - VCVT{A,N,P,M}.U32.F32 - with direct rounding
*   - VCVT{A,N,P,M}.S32.F32 - with direct rounding
*   - VCVT.F32.S32
*   - VCVT.F32.U32
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FPU_ENABLE==1)

    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_SpfpuConvertTest1
    
    /* Symbols defined outside but used within current module */
    CST_EXTERN m33_cst_test_tail_fpu_end
    CST_EXTERN m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    CST_EXTERN m33_cst_sum_fpu_s23_s31_to_r1_destr_r2
    CST_EXTERN m33_cst_sum_fpu_s16_s31_to_r1_destr_r2
    CST_EXTERN m33_cst_sum_fpu_s14_s31_to_r1_destr_r2
    
    /* Pre-signature */
    CST_SET(PRESIGNATURE_SEED_0, 0x7049)
    CST_SET(PRESIGNATURE_SEED_1, 0x3184)
    
    /* Values stored in registers:
       - R1  ... accumulated signature
       - R12 ... FPSCR register
    */

   /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
   /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_SpfpuConvertTest1" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_TYPE(M33_Cst_SpfpuConvertTest1, function)
    CST_THUMB2
M33_Cst_SpfpuConvertTest1:

    PUSH    {R4-R12,R14}
    MRS     R1,CONTROL  /* Store CONTROL prior first FPU instruction */
    PUSH    {R1}
    VPUSH   {S16-S31}

    /*-----------------------------------------------------------------------*/
    /* Test - preparation                                                    */
    /*-----------------------------------------------------------------------*/
    MOV     R12,#PRESIGNATURE_SEED_0
    MOVT    R12,#PRESIGNATURE_SEED_1
    MOV     R1,R12
    
    /*-----------------------------------------------------------------------*/
    /* Test - start                                                          */
    /*-----------------------------------------------------------------------*/
    VMRS        R12,FPSCR   /* Store FPSCR register */
    
    
    /*************************************************************************/
    /* Case1: Floating-point to Integer */
    /*************************************************************************/
    /* Let assume Floating-point numbers with exponent exp between 0 and 23 
       which have the following precision limitations when converted to 
       integer:
       
       - Float is converted to integer as exact/inexact depending on number
         of fraction bits set to 1 and exponent number.
       
     
     Case 1a) Floating-point to Integer  - Exact conversion
        
      NumA |exp |  mantissa1      |           NumB |exp |  mantissa2     |
     ------|----|-----------------|----      ------|----|----------------|-------
     Run1 :| 23 |  0b0000...0001/ | exact    Run1: | 23 | 0b1111...1111/ | exact
     Run2 :| 22 |  0b0000...001/0 | exact    Run2: | 22 | 0b1111...111/0 | exact
     Run3 :| 21 |  0b0000...01/00 | exact    Run3: | 21 | 0b1111...11/00 | exact
     Run4 :| 20 |  0b0000...1/000 | exact    Run4: | 20 | 0b1111...1/000 | exact
                                                                   
     Run20:|  4 |  0x0001/...0000 | exact    Run20:|  4 | 0b1111/...0000 | exact
     Run21:|  3 |  0b001/0...0000 | exact    Run21:|  3 | 0b111/0...0000 | exact
     Run22:|  2 |  0b01/00...0000 | exact    Run22:|  2 | 0b11/00...0000 | exact
     Run23:|  1 |  0b1/000...0000 | exact    Run23:|  1 | 0b1/000...0000 | exact
       
       
     Case 1b) Floating-point to Integer - Inexact conversion
     
      NumC |exp | mantissa1       |            NumD |exp | mantissa3      | 
     ------|----|-----------------|-----      ------|----|----------------|-------
     Run1 :| 22 |  0b0000...000/1 | inexact   Run1: | 22 | 0b1111...111/1 | inexact
     Run2 :| 21 |  0b0000...00/10 | inexact   Run1: | 21 | 0b1111...11/01 | inexact
     Run3 :| 20 |  0b0000...0/100 | inexact   Run1: | 20 | 0b1111...1/011 | inexact
     Run4 :| 19 |  0b0000.../1000 | inexact   Run1: | 19 | 0b1111.../0111 | inexact
                                                                           
     Run20:|  3 |  0x000/1...0000 | inexact   Run20:|  3 | 0x111/0...1111 | inexact
     Run21:|  2 |  0b00/10...0000 | inexact   Run21:|  2 | 0x11/01...1111 | inexact
     Run22:|  1 |  0b0/100...0000 | inexact   Run22:|  1 | 0b1/011...1111 | inexact
     Run23:|  0 |  0b/1000...0000 | inexact   Run23:|  0 | 0b/0111...1111 | inexact      
     */  
    
    MOV     R0,#0   /* Set Round to Nearest and clear FPSCR */
    
    MOV     R10,#((23+127)<<23)     /* 23(exp)+127(bias) */
    MOV     R11,#(1)                /* fraction bit */
    MOV     R5,#(1<<23)             /* R5=0x00800000 */
    SUB     R5,R5,R11               /* R5=0x007FFFFF Set fraction bits to 1 */
    MOV     R6,R5                   /* R6=0x007FFFFF Set fraction bits to 1 */
m33_cst_spfpu_convert_test1_loop1:

    /* Prepare NumA & NumB for exact conversions */
    ORR     R3,R10,R11              /* +NumA -> S4 */
    ORR     R4,R3,#(1<<31)          /* -NumA -> S5 */
    ORR     R7,R10,R6               /* +NumB -> S6 */
    ORR     R8,R7,#(1<<31)          /* -NumB -> S7 */

    VMOV    D2,R3,R4
    VMOV    D3,R7,R8
    
    /* Prepare NumC & NumD for inexact conversions */    
    SUB     R10,R10,#(1<<23)        /* exp = (exp - 1) */
    
    ORR     R3,R10,R11              /* +NumC -> S0 */
    ORR     R4,R3,#(1<<31)          /* -NumC -> S1 */
    ORR     R7,R10,R5               /* +NumD -> S2 */
    ORR     R8,R7,#(1<<31)          /* -NumD -> S3 */
    
    VMOV    D0,R3,R4
    VMOV    D1,R7,R8 
    
    /*  Case1a) - Exact Conversion */
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */
    VCVT.U32.F32    S31,S4      /* exact -> FPSCR.IXC = 0 */
    VCVT.S32.F32    S30,S5      /* exact -> FPSCR.IXC = 0 */
    VCVT.U32.F32    S29,S6      /* exact -> FPSCR.IXC = 0 */
    VCVT.S32.F32    S28,S7      /* exact -> FPSCR.IXC = 0 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1     
    
    /*  Case1b) - Inexact Conversion */
    VCVT.U32.F32    S27,S0      /* inexact -> FPSCR.IXC = 1 */
    VCVT.S32.F32    S26,S1      /* inexact -> FPSCR.IXC = 1 */
    VCVT.U32.F32    S25,S2      /* inexact -> FPSCR.IXC = 1 */
    VCVT.S32.F32    S24,S3      /* inexact -> FPSCR.IXC = 1 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1 
    
    /* Case 1c) - Inexact Conversion with direct rounding */
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */  
    VCVTA.U32.F32   S23,S0      /* inexact -> FPSCR.IXC = 1 */
    VCVTA.S32.F32   S22,S1      /* inexact -> FPSCR.IXC = 1 */
    VCVTN.U32.F32   S21,S0      /* inexact -> FPSCR.IXC = 1 */
    VCVTN.S32.F32   S20,S1      /* inexact -> FPSCR.IXC = 1 */
    VCVTP.U32.F32   S19,S0      /* inexact -> FPSCR.IXC = 1 */
    VCVTP.S32.F32   S18,S1      /* inexact -> FPSCR.IXC = 1 */
    VCVTM.U32.F32   S17,S0      /* inexact -> FPSCR.IXC = 1 */
    VCVTM.S32.F32   S16,S1      /* inexact -> FPSCR.IXC = 1 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s16_s31_to_r1_destr_r2
    
    /* Prepare new mantissas for next run */
    BIC     R6,R6,R11           /* mantissa2 */
    SUB     R5,R6,#1            /* mantissa3 */
    LSL     R11,R11,#1          /* mantissa1 - move fraction bit */
    
    CMP     R11,#(1<<22)
    BLS     m33_cst_spfpu_convert_test1_loop1   /* loop 23x */
    
    
    /* For Inexact conversions we will additionally check rounding for the
       following modes (RMode bits [22:23]):                 
        Run1: 0b00 Round to Nearest             (RN) mode 
        Run1: 0b01 Round towards Plus Infinity  (RP) mode
        Run2: 0b10 Round towards Minus Infinity (RM) mode
              0b11 Round towards Zero           (RZ) mode (tested above)
       Note, that (RZ) is default mode when {R} is not used. */
m33_cst_spfpu_convert_test1_loop1_rounding:
    
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */
    VCVTR.U32.F32    S31,S4     /* exact   -> No Rounding! */
    VCVTR.S32.F32    S30,S5     /* exact   -> No Rounding! */
    VCVTR.U32.F32    S29,S6     /* exact   -> No Rounding! */
    VCVTR.S32.F32    S28,S7     /* exact   -> No Rounding! */ 
    
    VCVTR.U32.F32    S27,S0     /* inexact -> Rounding */
    VCVTR.S32.F32    S26,S1     /* inexact -> Rounding */
    VCVTR.U32.F32    S25,S2     /* inexact -> Rounding */
    VCVTR.S32.F32    S24,S3     /* inexact -> Rounding */
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    ADD     R0,R0,#(1<<22)      /* Prepare new RMode bits */
    CMP     R0,#(2<<22)
    BLS     m33_cst_spfpu_convert_test1_loop1_rounding
    
    
    
    /*************************************************************************/
    /*  Case2: Integer to Floating-point */
    /*************************************************************************/
    /* Let assume integers between 2E25 and 2E26. These numbers have the 
       following precision limitations when converted to the Floating-point:
       
       - Every 4th number can be converted exactly, see below:
           0x02000000 - exact conversion
           0x02000001 - inexact conversion (depends on rounding mode)
           0x02000002 - inexact conversion (depends on rounding mode)
           0x02000003 - inexact conversion (depends on rounding mode)
           0x02000004 - exact conversion
        
        Perform the following test:
        ==========================
        - Force every fraction bit to "1" and "0"
        - Test inexact conversions
        - Additionally test rounding for all rounding modes in the FPSCR 
          register 
                 
        ----------------------------------------------------------------------        
          Integer         ->           Float                
        ----------------------------------------------------------------------
        Run_1:  
         0x02000000 -> exact   (+) (exp=25) 0b00000000000000000000000
         0x03FFFFFC -> exact   (+) (exp=25) 0b11111111111111111111111
        -0x02000001 -> inexact (-) (exp=25) 0b00000000000000000000000 (rounding)
         0x02000002 -> inexact (+) (exp=25) 0b00000000000000000000000 (rounding)
         0x02000003 -> inexact (+) (exp=25) 0b00000000000000000000001 (rounding)
        -0x03FFFFFD -> inexact (-) (exp=25) 0b11111111111111111111111 (rounding)
         0x03FFFFFE -> inexact (+) (exp=26) 0b00000000000000000000000 (rounding)
         0x03FFFFFF -> inexact (+) (exp=26) 0b00000000000000000000000 (rounding)
             
                  ...       ...      ...    ...           ....
              
        Run_12: 
         0x02001000 -> exact   (+) (exp=25) 0b00000000000010000000000
         0x03FFEFFC -> exact   (+) (exp=25) 0b11111111111101111111111
        -0x02001001 -> inexact (-) (exp=25) 0b00000000000010000000000 (rounding)
         0x02001002 -> inexact (+) (exp=25) 0b00000000000010000000000 (rounding)
         0x02001003 -> inexact (+) (exp=25) 0b00000000000010000000001 (rounding)
        -0x03FFFFFD -> inexact (-) (exp=25) 0b11111111111101111111111 (rounding)
         0x03FFFFFE -> inexact (+) (exp=25) 0b11111111111110000000000 (rounding)
         0x03FFFFFF -> inexact (+) (exp=25) 0b11111111111110000000000 (rounding)   
              
                  ...       ...      ...    ...           ....

        Run_24 is additionally tested for all rounding modes in the FPSCR!!
        
        Run_24:
         0x03000000 - exact   (+) (exp=25) 0b10000000000000000000000
         0x02FFFFFC - exact   (+) (exp=25) 0b01111111111111111111111
        -0x03000001 - inexact (-) (exp=25) 0b10000000000000000000000 (rounding)
         0x03000002 - inexact (+) (exp=25) 0b10000000000000000000000 (rounding)
         0x03000003 - inexact (+) (exp=25) 0b10000000000000000000001 (rounding)
        -0x02FFFFFD - inexact (-) (exp=25) 0b01111111111111111111111 (rounding)
         0x02FFFFFE - inexact (+) (exp=25) 0b10000000000000000000000 (rounding)
         0x02FFFFFF - inexact (+) (exp=25) 0b10000000000000000000000 (rounding)                      
     *************************************************************************/ 
    MOV     R0,#0               /* Initialize rounding mode */
    MOV     R3,#(1<<25)         /* 2exp25 = 0x02000000 -> S0 */
    MOV     R11,#2
    MOV     R4,#0x00FFFFFF
    ROR     R4,R4,#30           /*        = 0x03FFFFFC  -> S1 */
 m33_cst_spfpu_convert_test1_loop2:   
     
    /* Prepare values for exact/inexact conversion */
    RSB     R5,R3,#-1           /* +1                   -> S2 */
    ADD     R6,R3,#2            /* +2                   -> S3 */
    ADD     R7,R3,#3            /* +3                   -> S4 */
    RSB     R8,R4,#-1           /* +1                   -> S5 */ 
    ADD     R9,R4,#2            /* +2                   -> S6 */
    ADD     R10,R4,#3           /* +3                   -> S7 */
    
    VMOV    D0,R3,R4
    VMOV    D1,R5,R6
    VMOV    D2,R7,R8
    VMOV    D3,R9,R10 

    /*  Case2a) - Exact Conversion */
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */
    VCVT.F32.U32    S31,S0      /* exact -> FPSCR.IXC = 0 */
    VCVT.F32.U32    S30,S1      /* exact -> FPSCR.IXC = 0 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1        
   
    /*  Case2b) - Inexact Conversion */
    VCVT.F32.S32    S29,S2      /* inexact -> FPSCR.IXC = 1 */
    VCVT.F32.U32    S28,S3      /* inexact -> FPSCR.IXC = 1 */
    VCVT.F32.U32    S27,S4      /* inexact -> FPSCR.IXC = 1 */
    VCVT.F32.S32    S26,S5      /* inexact -> FPSCR.IXC = 1 */
    VCVT.F32.U32    S25,S6      /* inexact -> FPSCR.IXC = 1 */
    VCVT.F32.U32    S24,S7      /* inexact -> FPSCR.IXC = 1 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1     
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    /* Prepare new mantissas for next run */
    LSL     R11,R11,#1          /* shift to stimulate next fraction bit */
    ORR     R3,R11,#(1<<25)     
    MOV     R4,#(1<<26)
    SUB     R4,R4,#4
    BIC     R4,R4,R11           
     
    CMP     R11,#(1<<24)
    BLS     m33_cst_spfpu_convert_test1_loop2   /* Loop 24x */
    
    
    /* For Inexact conversions we will additionally check rounding for the
       following modes (RMode bits [22:23]): 
        Run1: 0b01 Round towards Plus Infinity  (RP) mode
        Run2: 0b10 Round towards Minus Infinity (RM) mode
        Run3: 0b11 Round towards Zero           (RZ) mode 
              0b00 Round to Nearest             (RN) mode (tested above) */
m33_cst_spfpu_convert_test1_loop2_rounding:

    ADD     R0,R0,#(1<<22)      /* Prepare new RMode bits */
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */
    VCVT.F32.U32    S31,S0      /* exact   -> No Rounding */
    VCVT.F32.U32    S30,S1      /* exact   -> No Rounding */
    
    VCVT.F32.S32    S29,S2      /* inexact -> Rounding */
    VCVT.F32.U32    S28,S3      /* inexact -> Rounding */
    VCVT.F32.U32    S27,S4      /* inexact -> Rounding */
    VCVT.F32.S32    S26,S5      /* inexact -> Rounding */
    VCVT.F32.U32    S25,S6      /* inexact -> Rounding */
    VCVT.F32.U32    S24,S7      /* inexact -> Rounding */
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s24_s31_to_r1_destr_r2
    
    CMP     R0,#(3<<22)
    BMI     m33_cst_spfpu_convert_test1_loop2_rounding
    
 
    /*************************************************************************/
    /*  Case3: Convert between half and single precision */
    /*************************************************************************/
    LDR     R1,=0x38800000  /* ~(+)6.1e-5  -> S10 (F16-smallest normal)    */
    LDR     R2,=0xC77FE000  /*  (-)65504   -> S11 (F16-largest normal)     */
    LDR     R3,=0x33800000  /* ~(+)5.96e-8 -> S12 (F16-smallest subnormal) */
    LDR     R4,=0xB87FC000  /* ~(-)6.1e-5  -> S13 (F16-largest subnormal ) */
    LDR     R5,=0x42AAA000  /*  (+)85.3125 -> S14 (F16 0xAAAA)             */
    LDR     R6,=0xBD554000  /* ~(-)0.05206 -> S15 (F16 0x5555)             */
    
    VMOV    D0,R1,R2
    VMOV    D1,R3,R4
    VMOV    D2,R5,R6
    
    /*  Case3a) - no error due to conversion    */
    /*  Convert F32->F16                        */
    /*            |sign | exp    | mantissa     */
    /* S31[31:16] |0b(0)| 0b00001| 0b0000000000 */
    /* S31[15:0]  |0b(1)| 0b11110| 0b1111111111 */
    /* S29[31:16] |0b(0)| 0b00000| 0b0000000001 */
    /* S29[15:0]  |0b(1)| 0b00000| 0b1111111111 */
    /* S30[31:16] |0b(0)| 0b10101| 0b0101010101 */
    /* S30[15:0]  |0b(1)| 0b01010| 0b1010101010 */
    MOV     R0,#0
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */
    VCVTT.F16.F32   S31,S0      /* exact -> FPSCR.IXC = 0 */
    VCVTB.F16.F32   S31,S1      /* exact -> FPSCR.IXC = 0 */
    VCVTT.F16.F32   S30,S2      /* exact -> FPSCR.IXC = 0 */
    VCVTB.F16.F32   S30,S3      /* exact -> FPSCR.IXC = 0 */
    VCVTT.F16.F32   S29,S4      /* exact -> FPSCR.IXC = 0 */
    VCVTB.F16.F32   S29,S5      /* exact -> FPSCR.IXC = 0 */
    
    /* Convert F16->F32 */ 
    VCVTT.F32.F16   S28,S31     /* exact -> FPSCR.IXC = 0 */
    VCVTB.F32.F16   S27,S31     /* exact -> FPSCR.IXC = 0 */
    VCVTT.F32.F16   S26,S30     /* exact -> FPSCR.IXC = 0 */
    VCVTB.F32.F16   S25,S30     /* exact -> FPSCR.IXC = 0 */
    VCVTT.F32.F16   S24,S29     /* exact -> FPSCR.IXC = 0 */
    VCVTB.F32.F16   S23,S29     /* exact -> FPSCR.IXC = 0 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s23_s31_to_r1_destr_r2
    
    /*  Case3b) - error due to conversion */
    LDR     R1,=0x3F800001  /* ~(+)1.000..1 -> S0 */
    LDR     R2,=0xBF7FFFFF  /* ~(-)0.999... -> S1 */
    LDR     R3,=0x3EAAAAAA  /* ~(+)0.333..3 -> S2 */
    LDR     R4,=0x3FD55555  /* ~(-)1.666..6 -> S3 */
    LDR     R5,=0x3DCCCCCC  /* ~(+)0.1      -> S4 */
    LDR     R6,=0xBF333333  /* ~(-)0.7      -> S5 */
    
    VMOV    D3,R1,R2
    VMOV    D4,R3,R4
    VMOV    D5,R5,R6
    
    /* RMode bits [23:23] 
        Run1: 0b00 Round to Nearest             (RN) mode
        Run2: 0b01 Round towards Plus Infinity  (RP) mode
        Run3: 0b10 Round towards Minus Infinity (RM) mode
        Run4: 0b11 Round towards Zero           (RZ) mode */
m33_cst_spfpu_convert_test1_runding_loop2:

    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */  
    VCVTT.F16.F32   S31,S6      /* inexact -> FPSCR.IXC = 1 */
    VCVTB.F16.F32   S31,S7      /* inexact -> FPSCR.IXC = 1 */
    VCVTT.F16.F32   S30,S8      /* inexact -> FPSCR.IXC = 1 */
    VCVTB.F16.F32   S30,S9      /* inexact -> FPSCR.IXC = 1 */
    VCVTT.F16.F32   S29,S10     /* inexact -> FPSCR.IXC = 1 */
    VCVTB.F16.F32   S29,S11     /* inexact -> FPSCR.IXC = 1 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */
    VCVTT.F32.F16   S28,S31     /* exact -> FPSCR.IXC = 0 */
    VCVTB.F32.F16   S27,S31     /* exact -> FPSCR.IXC = 0 */
    VCVTT.F32.F16   S26,S30     /* exact -> FPSCR.IXC = 0 */
    VCVTB.F32.F16   S25,S30     /* exact -> FPSCR.IXC = 0 */
    VCVTT.F32.F16   S24,S29     /* exact -> FPSCR.IXC = 0 */
    VCVTB.F32.F16   S23,S29     /* exact -> FPSCR.IXC = 0 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s23_s31_to_r1_destr_r2
    
    ADD     R0,R0,#(1<<22)      /* Prepare new RMode bits */
    CMP     R0,#(3<<22)
    BLS     m33_cst_spfpu_convert_test1_runding_loop2
    
    
    /*************************************************************************/
    /* Case4:                                                                */
    /*  -Test conversions of +-0, +-Inf, +-qNaN, +-sNaN                      */
    /*  - Test FPSCR.IOC, FPSCR.IDC, FPSCR.UFC, FPSCR.OFC flags              */
    /*  - Test Flush-to-zero                                                 */
    /*************************************************************************/
    MOV     R2,#0x7F800000  /* +Inf(F32) +0(F16L) */
    MOV     R3,#0x80000000
    ORR     R3,R3,#0xFC00   /* -0(F16H) | -Inf(F16L) */
    ORR     R4,R2,#(1<<22)  /* +qNaN(F32) +0 */
    ORR     R5,R3,#(1<<9)   /* -0(F16H) | -qNaN(F16L) */
    VMOV    D0,R2,R3
    VMOV    D1,R4,R5    
    
    MOV     R0,#0 
    VMSR    FPSCR,R0        /* Write RMode and clear FPSCR */
    VCVTT.F16.F32   S31,S0  /* +Inf(F32)    -> +Inf(F16) */
    VCVTB.F32.F16   S30,S1  /* -Inf(F16)    -> -Inf(F32) */
    VCVTB.F16.F32   S31,S30 /* -Inf(F32)    -> -Inf(16)  */  
    VCVTT.F32.F16   S29,S31 /* +Inf(F16)    -> +Inf(F32) */
    VCVTT.F32.F16   S28,S1  /* -0(F16)      -> -0(F32)   */
    VCVTB.F32.F16   S27,S0  /* +0(F16)      -> +0(F32)   */
    VCVTT.F16.F32   S26,S27 /* +0(F32)      -> +0(F16)   */
    VCVTB.F16.F32   S26,S28 /* -0(F32)      -> -0(F16)   */
    VCVTT.F16.F32   S25,S2  /* +qNaN(F32)   -> +qNaN(F16)*/
    VCVTB.F32.F16   S24,S3  /* -qNaN(F16)   -> -qNaN(F32)*/
    VCVTT.F32.F16   S23,S25 /* +qNaN(F16)   -> +qNaN(F32)*/ 
    VCVTB.F16.F32   S25,S24 /* -qNaN(F32)   -> -qNaN(F16)*/
    VCVT.S32.F32    S22,S28 /* -0(F32)      -> +0(S32)   */
    VCVT.U32.F32    S21,S28 /* -0(F32)      -> +0(U32)   */
    
    
    /* Test several situations which set FPSCR.IOC flag */
    ORR     R4,R2,#(1<<21)  /* +sNaN(F32) +0 */
    ORR     R5,R3,#(1<<8)   /* -0(F16H) | -sNaN(F16L) */
    VMOV    D1,R4,R5  
    
    VMSR    FPSCR,R0        /* Write RMode and clear FPSCR */
    VCVTT.F16.F32   S20,S2  /* +sNaN(F32) -> +qNaN(F16)     => FPSCR.IOC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR        /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    VMSR    FPSCR,R0        /* Write RMode and clear FPSCR */
    VCVTB.F32.F16   S19,S3  /* -sNaN(F16)   -> -qNaN(F32)   => FPSCR.IOC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR        /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    VMSR    FPSCR,R0        /* Write RMode and clear FPSCR */
    VCVT.S32.F32    S18,S30 /* -Inf(F32)    -> S32          => FPSCR.IOC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR        /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    VMSR    FPSCR,R0        /* Write RMode and clear FPSCR */
    VCVT.U32.F32    S17,S29 /* +Inf(F32)    -> S32          => FPSCR.IOC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR        /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    
    /* Test several situations which set FPSCR.IDC flag */
    MOV     R2,#(1<<22)     /* 0x00400000 - Denorm1 */
    SUB     R3,R2,#1        /* 0x003FFFFF - Denorm2 */    
    VMOV    D0,R2,R3
    
    MOV     R0,#(1<<24)     /* Set Flush-to-zero bit */ 
    VMSR    FPSCR,R0        /* Set Flush-to-zero bit and clear FPSCR */
    VCVTT.F16.F32   S16,S0  /* => FPSCR.IDC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR        /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    VMSR    FPSCR,R0        /* Set Flush-to-zero bit and clear FPSCR */
    VCVT.U32.F32    S15,S1  /* => FPSCR.IDC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR        /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1

    
    /* Test several situations which set FPSCR.UFC and FPSCR.OFC flags */
    LDR     R2,=0x33800000  /* Smallest F16 subnormal expressed in F32 */
    LDR     R3,=0xC77FE000  /* Largest F16 normal expressed in F32 */
    SUB     R2,R2,#1        /* Decrease smallest subnormal */
    ADD     R3,R3,#1        /* Increase largest normal */
    VMOV    D1,R2,R3
    
    MOV     R0,#(2<<22)     /* Set Round towards Minus Infinity */
    VMSR    FPSCR,R0        /* Write RMode and clear FPSCR */
    VCVTB.F16.F32   S14,S2  /* => FPSCR.UFC = 1 */
    VCVTT.F16.F32   S14,S3  /* => FPSCR.OFC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR        /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s14_s31_to_r1_destr_r2

    /**************************************************************************/
    /* Test of loading FPDSCR to FPSCR                                        */
    /**************************************************************************/
    LDR     R2,=M33_FPDSCR   /* Load address of FPDSCR to R2 */
    LDR     R3,[R2]         /* Backup M33_FPDSCR             !! DONT USE R3 !! */
    
    MOV     R5,#0x07C00000  /* Prepare FPDSCR mask */
    
    /* Prepare D0 for alternative half-precision format conversion test */
    LDR     R6,=0x7C00FC00  /* For AHP=1 -> T= 65536.0 B= -65536.0 */
    LDR     R7,=0x7FFFFFFF  /* For AHP=1 -> T=131008.0 B=-131008.0 */
    VMOV    D0,R6,R7
    
    MOV     R4,#0x05400000  /* Prepare test value to be stored to FPDSCR */
    BL      m33_cst_spfpu_convert_test1_check_fpdscr
    
    MOV     R4,#0x02800000  /* Prepare test value to be stored to FPDSCR */
    BL      m33_cst_spfpu_convert_test1_check_fpdscr
    
    LDR     R6,=0x5E2111E8
    CMP     R6,R7
    BNE     m33_cst_spfpu_convert_test1_failed
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
m33_cst_spfpu_convert_test1_end:
    /* Test result is returned in R0, according to the conventions */
    MOV     R0,R1 
m33_cst_spfpu_convert_test1_failed:
    STR     R3,[R2]         /* Restore FPDSCR */
    DSB
    VMSR    FPSCR,R12       /* Restore FPSCR register */
    B       m33_cst_test_tail_fpu_end

m33_cst_spfpu_convert_test1_check_fpdscr:
    STR     R4,[R2]         /* Store test value to FPDSCR */
    DSB
    
    /* Clear FPCA bit in CONTROL register to force loading FPDSCR to FPSCR */
    MRS     R6,CONTROL
    BFC     R6,#2,#1        /* Clear FPCA bit */
    MSR     CONTROL,R6
    ISB
    
    /* Convert F16->F32          AHP=1    AHP=0 */
    VCVTT.F32.F16   S2,S0   /*   65536.0   INF  */
    VCVTB.F32.F16   S3,S0   /*  -65536.0  -INF  */
    VCVTT.F32.F16   S4,S1   /*  131008.0   NAN  */
    VCVTB.F32.F16   S5,S1   /* -131008.0   NAN  */

    /* Sumarize results of F16->F32 conversions to R7 */
    VMOV    R8,S2
    EOR     R7,R8,R7,ROR #1
    VMOV    R8,S3
    EOR     R7,R8,R7,ROR #1
    VMOV    R8,S4
    EOR     R7,R8,R7,ROR #1
    VMOV    R8,S5
    EOR     R7,R8,R7,ROR #1
    
    /* Convert F32->F16 */
    VCVTT.F16.F32   S2,S2
    VCVTB.F16.F32   S2,S3
    VCVTT.F16.F32   S3,S4
    VCVTB.F16.F32   S3,S5
    
    /* Sumarize results of F32->F16 conversions to R7 */
    VMOV    R8,S2
    EOR     R7,R8,R7,ROR #1
    VMOV    R8,S3
    EOR     R7,R8,R7,ROR #1
    
    VMRS    R6,FPSCR        /* Load FPSCR to R6 */
    
    AND     R6,R6,R5        /* We care only about FPDSCR bits */

    CMP     R6,R4           /* Part of FPSCR should be same as FPDSCR */
    BNE     m33_cst_spfpu_convert_test1_failed
    
    BX      LR
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_FPU_ENABLE */

    CST_FILE_END

