/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
*
* Tests FPU comparison functionality provided by the cm33fpu_sp_add.v module.
* Only special situations are tested here. Internal adder functionality is 
* covered by the M33_Cst_SpfpuAdderTest test.
*
* Test procedure:
* --------------
* Tests the Floating-point comparison functionality.
*
* Compare the following special situations:
* ------------------------------------------

*  Case1: Incomparable if either is NaN
*  Case2: Both infinite, compare signs
*  Case3: Only A infinite, A larger if positive
*  Case4: Only B infinite, A larger if B negative
*  Case5: Both zero, equal regardless of signs
*  Case6: Signs differ, A larger if positive
*  Case7: Compare exponents
*    Case7a: Both positive, A larger if magnitude greater
*    Case7b: Both negative, A larger if magnitude lesser
*  Case8: Compare fractions
*    Case7a: Both positive, A larger if magnitude greater
*    Case7b: Both negative, A larger if magnitude lesser
*  Case9: Compare denormalized numbers
*
* SP FPU Decoder coverage:
* ------------------------
*  -VCMP{E}.F32 - this instruction sets flags => check NZCV flags
*              - this instruction sets FPSCR.IOC flag
*  -VMRS    - for moving flags from FPSCR to APSR_nzcv
*  -VMOV.F32 (immediate)
*  -VNEG.F32
*  -VSEL.F32
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FPU_ENABLE==1)

    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_SpfpuCmpTest

    /* Symbols defined outside but used within current module */
    CST_EXTERN m33_cst_test_tail_fpu_end
    
    /* Pre-signature */
    CST_SET(PRESIGNATURE_SEED_0, 0x9E3A)
    CST_SET(PRESIGNATURE_SEED_1, 0xA889)
    
    /* Values stored in registers:
       - R1  ... accumulated signature
       - R12 ... FPSCR
    */

   /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
   /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_SpfpuCmpTest" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_TYPE(M33_Cst_SpfpuCmpTest, function)
    CST_THUMB2
M33_Cst_SpfpuCmpTest:

    PUSH    {R4-R12,R14}
    MRS     R1,CONTROL  /* Store CONTROL prior first FPU instruction */
    PUSH    {R1}
    VPUSH   {S16-S31}

    /*-----------------------------------------------------------------------*/
    /* Test - preparation                                                    */
    /*-----------------------------------------------------------------------*/
    MOV     R10,#PRESIGNATURE_SEED_0
    MOVT    R10,#PRESIGNATURE_SEED_1
    MOV     R1,R10
    
    /*-----------------------------------------------------------------------*/
    /* Test - start                                                          */
    /*-----------------------------------------------------------------------*/
    VMRS        R12,FPSCR       /* Store FPSCR register */

    /* Prepare test vectors */
    MOV     R0,#0
    MOV     R3,#(255<<23)       /* +Inf  -> S0 */
    ORR     R4,R3,#(1<<31)      /* -Inf  -> S1 */
    ORR     R5,R3,#(1<<22)      /* +qNaN -> S2 */
    ORR     R6,R4,#(1<<22)      /* -qNaN -> S3 */
    ORR     R7,R3,#(1<<21)      /* +sNaN -> S4*/
    ORR     R8,R4,#(1<<21)      /* -sNaN -> S5*/
    MOV     R9,#(0<<31)         /* +0    -> S6 */
    ORR     R10,R9,#(1<<31)     /* -0    -> S7 */
    
    VMOV.F32    S8,#0.75        /* +Num  -> S8 */
    VMOV.F32    S9,#-0.75       /* -Num  -> S9 */
    VMOV    D0,R3,R4
    VMOV    D1,R5,R6
    VMOV    D2,R7,R8
    VMOV    D3,R9,R10
    
    
    /*************************************************************************/
    /* Case1: Incomparable if either A or B is NaN                           */
    /*************************************************************************/
    /*   A      B  |   N Z C V                                               */
    /*  -----------------------------------                                  */
    /*  NaN   -Num |   0 0 1 1   Unordered                                   */
    /*  Num   -NaN |   0 0 1 1   Unordered                                   */
    /*************************************************************************/
    VMSR    FPSCR,R0            /* Clear FPSCR */

    /* Compare +qNaN vs -Num */
    VCMP.F32    S2,S9           
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Test VSEL instruction (conditional MOV depending on APSR) */
    VMRS    APSR_nzcv,FPSCR     /* Test connection between FPSCR and APSR */
    VSELVS.F32  S10,S8,S9           /* Yes -> S10=0.75 */
    VADD.F32 S10,S10,S8         /* S10=1.5 */
    VSELGE.F32  S11,S9,S8           /* No -> S11=0.75 */
    VADD.F32 S11,S11,S10        /* S11=2.25*/
    VSELGT.F32  S10,S6,S11          /* No -> S10=2.25 */
    VADD.F32 S10,S11,S10        /* S10=4.5*/
    VSELEQ.F32  S11,S10,S8          /* No -> S11=0.75 */
    VADD.F32 S11,S11,S10        /* S11=5.25 */
    
    /* Compar{E} +qNaN vs -Num */ 
    VCMPE.F32   S2,S9           /*  Sets FPSCR.IOC flag */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VMSR    FPSCR,R0            /* Clear FPSCR */
    
    /* Compare sNaN vs -Num */
    VCMP.F32    S4,S9           /*  Sets FPSCR.IOC flag */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VMSR    FPSCR,R0            /* Clear FPSCR */
    
    /* Compare Num vs -qNaN */
    VCMP.F32    S8,S3
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Compar{E} Num vs -qNaN */ 
    VCMPE.F32   S8,S3           /*  Sets FPSCR.IOC flag */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VMSR    FPSCR,R0            /* Clear FPSCR */
    
    /* Compare Num vs -sNaN */ 
    VCMP.F32    S8,S5           /*  Sets FPSCR.IOC flag */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Test connection between FPSCR and APSR */
    VMRS    APSR_nzcv,FPSCR
    MRS     R2,APSR
    AND     R2,R2,#0xF0000000
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    
    /*************************************************************************/
    /* Case2: Both infinite, compare signs                                   */
    /*************************************************************************/
    /*   A      B  |   N Z C V                                               */
    /* --------------------------                                            */
    /*  +Inf  -Inf |   0 0 1 0    A > B                                      */
    /*  -Inf  +Inf |   1 0 0 0    A < B                                      */
    /*************************************************************************/
    /* Compare +Inf vs -Inf */
    VCMP.F32    S0,S1
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Test connection between FPSCR and APSR */
    VMRS    APSR_nzcv,FPSCR
    MRS     R2,APSR
    AND     R2,R2,#0xF0000000
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */

    /* Test VSEL instruction (conditional MOV depending on APSR) */
    VSELGE.F32  S10,S8,S11          /* Yes -> S10=0.75*/
    VADD.F32 S10,S10,S11        /* S10=6.0*/
    VSELGT.F32  S11,S8,S10          /* Yes -> S11=0.75*/
    VADD.F32 S11,S10,S11        /* S11=6.75*/
    VSELVS.F32  S10,S11,S8          /* No -> S10=0.75*/
    VADD.F32 S10,S10,S11        /* S10=7.5*/
    
    /* Compar{E} -Inf vs +Inf */
    VCMPE.F32  S1,S0
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    
    /*************************************************************************/
    /* Case3: Only A infinite, A larger if positive                          */
    /*************************************************************************/
    /*   A     B   |   N Z C V                                               */
    /* ------------------------                                              */
    /*  +Inf  +Num |   0 0 1 0     A > B                                     */
    /*  -Inf  -Num |   1 0 0 0     A < B                                     */
    /*************************************************************************/
    /* Compare +Inf vs +Num */
    VCMP.F32    S0,S8
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
             
    /* Compar{E} -Inf vs -Num */
    VCMPE.F32   S1,S9
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Test connection between FPSCR and APSR */
    VMRS    APSR_nzcv,FPSCR
    MRS     R2,APSR
    AND     R2,R2,#0xF0000000
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    
    /*************************************************************************/
    /* Case4: Only B infinite, A larger if B negative                        */
    /*************************************************************************/
    /*    A       B   |   N Z C V                                            */
    /*   ------------------------                                            */
    /*  -Num    -Inf  |   0 0 1 0     A > B                                  */
    /*  +Num    +Inf  |   1 0 0 0     A < B                                  */
    /*************************************************************************/
    /* Compare -Num vs -Inf */
    VCMP.F32    S9,S1
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Compar{E} +Num vs Inf */
    VCMPE.F32   S8,S0
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    
    /*************************************************************************/
    /* Case5: Both zero, equal regardless of signs                           */
    /*************************************************************************/
    /*   A       B   |   N Z C V                                             */
    /*  ------------------------                                             */
    /*   +0     -0   |   0 1 1 0     A = B                                   */
    /*   -0     +0   |   0 1 1 0     A = B                                   */
    /*************************************************************************/
    /* Compare +0 vs -0 */
    VCMP.F32    S6,S7
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */

    /* Test VSEL instruction (conditional MOV depending on APSR) */
    VMRS    APSR_nzcv,FPSCR     /* Test connection between FPSCR and APSR */
    VSELEQ.F32 S11,S8,S10          /* Yes -> S11=0.75*/
    VADD.F32 S11,S10,S11        /* S11=8.25*/
    VMOV    R2,S11
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VMSR    FPSCR,R0            /* Clear FPSCR */
    
    /* Compar{E} -0 vs +0 */
    VCMPE.F32   S7,S6
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VMSR    FPSCR,R0            /* Clear FPSCR */
    
    /* VCMP #immediate */
    /* Compare +0 vs #0 */
    VCMP.F32    S6,#0.0
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VMSR    FPSCR,R0            /* Clear FPSCR */
    
    /* Compar{E} -0 vs #0 */
    VCMPE.F32   S7,#0.0
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Test connection between FPSCR and APSR */
    VMRS    APSR_nzcv,FPSCR
    MRS     R2,APSR
    AND     R2,R2,#0xF0000000
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    
    /*************************************************************************/
    /* Case6: Sign differs, A larger if positive                             */
    /*************************************************************************/
    /*       A      B    |   N Z C V      |(+)Num| = |(-)Num|                */
    /*  -----------------                                                    */
    /*  (+)Num  (-)Num   |   0 0 1 0             A > B                       */
    /*  (-)Num  (+)Num   |   1 0 0 0             A < B                       */
    /*************************************************************************/
    /* Compare (+)Num vs (-)Num */
    VCMP.F32    S8,S9
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Compar{E} (-)Num vs (+)Num */
    VCMPE.F32   S9,S8
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    
    /*************************************************************************/
    /* Case7: Compare exponents                                              */
    /*************************************************************************/
    /* - Compare different exponents in the loops for both positive (Case7a)
         and negative (Case7b) numbers.
       - Compare exponents for equality (Case7c)
       - Additionally to (Case8c) fractions are compared for equality for two 
         different values 0x000000 and 0x7fffff.
       
       Proposed test vectors:
       ---------------------
       exp1: 0x80 0x40 0x20 ... 0x01
       exp2: 0x7F 0x3F 0x1F ... 0x00

     Note, that exponents are being additionally compared in the 
     m33_cst_spfpu_mimax_test test.       
    */
    
    MOV     R3,#(1<<31)
 m33_cst_spfpu_cmp_test_loop1:

    LSR     R3,R3,#1            /* Next exp1          */
    SUB     R4,R3,#1            /* Next exp2=(Exp1-1) */
    
    VMOV    S10,R3              /* (+) exp1 -> S10 */          
    VMOV    S11,R4              /* (+) exp2 -> S11 */ 
    VNEG.F32    S12,S10         /* (-) exp1 -> S12 */
    VNEG.F32    S13,S11         /* (-) exp2 -> S13 */
    VNEG.F32    S14,S12         /* (+) exp1 -> S14 */
    VNEG.F32    S15,S11         /* (+) exp2 -> S15 */
    
    
    /*************************************************************************/
    /* Case7a: Both positive, A larger if magnitude greater                  */
    /*************************************************************************/
    /*      A           B      |   N Z C V    (exp2 > exp1)                  */
    /*  ------------------------                                             */
    /*  exp1(S10)    exp2(S11) |   0 0 1 0        A > B                      */
    /*  exp2(S11)    exp1(S10) |   1 0 0 0        A < B                      */
    /*************************************************************************/
    /* Compare Exp1 vs Exp2 */
    VCMP.F32    S10,S11           
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    /* Compare Exp2 vs Exp1 */
    VCMP.F32    S11,S10
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    
    /*************************************************************************/
    /* Case7c: Both exponents are same                                       */
    /*************************************************************************/
    /*      A           B      |   N Z C V    (exp1 = exp1)                  */
    /*  ------------------------                                             */
    /*  exp1(S10)    exp1(S11) |   0 1 1 0        A = B                      */
    /*************************************************************************/
    /* Compare exp1 vs exp1 */
    VCMP.F32    S10,S14
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /*************************************************************************/
    /* Case7b: Both negative, A larger if magnitude lesser                   */
    /*************************************************************************/
    /*      A           B      |   N Z C V    (exp1 > exp2)                  */
    /*  ------------------------                                             */
    /*  exp1(S12)    exp2(S13) |   1 0 0 0        A < B                      */
    /*  exp2(S13)    exp1(S12) |   0 0 1 0        A > B                      */
    /*************************************************************************/
    /* Compare exp1 vs exp2 */
    VCMP.F32    S12,S13          
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    /* Compare exp2 vs exp1 */
    VCMP.F32    S13,S12           
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /*************************************************************************/
    /* Case7c: Both exponents are same                                       */
    /*************************************************************************/
    /*      A           B      |   N Z C V    (exp2 = exp2)                  */
    /*  ------------------------                                             */
    /*  exp2(S15)    exp2(S13) |   0 1 1 0        A = B                      */
    /*************************************************************************/
    /* Compare Exp2 vs Exp2 */
    VCMP.F32    S15,S13
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
     
    CMP     R3,#(1<<23)
    BNE     m33_cst_spfpu_cmp_test_loop1    /* Exponent loop 8x */
    

    /*************************************************************************/
    /* Case8: Compare fractions                                              */
    /*************************************************************************/
    /* - Compare different fractions in the loops for both positive (Case8a)
         and negative (Case8b) numbers.
       - Compare fractions for equality (Case8c)
       - Additionally to (Case7c) exponents are compared for equality for two 
         different values 0x55 and 0xAA.
    
     Proposed test vectors:
       ---------------------
        frac1: 0x400000 0x200000 0x100000 ...
        frac2: 0x3FFFFF 0x1FFFFF 0x0FFFFF ...
        
     Note, that fractions are being additionally compared in the 
     m33_cst_spfpu_mimax_test test.
     **************************************************************************/
    MOV     R7,#(1<<22)
m33_cst_spfpu_cmp_test_loop2:
    
    /* Prepare Next operands */
    ORR     R3,R7,#(0xAA<<23)   /* exp1 + frac1 */
    SUB     R4,R3,#1            /* exp1 + frac2 */
    ORR     R5,R7,#(0x55<<23)   /* exp2 + frac1 */
    SUB     R6,R5,#1            /* exp2 + frac2 */
    
    VMOV    S16,R3              /* (+) exp1 + frac1 -> S16 */
    VMOV    S17,R4              /* (+) exp1 + frac2 -> S17 */
    VMOV.F32    S20,S16         /* (+) exp1 + frac1 -> S20 */
    VMOV    D9,R5,R6 
    VNEG.F32    S18,S18         /* (-) exp2 + frac2 -> S18 */
    VNEG.F32    S19,S19         /* (-) exp2 + frac2 -> S19 */
    VMOV.F32    S21,S19         /* (-) exp2 + frac2 -> S19 */
    
    /*************************************************************************/
    /* Case8a: Both positive, A larger if magnitude greater                  */
    /*************************************************************************/
    /*      A           B        |   N Z C V    (frac1 > frac2)              */
    /*  ------------------------                                             */
    /*  frac1(S16)    frac2(S17) |   0 0 1 0         A > B                   */
    /*  frac2(S17)    frac1(S16) |   1 0 0 0         A < B                   */
    /*************************************************************************/
    VCMP.F32    S16,S17
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VCMP.F32    S17,S16
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /*************************************************************************/
    /* Case8c: Both fractions are same                                       */
    /*************************************************************************/
    /*      A           B        |   N Z C V    (frac1 = frac1)              */
    /*  ------------------------                                             */
    /*  frac1(S20)    frac1(S16) |   0 1 1 0         A = B                   */
    /*************************************************************************/
    VCMP.F32    S20,S16
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /*************************************************************************/
    /* Case8b: Both negative, A larger if magnitude lesser                   */
    /*************************************************************************/
    /*      A           B        |   N Z C V    (frac1 > frac2)              */
    /*  ------------------------                                             */
    /*  frac1(S18)    frac2(S19) |   1 0 0 0         A < B                   */
    /*  frac2(S19)    frac1(S18) |   0 0 1 0         A > B                   */
    /*************************************************************************/
    VCMP.F32    S18,S19
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    VCMP.F32    S19,S18
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /*************************************************************************/
    /* Case8c: Both fractions are same                                       */
    /*************************************************************************/
    /*      A           B        |   N Z C V  (frac2 = frac2)                */
    /*  ------------------------                                             */
    /*  frac2(S19)    frac2(S21) |   0 1 1 0       A = B                     */
    /*************************************************************************/
    VCMP.F32    S19,S21
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    LSRS     R7,R7,#1
    BNE     m33_cst_spfpu_cmp_test_loop2    /* Fraction loop 23x */ 
    
    
    /**************************************************************************/
    /* Case9: Compare denormalized numbers selection                          */
    /**************************************************************************/
    /* 
        Proposed test vectors:
       ---------------------
        frac1= 0x400000
        frac2= 0x3FFFFF 
       
       Test both Flush-to-zero Disabled/Enabled modes.
     **************************************************************************/
    MOV     R4,#(1<<21)         /* Num2 denorm */     
    ORR     R3,R4,#1            /* Num1 denorm */
        
    /* Write floating-point registers*/
    VMOV    S22,R3              /* (+) Num1 -> S20 */
    VMOV    S23,R4              /* (+) Num2 -> S21 */
    VNEG.F32    S24,S22         /* (-) Num1 -> S22 */
    VNEG.F32    S25,S23         /* (-) Num2 -> S23 */
    
    /* Clear FPSCR & Disable Flush-to-zero */
    VMSR    FPSCR,R0        
    /*************************************************************************/
    /* Case9a: Both positive, A larger if magnitude greater                  */
    /*************************************************************************/
    /*      A           B        |   N Z C V    (frac1 > frac2)              */
    /*  ------------------------                                             */
    /*     Num1        Num2      |   0 0 1 0         A > B                   */
    /*     Num2        Num1      |   1 0 0 0         A < B                   */
    /*************************************************************************/
    /* Compare Num1 vs Num2 */
    VCMP.F32    S22,S23
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    /* Compare Num2 vs Num1 */
    VCMP.F32    S23,S22     
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Enable Flush-to-zero */
    EOR     R2,R2,#(1<<24)      /* Enable Flush-to-zero */
    VMSR    FPSCR,R2
    /*************************************************************************/
    /* Case9c: Flush to zero enabled                                         */
    /*************************************************************************/
    /*      A           B        |   N Z C V   (|Num1| > |Num2|)             */
    /*  ------------------------                                             */
    /*    (+)Num1    (-)Num2     |   0 1 1 0         A = B                   */ 
    /*************************************************************************/
    VCMP.F32    S22,S25         /* Sets FPSCR.IXC flag */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Clear FPSCR & Disable Flush-to-zero */
    VMSR    FPSCR,R0
    /*************************************************************************/
    /* Case9b: Both negative, A larger if magnitude greater                  */
    /*************************************************************************/
    /*      A           B        |   N Z C V    (frac1 > frac2)              */
    /*  ------------------------                                             */
    /*     Num1        Num2      |   1 0 0 0         A < B                   */
    /*     Num2        Num1      |   0 0 1 0         A > B                   */
    /*************************************************************************/
    /* Compare (-)Num1 vs (-)Num2 */
    VCMP.F32    S24,S25         
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    /* Compare (-)Num2 vs (-)Num1 */
    VCMP.F32    S25,S24
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
    
    /* Enable Flush-to-zero */
    EOR     R2,R2,#(1<<24)         /* Enable Flush-to-zero */
    VMSR    FPSCR,R2
    /*************************************************************************/
    /* Case9c: Flush to zero enabled                                         */
    /*************************************************************************/
    /*      A           B        |   N Z C V   (|Num1| > |Num2|)             */
    /*  ------------------------                                             */
    /*    (+)Num2    (-)Num1     |   0 1 1 0         A = B                   */ 
    /*************************************************************************/
    VCMP.F32    S23,S25         /* Sets FPSCR.IXC flag */
    VMRS    R2,FPSCR
    EOR     R1,R2,R1,ROR #1     /* !! Update Signature !! */
           
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
    /* Test result is returned in R0, according to the conventions */
m33_cst_spfpu_cmp_test_end:
    VMSR        FPSCR,R12   /* Restore FPSCR register */
    /* Test result is returned in R0, according to the conventions */
    MOV         R0,R1 
    B           m33_cst_test_tail_fpu_end
    
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_FPU_ENABLE */

    CST_FILE_END

