/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
*   Tests forwarding logic functionality. 
*   - Verifies that results from one instruction can be used immediately 
*     as inputs in next instructions. Values are not taken from register file 
*     but they must be forwarded.
*
* The following cases are tested:
* -------------------------------
* 1) Forward data from the end of EX2 stage to ISS stage
*    1a) Forward data from noMAC W0 port to ISS stage
*    1b) Forward data from noMAC W1 port to ISS stage
*    1c) Forward data from noMAC W2 port to ISS stage
*    1d) Forward data from noMAC W3 port to ISS stage
*    1e) Forward data from MAC W0 port to ISS stage
*    1f) Forward data from MAC W1 port to ISS stage
* 2) Forward data from the end of EX1 stage to ISS stage
*    2a) Forward data from SHIFTER in dp0 to ISS stage
*    2b) Forward data from SHIFTER in dp1 to ISS stage
* 3) Forward R13 register (special case)
*
* For each case above data is forwarded to A0,A1,B0,B1 read ports and each
* forwarding path is stimulated with 20 pseudo-random test vectors.
* 
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FORWARD_ENABLE==1)
    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_ForwardingTest1

    /* Symbols defined outside but used within current module */
    CST_EXTERN CST_RAM_TARGET0
    CST_EXTERN m33_cst_test_tail_end
    

    CST_SET(PRESIGNATURE, 0x92E00114)
   
    /* Values stored in registers:
       - R0  ... accumulated signature
       - R2  ... CST_RAM_TARGET0 address 
       - R12 ... PRIMASK
    */
    /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
    /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_ForwardingTest1" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_THUMB2
    CST_TYPE(M33_Cst_ForwardingTest1, function)
M33_Cst_ForwardingTest1:

    PUSH    {R4-R12,R14}
    
    /*------------------------------------------------------------------------*/
    /* Test - preparation                                                     */
    /*------------------------------------------------------------------------*/
    LDR     R0,=PRESIGNATURE    /* Initialize results registers */
    
    
    /*------------------------------------------------------------------------*/
    /* Test - start                                                           */
    /*------------------------------------------------------------------------*/
    MRS     R12,PRIMASK                 /* Save PRIMASK  */
    
    MOV     R1,#0x00000000
    MSR     APSR_nzcvq,R1               /* Clear flags */
    
    LDR     R2,=CST_RAM_TARGET0        /* Do not use R2 !! */
    
    /**************************************************************************/
    /* Test forwarding from W0 & W1 write ports                               */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part1
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part1
    
    /**************************************************************************/
    /* Test forwarding from W2 & W3 write ports                               */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part2
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part2
    
    /**************************************************************************/
    /* Test forwarding from W0 & W2 write ports - MAC                         */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part3
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part3
    
    /**************************************************************************/
    /* Test forwarding from EX1(DP0) EX1(DP1)                                 */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part4
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test1_part4
    
    /**************************************************************************/
    /* Test forwarding from R13 register                                      */
    /**************************************************************************/
    MOV     R1,#0x00000001      /* Value to be forwarded (SP+R1) !! */
    BL      m33_cst_forwarding_test1_part5
    MOV     R1,#0x00000002      /* Value to be forwarded (SP+R1) !! */
    BL      m33_cst_forwarding_test1_part5
    MOV     R1,#0x00000003      /* Value to be forwarded (SP+R1) !! */
    BL      m33_cst_forwarding_test1_part5
    
    /*------------------------------------------------------------------------*/
    /* Test routine - End                                                     */
    /*------------------------------------------------------------------------*/    
m33_cst_forwarding_test1_end:
    /* Test result is returned in R0, according to the conventions */ 
    B       m33_cst_test_tail_end
    

    /*------------------------------------------------------------------------*/
    /* Test - internal routines                                               */
    /*------------------------------------------------------------------------*/
    /**************************************************************************/
    /* 1a) Forwarding from write ports W0->ISS                                */
    /* 1b) Forwarding from write ports W1->ISS                                */
    /**************************************************************************/
m33_cst_forwarding_test1_part1:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test1_loop1:
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ADD     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R3,R1,R10,ROR #1    /* Slot0 - Forward data to A0 read port !! */
    SUB     R4,R1,R10,ROR #2    /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    SUB     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R5,R10,R1,ROR #3    /* Slot0 - Forward data to B0 read port !! */
    ADD     R6,R10,R1,ROR #4    /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    ADD     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port */
    /* Interlock here to avoid Hazard !! */
    SUB     R7,R1,R10,ROR #5    /* Slot0 - Forward data to A0 read port !! */
    SUB     R8,R1,R10,ROR #6    /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0*/
    SUB     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port */
    /* Interlock here to avoid Hazard !! */
    ADD     R9,R10,R1,ROR #7    /* Slot0 - Forward data to B0 read port !! */
    ADD     R10,R10,R1,ROR #8   /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test1_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test1_loop1  
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 1c) Forwarding from write port W2->ISS                                 */
    /* 1d) Forwarding from write port W3->ISS                                 */
    /**************************************************************************/
m33_cst_forwarding_test1_part2:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test1_loop2:
    STR     R1,[R2]             /* Prepared data to be loaded */
    DSB
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R3,R1,R10,ROR #1    /* Slot0 - Forward data to A0 read port !! */
    ADD     R4,R1,R10,ROR #2    /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R5,R10,R1,ROR #3    /* Slot0 - Forward data to B0 read port !! */
    SUB     R6,R10,R1,ROR #4    /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    ADD     R7,R1,R10,ROR #5    /* Slot0 - Forward data to A0 read port !! */
    ADD     R8,R1,R10,ROR #6    /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    SUB     R9,R10,R1,ROR #7    /* Slot0 - Forward data to B0 read port !! */
    SUB     R10,R10,R1,ROR #8   /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test1_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test1_loop2
    
    POP     {R15}   /* Exit test subroutine */
    
      
    /**************************************************************************/
    /* 1e) Forwarding from write port W0(MAC)->ISS                            */
    /* - MAC result lower part [0:31] is connected to the W0 port             */
    /* 1f) Forwarding from write port W2(MAC)->ISS                            */
    /* - MAC result upper part [32:63] is connected to the W2 port            */
    /**************************************************************************/
m33_cst_forwarding_test1_part3: 
    MOV     R11,#10             /* Initialize loop counter */
    MOV     R9,#1
    PUSH    {R14}
m33_cst_forwarding_test1_loop3:
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    MUL     R10,R9,R1           /* Slot0 - MAC produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R3,R1,R10,ROR #11   /* Slot0 - Forward data to A0 read port !! */
    ADD     R4,R1,R10,ROR #12   /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    MUL     R10,R9,R1           /* Slot0 MAC produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R5,R10,R1,ROR #13   /* Slot0 - Forward data to B0 read port !! */
    SUB     R6,R10,R1,ROR #14   /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    UMULL   R7,R10,R1,R11       /* Slot0 - MAC produces result to W2 ports */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R7,R1,R10,ROR #15   /* Slot0 - Forward data to A0 read port !! */
    ADD     R8,R1,R10,ROR #16   /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    UMULL   R9,R10,R1,R11       /* Slot0 - MAC produces result to W2 ports */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R9,R10,R1,ROR #17   /* Slot0 - Forward data to B0 read port !! */
    SUB     R10,R10,R1,ROR #18  /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test1_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */      
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test1_loop3
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 2a) Forwarding from EX1(DP0) -> ISS                                    */
    /* 2b) Forwarding from EX1(DP1) -> ISS                                    */
    /**************************************************************************/
m33_cst_forwarding_test1_part4:
    PUSH    {R14}
    MOV     R11,#10             /* Initialize loop counter */
m33_cst_forwarding_test1_loop4:
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    REV     R10,R1              /* Slot0 - REV produces data at the end of EX1 */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R3,R1,R10,ROR #1    /* Slot0 - Forward data to A0 read port !! */     
    ADD     R4,R1,R10,ROR #3    /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    REV     R10,R1              /* Slot0 - REV produces data at the end of EX1 */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R5,R10,R1,ROR #5    /* Slot0 - Forward data to B0 read port !! */
    SUB     R6,R10,R1,ROR #7    /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    REV     R10,R1              /* Slot1 - REV produces data at the end of EX1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R7,R1,R10,ROR #9    /* Slot0 - Forward data to A0 read port !! */
    ADD     R8,R1,R10,ROR #11   /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    REV     R10,R1              /* Slot1 - REV produces data at the end of EX1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R9,R10,R1,ROR #13   /* Slot0 - Forward data to B0 read port !! */
    SUB     R10,R10,R1,ROR #15  /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test1_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */           
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test1_loop4
    
    POP     {R15}   /* Exit test subroutine */
    
    /**************************************************************************/
    /* R13 forwarding                                                         */
    /* - Forwarding logic must ensure bit0, bit1 masking !!                   */
    /**************************************************************************/
m33_cst_forwarding_test1_part5:
    MOV     R11,R13             /* Save SP register */
    STR     R1,[R2]             /* Prepared data to be loaded */
    DSB
    LSL     R10,R1,#4
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    LDR     R13,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R3,R13,R10,ROR #4   /* Slot0 - Forward data to B0 read port !! */
    ADD     R4,R13,R10,ROR #4   /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R13,[R2]            /* Slot1 - DPU produces result to the W2 port */
    /* Interlock here to avoid Hazard !! */
    SUB     R5,R13,R10,ROR #4   /* Slot0 - Forward data to B0 read port !! */
    SUB     R6,R13,R10,ROR #4   /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    ADD     R13,R1
    NOP
    /* Interlock here to avoid Hazard !! */
    ADD     R7,R13,R10,ROR #4   /* Slot0 - Forward data to B0 read port !! */
    ADD     R8,R13,R10,ROR #4   /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP
    ADD     R13,R1
    /* Interlock here to avoid Hazard !! */
    SUB     R9,R13,R10,ROR #4   /* Slot0 - Forward data to B0 read port !! */
    SUB     R10,R13,R10,ROR #4  /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MOV     R13,R11
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Continue - Update signature and return !! */
    
    
    /**************************************************************************/
    /* !! Update Signature !!                                                 */
    /**************************************************************************/
m33_cst_forwarding_test1_sum_r3_r10:
    ADC     R3,R3,R0,ROR #1
    EOR     R0,R0,R3
    ADC     R4,R4,R0,ROR #2
    EOR     R0,R0,R4
    ADC     R5,R5,R0,ROR #3
    EOR     R0,R0,R5
    ADC     R6,R6,R0,ROR #4
    EOR     R0,R0,R6
    ADC     R7,R7,R0,ROR #5
    EOR     R0,R0,R7
    ADC     R8,R8,R0,ROR #6
    EOR     R0,R0,R8
    ADC     R9,R9,R0,ROR #7
    EOR     R0,R0,R9
    ADC     R10,R10,R0,ROR #8
    EOR     R0,R0,R10
    
    BX      LR
    
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif /* CORTST_M33_FORWARD_ENABLE */
    
    CST_FILE_END

