/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
* This test focuses on fetch unit. Here we test that code which is modified
* during runtime is correctly fetched and then executed.
*
* Overall coverage:
* -----------------
* Covers DMB, DSB, ISB instructions - Dual issue
*  - DMB ensures that new code is stored before next data transfer begins.
*  - DSB ensures that new code is stored before next instruction execution.
*  - ISB ensures that new code is fetched.
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FETCH_ENABLE == 1U)
    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_FetchTest
    
    /* Symbols defined outside but used within current module */
    CST_EXTERN CST_RAM_TEST_CODE
    CST_EXTERN m33_cst_test_tail_end

    CST_SET(PRESIGNATURE,0xB6BF975C)

    /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
    CST_THUMB2
    /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label "M33_Cst_FetchTest" designates a function.
       In combination with the above specified ".thumb2" directive, this would cause setting
       the least significant bit to '1' within any pointer to this function,
       causing change to Thumb mode whenever this function is called. */
    CST_TYPE(M33_Cst_FetchTest, function)
M33_Cst_FetchTest:
    PUSH.W  {R4-R12,R14}
    
    /*------------------------------------------------------------------------*/
    /* Test - preparation                                                     */
    /*------------------------------------------------------------------------*/
    LDR     LR,=PRESIGNATURE
    
    /* Interrupts will be disabled, store PRIMASK to R12. */
    MRS     R12,PRIMASK     /* Do not use R12 until PRIMASK is restored !! */
    
    /* Load the SRAM memory address where the test code will be stored.
       This is start address of the "m33_cst_ram_test_code" section. */
    LDR     R0,=CST_RAM_TEST_CODE
    ORR     R1,R0,#1        /* Load RAM address with THUMB bit set to R1*/
    
    /* Set branch address for fault case */
    LDR     R11,=m33_cst_fetch_test_end
    ORR     R11,R11,#1
    
    /***************************************************************************
    * Instructions:
    *   - DSB, ISB - Dual issue
	*
    *   Note: Encoding T1 (32-bit)
    ***************************************************************************/
    
    /*  Following code is stored to SRAM. This code will be partially modified 
        during the runtime by the STR.W R8,[R0,#20] instruction.
        
        This code is checking DSB and ISB instructions functionality
        
        For dual issue testing, code is repeated 2 times
         1. pass
          - opcode "BX R11, BX R10" at offset #20 is changed to "BX R1 , BX R11"
          - DSB is in slot 0, ISB in slot 1
         2. pass
          - opcode "BX R1 , BX R11" at offset #20 is changed to "BX R10, BX R11"
          - DSB is in slot 1, ISB in slot 0
        
    Test code in SRAM
    Branch   Offset  Instruction       Description
    ----------------------------------------------------------------------------
    BX R1 -> #0      ADDW  R8,R8,#0x48 Modify opcode in R8, BX R1 -> BX R10
    enter -> #4      ROR   LR,LR,#1    Update signature
             #8      STR.W R8,[R0,#28] Store modified opcode at offset #28
             #12     DSB               Wait until end of data transfer
             #16     STR.N R0,[R3]     Clean data cache by MVA to PoU
             #18     STR.N R0,[R4]     Invalidate instruction cache by MVA
             #20     DSB               Wait until end of data transfer
             #24     ISB               Flush pipeline, fetch instructions again
    
    Next instructions vary depending on how many times STR.W(#8) was called
    0x       #28     BX    R11         -> branch to end label (error)
    1x       #28     BX    R1          -> branch to offset #0
    2x       #28     BX    R10         -> branch to address in R10 (success)
    
    0x       #30     BX    R10         This branch is used to check if data was
    1x,2x    #30     BX    R11         successfully stored (by branch here)
    ----------------------------------------------------------------------------
    */
    
    /* Prepare test code for checking DSB and ISB instructions
    1. pass : DSB in slot 1, ISB in slot 0
    2. pass : DSB in slot 0, ISB in slot 1 */
    LDR     R3,=0x0848F208  /* Load to R3  opcode ADDW  R8,R8,#0x48        */
    LDR     R4,=0x0E7EEA4F  /* Load to R4  opcode ROR   LR,LR,#1           */
    LDR     R5,=0x801CF8C0  /* Load to R5  opcode STR.W R8,[R0,#28]        */
    LDR     R6,=0x8F4FF3BF  /* Load to R6  opcode DSB                      */
    LDR     R7,=0x60206018  /* Load to R7  opcode STR R0,[R3], STR R0,[R4] */
    LDR     R8,=0x8F4FF3BF  /* Load to R8  opcode DSB                      */
    LDR     R9,=0x8F6FF3BF  /* Load to R9  opcode ISB                      */
    LDR     R10,=0x47504758 /* Load to R10 opcode BX R11, BX R10           */
    
    ADD     R2,R0,#30       /* Set enter address for RAM code execution */
    
    /* Disable all interrupts to ensure that test code running from SRAM is not
    interrupted to avoid scenario that pipeline is flushed by external interrupt
    and fetched again when returns from interrupt. */
    CPSID   i
    
    /* Store self-modifiable code into "m33_cst_ram_test_code" SRAM section */
    STM.W   R0,{R3-R10}
    DSB              /* Ensure the data has been written to the cache */

    LDR     R3,=M33_DCCMVAU
    LDR     R4,=M33_ICIMVAU
    
    /* Set branch address for fault-free case */
    LDR     R10,=m33_cst_fetch_test_dsb
    ORR     R10,R10,#1
    
    STR     R0,[R3]  /* Clean data cache by MVA to point of unification (PoU) */
    STR     R0,[R4]  /* Invalidate instruction cache by MVA to PoU */
    DSB              /* Ensure completion of the invalidations */
    ISB              /* Synchronize fetched instruction stream */
    
    /* Verify code in SRAM memory */
    MOV     PC,R2           /* Execute test code in SRAM                  */
                            /*  - should branch to m33_cst_fetch_test_dsb */

m33_cst_fetch_test_dsb:
    MSR     PRIMASK,R12     /* Restore PRIMASK from R12 to enable interrupts */
    
    LDR     R8,=0x47584708  /* Load to R8 opcodes for "BX R1" and "BX R11" */
    
    ADD     R2,R0,#4        /* Set enter address for RAM code execution */
    
    /* Set branch address for fault-free case */
    LDR     R10,=m33_cst_fetch_test_dsb_ok
    ORR     R10,R10,#1

    CPSID   i               /* Disable all interrupts */
    
    MOV     PC,R2           /* Execute test code in SRAM                     */
                            /*  - should branch to m33_cst_fetch_test_dsb_ok */

    /*------------------------------------------------------------------------*/
    /* Test - OK - This label is reached if all tests succeeded               */
    /*------------------------------------------------------------------------*/
m33_cst_fetch_test_dmb_ok:
    ROR     LR,LR,#1        /* Update signature */
    B       m33_cst_fetch_test_end

    /***************************************************************************
    * Instructions:
    *   - DMB, ISB - Dual issue
	*
    *   Note: Encoding T1 (32-bit)
    ***************************************************************************/
m33_cst_fetch_test_dsb_ok:
    MSR     PRIMASK,R12     /* Restore PRIMASK from R12 to enable interrupts */
    ROR     LR,LR,#1        /* Update signature */
    
    /* Prepare test code for checking DMB and ISB instructions
    1. pass : DMB and ISB in slot 0
    2. pass : DMB and ISB in slot 1 */
    LDR     R3,=0x0948F209  /* Load to R3  opcode ADDW  R9,R9,#0x48        */
    LDR     R4,=0x0E7EEA4F  /* Load to R4  opcode ROR   LR,LR,#1           */
    LDR     R5,=0x901CF8C0  /* Load to R5  opcode STR.W R9,[R0,#28]        */
    LDR     R6,=0x8F5FF3BF  /* Load to R6  opcode DMB                      */
    LDR     R7,=0x60206018  /* Load to R7  opcode STR R0,[R3], STR R0,[R4] */
    LDR     R8,=0x8F4FF3BF  /* Load to R8  opcode DSB                      */
    LDR     R9,=0x8F6FF3BF  /* Load to R9  opcode ISB                      */
    LDR     R10,=0x47504758 /* Load to R10 opcode BX R11, BX R10           */

    ADD     R2,R0,#30       /* Set enter address for SRAM code execution */
    
    CPSID   i               /* Disable all interrupts */
    
    /* Store self-modifiable code into "m33_cst_ram_test_code" SRAM section */
    STM.W   R0,{R3-R10}
    DSB              /* Ensure the data has been written to the cache */

    LDR     R3,=M33_DCCMVAU
    LDR     R4,=M33_ICIMVAU

    /* Set branch address for fault-free case */
    LDR     R10,=m33_cst_fetch_test_dmb
    ORR     R10,R10,#1

    STR     R0,[R3]  /* Clean data cache by MVA to point of unification (PoU) */
    STR     R0,[R4]  /* Invalidate instruction cache by MVA to PoU */
    DSB              /* Ensure completion of the invalidations */
    ISB              /* Synchronize fetched instruction stream */
    
    MOV     PC,R2           /* Execute test code in SRAM                  */
                            /*  - should branch to m33_cst_fetch_test_dmb */
    
m33_cst_fetch_test_dmb:
    MSR     PRIMASK,R12     /* Restore PRIMASK from R12 to enable interrupts */
    
    LDR     R9,=0x47584708  /* Load to R9 opcodes for "BX R1" and "BX R11" */
    
    ADD     R2,R0,#4        /* Set enter address for RAM code execution */
    
    /* Set branch address for fault-free case */
    LDR     R10,=m33_cst_fetch_test_dmb_ok
    ORR     R10,R10,#1
    
    CPSID   i               /* Disable all interrupts */
    
    MOV     PC,R2           /* Execute test code in SRAM                     */
                            /*  - should branch to m33_cst_fetch_test_dmb_ok */
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
m33_cst_fetch_test_end:
    MSR     PRIMASK,R12     /* Restore PRIMASK from R12 */
    MVN     R0,LR           /* Test result is returned in R0 */
    B       m33_cst_test_tail_end

    CST_ALIGN_BYTES_4
    CST_LTORG  /* Marks the current location for dumping psuedoinstruction pools (literal pool)
                   (containing numeric values for used symbolic names used within LDR instruction).
                   It is 4-byte aligned, as 2-byte alignment causes incorrect work. */    

#endif /* CORTST_M33_FETCH_ENABLE */

    CST_FILE_END

