/**
*   @file    CddUart.c
*   @version
*   @brief   AUTOSAR CddUart module interface
*   @details API implementation for CddUart driver
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : uart
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.3.0
*
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
*                                        INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "CddUart_Lld_LinFlexD.h"
#include "SchM_CddUart.h"
#if(CDDUART_DMA_USED == STD_ON)
#include "CddDma.h"
#endif
/*==================================================================================================
*                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/

#define CDDUART_VENDOR_ID_LLD_FLEXDUART_C                      (180)
#define CDDUART_AR_RELEASE_MAJOR_VERSION_LLD_FLEXUART_C        (4)
#define CDDUART_AR_RELEASE_MINOR_VERSION_LLD_FLEXUART_C        (4)
#define CDDUART_AR_RELEASE_REVISION_VERSION_LLD_FLEXUART_C     (0)
#define CDDUART_SW_MAJOR_VERSION_LLD_FLEXUART_C                (2)
#define CDDUART_SW_MINOR_VERSION_LLD_FLEXUART_C                (3)
#define CDDUART_SW_PATCH_VERSION_LLD_FLEXUART_C                (0)
/*==================================================================================================
                                      FILE VERSION CHECKS
==================================================================================================*/
/* Check if current file and CddUart header file are of the same Autosar version */
#if (CDDUART_VENDOR_ID_LLD_FLEXDUART_C != CDDUART_VENDOR_ID_LINFLEXDUART_CFG)
#error "CddUart_Lld_LinFlexDUart.c and CddUart_Lld_LinFlexDUartCfg.h have different vendor ids"
#endif

/* Check if source file and CddUart header file are of the same Autosar version */
#if ((CDDUART_AR_RELEASE_MAJOR_VERSION_LLD_FLEXUART_C != CDDUART_AR_RELEASE_MAJOR_VERSION_LINFLEXDUART_CFG) || \
     (CDDUART_AR_RELEASE_MINOR_VERSION_LLD_FLEXUART_C != CDDUART_AR_RELEASE_MINOR_VERSION_LINFLEXDUART_CFG) || \
     (CDDUART_AR_RELEASE_REVISION_VERSION_LLD_FLEXUART_C != CDDUART_AR_RELEASE_REVISION_VERSION_LINFLEXDUART_CFG) \
    )
#error "AutoSar Version Numbers of CddUart_Lld_LinFlexDUart.c  and CddUart_Lld_LinFlexDUartCfg.h are different"
#endif

/* Check if source file and CddUart header file are of the same Software version */
#if ((CDDUART_SW_MAJOR_VERSION_LLD_FLEXUART_C != CDDUART_SW_MAJOR_VERSION_LINFLEXDUART_CFG) || \
     (CDDUART_SW_MINOR_VERSION_LLD_FLEXUART_C != CDDUART_SW_MINOR_VERSION_LINFLEXDUART_CFG) || \
     (CDDUART_SW_PATCH_VERSION_LLD_FLEXUART_C != CDDUART_SW_PATCH_VERSION_LINFLEXDUART_CFG) \
    )
#error "Software Version Numbers of CddUart_Lld_LinFlexDUart.c  and CddUart_Lld_LinFlexDUartCfg.h are different"
#endif
/*==================================================================================================
                                       GLOBAL VARIABLES
==================================================================================================*/
#if(CDDUART_DMA_USED == STD_ON)
extern const CddDma_TransferConfigType *DmaChannelTransferConfigArray[CDDDMA_TRANSFER_CONFIG_COUNT];
#endif
/**
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 2985 Rule-2.2: This operation is redundant. The value of the result is always that of the left-hand operand.
*/
/*==================================================================================================
                                       LOCAL VARIABLES
==================================================================================================*/
#define CDDUART_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "CddUart_MemMap.h"
/* Array of pointers to UART driver runtime state structures */
CDDUART_VAR static LinFlexd_UartStateType UartLINFlexDStatePtr[LINFlexD_INSTANCE_COUNT];
CDDUART_VAR static uint32 LinFlexUart_ClockFrequency[LINFlexD_INSTANCE_COUNT];
#if(CDDUART_DMA_USED == STD_ON)
CDDUART_VAR static CddDma_TransferConfigType RxDmaTransferConfig;
CDDUART_VAR static CddDma_TransferConfigType TxDmaTransferConfig;
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
/* Dummy transfer configuration is used for Rx channel trigger after Rx minor loop done each time */
CDDUART_VAR static CddDma_LoopTransferConfigType RxChannelTrigger;
CDDUART_VAR static CddDma_LoopTransferConfigType TxChannelTrigger;
CDDUART_VAR static CddDma_TransferConfigType DummyDmaTransferConfig;
#endif /* CPU_YTM32B1ME0 || CPU_YTM32B1MD1 */
#endif /* CDDUART_DMA_USED == STD_ON */
#define CDDUART_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "CddUart_MemMap.h"

#define CDDUART_START_SEC_CONST_UNSPECIFIED
#include "CddUart_MemMap.h"
/*! @brief Table of Base addresses for LINFlexD instances. */
CDDUART_CONST static volatile LINFlexD_Type *const UartLINFlexDBase[LINFlexD_INSTANCE_COUNT] = LINFlexD_BASE_PTRS;
#define CDDUART_STOP_SEC_CONST_UNSPECIFIED
#include "CddUart_MemMap.h"

#if((CDDUART_DMA_USED == STD_ON) && (defined(CPU_YTM32B1ME0) || defined(CPU_YTM32B1MD1)))
#define DUMMY_ADDRESS ((uint32)0x4007b040U) /**< @brief CIM DIEINFO address */
#endif /* CDDUART_DMA_USED == STD_ON && (defined(CPU_YTM32B1ME0) || defined(CPU_YTM32B1MD1)) */
/*==================================================================================================
*                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

static CddUart_StatusType LinFlexD_Lld_StartSendUsingInt(uint32 Instance, uint8 *TxBuff, uint32 TxSize);
static CddUart_StatusType LinFlexD_Lld_StartReceiveUsingInt(uint32 Instance, uint8 *RxBuff, uint32 RxSize);
static void LinFlexD_Lld_CompleteSendUsingInt(uint32 Instance);
static void LinFlexD_Lld_CompleteRxUsingInt(uint32 Instance);
static void LinFlexD_Lld_PutData(uint32 Instance);
static void LinFlexD_Lld_GetData(uint32 Instance);
static inline void LinFlexD_Lld_EnterInitMode(volatile LINFlexD_Type *Base);
#if (STD_ON == CDDUART_DMA_USED)
static void LinFlexD_Lld_FlushRxFifo(volatile LINFlexD_Type *Base, uint8 FifoType);
static CddUart_StatusType LinFlexD_Lld_StartSendUsingDma(uint32 Instance, uint8 *TxBuff, uint32 TxSize);
static CddUart_StatusType LinFlexD_Lld_StartReceiveUsingDma(uint32 Instance, uint8 *RxBuff, uint32 RxSize);
static void LinFlexD_Lld_CompleteSendUsingDma(void *Parameter);
static void LinFlexD_Lld_CompleteRxUsingDma(void *Parameter);
static void UART_Lld_DmaTxError(void *Parameter);
static void UART_Lld_DmaRxError(void *Parameter);
static void LinFlexD_Lld_StopRxUsingDma(uint32 Instance);
#endif
static void LinFlexD_Lld_RxIRQHandler(uint8 Instance);
static void LinFlexD_Lld_TxIRQHandler(uint8 Instance);
static void LinFlexD_Lld_ErrIRQHandler(uint32 Instance);
#define CDDUART_START_SEC_CODE
#include "CddUart_MemMap.h"
/*==================================================================================================
*                                    LOCAL FUNCTION
==================================================================================================*/
/*!
 * @brief Requests LINFlexD module to enter init mode.
 *
 * This function enters initialization mode.
 *
 * @param Base LINFlexD Base pointer.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_EnterInitMode(volatile LINFlexD_Type *Base)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_00();
    RegValTemp = Base->LINCR1;
    /* Exit sleep mode */
    RegValTemp &= ~LINFlexD_LINCR1_SLEEP_MASK;
    /* Enter init mode. */
    RegValTemp |= LINFlexD_LINCR1_INIT(1U);
    Base->LINCR1 = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_00();
}

/*!
 * @brief Requests LINFlexD module to enter normal mode.
 *
 * This function enters normal mode. *
 *
 * @param Base LINFlexD Base pointer.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_EnterNormalMode(volatile LINFlexD_Type *Base)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_01();
    RegValTemp = Base->LINCR1;
    /* Exit sleep mode */
    RegValTemp &= ~LINFlexD_LINCR1_SLEEP_MASK;
    RegValTemp &= ~LINFlexD_LINCR1_INIT_MASK;
    /* Exit init mode. */
    Base->LINCR1 = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_01();
}

/*!
 * @brief Sets the LINFlexD mode.
 *
 * This function sets LIN or UART mode, based on the parameter received.
 *
 * @param Base LINFlexD Base pointer.
 * @param Mode LINFlexD mode - UART/LIN.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetMode(volatile LINFlexD_Type *Base, LinFlexd_ModeType Mode)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_02();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_UART_MASK;
    RegValTemp |= LINFlexD_UARTCR_UART((uint32)Mode);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_02();
}

/*!
 * @brief Returns the current LINFlexD state.
 *
 * This function returns the state of the LINFlexD Instance. The possible states are:
 *    - Sleep mode
 *    - Init mode
 *    - Idle mode
 *    - Sync Break Reception/Transmission
 *    - Sync Delimiter Reception/Transmission
 *    - Sync Field Reception/Transmission
 *    - Identifier Field Reception/Transmission
 *    - Header Reception/Transmission
 *    - Data Reception/Transmission
 *    - Checksum Reception/Transmission
 *
 * @param Base LINFlexD Base pointer.
 * @return LINFlexD mode - UART/LIN.
 */
CDDUART_FUNC static inline LinFlexd_HW_StateType LinFlexD_Lld_GetLinState(volatile LINFlexD_Type const *Base)
{
    uint8 State = (uint8)((Base->LINSR & LINFlexD_LINSR_LINS_MASK) >> LINFlexD_LINSR_LINS_SHIFT);
    return (LinFlexd_HW_StateType)State;
}

/*!
 * @brief Sets the word length.
 *
 * This function sets the word length in UART mode.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Length UART mode word length.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetUartWordLength(volatile LINFlexD_Type *Base, LinFlexd_UartWordLengthType Length)
{
    uint32 RegValTemp;
    uint32 Temp = 0;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_03();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_WL0_MASK;
    Temp = (uint32)((uint32)Length & 1U);
    RegValTemp |= LINFlexD_UARTCR_WL0(Temp);/*PRQA S 2985*/
    RegValTemp &= ~LINFlexD_UARTCR_WL1_MASK;
    RegValTemp |= LINFlexD_UARTCR_WL1(((uint32)Length >> 1U) & 1U);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_03();
}

/*!
 * @brief Enables/disables parity bit.
 *
 * This function enables or disables the parity bit from UART bytes.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable TRUE - Enable parity queuing/checking, FALSE - disable parity queuing/checking.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetParityControl(volatile LINFlexD_Type *Base, boolean Enable)
{
    uint32 RegValTemp;
    uint8 Temp = 0;
    if (TRUE == Enable)
    {
        Temp = 1;
    }
    else
    {
        Temp = 0;
    }
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_04();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_PCE_MASK;
    RegValTemp |= LINFlexD_UARTCR_PCE(Temp);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_04();
}

/*!
 * @brief Sets parity type.
 *
 * This function sets the type of parity to be used for UART bytes. Available options are:
 *    - odd
 *    - even
 *    - always 0
 *    - always 1
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param ParityType the type of parity.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetParityType(volatile LINFlexD_Type *Base, LinFlexd_UartParityType ParityType)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_05();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_PC0_MASK;
    RegValTemp |= LINFlexD_UARTCR_PC0((uint32)ParityType & 1U);/*PRQA S 2985*/
    RegValTemp &= ~LINFlexD_UARTCR_PC1_MASK;
    RegValTemp |= LINFlexD_UARTCR_PC1(((uint32)ParityType >> 1U) & 1U);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_05();
}

/*!
 * @brief Enables/disables transmitter.
 *
 * This function enables or disables the UART transmitter, Based on the parameter received.
 * Should be called only when the module is in UART mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable Enable/disable transmitter.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetTransmitterState(volatile LINFlexD_Type *Base, boolean Enable)
{
    uint32 RegValTemp;
    uint8 Temp = 0;
    if (TRUE == Enable)
    {
        Temp = 1;
    }
    else
    {
        Temp = 0;
    }
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_06();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_TxEn_MASK;
    RegValTemp |= LINFlexD_UARTCR_TxEn(Temp);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_06();
}

#if (STD_ON == CDDUART_DMA_USED)
CDDUART_FUNC static inline void LinFlexD_Lld_SetTransmitterDMA(volatile LINFlexD_Type *Base, boolean Enable)
{
    if (TRUE == Enable)
    {
        Base->DMATXE = 1U;
    }
    else
    {
        Base->DMATXE = 0U;
    }
}

CDDUART_FUNC static inline void LinFlexD_Lld_SetRxDMA(volatile LINFlexD_Type *Base, boolean Enable)
{
    if (TRUE == Enable)
    {
        Base->DMARXE = 1U;
    }
    else
    {
        Base->DMARXE = 0U;
    }
}
#endif

/*!
 * @brief Enables/disables receiver.
 *
 * This function enables or disables the UART receiver, based on the parameter received.
 * Should be called only when the module is in UART mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable Enable/disable receiver.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetReceiverState(volatile LINFlexD_Type *Base, boolean Enable)
{
    uint32 RegValTemp;
    uint8 Temp = 0;
    if (TRUE == Enable)
    {
        Temp = 1;
    }
    else
    {
        Temp = 0;
    }
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_07();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_RxEn_MASK;
    RegValTemp |= LINFlexD_UARTCR_RxEn(Temp);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_07();
}

/*!
 * @brief Sets the transmission mode (FIFO/Buffer).
 *
 * This function sets the transmission mode (FIFO/Buffer).
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Mode tx mode: FIFO/Buffer.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetTxMode(volatile LINFlexD_Type *Base, LinFlexd_UartRxTxType Mode)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_08();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_TFBM_MASK;
    RegValTemp |= LINFlexD_UARTCR_TFBM((uint32)Mode);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_08();
}

/*!
 * @brief Sets the reception mode (FIFO/Buffer).
 *
 * This function sets the reception mode (FIFO/Buffer).
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Mode rx mode: FIFO/Buffer.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetRxMode(volatile LINFlexD_Type *Base, LinFlexd_UartRxTxType Mode)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_09();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_RFBM_MASK;
    RegValTemp |= LINFlexD_UARTCR_RFBM((uint32)Mode);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_09();
}

/*!
 * @brief Sets the Monitor idle state.
 *
 * @details This function sets the uart Monitor idle state.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable Enable/disable receiver.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetMonitorIdleState(volatile LINFlexD_Type *Base,  boolean Enable)
{
    uint32 RegValTemp;
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~(LINFlexD_UARTCR_MIS_MASK);
    RegValTemp |= LINFlexD_UARTCR_MIS(Enable ? 1UL : 0UL);
    Base->UARTCR = RegValTemp;
}

/*!
 * @brief Sets the Monitor idle state.
 *
 * @details This function sets the uart disable timeout.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable Enable/disable receiver.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetDisableTimeout(volatile LINFlexD_Type *Base,  boolean Enable)
{
    uint32 RegValTemp;
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~(LINFlexD_UARTCR_DTU_MASK);
    RegValTemp |= LINFlexD_UARTCR_DTU(Enable ? 1UL : 0UL);
    Base->UARTCR = RegValTemp;
}

/*!
 * @brief Sets the timeout value.
 *
 * @details This function sets the timeout value.
 *
 * @param Base LINFlexD Base pointer.
 * @param TimeoutValue timeout value.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetTimeoutValue(volatile LINFlexD_Type *Base, uint16 TimeoutValue)
{
    Base->UARTPTO = LINFlexD_UARTPTO_PTO(TimeoutValue);
}

/*!
 * @brief Sets the expected number of frames.
 *
 * @details This function sets the expected number of frames. If the DTU bit is set, then the UART timeout counter is reset after the configured
 *          number of frames have been received
 *
 * @param Base LINFlexD Base pointer.
 * @param Frames expected number of frames.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetExpectedFrame(volatile LINFlexD_Type *Base, uint8 Frames)
{
    uint32 RegValTemp;
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_NEF_MASK;
    RegValTemp |= LINFlexD_UARTCR_NEF(Frames);
    Base->UARTCR = RegValTemp;
}

/*!
 * @brief Sets the reception data field length/FIFO depth.
 *
 * This function sets either the reception data field length or the number of bytes in the Rx FIFO, according to
 * the current reception mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Length data field length/FIFO depth.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetRxDataFieldLength(volatile LINFlexD_Type *Base, uint8 Length)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_10();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_RDFL_RFC_MASK;
    RegValTemp |= LINFlexD_UARTCR_RDFL_RFC(Length);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_10();
}

/*!
 * @brief Sets the transmission data field length/FIFO depth.
 *
 * This function sets either the transmission data field length or the number of bytes in the Tx FIFO, according to
 * the current transmission mode.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Length data field length/FIFO depth.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetTxDataFieldLength(volatile LINFlexD_Type *Base, uint8 Length)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_11();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_TDFL_TFC_MASK;
    RegValTemp |= LINFlexD_UARTCR_TDFL_TFC(Length);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_11();
}

/*!
 * @brief Sets the number of stop bits for Rx.
 *
 * This function sets the number of stop bits for received bytes.
 *
 * @param Base LINFlexD Base pointer.
 * @param StopBitsCount number of stop bits.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetRxStopBitsCount(volatile LINFlexD_Type *Base, LinFlexd_UartStopBitsType StopBitsCount)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_12();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_SBUR_MASK;
    RegValTemp |= LINFlexD_UARTCR_SBUR(StopBitsCount);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_12();
}

/*!
 * @brief Sets fractional baud rate.
 *
 * This function configures the bits that decide the fractional part of the LIN Baud Rate.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Fbr fractional baud rate.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetFractionalBaudRate(volatile LINFlexD_Type *Base, uint8 Fbr)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_13();
    RegValTemp = Base->LINFBRR;
    RegValTemp &= ~LINFlexD_LINFBRR_FBR_MASK;
    RegValTemp |= LINFlexD_LINFBRR_FBR(Fbr);/*PRQA S 2985*/
    Base->LINFBRR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_13();
}

/*!
 * @brief Gets fractional baud rate.
 *
 * This function returns the bits that decide the fractional part of the LIN Baud Rate.
 *
 * @param Base LINFlexD Base pointer.
 * @return fractional baud rate.
 */
CDDUART_FUNC static inline uint8 LinFlexD_Lld_GetFractionalBaudRate(volatile LINFlexD_Type const *Base)
{
    return (uint8)((Base->LINFBRR & LINFlexD_LINFBRR_FBR_MASK) >> LINFlexD_LINFBRR_FBR_SHIFT);
}

/*!
 * @brief Sets integer baud rate.
 *
 * This function configures the bits that decide the integer part of the LIN Baud Rate.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Ibr integer baud rate.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_Lld_SetIntegerBaudRate(volatile LINFlexD_Type *Base, uint32 Ibr)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_14();
    RegValTemp = Base->LINIBRR;
    RegValTemp &= ~LINFlexD_LINIBRR_IBR_MASK;
    RegValTemp |= LINFlexD_LINIBRR_IBR(Ibr);/*PRQA S 2985*/
    Base->LINIBRR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_14();
}

/*!
 * @brief Gets integer baud rate.
 *
 * This function returns the bits that decide the integer part of the LIN Baud Rate.
 *
 * @param Base LINFlexD Base pointer.
 * @return integer baud rate.
 */
CDDUART_FUNC static inline uint32 LinFlexD_Lld_GetIntegerBaudRate(volatile LINFlexD_Type const *Base)
{
    return ((Base->LINIBRR & LINFlexD_LINIBRR_IBR_MASK) >> LINFlexD_LINIBRR_IBR_SHIFT);
}

/*!
 * @brief Sets the number of stop bits for Tx.
 *
 * This function sets the number of stop bits for transmitted bytes.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param DefaultOffset - TRUE - use the memory map defined in the header file to access the register;
 *                      - FALSE - use custom memory map to access the register.
 * @param StopBitsCount number of stop bits.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetTxStopBitsCount(volatile LINFlexD_Type *Base, LinFlexd_UartStopBitsType StopBitsCount)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_15();
    RegValTemp = Base->GCR;
    RegValTemp &= ~LINFlexD_GCR_STOP_MASK;
    RegValTemp |= LINFlexD_GCR_STOP(StopBitsCount);
    Base->GCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_15();
}

/*!
 * @brief Clears an UART interrupt flag.
 *
 * This function clears the UART status flag received as parameter.
 *
 * @param Base LINFlexD Base pointer.
 * @param Flag UART status flag.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_ClearStatusFlag(volatile LINFlexD_Type *Base, LinFlexD_UartStatusType Flag)
{
    Base->UARTSR = (uint32)Flag;
}

/*!
 * @brief Returns an UART interrupt flag.
 *
 * This function returns the value of the UART status flag received as parameter.
 *
 * @param Base LINFlexD Base pointer.
 * @param Flag LIN status flag.
 * @return TRUE - UART status flag set, FALSE - UART status flag reset.
 */
CDDUART_FUNC static inline boolean LinFlexD_Lld_GetStatusFlag(volatile LINFlexD_Type const *Base, LinFlexD_UartStatusType Flag)
{
    boolean State = FALSE;
    if ((Base->UARTSR & (uint32)Flag) != 0U)
    {
        State = TRUE;
    }
    return State;
}

/*!
 * @brief Enables/disables an UART interrupt.
 *
 * This function configures whether the UART event will trigger an interrupt.
 *
 * @param Base LINFlexD Base pointer.
 * @param IntSrc UART interrupt source.
 * @param Enable TRUE - Enable interrupt, FALSE - disable interrupt.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetInterruptMode(volatile LINFlexD_Type *Base, LinFlexd_InterruptType IntSrc, boolean Enable)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_16();
    RegValTemp = Base->LINIER;
    if (TRUE == Enable)
    {
        RegValTemp |= (uint32)IntSrc;
    }
    else
    {
        RegValTemp &= ~(uint32)IntSrc;
    }
    Base->LINIER = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_16();
}

/*!
 * @brief Returns the state of an UART interrupt.
 *
 * This function returns whether the UART event will trigger an interrupt.
 *
 * @param Base LINFlexD Base pointer.
 * @param IntSrc UART interrupt source.
 * @return TRUE - interrupt enabled, FALSE - interrupt disabled.
 */
CDDUART_FUNC static inline boolean LinFlexD_Lld_IsInterruptEnabled(volatile LINFlexD_Type const *Base, uint32 IntSrc)
{
    boolean State = FALSE;
    if ((Base->LINIER & IntSrc) != 0U)
    {
        State = TRUE;
    }
    return State;
}

/*!
 * @brief Sets the first byte of the tx data buffer.
 *
 * This function writes one byte to the tx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @param Data data byte.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetTxDataBuffer1Byte(volatile LINFlexD_Type *Base, uint8 Data)
{
    Base->DATA.DATA8[0] = Data;
}

/*!
 * @brief Sets the first half-word of the tx data buffer.
 *
 * This function writes two bytes to the tx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @param Data data half-word.
 */
CDDUART_FUNC static inline void LinFlexD_Lld_SetTxDataBuffer2Bytes(volatile LINFlexD_Type *Base, uint16 Data)
{
    Base->DATA.DATA16[0] = Data;
}

/*!
 * @brief Gets the first byte of the rx data buffer.
 *
 * This function retrieves one byte from the rx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @return data byte.
 */
CDDUART_FUNC static inline uint8 LinFlexD_Lld_GetRxDataBuffer1Byte(volatile LINFlexD_Type const *Base)
{
    return Base->DATA.DATA8[4];
}

/*!
 * @brief Gets the first half-word of the rx data buffer.
 *
 * This function retrieves two bytes from the rx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @return data half-word.
 */
CDDUART_FUNC static inline uint16 LinFlexD_Lld_GetRxDataBuffer2Bytes(volatile LINFlexD_Type const *Base)
{
    return Base->DATA.DATA16[2];
}

/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
/* implements LinFlexD_Lld_SetBaudRate Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_SetBaudRate(uint32 Instance, uint32 FlexUartClock, uint32 Baudrate)
{
    float32 Prescaler;
    float32 Fraction;
    uint32 Mantissa;
    uint32 FractionDenominator;
    uint8 FractionNumerator;
    uint32 UartSourceClock = FlexUartClock;
    volatile LINFlexD_Type *Base;
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    boolean ResetIdle = FALSE;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else if ((TRUE == UartLINFlexDStatePtr[Instance].IsTxBusy)
             || (TRUE == UartLINFlexDStatePtr[Instance].IsRxBusy))
    {
        UartStatus = CDDUART_STATUS_OPERATION_ONGOING;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        LinFlexUart_ClockFrequency[Instance] = FlexUartClock;
        /* Compute the values for baud rate divider Mantissa and Fraction */
        Prescaler = (float32)UartSourceClock / ((float32)Baudrate * (float32)DEFAULT_OSR);
        Mantissa = (uint32)Prescaler;
        Fraction = Prescaler - (float32)Mantissa;
        FractionDenominator = ((uint32)1U << (uint32)BAUDRATE_FRACTION_WIDTH);
        float32 Float32Numerator = (Fraction * (float32)FractionDenominator) + 0.5F;
        FractionNumerator = (uint8)(Float32Numerator);
        if (FractionNumerator == FractionDenominator)
        {
            FractionNumerator = 0;
            ++Mantissa;
        }
        if (LinFlexD_Lld_GetLinState(Base) != LINFLEXD_STATE_INIT)
        {
            /* Request init mode and wait until the mode entry is complete */
            LinFlexD_Lld_EnterInitMode(Base);
            while (LinFlexD_Lld_GetLinState(Base) != LINFLEXD_STATE_INIT)
            {
                /*do nothing*/
            }
            ResetIdle = TRUE;
        }
        /* Write the computed values to registers */
        LinFlexD_Lld_SetIntegerBaudRate(Base, Mantissa);
        LinFlexD_Lld_SetFractionalBaudRate(Base, FractionNumerator);
        if (TRUE == ResetIdle)
        {
            /* Enter normal mode */
            LinFlexD_Lld_EnterNormalMode(Base);
        }
    }
    return UartStatus;
}

/* implements LinFlexD_Lld_GetBaudRate Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_GetBaudRate(uint32 Instance, uint32 *Baudrate)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    uint32 Mantissa;
    uint8 Fraction;
    uint32 UartSourceClock;
    LINFlexD_Type const volatile *Base;
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == Baudrate))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        UartSourceClock = LinFlexUart_ClockFrequency[Instance];
        /* Get the baud rate divider integer and fractional parts from registers */
        Mantissa = LinFlexD_Lld_GetIntegerBaudRate(Base);
        Fraction = LinFlexD_Lld_GetFractionalBaudRate(Base);
        /* Compute the baud rate Based on protocol clock value */
        *Baudrate = UartSourceClock / ((Mantissa * DEFAULT_OSR) + (uint32)Fraction);
    }
    return UartStatus;
}
/* implements LinFlexD_Lld_Init Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_Init(uint32 Instance, uint32 FlexUartClock, const LinFlexd_UartUserConfigType *UartUserConfig)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartStatePtr = &UartLINFlexDStatePtr[Instance];
    uint32 Idx = 0;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else if (NULL_PTR == UartUserConfig)
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        /* Clear the state struct for this Instance. */
        uint8 *ClearStructPtr = (uint8 *)UartStatePtr;
        for (; Idx < sizeof(LinFlexd_UartStateType); ++Idx)
        {
            ClearStructPtr[Idx] = 0;
        }
        /* Save runtime structure pointer.*/
        UartLINFlexDStatePtr[Instance] = (*UartStatePtr);
        UartStatePtr->IdleLineIntEn = UartUserConfig->IdleLineIntEn;
        UartStatePtr->IdleTimeoutValue = UartUserConfig->IdleTimeoutValue;
        /* Request init mode and wait until the mode entry is complete */
        LinFlexD_Lld_EnterInitMode(Base);
        while (LinFlexD_Lld_GetLinState(Base) != LINFLEXD_STATE_INIT)
        {
            /*do nothing*/
        }
        /* Set UART mode */
        LinFlexD_Lld_SetMode(Base, LINFLEXD_UART_MODE);
        /* Set the baud rate */
        UartStatus = LinFlexD_Lld_SetBaudRate(Instance, FlexUartClock, UartUserConfig->BaudRate);
        if (CDDUART_STATUS_NO_ERROR == UartStatus)
        {
            /* Set word length */
            LinFlexD_Lld_SetUartWordLength(Base, UartUserConfig->WordLength);
            LinFlexD_Lld_SetTxDataFieldLength(Base, (uint8)((uint8)UartUserConfig->WordLength >> 1U));
            LinFlexD_Lld_SetRxDataFieldLength(Base, (uint8)((uint8)UartUserConfig->WordLength >> 1U));
            UartLINFlexDStatePtr[Instance].WordLength = UartUserConfig->WordLength;
            /* Set parity */
            if (TRUE == UartUserConfig->ParityCheck)
            {
                LinFlexD_Lld_SetParityControl(Base, TRUE);
                LinFlexD_Lld_SetParityType(Base, UartUserConfig->ParityType);
            }
            else
            {
                LinFlexD_Lld_SetParityControl(Base, FALSE);
            }
            /* Set stop bits count */
            LinFlexD_Lld_SetRxStopBitsCount(Base, UartUserConfig->StopBitsCount);
            LinFlexD_Lld_SetTxStopBitsCount(Base, UartUserConfig->StopBitsCount);
            /* Enable FIFO for DMA based communication, or buffer mode for interrupt based communication */
            if (UartUserConfig->TxTransferType == LINFLEXD_UART_USING_INTS)
            {
                LinFlexD_Lld_SetTxMode(Base, LINFLEXD_UART_BUFFER_MODE);
            }
            else
            {
                LinFlexD_Lld_SetTxMode(Base, LINFLEXD_UART_FIFO_MODE);
            }
            if (UartUserConfig->RxTransferType == LINFLEXD_UART_USING_INTS)
            {
                LinFlexD_Lld_SetRxMode(Base, LINFLEXD_UART_BUFFER_MODE);
            }
            else
            {
                LinFlexD_Lld_SetRxMode(Base, LINFLEXD_UART_FIFO_MODE);
            }
            if (TRUE == UartUserConfig->IdleLineIntEn)
            {
                /* Enable monitors the idle state of the reception line */
                LinFlexD_Lld_SetMonitorIdleState(Base, TRUE);
                /* Timeout in UART mode is disabled automatically */
                LinFlexD_Lld_SetDisableTimeout(Base, TRUE);
                LinFlexD_Lld_SetExpectedFrame(Base, 1U);
            }
            /* Enable Frame Error Interrupt */
            LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_FRAME_ERROR, TRUE);
            /* Enable Buffer Overrun Interrupt */
            LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_BUFFER_OVERRUN, TRUE);
            /* Enter normal mode */
            LinFlexD_Lld_EnterNormalMode(Base);
            /* initialize last driver operation status */
            UartLINFlexDStatePtr[Instance].TransmitStatus = CDDUART_STATUS_NO_ERROR;
            UartLINFlexDStatePtr[Instance].ReceiveStatus = CDDUART_STATUS_NO_ERROR;
            /* Save the transfer type and DMA channels in driver state */
            UartLINFlexDStatePtr[Instance].TxTransferType = UartUserConfig->TxTransferType;
            UartLINFlexDStatePtr[Instance].RxTransferType = UartUserConfig->RxTransferType;
#if (STD_ON == CDDUART_DMA_USED)
            UartLINFlexDStatePtr[Instance].RxDMAChannel = UartUserConfig->RxDMAChannel;
            UartLINFlexDStatePtr[Instance].TxDMAChannel = UartUserConfig->TxDMAChannel;
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
            UartLINFlexDStatePtr[Instance].DummyDMAChannel = UartUserConfig->DummyDMAChannel;
#endif /* CPU_YTM32B1ME0 || CPU_YTM32B1MD1 */
#endif
            /* Initialize callback functions and parameters */
            UartLINFlexDStatePtr[Instance].RxCallback = UartUserConfig->ComplexCallback;
            UartLINFlexDStatePtr[Instance].TxCallback = UartUserConfig->ComplexCallback;
            UartLINFlexDStatePtr[Instance].ErrorCallback = UartUserConfig->ComplexCallback;
            UartLINFlexDStatePtr[Instance].RxCallbackParam = UartUserConfig->LogicChannel;
            UartLINFlexDStatePtr[Instance].TxCallbackParam = UartUserConfig->LogicChannel;
            UartLINFlexDStatePtr[Instance].ErrorCallbackParam = NULL_PTR;
            UartLINFlexDStatePtr[Instance].WordLength = UartUserConfig->WordLength;
        }
    }
    return UartStatus;
}

/* implements LinFlexD_Lld_Deinit Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_Deinit(uint32 Instance)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartStatePtr = &UartLINFlexDStatePtr[Instance];
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        /* Disable error interrupts */
        LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_FRAME_ERROR, FALSE);
        LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_BUFFER_OVERRUN, FALSE);
        /* Clear the state struct for this Instance. */
        uint8 *ClearStructPtr = (uint8 *)UartStatePtr;
        for (uint32 Idx = 0; Idx < sizeof(LinFlexd_UartStateType); ++Idx)
        {
            ClearStructPtr[Idx] = 0;
        }
        /* Save runtime structure pointer.*/
        UartLINFlexDStatePtr[Instance] = (*UartStatePtr);
    }
    return UartStatus;
}
/* implements LinFlexD_Lld_SendData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_SendData(uint32 Instance, uint8 *TxBuff, uint32 TxSize)
{
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType const *UartState;
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == TxBuff)
            || (0U == TxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState = &UartLINFlexDStatePtr[Instance];
        if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
        {
            /* Start the transmission process using interrupts */
            RetVal = LinFlexD_Lld_StartSendUsingInt(Instance, TxBuff, TxSize);
        }
#if (STD_ON == CDDUART_DMA_USED)
        else
        {
            /* Start the transmission process using DMA */
            RetVal = LinFlexD_Lld_StartSendUsingDma(Instance, TxBuff, TxSize);
        }
#endif
        /* Start the transmission process */
    }
    return RetVal;
}
/* implements LinFlexD_Lld_SendDataBlocking Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_SendDataBlocking(uint32 Instance, uint8 *TxBuff, uint32 TxSize, uint32 Timeout)
{
    uint8 TempSendCtl = 0;
    uint16 TempData = 0;
    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == TxBuff)
            || (0U == TxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        /* Enable the transmitter */
        LinFlexD_Lld_SetTransmitterState(Base, TRUE);
        while (TxSize > 0U)
        {
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
            if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
            {
                LinFlexD_Lld_SetTxDataBuffer1Byte(Base, *TxBuff);
                ++TxBuff;
                --TxSize;
            }
            else
            {
                TempData = (*TxBuff);
                TempData = (uint16)(TempData << 8);
                ++TxBuff;
                TempData |= (*TxBuff);
                ++TxBuff;
                TxSize -= 2U;
                LinFlexD_Lld_SetTxDataBuffer2Bytes(Base, TempData);
            }
            TempSendCtl = 1;
            while (TempSendCtl > 0U)
            {
                /*In DMA mode, both sending and receiving must be in FIFO mode, and the TFF flag bit is setted indicates the BUFF is full.
                   Conversely, in BUFF mode, when the TFF flag bit is set, it indicates that the sending is complete.*/
                if ((UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
                        && (TRUE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED)))
                {
                    TempSendCtl = 0;
                }
                else if ((UartState->TxTransferType == LINFLEXD_UART_USING_DMA)
                         && (FALSE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED)))
                {
                    TempSendCtl = 0;
                }
                else
                {
                    /*do nothing*/
                }
                --Timeout;
                if (Timeout == 0U)
                {
                    RetVal = CDDUART_STATUS_TIMEOUT;
                    UartState->TransmitStatus = CDDUART_STATUS_TIMEOUT;
                    TxSize = 0;
                    TempSendCtl = 0;
                }
                else
                {
                    UartState->TransmitStatus = CDDUART_STATUS_NO_ERROR;
                }
            }
        }
        /* Call the callback to notify application that the transfer is complete */
        if (UartState->TxCallback != NULL_PTR)
        {
            UartState->TxCallback(UartState->TxCallbackParam, UART_EVENT_END_TX);
        }
        /* Disable the transmitter */
        LinFlexD_Lld_SetTransmitterState(Base, FALSE);
    }
    return RetVal;
}

CDDUART_FUNC void  LinFlexD_Lld_SetTransmitStatus(uint32 Instance, CddUart_StatusType Status)
{
    LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    UartState->TransmitStatus = Status;
}

/* implements LinFlexD_Lld_GetTransmitStatus Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_GetTransmitStatus(uint32 Instance, uint32 *BytesRemaining)
{
    const LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        if (BytesRemaining != NULL_PTR)
        {
            /* Fill in the number of bytes yet to be transferred and update the return value if needed */
            if (TRUE == UartState->IsTxBusy)
            {
                /* Fill in the bytes not transferred yet. */
                if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
                {
                    /* In interrupt-based communication, the remaining bytes are retrieved
                     * from the state structure
                     */
                    *BytesRemaining = UartState->TxSize;
                }
#if(STD_ON == CDDUART_DMA_USED)
                else
                {
                    /* In DMA-based communication, the remaining bytes are retrieved
                     * from the current DMA trigger loop count
                     */
                    *BytesRemaining =  CddDma_GetChannelRemainingTriggerCount(UartState->TxDMAChannel);
                }
#endif
            }
            else
            {
                *BytesRemaining = 0;
            }
        }
        RetVal = UartState->TransmitStatus;
    }
    return RetVal;
}

/* implements LinFlexD_Lld_AbortSendingData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_AbortSendingData(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    UartState = &UartLINFlexDStatePtr[Instance];
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else if (TRUE == UartState->IsTxBusy)
    {
        /* Update the tx status */
        UartState->TransmitStatus = CDDUART_STATUS_ABORTED;
        /* Stop the running transfer. */
        if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
        {
            LinFlexD_Lld_CompleteSendUsingInt(Instance);
        }
#if (STD_ON == CDDUART_DMA_USED)
        else
        {
            LinFlexD_Lld_CompleteSendUsingDma((void *)(Instance));
        }
#endif
    }
    else
    {
        /*do nothing*/
    }
    return RetVal;
}
/* implements LinFlexD_Lld_ReceiveDataBlocking Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_ReceiveDataBlocking(uint32 Instance, uint8 *RxBuff, uint32 RxSize, uint32 Timeout)
{
    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    uint32 TimeoutVal = 0;
    TimeoutVal = Timeout;
    UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == RxBuff)
            || (0U == RxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
        {
            /* Start the reception process using interrupts */
            RetVal = LinFlexD_Lld_StartReceiveUsingInt(Instance, RxBuff, RxSize);
        }
#if(STD_ON == CDDUART_DMA_USED)
        else
        {
            /* Start the reception process using DMA */
            RetVal = LinFlexD_Lld_StartReceiveUsingDma(Instance, RxBuff, RxSize);
        }
#endif
        while ((TRUE == UartState->IsRxBusy) && (CDDUART_STATUS_NO_ERROR == RetVal))
        {
            /* Wait until to transmit is complete. */
            if (TimeoutVal == 0U)
            {
                RetVal = CDDUART_STATUS_TIMEOUT;
                break;
            }
            TimeoutVal--;
        }
        /* Finish the reception if Timeout expired */
        if (RetVal == CDDUART_STATUS_TIMEOUT)
        {
            UartState->IsRxBusy = FALSE;
            UartState->ReceiveStatus = CDDUART_STATUS_TIMEOUT;
            if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
            {
                LinFlexD_Lld_CompleteRxUsingInt(Instance);
            }
#if(STD_ON == CDDUART_DMA_USED)
            else
            {
                LinFlexD_Lld_CompleteRxUsingDma((void *)((uint32)(Instance)));
            }
#endif
        }
        else
        {
            if (UartState->ReceiveStatus != CDDUART_STATUS_NO_ERROR)
            {
                UartState->ReceiveStatus = CDDUART_STATUS_OTHER_ERROR;
            }
        }
    }
    return RetVal;
}

/* implements LinFlexD_Lld_ReceiveData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_ReceiveData(uint32 Instance, uint8 *RxBuff, uint32 RxSize)
{
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType const *UartState;
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == RxBuff)
            || (0U == RxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState = &UartLINFlexDStatePtr[Instance];
        if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
        {
            /* Start the transmission process using interrupts */
            RetVal = LinFlexD_Lld_StartReceiveUsingInt(Instance, RxBuff, RxSize);
        }
#if(STD_ON == CDDUART_DMA_USED)
        else
        {
            /* Start the transmission process using interrupts */
            RetVal = LinFlexD_Lld_StartReceiveUsingDma(Instance, RxBuff, RxSize);
        }
#endif
    }
    return RetVal;
}

CDDUART_FUNC void LinFlexD_Lld_SetReceiveStatus(uint32 Instance, CddUart_StatusType Status)
{
    LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    UartState->ReceiveStatus = Status;
}

/* implements LinFlexD_Lld_GetReceiveStatus Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_GetReceiveStatus(uint32 Instance, uint32 *BytesRemaining)
{
    const LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        if (BytesRemaining != NULL_PTR)
        {
            if (TRUE == UartState->IsRxBusy)
            {
                /* Fill in the number of bytes yet to be received and update the return value if needed */
                if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
                {
                    /* In interrupt-based communication, the remaining bytes are retrieved
                     * from the state structure
                     */
                    *BytesRemaining = UartState->RxSize;
                }
#if(STD_ON == CDDUART_DMA_USED)
                else
                {
                    /* In DMA-based communication, the remaining bytes are retrieved
                     * from the current DMA trigger loop count*/
                    *BytesRemaining = CddDma_GetChannelRemainingTriggerCount(UartState->RxDMAChannel);
                }
#endif
            }
            else
            {
                *BytesRemaining = 0;
            }
        }
        RetVal = UartState->ReceiveStatus;
    }
    return RetVal;
}

/* implements LinFlexD_Lld_AbortReceivingData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_AbortReceivingData(uint32 Instance)
{
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else if (TRUE == UartState->IsRxBusy)
    {
        /* Check if a transfer is running. */
        /* Update the rx status */
        UartState->ReceiveStatus = CDDUART_STATUS_ABORTED;
        /* Stop the running transfer. */
        if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
        {
            LinFlexD_Lld_CompleteRxUsingInt(Instance);
        }
#if(STD_ON == CDDUART_DMA_USED)
        else
        {
            LinFlexD_Lld_StopRxUsingDma(Instance);
        }
#endif
    }
    else
    {
        /*do nothing*/
    }
    return RetVal;
}

/* implements LinFlexD_Lld_SetTxBuffer Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_SetTxBuffer(uint32 Instance, uint8 *TxBuff, uint32 TxSize)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == TxBuff)
            || (0U == TxSize))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState->TxBuff = TxBuff;
        UartState->TxSize = TxSize;
    }
    return UartStatus;
}

/* implements LinFlexD_Lld_SetCallback Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_SetCallback(uint32 Instance, CddUart_General_CallbackType CallBackFunction)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == CallBackFunction))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState->TxCallback = CallBackFunction;
        UartState->RxCallback = CallBackFunction;
        UartState->ErrorCallback = CallBackFunction;
    }
    return UartStatus;
}

/* implements LinFlexD_Lld_SetRxBuffer Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Lld_SetRxBuffer(uint32 Instance, uint8 *RxBuff, uint32 RxSize)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == RxBuff)
            || (0U == RxSize))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState->RxBuff = RxBuff;
        UartState->RxSize = RxSize;
    }
    return UartStatus;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_Lld_RxIRQHandler
 * Description   : Rx interrupt handler for UART.
 * This handler uses the rx buffer stored in the state structure to receive
 * data. This is not a public API as it is called by IRQ whenever an interrupt
 * occurs.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_Lld_RxIRQHandler(uint8 Instance)
{
    LinFlexd_UartStateType const *UartState;
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Exit if there is no reception in progress */
    if (TRUE == UartState->IsRxBusy)
    {
        /* Retrieve the data */
        LinFlexD_Lld_GetData(Instance);
        /* Check if this was the last byte in the current buffer */
        if (0U == UartState->RxSize)
        {
            /* Invoke the callback when the buffer is finished;
             * Application can provide another buffer inside the callback by calling LINFlexD_UART_DRV_SetRxBuffer */
            if (UartState->RxCallback != NULL_PTR)
            {
                UartState->RxCallback(UartState->RxCallbackParam, UART_EVENT_RX_FULL);
            }
        }
        /* Finish reception if this was the last byte received */
        if (0U == UartState->RxSize)
        {
            /* Complete transfer (disable rx logic) */
            LinFlexD_Lld_CompleteRxUsingInt(Instance);
            /* Invoke callback if there is one */
            if (UartState->RxCallback != NULL_PTR)
            {
                UartState->RxCallback(UartState->RxCallbackParam, UART_EVENT_END_RX);
            }
        }
#ifndef CPU_YTM32B1HA0
        else
        {
            if (TRUE == UartState->IdleLineIntEn)
            {
                LinFlexD_Lld_SetTimeoutValue(Base, UartState->IdleTimeoutValue);
                LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TIMEOUT);
                LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_UART_TIMEOUT, TRUE);
            }
        }
#endif
        /* Clear the flag */
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_RXCOMPLETE);
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_BUFFER_FULL);
    }
}

#if defined (CPU_YTM32B1HA0)
/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_Lld_RxStartSignalIRQHandler
 * Description   : Interrupt handler for UART start signal.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_Lld_RxStartSignalIRQHandler(uint8 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    if (TRUE == UartState->IdleLineIntEn)
    {
        LinFlexD_Lld_SetTimeoutValue(Base, UartState->IdleTimeoutValue);
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TIMEOUT);
        LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_UART_TIMEOUT, TRUE);
    }
    /* Disable start detection interrupt*/
    LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_START_DETECT, FALSE);
    /* Clear the flag */
    LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_START_DETECT);
}
#endif

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_Lld_RxTimeoutIRQHandler
 * Description   : Interrupt handler for UART IDLE.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_Lld_RxTimeoutIRQHandler(uint8 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Disable the timeout interrupt */
    LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_UART_TIMEOUT, FALSE);
    /* Clear the flag */
    LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TIMEOUT);
    if (UartState->RxSize > 0U)
    {
        SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_25();
        /* Update the tx status */
        UartState->ReceiveStatus = CDDUART_STATUS_TIMEOUT;
        /* Stop the running transfer. */
        if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
        {
            LinFlexD_Lld_CompleteRxUsingInt(Instance);
        }
#if (STD_ON == CDDUART_DMA_USED)
        else
        {
            LinFlexD_Lld_StopRxUsingDma(Instance);
            UartState->RxSize = CddDma_GetChannelRemainingTriggerCount(UartState->RxDMAChannel);
        }
#endif
        /* Ensure no new receive is required from the Idle callback*/
        UartState->IsRxBusy = TRUE;
        if (UartState->RxCallback != NULL_PTR)
        {
            UartState->RxCallback(UartState->RxCallbackParam, UART_EVENT_IDLE);
        }
        UartState->RxSize = 0U;
        UartState->IsRxBusy = FALSE;
        SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_25();
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_Lld_TxIRQHandler
 * Description   : Tx interrupt handler for UART.
 * This handler uses the tx buffer stored in the state structure to transmit
 * data. This is not a public API as it is called by IRQ whenever an interrupt
 * occurs.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_Lld_TxIRQHandler(uint8 Instance)
{
    LinFlexd_UartStateType const *UartState;
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Exit if there is no transmission in progress */
    if (TRUE == UartState->IsTxBusy)
    {
        /* Check if there are any more bytes to send */
        if (UartState->TxSize > 0U)
        {
            /* Clear the flag */
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
            /* Send data */
            LinFlexD_Lld_PutData(Instance);
        }
        else
        {
            /* Invoke the callback when the buffer is finished;
             * Application can provide another buffer inside the callback by calling LINFlexD_UART_DRV_SetTxBuffer
             */
            if (UartState->TxCallback != NULL_PTR)
            {
                UartState->TxCallback(UartState->TxCallbackParam, UART_EVENT_TX_EMPTY);
            }
            /* If there is no more data to send, complete the transmission */
            if (0U == UartState->TxSize)
            {
                LinFlexD_Lld_CompleteSendUsingInt(Instance);
                /* Call the callback to notify application that the transfer is complete */
                if (UartState->TxCallback != NULL_PTR)
                {
                    UartState->TxCallback(UartState->TxCallbackParam, UART_EVENT_END_TX);
                }
                /* Clear the flag */
                LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
            }
        }
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_Lld_ErrIRQHandler
 * Description   : Error interrupt handler for UART.
 * This handler calls the user callback to treat error conditions.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_Lld_ErrIRQHandler(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Update the reception status according to the error occurred */
    if (TRUE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_BUF_OVER))
    {
        /* Update the status */
        UartState->ReceiveStatus = CDDUART_STATUS_RX_OVERRUN_ERROR;
        /* Clear the flag */
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_BUF_OVER);
    }
    else if (TRUE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_FRM_ERR))
    {
        /* Update the status */
        UartState->ReceiveStatus = CDDUART_STATUS_FRAMING_ERROR;
        /* Clear the flag */
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_FRM_ERR);
    }
    else
    {
        /* This branch should never be reached - avoid MISRA violations */
    }
    /* Terminate the current reception */
    if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
    {
        LinFlexD_Lld_CompleteRxUsingInt(Instance);
    }
#if(STD_ON == CDDUART_DMA_USED)
    else
    {
        LinFlexD_Lld_StopRxUsingDma(Instance);
    }
#endif
    /* Invoke the callback, if any */
    if (UartState->ErrorCallback != NULL_PTR)
    {
        UartState->ErrorCallback(Instance, UART_EVENT_ERROR);
    }
}

/* implements LinFlexD_Lld_IRQHandler Activity*/
CDDUART_FUNC void LinFlexD_Lld_IRQHandler(uint8 Instance)
{
    LINFlexD_Type volatile *Base;
    Base = UartLINFlexDBase[Instance];
    /* Handle the error interrupts when error interrupt triggered */
    if (TRUE == LinFlexD_Lld_GetStatusFlag(Base, (LinFlexD_UartStatusType)((uint32)LINFLEXD_UARTSTS_BUF_OVER | (uint32)LINFLEXD_UARTSTS_FRM_ERR)))
    {
        /*GCOVR_EXCL_START*/
        if ((TRUE == LinFlexD_Lld_IsInterruptEnabled(Base, (uint32)LINFLEXD_INT_FRAME_ERROR)) ||
                (TRUE == LinFlexD_Lld_IsInterruptEnabled(Base, (uint32)LINFLEXD_INT_BUFFER_OVERRUN)))
        {
            /* Call the error handler */
            LinFlexD_Lld_ErrIRQHandler(Instance);
        }
        /* GCOVR_EXCL_STOP*/
    }
    else
    {
#if defined (CPU_YTM32B1HA0)
        /* Handle receive start signal interrupt*/
        if (TRUE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_START_DETECT))
        {
            if (TRUE == LinFlexD_Lld_IsInterruptEnabled(Base, (uint32)LINFLEXD_INT_START_DETECT))
            {
                LinFlexD_Lld_RxStartSignalIRQHandler(Instance);
            }
        }
#endif /* defined CPU_YTM32B1HA0*/
        /* Handle receive data full interrupt */
        if (TRUE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_RXCOMPLETE))
        {
            if (TRUE == LinFlexD_Lld_IsInterruptEnabled(Base, (uint32)LINFLEXD_INT_RX_COMPLETE))
            {
                LinFlexD_Lld_RxIRQHandler(Instance);
            }
        }
        /* Handle transmitter data register empty interrupt */
        if (TRUE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED))
        {
            if (TRUE == LinFlexD_Lld_IsInterruptEnabled(Base, (uint32)LINFLEXD_INT_TRANSMITTED))
            {
                LinFlexD_Lld_TxIRQHandler(Instance);
            }
        }
        /*Handle timeout / Idle interrupt*/
        if (TRUE == LinFlexD_Lld_GetStatusFlag(Base, LINFLEXD_UARTSTS_TIMEOUT))
        {
            if (TRUE == LinFlexD_Lld_IsInterruptEnabled(Base, LINFLEXD_INT_UART_TIMEOUT))
            {
                LinFlexD_Lld_RxTimeoutIRQHandler(Instance);
            }
        }
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_Lld_StartSendUsingInt
 * Description   : Initiate (start) a transmit by beginning the process of
 * sending data and enabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 *END**************************************************************************/
CDDUART_FUNC static CddUart_StatusType LinFlexD_Lld_StartSendUsingInt(uint32 Instance, uint8 *TxBuff, uint32 TxSize)
{
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Check it's not busy transmitting data from a previous function call */
    if (TRUE == UartState->IsTxBusy)
    {
        RetVal = CDDUART_STATUS_OPERATION_ONGOING;
    }
    else
    {
        /* Initialize the module driver state structure */
        UartState->TxBuff = TxBuff;
        UartState->TxSize = TxSize;
        UartState->IsTxBusy = TRUE;
        UartState->TransmitStatus = CDDUART_STATUS_OPERATION_ONGOING;
        /* Enable the transmitter */
        LinFlexD_Lld_SetTransmitterState(Base, TRUE);
        /* Clear the tx empty flag to make sure the transmission of the first byte
         * doesn't occur right after enabling the tx interrupt
         */
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
        /* Enable transmission complete interrupt */
        LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_TRANSMITTED, TRUE);
        /* Transmit the first word */
        LinFlexD_Lld_PutData(Instance);
    }
    return RetVal;
}

/*!
 * @brief Initiate (start) a receive by beginning the process of
 * receiving data and enabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @param RxBuff    buff address
 * @param RxSize    size of buff
 * @return CddUart_StatusType An error code or CDDUART_STATUS_NO_ERROR
 */
CDDUART_FUNC static CddUart_StatusType LinFlexD_Lld_StartReceiveUsingInt(uint32 Instance, uint8 *RxBuff, uint32 RxSize)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];
    /* Check it's not busy receiving data from a previous function call */
    if (TRUE == UartState->IsRxBusy)
    {
        RetVal = CDDUART_STATUS_OPERATION_ONGOING;
    }
    else
    {
        /* Initialize the module driver state struct to indicate transfer in progress
         * and with the buffer and byte count data. */
        UartState->IsRxBusy = TRUE;
        UartState->RxBuff = RxBuff;
        UartState->RxSize = RxSize;
        UartState->ReceiveStatus = CDDUART_STATUS_OPERATION_ONGOING;
        /* Enable receive data full interrupt */
        LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_RX_COMPLETE, TRUE);
#if defined (CPU_YTM32B1HA0)
        /* Idle interrupt feature is enabled from the start detection interrupt*/
        if (TRUE == UartState->IdleLineIntEn)
        {
            /* Clear start detection flag*/
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_START_DETECT);
            /* Enable start detection interrupt*/
            LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_START_DETECT, TRUE);
        }
#endif /* defined CPU_YTM32B1HA0*/
        /* Enable the receiver */
        LinFlexD_Lld_SetReceiverState(Base, TRUE);
    }
    return RetVal;
}

/*!
 * @brief Finish up a transmit by completing the process of sending
 * data and disabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_Lld_CompleteSendUsingInt(uint32 Instance)
{
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartState;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Disable transmission complete interrupt */
    LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_TRANSMITTED, FALSE);
    /* Disable the transmitter */
    LinFlexD_Lld_SetTransmitterState(Base, FALSE);
    /* Update the information of the module driver state */
    UartState->IsTxBusy = FALSE;
    /* If the current transmission hasn't been aborted, update the status */
    if (UartState->TransmitStatus == CDDUART_STATUS_OPERATION_ONGOING)
    {
        UartState->TransmitStatus = CDDUART_STATUS_NO_ERROR;
    }
}

#if (STD_ON == CDDUART_DMA_USED)
/**
 * @brief Stops the reception of data using DMA for the LinFlexD module.
 *
 * @details This function disables the DMA channel associated with the LinFlexD
 *          module for receiving data. It ensures that the DMA transfer is stopped
 *          and any necessary cleanup is performed to prevent further data reception.
 *
 * @param[in] Instance The ID of the LinFlexD instance to stop the DMA reception for.
 *
 * @return void
 *
 * @note This function should be called when the DMA-based reception is no longer needed
 *       or when transitioning to a different mode of operation.
 */
CDDUART_FUNC static void LinFlexD_Lld_StopRxUsingDma(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];
    /* Stop Dma Rx channel*/
    CddDma_StopChannel((CddDma_ChannelType)UartState->RxDMAChannel);
    if (TRUE == UartState->IsRxBusy)
    {
        LinFlexD_Lld_SetRxDMA(Base, FALSE);
        /* Disable the receiver */
        LinFlexD_Lld_SetReceiverState(Base, FALSE);
        /* Clear the flag */
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_RXCOMPLETE);
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_BUFFER_FULL);
#if defined (CPU_YTM32B1HA0)
        if (TRUE == UartState->IdleLineIntEn)
        {
            LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_UART_TIMEOUT, FALSE);
            LinFlexD_Lld_SetTimeoutValue(Base, UartState->IdleTimeoutValue);
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TIMEOUT);
        }
#endif /* CPU_YTM32B1HA0 */
        /* If the current transmission hasn't been aborted, update the status */
        if (UartState->ReceiveStatus == CDDUART_STATUS_OPERATION_ONGOING)
        {
            UartState->ReceiveStatus = CDDUART_STATUS_NO_ERROR;
        }
        UartState->IsRxBusy = FALSE;
    }
}
#endif

/*!
 * @brief Finish up a receive by completing the process of receiving data
 * and disabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_Lld_CompleteRxUsingInt(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];
    /* Disable receive data full interrupt. */
    LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_RX_COMPLETE, FALSE);
    /* Disable the receiver */
    LinFlexD_Lld_SetReceiverState(Base, FALSE);
    if (TRUE == UartState->IdleLineIntEn)
    {
        LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_UART_TIMEOUT, FALSE);
        LinFlexD_Lld_SetTimeoutValue(Base, UartState->IdleTimeoutValue);
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TIMEOUT);
    }
    /* If the current reception hasn't been aborted and no error occurred, update the status */
    if (UartState->ReceiveStatus == CDDUART_STATUS_OPERATION_ONGOING)
    {
        UartState->ReceiveStatus = CDDUART_STATUS_NO_ERROR;
    }
    /* Update the information of the module driver state */
    UartState->IsRxBusy = FALSE;
}

/*!
 * @brief Write data to the buffer register, according to configured
 * word length.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_Lld_PutData(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    uint16 TempData16 = 0;
    uint8 TempData8 = 0;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];
    if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
    {
        TempData8 = *UartState->TxBuff;
        /* Update the state structure */
        ++UartState->TxBuff;
        UartState->TxSize -= 1U;
        /* Transmit the data */
        LinFlexD_Lld_SetTxDataBuffer1Byte(Base, TempData8);
    }
    else
    {
        TempData16 = *(UartState->TxBuff);
        TempData16 = (uint16)(TempData16 << 8);
        ++UartState->TxBuff;
        UartState->TxSize -= 1U;
        if (UartState->TxSize > 0U)
        {
            TempData16 |= *(UartState->TxBuff);
            ++UartState->TxBuff;
            UartState->TxSize -= 1U;
        }
        LinFlexD_Lld_SetTxDataBuffer2Bytes(Base, TempData16);
    }
}

/*!
 * @brief Read data from the buffer register, according to configured
 * word length.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_Lld_GetData(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    LINFlexD_Type const volatile *Base;
    uint16 TempData16 = 0;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];
    if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
    {
        /* Get the data */
        *(UartState->RxBuff) = LinFlexD_Lld_GetRxDataBuffer1Byte(Base);
        /* Update the state structure */
        ++UartState->RxBuff;
        UartState->RxSize -= 1U;
    }
    else
    {
        TempData16 = LinFlexD_Lld_GetRxDataBuffer2Bytes(Base);
        *(UartState->RxBuff) = (uint8)(TempData16 >> 8);
        ++UartState->RxBuff;
        UartState->RxSize -= 1U;
        if (UartState->RxSize > 0U)
        {
            *(UartState->RxBuff) = (uint8)(TempData16 & 0x00FFU);
            ++UartState->RxBuff;
            UartState->RxSize -= 1U;
        }
        else
        {
            /*do nothing*/
        }
    }
}

#if (STD_ON == CDDUART_DMA_USED)
/*!
 * @brief Flushes the rx FIFO.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Base  one of LINFlexD Instance registers.
 * @return void
 */
CDDUART_FUNC static void LinFlexD_Lld_FlushRxFifo(volatile LINFlexD_Type *Base, uint8 FifoType)
{
    uint16 Dummy;
    volatile const uint8 *FifoBase = NULL_PTR;
    volatile const uint16 *FifoBase1 = NULL_PTR;
    if (0U == FifoType)
    {
        /* Get the address of the FIFO */
        FifoBase = (volatile const uint8 *)(&(Base->DATA.DATA32[1]));
        /* Four Dummy reads, to flush the FIFO contents */
        Dummy = (uint16)(*FifoBase);
        Dummy = (uint16)(*FifoBase);
        Dummy = (uint16)(*FifoBase);
        Dummy = (uint16)(*FifoBase);
    }
    else
    {
        /* Get the address of the FIFO */
        FifoBase1 = (&(Base->DATA.DATA16[2]));
        /* Four Dummy reads, to flush the FIFO contents */
        Dummy = (*FifoBase1);
        Dummy = (*FifoBase1);
    }
    (void)Dummy;
}

/*!
 * @brief Reserve API
 */
CDDUART_FUNC static CddUart_StatusType LinFlexD_Lld_StartSendUsingDma(uint32 Instance, uint8 *TxBuff, uint32 TxSize)
{
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartState;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    TxDmaTransferConfig = *DmaChannelTransferConfigArray[UartState->TxDMAChannel];
    /*config Dma*/
    UartState->TxBuff = TxBuff;
    UartState->TxSize = TxSize;
    UartState->IsTxBusy = TRUE;
    UartState->TransmitStatus = CDDUART_STATUS_OPERATION_ONGOING;
    if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
    {
        SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_17();
        TxDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_1_BYTE;
        TxDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_1_BYTE;
        TxDmaTransferConfig.SrcOffset = (sint16)(1U << (uint16)DMA_TRANSFER_SIZE_1_BYTE);
        TxDmaTransferConfig.DestOffset = 0;
        TxDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_1_BYTE;
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
        /*Dummy channel configuration*/
        DummyDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.SrcOffset = 0;
        DummyDmaTransferConfig.DestOffset = 0;
        DummyDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.TriggerCount = (uint32)(TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
        DummyDmaTransferConfig.ChannelPollingMode = TRUE;
        (void)CddDma_SetLogicChannelTransfer(UartState->DummyDMAChannel, DUMMY_ADDRESS, DUMMY_ADDRESS, &DummyDmaTransferConfig);
        /* Tx channel link configuration */
        TxChannelTrigger.TransferLoopChnLinkEnable = TRUE;
        TxChannelTrigger.TransferLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        TxChannelTrigger.TriggerLoopChnLinkEnable = TRUE;
        TxChannelTrigger.TriggerLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        TxChannelTrigger.TriggerLoopIterationCount = (uint32)(TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
        /* Link the Tx channel to the Dummy channel */
        TxDmaTransferConfig.LoopTransferConfig = &TxChannelTrigger;
#else
        TxDmaTransferConfig.TriggerCount = (uint32)(TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
#endif
        SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_17();
        (void)CddDma_SetLogicChannelTransfer(UartState->TxDMAChannel, (uint32)(UartState->TxBuff),
                                             (uint32)(&(Base->DATA.DATA8[0])), &TxDmaTransferConfig);
    }
    else
    {
        SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_18();
        TxDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_2_BYTE;
        TxDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_2_BYTE;
        TxDmaTransferConfig.SrcOffset = (sint16)(1U << (uint16)DMA_TRANSFER_SIZE_2_BYTE);
        TxDmaTransferConfig.DestOffset = 0;
        TxDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_2_BYTE;
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
        /*Dummy channel configuration*/
        DummyDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.SrcOffset = 0;
        DummyDmaTransferConfig.DestOffset = 0;
        DummyDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.TriggerCount = (uint32)(TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
        DummyDmaTransferConfig.ChannelPollingMode = TRUE;
        (void)CddDma_SetLogicChannelTransfer(UartState->DummyDMAChannel, DUMMY_ADDRESS, DUMMY_ADDRESS, &DummyDmaTransferConfig);
        /* Tx channel link configuration */
        TxChannelTrigger.TransferLoopChnLinkEnable = TRUE;
        TxChannelTrigger.TransferLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        TxChannelTrigger.TriggerLoopChnLinkEnable = TRUE;
        TxChannelTrigger.TriggerLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        TxChannelTrigger.TriggerLoopIterationCount = (uint32)(TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
        /* Link the Tx channel to the Dummy channel */
        TxDmaTransferConfig.LoopTransferConfig = &TxChannelTrigger;
#else
        /* Set the trigger count for the DMA transfer */
        TxDmaTransferConfig.TriggerCount = (uint32)(TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
#endif
        SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_18();
        (void)CddDma_SetLogicChannelTransfer(UartState->TxDMAChannel, (uint32)(UartState->TxBuff),
                                             (uint32)(&(Base->DATA.DATA16[0])), &TxDmaTransferConfig);
    }
    CddDma_InstallCallback((CddDma_ChannelType)UartState->TxDMAChannel, LinFlexD_Lld_CompleteSendUsingDma, (void *)Instance);
    CddDma_InstallErrorCallback((CddDma_ChannelType)UartState->TxDMAChannel, UART_Lld_DmaTxError, (void *)Instance);
    /* Start the DMA channel */
    CddDma_StartChannel((CddDma_ChannelType)UartState->TxDMAChannel);
    /* Enable the transmitter */
    LinFlexD_Lld_SetTransmitterState(Base, TRUE);
    LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
    LinFlexD_Lld_SetTransmitterDMA(Base, TRUE);
    return (CDDUART_STATUS_NO_ERROR);
}

/*!
 * @brief Reserve API
 */
CDDUART_FUNC static CddUart_StatusType LinFlexD_Lld_StartReceiveUsingDma(uint32 Instance, uint8 *RxBuff, uint32 RxSize)
{
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartState;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    RxDmaTransferConfig = *DmaChannelTransferConfigArray[UartState->RxDMAChannel];
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
    DummyDmaTransferConfig = *DmaChannelTransferConfigArray[UartState->DummyDMAChannel];
#endif
    /*config Dma*/
    UartState->RxBuff = RxBuff;
    UartState->RxSize = RxSize;
    UartState->IsRxBusy = TRUE;
    UartState->ReceiveStatus = CDDUART_STATUS_OPERATION_ONGOING;
    if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
    {
        SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_19();
        RxDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_1_BYTE;
        RxDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_1_BYTE;
        RxDmaTransferConfig.SrcOffset = 0;
        RxDmaTransferConfig.DestOffset = (sint16)(1U << (uint16)DMA_TRANSFER_SIZE_1_BYTE);
        RxDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_1_BYTE;
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
        /*Dummy channel configuration*/
        DummyDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.SrcOffset = 0;
        DummyDmaTransferConfig.DestOffset = 0;
        DummyDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.TriggerCount = (uint32)(RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
        DummyDmaTransferConfig.ChannelPollingMode = TRUE;
        (void)CddDma_SetLogicChannelTransfer(UartState->DummyDMAChannel, DUMMY_ADDRESS, DUMMY_ADDRESS, &DummyDmaTransferConfig);
        /* Rx channel link configuration */
        RxChannelTrigger.TransferLoopChnLinkEnable = TRUE;
        RxChannelTrigger.TransferLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        RxChannelTrigger.TriggerLoopChnLinkEnable = TRUE;
        RxChannelTrigger.TriggerLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        RxChannelTrigger.TriggerLoopIterationCount = (uint32)(RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
        /* Link the Rx channel to the Dummy channel */
        RxDmaTransferConfig.LoopTransferConfig = &RxChannelTrigger;
#else
        RxDmaTransferConfig.TriggerCount = (uint32)(RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
#endif
        SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_19();
        (void)CddDma_SetLogicChannelTransfer(UartState->RxDMAChannel, (uint32)(&(Base->DATA.DATA8[4])),
                                             (uint32)(UartState->RxBuff), &RxDmaTransferConfig);
    }
    else
    {
        SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_20();
        RxDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_2_BYTE;
        RxDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_2_BYTE;
        RxDmaTransferConfig.SrcOffset = 0;
        RxDmaTransferConfig.DestOffset = (sint16)(1U << (uint16)DMA_TRANSFER_SIZE_2_BYTE);
        RxDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_2_BYTE;
        RxDmaTransferConfig.TriggerCount = (uint32)(RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
        /*Dummy channel configuration*/
        DummyDmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.SrcOffset = 0;
        DummyDmaTransferConfig.DestOffset = 0;
        DummyDmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_4_BYTE;
        DummyDmaTransferConfig.TriggerCount = (uint32)(RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
        DummyDmaTransferConfig.ChannelPollingMode = TRUE;
        (void)CddDma_SetLogicChannelTransfer(UartState->DummyDMAChannel, DUMMY_ADDRESS, DUMMY_ADDRESS, &DummyDmaTransferConfig);
        /* Rx channel link configuration */
        RxChannelTrigger.TransferLoopChnLinkEnable = TRUE;
        RxChannelTrigger.TransferLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        RxChannelTrigger.TriggerLoopChnLinkEnable = TRUE;
        RxChannelTrigger.TriggerLoopChnLinkNumber = (uint8)UartState->DummyDMAChannel;
        RxChannelTrigger.TriggerLoopIterationCount = (uint32)(RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
        /* Link the Rx channel to the Dummy channel */
        RxDmaTransferConfig.LoopTransferConfig = &RxChannelTrigger;
#else
        /* Set the trigger count for the DMA transfer */
        RxDmaTransferConfig.TriggerCount = (uint32)(RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
#endif
        SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_20();
        (void)CddDma_SetLogicChannelTransfer(UartState->RxDMAChannel, (uint32)(&(Base->DATA.DATA16[2])),
                                             (uint32)(UartState->RxBuff), &RxDmaTransferConfig);
    }
    CddDma_InstallCallback((CddDma_ChannelType)UartState->RxDMAChannel, LinFlexD_Lld_CompleteRxUsingDma, (void *)Instance);
    CddDma_InstallErrorCallback((CddDma_ChannelType)UartState->RxDMAChannel, UART_Lld_DmaRxError, (void *)Instance);
    /* Start the DMA channel */
    CddDma_StartChannel((CddDma_ChannelType)UartState->RxDMAChannel);
    /* Flush the rx FIFO to discard any junk data received while the driver was idle */
    if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
    {
        LinFlexD_Lld_FlushRxFifo(Base, 0U);
    }
    else
    {
        LinFlexD_Lld_FlushRxFifo(Base, 1U);
    }
    /* Idle interrupt feature is enabled from the start detection interrupt*/
#if defined (CPU_YTM32B1HA0)
    if (TRUE == UartState->IdleLineIntEn)
    {
        /* Clear start detection flag*/
        LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_START_DETECT);
        /* Enable start detection interrupt*/
        LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_START_DETECT, TRUE);
    }
#endif /* CPU_YTM32B1HA0 */
    /* Enable the receiver */
    LinFlexD_Lld_SetReceiverState(Base, TRUE);
    LinFlexD_Lld_SetRxDMA(Base, TRUE);
    return CDDUART_STATUS_NO_ERROR;
}

/*!
 * @brief Reserve API
 */
CDDUART_FUNC static void LinFlexD_Lld_CompleteSendUsingDma(void *Parameter)
{
    uint32 Instance;
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    Instance = ((uint32)Parameter);
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    UartState->TxSize = 0U;
    if (TRUE == UartState->IsTxBusy)
    {
        if (UartState->TxCallback != NULL_PTR)
        {
            UartState->TxCallback(UartState->TxCallbackParam, UART_EVENT_TX_EMPTY);
        }
        if (UartState->TxSize > 0U)
        {
            /*config Dma*/
            if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
            {
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
                TxChannelTrigger.TriggerLoopIterationCount = (uint32)(UartState->TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
#else
                TxDmaTransferConfig.TriggerCount = (uint32)(UartState->TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
#endif
                (void)CddDma_SetLogicChannelTransfer(UartState->TxDMAChannel, (uint32)(UartState->TxBuff),
                                                     (uint32)(&(Base->DATA.DATA8[0])), &TxDmaTransferConfig);
            }
            else
            {
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
                TxChannelTrigger.TriggerLoopIterationCount = (uint32)(UartState->TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
#else
                TxDmaTransferConfig.TriggerCount = (uint32)(UartState->TxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
#endif
                (void)CddDma_SetLogicChannelTransfer(UartState->TxDMAChannel, (uint32)(UartState->TxBuff),
                                                     (uint32)(&(Base->DATA.DATA16[0])), &TxDmaTransferConfig);
            }
            /* Start the DMA channel */
            CddDma_StartChannel((CddDma_ChannelType)UartState->TxDMAChannel);
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
        }
        else
        {
            /* Disable the transmitter */
            LinFlexD_Lld_SetTransmitterState(Base, FALSE);
            /* Clear the flag */
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
            LinFlexD_Lld_SetTransmitterDMA(Base, FALSE);
            if (UartState->TxCallback != NULL_PTR)
            {
                UartState->TxCallback(UartState->TxCallbackParam, UART_EVENT_END_TX);
            }
            UartState->IsTxBusy = FALSE;
            /* If the current transmission hasn't been aborted, update the status */
            if (UartState->TransmitStatus == CDDUART_STATUS_OPERATION_ONGOING)
            {
                UartState->TransmitStatus = CDDUART_STATUS_NO_ERROR;
            }
        }
    }
}

/*!
 * @brief This function is called when an error occurs during a DMA transfer.
 *
 * @param[in] Parameter The UART instance here to use.
 * @return void
 */

CDDUART_FUNC static void UART_Lld_DmaTxError(void *Parameter)
{
    /*GCOVR_EXCL_START*/
    uint32 Instance;
    LinFlexd_UartStateType *UartState;
    Instance = ((uint32)Parameter);
    UartState = &UartLINFlexDStatePtr[Instance];
    UartState->TransmitStatus = CDDUART_STATUS_DMA_ERROR;
    /* Stop the transfer */
    CddDma_StopChannel(UartState->TxDMAChannel);
    /* Disable the transmitter */
    LinFlexD_Lld_SetTransmitterState(UartLINFlexDBase[Instance], FALSE);
    /* Notify the application that an error occurred */
    if (UartState->TxCallback != NULL_PTR)
    {
        UartState->TxCallback(UartState->TxCallbackParam, UART_EVENT_ERROR);
    }
    /*GCOVR_EXCL_STOP*/
}

/*!
 * @brief This function is called when an error occurs during a DMA receiver.
 *
 * @param[in] Parameter The UART instance here to use.
 * @return void
 */
CDDUART_FUNC static void UART_Lld_DmaRxError(void *Parameter)
{
    /*GCOVR_EXCL_START*/
    uint32 Instance;
    LinFlexd_UartStateType *UartState;
    Instance = ((uint32)Parameter);
    UartState = &UartLINFlexDStatePtr[Instance];
    UartState->ReceiveStatus = CDDUART_STATUS_DMA_ERROR;
    /* Stop the transfer */
    CddDma_StopChannel(UartState->RxDMAChannel);
    /* Disable the receiver */
    LinFlexD_Lld_SetReceiverState(UartLINFlexDBase[Instance], FALSE);
    /* Notify the application that an error occurred */
    if (UartState->RxCallback != NULL_PTR)
    {
        UartState->RxCallback(UartState->RxCallbackParam, UART_EVENT_ERROR);
    }
    /*GCOVR_EXCL_STOP*/
}

/*!
 * @brief Reserve API
 */
CDDUART_FUNC static void LinFlexD_Lld_CompleteRxUsingDma(void *Parameter)
{
    uint32 Instance;
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    Instance = ((uint32)Parameter);
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    UartState->RxSize = 0U;
    if (TRUE == UartState->IsRxBusy)
    {
        if (UartState->RxCallback != NULL_PTR)
        {
            UartState->RxCallback(UartState->RxCallbackParam, UART_EVENT_RX_FULL);
        }
        if (UartState->RxSize > 0U)
        {
            /*config Dma*/
            if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
            {
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
                RxChannelTrigger.TriggerLoopIterationCount = (uint32)(UartState->RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
#else
                RxDmaTransferConfig.TriggerCount = (uint32)(UartState->RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_1_BYTE));
#endif
                (void)CddDma_SetLogicChannelTransfer(UartState->RxDMAChannel, (uint32)(&(Base->DATA.DATA8[4])),
                                                     (uint32)(UartState->RxBuff), &RxDmaTransferConfig);
            }
            else
            {
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)
                RxChannelTrigger.TriggerLoopIterationCount = (uint32)(UartState->RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
#else
                RxDmaTransferConfig.TriggerCount = (uint32)(UartState->RxSize / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_2_BYTE));
#endif
                (void)CddDma_SetLogicChannelTransfer(UartState->RxDMAChannel, (uint32)(&(Base->DATA.DATA16[2])),
                                                     (uint32)(UartState->RxBuff), &RxDmaTransferConfig);
            }
            /* Start the DMA channel */
            CddDma_StartChannel((CddDma_ChannelType)UartState->RxDMAChannel);
            /* Clear the flag */
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_RXCOMPLETE);
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_BUFFER_FULL);
        }
        else
        {
            LinFlexD_Lld_SetRxDMA(Base, FALSE);
            /* Disable the receiver */
            LinFlexD_Lld_SetReceiverState(Base, FALSE);
            /* Clear the flag */
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_RXCOMPLETE);
            LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_BUFFER_FULL);
#if defined (CPU_YTM32B1HA0)
            if (TRUE == UartState->IdleLineIntEn)
            {
                LinFlexD_Lld_SetInterruptMode(Base, LINFLEXD_INT_UART_TIMEOUT, FALSE);
                LinFlexD_Lld_SetTimeoutValue(Base, UartState->IdleTimeoutValue);
                LinFlexD_Lld_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TIMEOUT);
            }
#endif /* CPU_YTM32B1HA0 */
            if (UartState->RxCallback != NULL_PTR)
            {
                UartState->RxCallback(UartState->RxCallbackParam, UART_EVENT_END_RX);
            }
            /* If the current transmission hasn't been aborted, update the status */
            if (UartState->ReceiveStatus == CDDUART_STATUS_OPERATION_ONGOING)
            {
                UartState->ReceiveStatus = CDDUART_STATUS_NO_ERROR;
            }
            UartState->IsRxBusy = FALSE;
        }
    }
}
#endif

#define CDDUART_STOP_SEC_CODE
#include "CddUart_MemMap.h"
