/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @file devassert.h
 * @version 1.4.0
 */

#ifndef DEVASSERT_H
#define DEVASSERT_H

#include "Platform_Types.h"
#include "Platform_Cfg.h"
#include "Mcal.h"
#include <stdbool.h>

/**
\page Error_detection_and_reporting Error detection and reporting

YTM32 SDK drivers can use a mechanism to validate data coming from upper software layers (application code) by performing
a number of checks on input parameters' range or other invariants that can be statically checked (not dependent on
runtime conditions). A failed validation is indicative of a software bug in application code, therefore it is important
to use this mechanism during development.

The validation is performed by using DEV_ASSERT macro.
A default implementation of this macro is provided in this file. However, application developers can provide their own
implementation in a custom file. This requires defining the CUSTOM_DEVASSERT symbol with the specific file name in the
project configuration (for example: -DCUSTOM_DEVASSERT="custom_devassert.h")

The default implementation accommodates two behaviors, based on DEV_ERROR_DETECT symbol:
 - When DEV_ERROR_DETECT symbol is defined in the project configuration (for example: -DDEV_ERROR_DETECT), the validation
   performed by the DEV_ASSERT macro is enabled, and a failed validation triggers a software breakpoint and further execution is
   prevented (application spins in an infinite loop)
   This configuration is recommended for development environments, as it prevents further execution and allows investigating
   potential problems from the point of error detection.
 - When DEV_ERROR_DETECT symbol is not defined, the DEV_ASSERT macro is implemented as no-op, therefore disabling all validations.
   This configuration can be used to eliminate the overhead of development-time checks.

It is the application developer's responsibility to decide the error detection strategy for production code: one can opt to
disable development-time checking altogether (by not defining DEV_ERROR_DETECT symbol), or one can opt to keep the checks
in place and implement a recovery mechanism in case of a failed validation, by defining CUSTOM_DEVASSERT to point
to the file containing the custom implementation.
*/

#define DEVASSERT_VENDOR_ID                    (180)
#define DEVASSERT_AR_RELEASE_MAJOR_VERSION     (4)
#define DEVASSERT_AR_RELEASE_MINOR_VERSION     (4)
#define DEVASSERT_AR_RELEASE_REVISION_VERSION  (0)
#define DEVASSERT_SW_MAJOR_VERSION             (2)
#define DEVASSERT_SW_MINOR_VERSION             (3)
#define DEVASSERT_SW_PATCH_VERSION             (0)

//#ifdef CCOV_ENABLE
//#define DevAssert(Value)
//#else

#if ((MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64) || \
     (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH32) || \
     (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH))
// #define BREAKPOINT_INSTR        "HLT #0"
#define BREAKPOINT_INSTR        "BKPT #0"
#elif (MCAL_PLATFORM_ARM == MCAL_ARM_MARCH)
#define BREAKPOINT_INSTR        "BKPT #0"
#else
#error "Unsupported architecture!"
#endif

//#if defined (CUSTOM_DEVASSERT)
//    /* If the CUSTOM_DEVASSERT symbol is defined, then add the custom implementation */
//    #include CUSTOM_DEVASSERT
//#elif  defined (DEV_ERROR_DETECT)
#define DEV_ERROR_DETECT
    /* Implement default assert macro */
//static inline void DevAssert(volatile bool x)
//{
//    if(x) { } else { BKPT_ASM; for(;;) {} }
//}
static inline void DevAssert(boolean Value)
{
    if (Value)  /*NOSONAR, Value is a Boolean data, but the boolean does not recognized by Sonar */
    {
        /* do nothing */
    }
    else
    {
        for (;;)
        {
#ifdef PLATFORM_COMP_TEST   /* for the Platform integration test only */
            break;
#else
            __ASM(BREAKPOINT_INSTR);
#endif
        }
    }
}

    #define DEV_ASSERT(x) DevAssert(x)
//#else
//    /* Assert macro does nothing */
//    #define DEV_ASSERT(x) ((void)0)
//#endif
//#endif

#endif /* DEVASSERT_H */

/*******************************************************************************
 * EOF
 ******************************************************************************/
