/**
 * @file    WdgTst.c
 * @version V2.0.0
 *
 * @brief   Safety library WdgTst module interface
 * @details API implementation for WdgTst driver
 *
 * @addtogroup WDGTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 SafLib
 *   Platform             : ARM
 *   Peripheral           : WDG
 *   Dependencies         : None
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 0306 Rule 11.4: A conversion should not be performed between a pointer to object and an integer type.
 * PRQA S 3219 Rule 2.1: A project shall not contain unreachable code.
 * PRQA S 4700: Metric value out of threshold range: %s.
 */
/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "WdgTst_Lld.h"
#if (WDGTST_WDG_ENABLE == STD_ON)
#if defined(CPU_YTM32B1ME0)
#include "YTM32B1ME0_features.h"
#elif defined(CPU_YTM32B1MD1)
#include "YTM32B1MD1_features.h"
#elif defined(CPU_YTM32B1HA0)
#include "YTM32B1HA0_features.h"
#elif defined(CPU_YTM32B1MC0)
#include "YTM32B1MC0_features.h"
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
#endif
#include "SchM_WdgTst.h"
#ifdef WDGTST_USER_MODE_SUPPORT
#include "OsIf.h"
#endif
#include "Platform.h"

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define WDGTST_LLD_VENDOR_ID_C                      (180)
#define WDGTST_LLD_AR_RELEASE_MAJOR_VERSION_C       (4)
#define WDGTST_LLD_AR_RELEASE_MINOR_VERSION_C       (4)
#define WDGTST_LLD_AR_RELEASE_REVISION_VERSION_C    (0)
#define WDGTST_LLD_SW_MAJOR_VERSION_C               (2)
#define WDGTST_LLD_SW_MINOR_VERSION_C               (0)
#define WDGTST_LLD_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if WdgTst.c and WdgTst.h are of the same vendor */
#if (WDGTST_LLD_VENDOR_ID_C != WDGTST_LLD_VENDOR_ID)
#error "WdgTst.c and WdgTst.h have different vendor ids"
#endif

/* Check if WdgTst.c and WdgTst.h are of the same Autosar version */
#if ((WDGTST_LLD_AR_RELEASE_MAJOR_VERSION_C != WDGTST_LLD_AR_RELEASE_MAJOR_VERSION) || \
     (WDGTST_LLD_AR_RELEASE_MINOR_VERSION_C != WDGTST_LLD_AR_RELEASE_MINOR_VERSION) || \
     (WDGTST_LLD_AR_RELEASE_REVISION_VERSION_C != WDGTST_LLD_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of WdgTst.c and WdgTst.h are different"
#endif

/* Check if WdgTst.c and WdgTst.h are of the same Software version */
#if ((WDGTST_LLD_SW_MAJOR_VERSION_C != WDGTST_LLD_SW_MAJOR_VERSION) || \
     (WDGTST_LLD_SW_MINOR_VERSION_C != WDGTST_LLD_SW_MINOR_VERSION) || \
     (WDGTST_LLD_SW_PATCH_VERSION_C != WDGTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of WdgTst.c and WdgTst.h are different"
#endif

/* Check if WdgTst.c and SchM_WdgTst.h file are of the same vendor */
#if (WDGTST_LLD_VENDOR_ID_C != SCHM_WDGTST_VENDOR_ID)
#error "WdgTst_Lld.c and SchM_WdgTst.h have different vendor ids"
#endif

/* Check if WdgTst.c and SchM_WdgTst.h file are of the same Autosar version */
#if (( WDGTST_LLD_AR_RELEASE_MAJOR_VERSION_C != SCHM_WDGTST_AR_RELEASE_MAJOR_VERSION) || \
      ( WDGTST_LLD_AR_RELEASE_MINOR_VERSION_C != SCHM_WDGTST_AR_RELEASE_MINOR_VERSION) || \
      ( WDGTST_LLD_AR_RELEASE_REVISION_VERSION_C != SCHM_WDGTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of WdgTst_Lld.c and SchM_WdgTst.h are different"
#endif

/* Check if WdgTst.c and SchM_WdgTst.h file are of the same Software version */
#if (( WDGTST_LLD_SW_MAJOR_VERSION_C != SCHM_WDGTST_SW_MAJOR_VERSION) || \
      ( WDGTST_LLD_SW_MINOR_VERSION_C != SCHM_WDGTST_SW_MINOR_VERSION) || \
      ( WDGTST_LLD_SW_PATCH_VERSION_C != SCHM_WDGTST_SW_PATCH_VERSION))
#error "Software Version Numbers of WdgTst_Lld.c and SchM_WdgTst.h are different"
#endif

/*==================================================================================================
 *                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/
extern const WdgTst_ConfigType WdgTst_Config;


/*==================================================================================================
 *                                      LOCAL MACROS
==================================================================================================*/

#define WDGTST_TIMEOUT_COUNT (65535U)

#define WDGTST_WDG_CR_VALUE0 (0x42U)
#define WDGTST_WDG_WVR_VALUE0 (0U)
#define WDGTST_WDG_TOVR_VALUE0 (0x300U)
#define WDGTST_WDG_CR_VALUE1 (0x44U)
#define WDGTST_WDG_WVR_VALUE1 (0x10U)
#define WDGTST_WDG_TOVR_VALUE1 (0xFFU)

#define WDGTST_WDG_INT_TEST_TOVR_VALUE (0x300U)

#define WDGTST_EWDG_CMPH_VALUE (100U)

#define WDGTST_SYSTICK_OFFSET (0x100U)

/*==================================================================================================
 *                                       GLOBAL VARIABLES
==================================================================================================*/
#define WDGTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "WdgTst_MemMap.h"

/**
 * @brief     Stores detailed test results for the WdgTst module.
 * @details   This variable holds the results of various tests performed by the WdgTst module.
 *            It includes specific test results for soft lock, timeout interrupt. The structure is updated during the execution
 *            of the respective tests and is used to provide detailed feedback about the test
 *            outcomes.
 * @private
 */
WDGTST_VAR static volatile WdgTst_TestDetailsType WdgTst_DetailResult;

#define WDGTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "WdgTst_MemMap.h"


#define WDGTST_START_SEC_VAR_CLEARED_8
#include "WdgTst_MemMap.h"

/**
 * @brief     WdgTst driver interrupt trigger mask
 * @details   Bit Description:\n
 *            Bit 3: EWDG Start - Indicates the start of the EWDG (External Watchdog) with a value of 0x08.\n
 *            Bit 2: WDG Start  - Indicates the start of the WDG (Watchdog) with a value of 0x04.\n
 *            Bit 1: EWDG Interrupt Triggered - Indicates that the EWDG interrupt has been triggered with a value of 0x02.\n
 *            Bit 0: WDG Interrupt Triggered  - Indicates that the WDG interrupt has been triggered with a value of 0x01.
 * @private
 */
WDGTST_VAR static volatile uint8 WdgTst_TrgMask;

/**
 * @brief    WdgTst driver WDG soft lock test mask
 * @details  Bit Description:\n
 *           Bit 6: TOVR inject test trigger bus fault.\n
 *           Bit 5: WVR inject test trigger bus fault.\n
 *           Bit 4: CR inject test trigger bus fault.\n
 *           Bit 2: TOVR inject test start.\n
 *           Bit 1: WVR inject test start.\n
 *           Bit 0: CR inject test start.
 * @private
 */
WDGTST_VAR static volatile uint8 WdgTst_SoftLockTestMask;

#define WDGTST_STOP_SEC_VAR_CLEARED_8
#include "WdgTst_MemMap.h"

#define WDGTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "WdgTst_MemMap.h"
/**
 * @brief   Used to store the interrupt service function of the wdg before executing the wdgtst test
 * @private
 */
WDGTST_VAR static Platform_IrqHandlerType WdgTst_BackupBusFaultHandler = NULL_PTR;
/**
 * @brief   Used to pass parameters when install irq handler
 * @private
 */
WDGTST_VAR static Platform_IrqHandlerType WdgTst_DummyBusFaultIrqHandler = NULL_PTR;
#define WDGTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "WdgTst_MemMap.h"
/*==================================================================================================
 *                                       LOCAL FUNCTIONS
==================================================================================================*/
#define WDGTST_START_SEC_CODE
#include "WdgTst_MemMap.h"

#if (WDGTST_WDG_ENABLE == STD_ON)
/**
 * @brief       Verifies if the WDG is hard-locked.
 * @details     This function verifies if the WDG is hard-locked. If the WDG is hard-locked,
 *              reconfiguration of the registers is not allowed.
 * @param[in]   Base WDG base pointer
 * @return      uint8
 * @retval      TRUE if the WDG is hard-locked
 * @retval      FALSE if the WDG is not hard-locked
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_664
 */
WDGTST_FUNC static uint8 WdgTst_Lld_WdgIsHardLocked(WDG_Type const volatile *Base)
{
    return (uint8)(((Base->LR) & WDG_LR_HL_MASK) >> WDG_LR_HL_SHIFT);
}
/**
 * @brief       Verifies if the WDG is soft-locked.
 * @details     This function verifies if the WDG is soft-locked. If the WDG is soft-locked,
 *              reconfiguration of the registers is not allowed.
 * @param[in]   Base WDG base pointer
 * @return      uint8
 * @retval      TRUE if the WDG is soft-locked
 * @retval      FALSE if the WDG is not soft-locked
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_665
 */
WDGTST_FUNC static uint8 WdgTst_Lld_WdgIsSoftLocked(WDG_Type const volatile *Base)
{
    return (uint8)(((Base->LR) & WDG_LR_SL_MASK) >> WDG_LR_SL_SHIFT);
}

/**
 * @brief       Set the WDG soft lock test result to not ok.
 * @private
 */
WDGTST_FUNC LOCAL_INLINE void WdgTst_Lld_SetWdgSLTestNOK(void)
{
    WdgTst_DetailResult.WdgSoftLockTestResult = WDGTST_RESULT_NOT_OK;
    WdgTst_DetailResult.TestResult = WDGTST_RESULT_NOT_OK;
}

/**
 * @brief       Unlocks the WDG.
 * @details     This function unlocks the WDG. The WDG can be unlocked only if the module is
 *              locked.
 * @param[in]   Base WDG base pointer
 * @return      Std_ReturnType
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_666
 */
WDGTST_FUNC static Std_ReturnType WdgTst_Lld_WdgUnlock(volatile WDG_Type *const Base)
{
    uint32 Timeout = WDGTST_TIMEOUT_COUNT;
    Std_ReturnType Ret = E_OK;
    SchM_Enter_WdgTst_WDGTST_EXCLUSIVE_AREA_00();
    Base->SVCR = FEATURE_WDG_UNLOCK_VALUE_1;
    /* Avoid these 2 steps are optimized by compiler. */
    (void)Base->SVCR;
    Base->SVCR = FEATURE_WDG_UNLOCK_VALUE_2;
    SchM_Exit_WdgTst_WDGTST_EXCLUSIVE_AREA_00();
    /* Wait until registers are unlocked or timeout. */
    while ((TRUE == WdgTst_Lld_WdgIsSoftLocked(Base)) && (Timeout > 0U))
    {
        --Timeout;
    }
    /* Timeout. */
    if (0U == Timeout)
    {
        /* Soft lock is enabled again to avoid unlock sequence successfully after timeout expired. */
        Base->LR |= WDG_LR_SL(1);
        WdgTst_Lld_SetWdgSLTestNOK();
        Ret = E_NOT_OK;
    }
    return Ret;
}

#if (defined CPU_YTM32B1ME0) || (defined CPU_YTM32B1MC0) || (defined CPU_YTM32B1MD1)
/**
 * @brief       Handles bus fault events for watchdog test lock conditions.
 * @details     This function checks if a bus fault occurred while attempting to access
 *              specific watchdog registers (CR, WVR, or TOVR) during a lock test.
 *              If the conditions match, it sets the corresponding bus fault mask flag.
 *              Finally, it clears the bus fault status register.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
/* MR12 RULE 2.1 VIOLATION: The function is called by the function WdgTst_BusFaultHandler, but the compiler doesn't recognize it. */
/* SONAR VIOLATION: The function is called by the function WdgTst_BusFaultHandler. */
WDGTST_FUNC static __attribute__((used)) void WdgTst_LockBusFaultHandler(void) /* PRQA S 3219*/ /*NOSONAR*/
{
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    uint32 BusFaultAddress = SCB->BFAR; /* PRQA S 0306 */
    if ((WDGTST_LOCK_CR_START_MASK == (WdgTst_SoftLockTestMask & WDGTST_LOCK_CR_START_MASK))
            && (BusFaultAddress == (WDG_BASE_ADDR32 + WDG_CR_OFFSET32)))
    {
        WdgTst_SoftLockTestMask |= WDGTST_LOCK_CR_BUSFAULT_MASK;
    }
    else if ((WDGTST_LOCK_WVR_START_MASK == (WdgTst_SoftLockTestMask & WDGTST_LOCK_WVR_START_MASK))
             && (BusFaultAddress == (WDG_BASE_ADDR32 + WDG_WVR_OFFSET32)))
    {
        WdgTst_SoftLockTestMask |= WDGTST_LOCK_WVR_BUSFAULT_MASK;
    }
    else if ((WDGTST_LOCK_TOVR_START_MASK == (WdgTst_SoftLockTestMask & WDGTST_LOCK_TOVR_START_MASK))
             && (BusFaultAddress == (WDG_BASE_ADDR32 + WDG_TOVR_OFFSET32)))
    {
        WdgTst_SoftLockTestMask |= WDGTST_LOCK_TOVR_BUSFAULT_MASK;
    }
    else
    {
        /* nothing */
    }
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk; /* PRQA S 0306 */
}

/**
 * @brief Bus fault handler function for watchdog testing.
 * @details This assembly-implemented function handles bus faults by:
 *          1.Saving the link register.
 *          2.Calling WdgTst_LockBusFaultHandler to process the fault.
 *          3.Adjusting the return address to skip the faulting instruction.
 *          4.Returning from the exception.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 */
WDGTST_FUNC static __attribute__((naked)) void WdgTst_BusFaultHandler(void)
{
#ifndef TESSY
    /**
     * M3CM-2_2 Dir-1.1 VIOLATION: This line uses the __ASM() macro for inline assembly which is not
     * a part of standard C. In this specific case, We use assembly to ensure the stack position of the PC register is deterministic,
     * hence we couldn't adhere to the Dir-1.1
     */
    __ASM volatile( /* PRQA S 1006*/
        "PUSH {LR}                      \n" /* push R1 to stack */
        "PUSH {LR}                      \n" /* push R1 to stack */
        "bl WdgTst_LockBusFaultHandler  \n"
        "POP {R1}                       \n" /* pop R1 from stack */
        "MOV LR, R1                     \n" /* Move R1 to LR */
        "MOVS R0, #4                    \n" /* Load immediate value 4 into R0 */
        "TST R1, R0                     \n" /* Test if LR[2] is set */
        "BNE PSP_PROCESS                \n" /* If LR[2] is not zero, branch to PSP_PROCESS */
        "MRS R1, MSP                    \n" /* Move Main Stack Pointer (MSP) into R1 */
        "B MSP_PROCESS                  \n" /* Branch to MSP_PROCESS */
        "PSP_PROCESS:                   \n"
        "MRS R1, PSP                    \n" /* Move Process Stack Pointer (PSP) into R1 */
        "MSP_PROCESS:                   \n"
        "LDR R0, [R1,#28]               \n" /* Load the instruction from the address */
        "ADD R0, R0, #2                 \n" /* Add 2 to the instruction to get the next instruction */
        "STR R0, [R1,#28]               \n" /* Store the next instruction back to the address */
        "POP {PC}                       \n" /* Pop the next instruction to LR */
    );
#endif /* #ifndef TESSY */
}
#endif

#if defined CPU_YTM32B1HA0
/**
 * @brief       Handles bus fault events for watchdog lock test.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_WdgTst_937
 */
WDGTST_FUNC static void WdgTst_BusFaultHandler(void)
{
    if ((WdgTst_SoftLockTestMask & WDGTST_LOCK_CR_START_MASK) == WDGTST_LOCK_CR_START_MASK)
    {
        WdgTst_SoftLockTestMask |= WDGTST_LOCK_CR_BUSFAULT_MASK;
    }
    else if ((WdgTst_SoftLockTestMask & WDGTST_LOCK_WVR_START_MASK) == WDGTST_LOCK_WVR_START_MASK)
    {
        WdgTst_SoftLockTestMask |= WDGTST_LOCK_WVR_BUSFAULT_MASK;
    }
    else if ((WdgTst_SoftLockTestMask & WDGTST_LOCK_TOVR_START_MASK) == WDGTST_LOCK_TOVR_START_MASK)
    {
        WdgTst_SoftLockTestMask |= WDGTST_LOCK_TOVR_BUSFAULT_MASK;
    }
    else
    {
        /* nothing */
    }
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk; /* PRQA S 0306 */
}
#endif

/**
 * @brief       Test the soft lock function of WDG module.
 * @details     This function tests the soft lock functionality of the watchdog by writing specific values
 *              to the watchdog control registers (CR, WVR, TOVR) and verifying their contents. If the
 *              registers do not hold the expected values, the test is marked as failed, and an error
 *              notification is triggered. If the soft lock test passes, the watchdog is unlocked using
 *              the `WdgTst_Lld_WdgUnlock` function.\n
 *              This function is static and intended for internal use within the WdgTst module.
 * @param[in]   Base Pointer to the watchdog register base address. This parameter must not be NULL.
 * @return      Std_ReturnType
 * @retval      E_OK: The soft lock test passed, and the watchdog was successfully unlocked.
 * @retval      E_NOT_OK: The soft lock test failed, or an error occurred during the test.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_660
 */
/* M3CM 4700 VALIDATION: Metric value out of threshold range: %s. This function governs the execution flow of the WDG module lock test.
 * It needs to complete the test steps in turn, and make multiple condition judgments and timeout judgments. Therefore, the STPTH cannot follow the standard.
 */
WDGTST_FUNC static Std_ReturnType WdgTst_Lld_WdgLockTest(volatile WDG_Type *const Base) /*PRQA S 4700*/
{
    uint32 Timeout = 0U;
    Std_ReturnType Ret = E_OK;
    uint32 TmpCRValue;
    uint32 TmpWVRValue;
    uint32 TmpTOVRValue;
    Base->CR = WDGTST_WDG_CR_VALUE0;
    Base->WVR = WDGTST_WDG_WVR_VALUE0;
    Base->TOVR = WDGTST_WDG_TOVR_VALUE0;
    TmpCRValue = Base->CR;
    TmpWVRValue = Base->WVR;
    TmpTOVRValue = Base->TOVR;
    if ((TmpCRValue != WDGTST_WDG_CR_VALUE0) || (TmpWVRValue != WDGTST_WDG_WVR_VALUE0) || (TmpTOVRValue != WDGTST_WDG_TOVR_VALUE0))
    {
        WdgTst_Lld_SetWdgSLTestNOK();
        Ret = E_NOT_OK;
    }
    else
    {
        /* Soft lock */
        Base->LR |= WDG_LR_SL(1);
        WdgTst_SoftLockTestMask = WDGTST_LOCK_CR_START_MASK;
        /* inject CR */
        Base->CR = WDGTST_WDG_CR_VALUE1;
        while (0U == (WdgTst_SoftLockTestMask & WDGTST_LOCK_CR_BUSFAULT_MASK))
        {
            ++Timeout;
            if (Timeout > WDGTST_TIMEOUT_COUNT)
            {
                WdgTst_Lld_SetWdgSLTestNOK();
                break;
            }
        }
        if (WdgTst_DetailResult.WdgSoftLockTestResult != WDGTST_RESULT_NOT_OK)
        {
            Timeout = 0U;
            WdgTst_SoftLockTestMask = WDGTST_LOCK_WVR_START_MASK;
            /* inject WVR */
            Base->WVR = WDGTST_WDG_WVR_VALUE1;
            while (0U == (WdgTst_SoftLockTestMask & WDGTST_LOCK_WVR_BUSFAULT_MASK))
            {
                ++Timeout;
                if (Timeout > WDGTST_TIMEOUT_COUNT)
                {
                    WdgTst_Lld_SetWdgSLTestNOK();
                    break;
                }
            }
        }
        if (WdgTst_DetailResult.WdgSoftLockTestResult != WDGTST_RESULT_NOT_OK)
        {
            Timeout = 0U;
            WdgTst_SoftLockTestMask = WDGTST_LOCK_TOVR_START_MASK;
            /* inject TOVR */
            Base->TOVR = WDGTST_WDG_TOVR_VALUE1;
            while (0U == (WdgTst_SoftLockTestMask & WDGTST_LOCK_TOVR_BUSFAULT_MASK))
            {
                ++Timeout;
                if (Timeout > WDGTST_TIMEOUT_COUNT)
                {
                    WdgTst_Lld_SetWdgSLTestNOK();
                    break;
                }
            }
        }
        WdgTst_SoftLockTestMask = 0U;
        if (WdgTst_DetailResult.WdgSoftLockTestResult != WDGTST_RESULT_NOT_OK)
        {
            WdgTst_DetailResult.WdgSoftLockTestResult = WDGTST_RESULT_OK;
            /* Unlocks the WDG. */
            Ret = WdgTst_Lld_WdgUnlock(Base);
        }
        else
        {
            Ret = E_NOT_OK;
        }
    }
    return Ret;
}



/**
 * @brief       Test the interrupt function of WDG module when timeout.
 * @param[in]   Base WDG base pointer
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_663
 */
WDGTST_FUNC static void WdgTst_Lld_WdgInterruptTest(volatile WDG_Type *const Base)
{
    uint32 Cr;
    uint32 SysTickValue;
    Cr = WDG_CR_RIA(1);
    /* Enable interrupt */
    Cr |= WDG_CR_IBR(1);
    /* Clock source */
#if defined(CPU_YTM32B1ME0) || defined(CPU_YTM32B1MC0)
    Cr |= WDG_CR_CLKSRC(WDGTST_WDG_CLKSRC_SIRC);
#elif defined(CPU_YTM32B1MD1) || defined(CPU_YTM32B1HA0)
    Cr |= WDG_CR_CLKSRC(WDGTST_WDG_CLKSRC_BUS);
#endif
    /* Disable in debug mode */
    Cr |= WDG_CR_DBGDIS(1);
    /* Update CR */
    Base->CR = Cr;
    /* Clears the Interrupt Flag. */
    Base->INTF |= WDG_INTF_IF_MASK;
    /* Updates the WDG timer overflow register. */
    Base->TOVR = WDGTST_WDG_INT_TEST_TOVR_VALUE;
    /* Updates the WDG window value register. */
    Base->WVR = 0UL;
    WdgTst_TrgMask |= WDGTST_WDG_STARTED_MASK;
    WdgTst_DetailResult.WdgTimeoutInterruptTestResult = WDGTST_RESULT_NOT_TESTED;
    /* Enable WDG. */
    Base->CR |= WDG_CR_EN_MASK;
    /* Lock WDG*/
    Base->LR |= WDG_LR_SL(1);
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    SysTick->VAL = WDGTST_WDG_INTERRUPT_TIMEOUT + WDGTST_SYSTICK_OFFSET; /* PRQA S 0306 */
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    SysTick->LOAD = WDGTST_WDG_INTERRUPT_TIMEOUT + WDGTST_SYSTICK_OFFSET; /* PRQA S 0306 */
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    SysTick->CTRL = SysTick_CTRL_CLKSOURCE_Msk | SysTick_CTRL_ENABLE_Msk; /* PRQA S 0306 */
    while ((WdgTst_TrgMask & WDGTST_INT_MASK_WDG) != WDGTST_INT_MASK_WDG)
    {
        /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
        SysTickValue = SysTick->VAL; /* PRQA S 0306 */
        if (SysTickValue < WDGTST_SYSTICK_OFFSET)
        {
            /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
            SysTick->CTRL  &= ~SysTick_CTRL_ENABLE_Msk; /* PRQA S 0306 */
            WdgTst_DetailResult.WdgTimeoutInterruptTestResult = WDGTST_RESULT_NOT_OK;
            WdgTst_DetailResult.TestResult = WDGTST_RESULT_NOT_OK;
            break;
        }
    }
    if (WdgTst_DetailResult.WdgTimeoutInterruptTestResult != WDGTST_RESULT_NOT_OK)
    {
        /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
        SysTick->CTRL  &= ~SysTick_CTRL_ENABLE_Msk; /* PRQA S 0306 */
        WdgTst_DetailResult.WdgTimeoutInterruptTestResult = WDGTST_RESULT_OK;
    }
    /* Service WDG to avoid reset before the test is complete. */
    SchM_Enter_WdgTst_WDGTST_EXCLUSIVE_AREA_01();
    /* Write first value of the service key */
    Base->SVCR = FEATURE_WDG_TRIGGER_VALUE_1;
    /* Avoid these 2 steps are optimized by compiler. */
    (void)Base->SVCR;
    /* Write second value of the service key */
    Base->SVCR = FEATURE_WDG_TRIGGER_VALUE_2;
    SchM_Exit_WdgTst_WDGTST_EXCLUSIVE_AREA_01();
}

#endif

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/

/**
 * @brief       Initializes the WdgTst module.
 * @return      void
 * @private
 */
WDGTST_FUNC void WdgTst_Lld_Init(void)
{
    WdgTst_DetailResult.TestResult = WDGTST_RESULT_NOT_TESTED;
    WdgTst_TrgMask = WDGTST_INT_MASK_NONE;
    WdgTst_SoftLockTestMask = 0U;
#if (WDGTST_WDG_ENABLE == STD_ON)
    WdgTst_DetailResult.WdgSoftLockTestResult = WDGTST_RESULT_NOT_TESTED;
    WdgTst_DetailResult.WdgTimeoutInterruptTestResult = WDGTST_RESULT_NOT_TESTED;
#endif
#if (WDGTST_EWDG_ENABLE == STD_ON)
    WdgTst_DetailResult.EwdgTestResult = WDGTST_RESULT_NOT_TESTED;
#endif
}

#if (WDGTST_WDG_ENABLE == STD_ON)
WDGTST_FUNC LOCAL_INLINE void WdgTst_Lld_SetSCB(void)
{
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    SCB->SHCSR |= SCB_SHCSR_BUSFAULTENA_Msk; /* PRQA S 0306 */
    /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
    SCB->CCR |= SCB_CCR_BFHFNMIGN_Msk; /* PRQA S 0306 */
}

/**
 * @brief       Preparation to start testing.
 * @return      void
 * @private
 */
WDGTST_FUNC void WdgTst_Lld_StartInit(void)
{
#ifdef WDGTST_USER_MODE_SUPPORT
    OsIf_Trusted_Call(WdgTst_Lld_SetSCB);
#else
    WdgTst_Lld_SetSCB();
#endif
    /* Install the Bus fault interrupt handler */
    (void)Platform_InstallIrqHandler(BusFault_IRQn, WdgTst_BusFaultHandler, &WdgTst_BackupBusFaultHandler);
}
#endif

/**
 * @brief       Update the state and restore the bus fault handler.
 * @private
 */
WDGTST_FUNC void WdgTst_Lld_TestComplete(void)
{
    if (WdgTst_DetailResult.TestResult != WDGTST_RESULT_NOT_OK)
    {
        WdgTst_DetailResult.TestResult = WDGTST_RESULT_OK;
    }
    /* Restore the Bus fault interrupt handler */
    (void)Platform_InstallIrqHandler(BusFault_IRQn, WdgTst_BackupBusFaultHandler, &WdgTst_DummyBusFaultIrqHandler);
}

/**
 * @brief       Get test result.
 * @return      WdgTst_TestResultType
 * @private
 */
WDGTST_FUNC WdgTst_TestResultType WdgTst_Lld_GetResult(void)
{
    return WdgTst_DetailResult.TestResult;
}


/**
 * @brief       Get details test result.
 * @param[out]  TestDetails
 * @private
 */
WDGTST_FUNC void WdgTst_Lld_GetDetailsResult(WdgTst_TestDetailsType *TestDetails)
{
    TestDetails->TestResult = WdgTst_DetailResult.TestResult;
#if (WDGTST_EWDG_ENABLE == STD_ON)
    TestDetails->EwdgTestResult = WdgTst_DetailResult.EwdgTestResult;
#endif
#if (WDGTST_WDG_ENABLE == STD_ON)
    TestDetails->WdgSoftLockTestResult = WdgTst_DetailResult.WdgSoftLockTestResult;
    TestDetails->WdgTimeoutInterruptTestResult = WdgTst_DetailResult.WdgTimeoutInterruptTestResult;
#endif
}



#if (WDGTST_WDG_ENABLE == STD_ON)
/**
 * @brief       Execute the tests for WDG
 * @return      Std_ReturnType
 * @retval      E_OK Test running properly
 * @retval      E_NOT_OK Test running failed
 * @private
 */
WDGTST_FUNC Std_ReturnType WdgTst_Lld_WdgStart(volatile WDG_Type *Base)
{
    Std_ReturnType Ret = E_OK;
#ifdef WDGTST_USER_MODE_SUPPORT
    if (TRUE == (uint8)OsIf_Trusted_Call_Return1param(WdgTst_Lld_WdgIsHardLocked, Base))
#else
    if (TRUE == WdgTst_Lld_WdgIsHardLocked(Base))
#endif
    {
        /* When hard lock bit set, the CR, WIN, TO are read-only. once set, the bit can only be clear by reset. */
        Ret = E_NOT_OK;
    }
    /* Soft lock, when this bit set, the CR, WIN, TO are read-only. this bit can be clear by
     * writing a valid sequence to SVR register. */
#ifdef WDGTST_USER_MODE_SUPPORT
    else if (TRUE == (uint8)OsIf_Trusted_Call_Return1param(WdgTst_Lld_WdgIsSoftLocked, Base))
#else
    else if (TRUE == WdgTst_Lld_WdgIsSoftLocked(Base))
#endif
    {
        /* Unlocks the WDG. */
#ifdef WDGTST_USER_MODE_SUPPORT
        Ret = (Std_ReturnType)OsIf_Trusted_Call_Return1param(WdgTst_Lld_WdgUnlock, Base);
#else
        Ret = WdgTst_Lld_WdgUnlock(Base);
#endif
    }
    else
    {
        /* nothing */
    }
    if (E_OK == Ret)
    {
        /* Test the lock function. */
#ifdef WDGTST_USER_MODE_SUPPORT
        Ret = (Std_ReturnType)OsIf_Trusted_Call_Return1param(WdgTst_Lld_WdgLockTest, Base);
#else
        Ret = WdgTst_Lld_WdgLockTest(Base);
#endif
        if (E_OK == Ret)
        {
#ifdef WDGTST_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(WdgTst_Lld_WdgInterruptTest, Base);
#else
            WdgTst_Lld_WdgInterruptTest(Base);
#endif
#if(WDGTST_ERROR_NOTIFICATION == STD_ON)
            if (WdgTst_DetailResult.WdgTimeoutInterruptTestResult == WDGTST_RESULT_NOT_OK)
            {
                (*(WdgTst_Config.ErrNotification))(WDGTST_LLD_MODULE_ID);
            }
#endif
        }
        else
        {
#if(WDGTST_ERROR_NOTIFICATION == STD_ON)
            (*(WdgTst_Config.ErrNotification))(WDGTST_LLD_MODULE_ID);
#endif
        }
    }
    return Ret;
}


/**
 * @brief       WDG instance interrupt handler.
 * @details     This function handles the different WDG instances interrupt.
 * @param[in]   Base WDG base pointer
 * @return      void
 * @private
 */
WDGTST_FUNC void WdgTst_Lld_WdgIrqHandler(volatile WDG_Type *Base)
{
    uint32 IntFlag = ((Base->INTF) & WDG_INTF_IF_MASK) >> WDG_INTF_IF_SHIFT;
    if (1U == IntFlag)
    {
        /* Clear interrupt flag */
        Base->INTF |= WDG_INTF_IF_MASK;
        if (1U == (((Base->CR) & WDG_CR_IBR_MASK) >> WDG_CR_IBR_SHIFT))
        {
            /* Check whether the WDG test of WdgTst trigger the interrupt */
            if (WDGTST_WDG_STARTED_MASK == (WdgTst_TrgMask & WDGTST_WDG_STARTED_MASK))
            {
                WdgTst_TrgMask |= (WdgTst_MaskType)WDGTST_INT_MASK_WDG;
            }
        }
    }
}

#endif

#if (WDGTST_EWDG_ENABLE == STD_ON)
/**
 * @brief       Execute the interrupt test for EWDG
 * @return      Std_ReturnType
 * @private
 */
WDGTST_FUNC LOCAL_INLINE Std_ReturnType WdgTst_Lld_EwdgInterruptTest(volatile EWDG_Type *Base)
{
    Std_ReturnType Ret = E_OK;
    /* Flag to store if the module is enabled */
    uint8 RegTmp = 0U;
    uint32 SysTickValue = 0U;
    /* Check if the EWDG instance is already enabled or if the windows values are not correct */
    if (TRUE == ((Base->CTRL & EWDG_CTRL_EN_MASK) >> EWDG_CTRL_EN_SHIFT))
    {
        Ret = E_NOT_OK;
    }
    else
    {
#if (WDGTST_EWDG_HAS_CLKSEL == STD_ON)
        /* Set clock source */
        Base->CLK_CTRL &= ~EWDG_CLK_CTRL_CLKSEL_MASK;
        Base->CLK_CTRL |= EWDG_CLK_CTRL_CLKSEL(WDGTST_EWDG_CLKSRC_SIRC);
#endif
        /* Set compare high and low values */
        Base->CMPH = WDGTST_EWDG_CMPH_VALUE;
        Base->CMPL = 0U;
        /* Configure the Control register and enable the instance.
         * Set the values that are not affected by the input pin.
         */
        RegTmp |= (uint8)EWDG_CTRL_EN(1U);
        RegTmp |= (uint8)EWDG_CTRL_INTEN(1U);
        /* Write the configuration into the Control register */
        Base->CTRL = RegTmp;
        WdgTst_TrgMask |= WDGTST_EWDG_STARTED_MASK;
        /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
        SysTick->VAL = WDGTST_WDG_INTERRUPT_TIMEOUT + WDGTST_SYSTICK_OFFSET; /* PRQA S 0306 */
        /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
        SysTick->LOAD = WDGTST_EWDG_INTERRUPT_TIMEOUT + WDGTST_SYSTICK_OFFSET; /* PRQA S 0306 */
        /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
        SysTick->CTRL = SysTick_CTRL_CLKSOURCE_Msk | SysTick_CTRL_ENABLE_Msk; /* PRQA S 0306 */
        while ((WdgTst_TrgMask & WDGTST_INT_MASK_EWDG) != WDGTST_INT_MASK_EWDG)
        {
            /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
            SysTickValue = SysTick->VAL; /* PRQA S 0306 */
            if (SysTickValue < WDGTST_SYSTICK_OFFSET)
            {
                /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
                SysTick->CTRL  &= ~SysTick_CTRL_ENABLE_Msk; /* PRQA S 0306 */
                WdgTst_DetailResult.EwdgTestResult = WDGTST_RESULT_NOT_OK;
                WdgTst_DetailResult.TestResult = WDGTST_RESULT_NOT_OK;
                Ret = E_NOT_OK;
                break;
            }
        }
        if (WdgTst_DetailResult.EwdgTestResult != WDGTST_RESULT_NOT_OK)
        {
            /* MR12 RULE 11.4 VIOLATION: The integer is needed as the base address to manipulate the registers. */
            SysTick->CTRL  &= ~SysTick_CTRL_ENABLE_Msk; /* PRQA S 0306 */
            WdgTst_DetailResult.EwdgTestResult = WDGTST_RESULT_OK;
        }
    }
    return Ret;
}

/**
 * @brief       Execute the tests for EWDG
 * @return      Std_ReturnType
 * @retval      E_OK EWDG test execute properly
 * @retval      E_NOT_OK EWDG has been enabled and cannot be tested
 * @private
 */
WDGTST_FUNC Std_ReturnType WdgTst_Lld_EwdgStart(volatile EWDG_Type *Base)
{
#ifdef WDGTST_USER_MODE_SUPPORT
    Std_ReturnType Ret = (Std_ReturnType)OsIf_Trusted_Call_Return1param(WdgTst_Lld_EwdgInterruptTest, Base);
#else
    Std_ReturnType Ret = WdgTst_Lld_EwdgInterruptTest(Base);
#endif
#if(WDGTST_ERROR_NOTIFICATION == STD_ON)
    if (WdgTst_DetailResult.EwdgTestResult == WDGTST_RESULT_NOT_OK)
    {
        (*(WdgTst_Config.ErrNotification))(WDGTST_LLD_MODULE_ID);
    }
#endif
    return Ret;
}


/**
 * @brief       EWDG instance interrupt handler.
 * @details     This function handles the different EWDG instances interrupt.
 * @param[in]   Base EWDG base pointer
 * @return      void
 * @private
 */
WDGTST_FUNC void WdgTst_Lld_EwdgIrqHandler(volatile EWDG_Type *Base)
{
    /* Check interrupt enable bit */
    if (1U == (((Base->CTRL) & EWDG_CTRL_INTEN_MASK) >> EWDG_CTRL_INTEN_SHIFT))
    {
        /* Check whether the EWDG test of WdgTst trigger the interrupt */
        if ((WdgTst_TrgMask & WDGTST_EWDG_STARTED_MASK) == WDGTST_EWDG_STARTED_MASK)
        {
            Base->CTRL &= ~EWDG_CTRL_INTEN_MASK;
            WdgTst_TrgMask |= (WdgTst_MaskType)WDGTST_INT_MASK_EWDG;
        }
    }
}

#endif


#define WDGTST_STOP_SEC_CODE
#include "WdgTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

