/**
 * @file    WdgTst.h
 * @version V2.0.0
 *
 * @brief   Safety library WdgTst module interface
 * @details API implementation for WdgTst driver
 *
 * @addtogroup WDGTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 SafLib
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : None
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
 *=================================================================================================*/

#ifndef WDGTST_H
#define WDGTST_H

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
 *=================================================================================================*/
#include "WdgTst_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
 *=================================================================================================*/
#define WDGTST_VENDOR_ID                      (180)
#define WDGTST_MODULE_ID                      (524)
#define WDGTST_AR_RELEASE_MAJOR_VERSION       (4)
#define WDGTST_AR_RELEASE_MINOR_VERSION       (4)
#define WDGTST_AR_RELEASE_REVISION_VERSION    (0)
#define WDGTST_SW_MAJOR_VERSION               (2)
#define WDGTST_SW_MINOR_VERSION               (0)
#define WDGTST_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
 *=================================================================================================*/
/* Check if WdgTst.h and WdgTst_Types.h are of the same vendor */
#if (WDGTST_VENDOR_ID != WDGTST_TYPES_VENDOR_ID)
#error "WdgTst.h and WdgTst_Types.h have different vendor ids"
#endif

/* Check if WdgTst.h and WdgTst_Types.h are of the same Autosar version */
#if ((WDGTST_AR_RELEASE_MAJOR_VERSION != WDGTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
       (WDGTST_AR_RELEASE_MINOR_VERSION != WDGTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
       (WDGTST_AR_RELEASE_REVISION_VERSION != WDGTST_TYPES_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of WdgTst.h and WdgTst_Types.h are different"
#endif

/* Check if WdgTst.h and WdgTst_Types.h are of the same software version */
#if ((WDGTST_SW_MAJOR_VERSION != WDGTST_TYPES_SW_MAJOR_VERSION) || \
       (WDGTST_SW_MINOR_VERSION != WDGTST_TYPES_SW_MINOR_VERSION) || \
       (WDGTST_SW_PATCH_VERSION != WDGTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of WdgTst.h and WdgTst_Types.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
 *=================================================================================================*/
/**
 * @defgroup  WdgTst_Service_ID
 * @brief     Service ID number for all WDGTST driver services
 * @sw_type   sw_detail
 * @trace YTSDS_WdgTst_884
 * @{
 */
/** API Service ID for WdgTst_Init */
#define WDGTST_SID_INIT                 (0x00U)
/** API Service ID for WdgTst_Start */
#define WDGTST_SID_START                (0x01U)
/** API Service ID for WdgTst_GetResult */
#define WDGTST_SID_GET_RESULT           (0x02U)
/** API Service ID for WdgTst_GetDetailsResult */
#define WDGTST_SID_GET_DETAILS_RESULT   (0x03U)
/** API Service ID for WdgTst_GetVersionInfo */
#define WDGTST_SID_GET_VERSION_INFO     (0x04U)
/** @} */

/**
 * @defgroup  WdgTst_Development_Errors
 * @brief     WdgTst development errors
 * @sw_type   sw_detail
 * @trace YTSDS_WdgTst_883
 * @{
 */
/** API WdgTst_Init service called while the WdgTst driver has already been initialised. */
#define WDGTST_E_ALREADY_INITIALIZED    (0x00U)
/** API WdgTst_Init service called with wrong parameter. */
#define WDGTST_E_INIT_FAILED            (0x01U)
/** API service used without module initialization. */
#define WDGTST_E_UNINIT                 (0x02U)
/** WdgTst API service is called using an invalid pointer (e.g. the pointer should not be NULL). */
#define WDGTST_E_PARAM_POINTER          (0x03U)
/** @} */


/*==================================================================================================
 *                                    FUNCTION PROTOTYPES
 *=================================================================================================*/
#define WDGTST_START_SEC_CODE
#include "WdgTst_MemMap.h"

/**
 * @brief       Service for WdgTst initialization.
 * @details     This function initializes the WdgTst driver, include state variables, structures, etc.
 * @param[in]   ConfigPtr Pointer to configuration set
 * @table       @service_id:       0x00        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      void
 * @sw_type     sw_arch
 */
WDGTST_FUNC void WdgTst_Init(const WdgTst_ConfigType *ConfigPtr);

/**
 * @brief       Service for executing Watchdog Test.
 * @table       @service_id:       0x01        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      Std_ReturnType
 * @retval      E_OK Executed successfully
 * @retval      E_NOT_OK Executed failed
 * @sw_type     sw_arch
 */
WDGTST_FUNC Std_ReturnType WdgTst_Start(void);

#if (WDGTST_GET_RESULT_API == STD_ON)
/**
 * @brief       Service returns the result of the last executed Watchdog Test.
 * @table       @service_id:       0x02        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      WdgTst_TestResultType
 * @retval      WDGTST_RESULT_OK Test passed
 * @retval      WDGTST_RESULT_NOT_OK Test failed
 * @retval      WDGTST_RESULT_NOT_TESTED Test not executed
 * @sw_type     sw_arch
 */
WDGTST_FUNC WdgTst_TestResultType WdgTst_GetResult(void);

/**
 * @brief       Service returns the detail result of the Watchdog Test.
 * @table       @service_id:       0x03        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @param[out]  WdgTst_TestDetailsType *ResultDetail
 * @return      void
 * @sw_type     sw_arch
 */
WDGTST_FUNC void WdgTst_GetDetailsResult(WdgTst_TestDetailsType *ResultDetail);
#endif

#if (WDGTST_VERSION_INFO_API == STD_ON)
/**
 * @brief       Service returns the version information of this module.
 * @details     The version information includes: Module Id, Vendor Id, Vendor specific version number.
 * @param[out]  VersionInfo Pointer to where to store the version information of this module.
 * @table       @service_id:       0x04        \n
 *              @is_reentrant:     true       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return     void
 * @sw_type    sw_arch
 */
WDGTST_FUNC void WdgTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif


#define WDGTST_STOP_SEC_CODE
#include "WdgTst_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* WDGTST_H */

/** @} */
