/**
* @file    RamTst_Lld.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : RamTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations List
 * PRQA S 0488 Rule 18.4: The +, -, += and -= operators should not be applied to an expression of pointer type.
 * PRQA S 0306 Rule 11.4: A conversion should not be performed between a pointer to object and an integer type.
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "RamTst_Lld.h"
#include "OsIf.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define RAMTST_LLD_VENDOR_ID_C                     (180)
#define RAMTST_LLD_AR_REL_MAJOR_VER_C              (4)
#define RAMTST_LLD_AR_REL_MINOR_VER_C              (4)
#define RAMTST_LLD_AR_REL_REVISION_VER_C           (0)
#define RAMTST_LLD_SW_MAJOR_VER_C                  (2)
#define RAMTST_LLD_SW_MINOR_VER_C                  (0)
#define RAMTST_LLD_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and RAMTST_LLD header file are of the same vendor */
#if (RAMTST_LLD_VENDOR_ID_C != RAMTST_LLD_VENDOR_ID)
#error "RamTst_Lld.c and RamTst_Lld.h have different vendor ids"
#endif

/* Check if source file and RAMTST_LLD header file are of the same Autosar version */
#if (( RAMTST_LLD_AR_REL_MAJOR_VER_C != RAMTST_LLD_AR_REL_MAJOR_VER) || \
     ( RAMTST_LLD_AR_REL_MINOR_VER_C != RAMTST_LLD_AR_REL_MINOR_VER) || \
     ( RAMTST_LLD_AR_REL_REVISION_VER_C != RAMTST_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of RamTst_Lld.c and RamTst_Lld.h are different"
#endif

/* Check if source file and RAMTST_LLD header file are of the same Software version */
#if (( RAMTST_LLD_SW_MAJOR_VER_C != RAMTST_LLD_SW_MAJOR_VER) || \
     ( RAMTST_LLD_SW_MINOR_VER_C != RAMTST_LLD_SW_MINOR_VER) || \
     ( RAMTST_LLD_SW_PATCH_VER_C != RAMTST_LLD_SW_PATCH_VER))
#error "Software Version Numbers of RamTst_Lld.c and RamTst_Lld.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
extern RamTst_StateType RamTstState;

/*==================================================================================================
                                                LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#define RAMTST_START_SEC_CODE
#include "RamTst_MemMap.h"

#if (RAMTST_CHECKERBOARD_TEST_SELECTED == STD_ON)
RAMTST_FUNC static Std_ReturnType RamTst_Lld_CheckerboardAlgorithm(uint32 *const StartAddress, const uint32 Length);
#endif

#if (RAMTST_MARCH_TEST_SELECTED == STD_ON)
RAMTST_FUNC static Std_ReturnType RamTst_Lld_MarchAlgorithm(uint32 *const StartAddress, const uint32 Length);
#endif

#if (RAMTST_WALK_PATH_TEST_SELECTED == STD_ON)
RAMTST_FUNC static Std_ReturnType RamTst_Lld_WalkPathAlgorithm(uint32 *const StartAddress, const uint32 Length);
#endif

#if (RAMTST_GALPAT_TEST_SELECTED == STD_ON)
RAMTST_FUNC static Std_ReturnType RamTst_Lld_GalpatAlgorithm(uint32 *const StartAddress, const uint32 Length);
#endif

#if (RAMTST_TRANSP_GALPAT_TEST_SELECTED == STD_ON)
RAMTST_FUNC static Std_ReturnType RamTst_Lld_TranspGalpatAlgorithm(uint32 *const StartAddress, const uint32 Length);
#endif

#if (RAMTST_ABRAHAM_TEST_SELECTED == STD_ON)
RAMTST_FUNC static Std_ReturnType RamTst_Lld_AbrahamAlgorithm(uint32 *const StartAddress, const uint32 Length);
#endif

#define RAMTST_STOP_SEC_CODE
#include "RamTst_MemMap.h"

/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
#define RAMTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "RamTst_MemMap.h"
/**
 * @brief   The main state of the RAM test
 * @details After executing RamTst_SelectAlgParams(), the RamTst_MainFunction() can resume execution from where it was previously interrupted
 *          to continue with the new algorithm test, rather than starting over from the beginning.
 * @private
 */
RAMTST_VAR static RamTst_MainStateType MainState;
#define RAMTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "RamTst_MemMap.h"

#define RAMTST_START_SEC_CONST_UNSPECIFIED
#include "RamTst_MemMap.h"
/**
 * @brief   The array of RAM test algorithm functions.
 * @details The array of RAM test algorithm functions is used to select the appropriate algorithm function based on the selected algorithm.
 * @private
 */
RAMTST_CONST static const RamTst_AlgorithmFunctionType RamTst_AlgorithmFunction[7U] =
{
    NULL_PTR, /* RAMTST_ALGORITHM_UNDEFINED */
#if (RAMTST_CHECKERBOARD_TEST_SELECTED == STD_ON)
    RamTst_Lld_CheckerboardAlgorithm,
#else
    NULL_PTR,
#endif
#if (RAMTST_MARCH_TEST_SELECTED == STD_ON)
    RamTst_Lld_MarchAlgorithm,
#else
    NULL_PTR,
#endif
#if (RAMTST_WALK_PATH_TEST_SELECTED == STD_ON)
    RamTst_Lld_WalkPathAlgorithm,
#else
    NULL_PTR,
#endif
#if (RAMTST_GALPAT_TEST_SELECTED == STD_ON)
    RamTst_Lld_GalpatAlgorithm,
#else
    NULL_PTR,
#endif
#if (RAMTST_TRANSP_GALPAT_TEST_SELECTED == STD_ON)
    RamTst_Lld_TranspGalpatAlgorithm,
#else
    NULL_PTR,
#endif
#if (RAMTST_ABRAHAM_TEST_SELECTED == STD_ON)
    RamTst_Lld_AbrahamAlgorithm,
#else
    NULL_PTR,
#endif
};
#define RAMTST_STOP_SEC_CONST_UNSPECIFIED
#include "RamTst_MemMap.h"

/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
#define RAMTST_START_SEC_CODE
#include "RamTst_MemMap.h"

/**
 * @brief            Reading the data from the RAM.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        Address: the address of data that need to be read
 * @return           uint32: the data which read from the RAM address
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_358
 */
RAMTST_FUNC static inline uint32 RamTst_Lld_ReadDataFromRam(const uint32 *Address)
{
    return *Address;
}

/**
 * @brief            Saving the data from the source address to the backup address.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        SourceAddress: the source address of data that need to be saved
 * @param[in]        BackupAddress: the backup address of data to be saved
 * @param[in]        Length: the length of data to be saved
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_361
 */
RAMTST_FUNC static void RamTst_Lld_SaveData(const uint32 *SourceAddress, uint32 *BackupAddress, const uint32 Length)
{
    const uint32 *Src = SourceAddress;
    uint32 *Dst = BackupAddress;
    for (uint32 Index = 0; Index < Length; ++Index)
    {
        *Dst = *Src;
        ++Src;
        ++Dst;
    }
}

/**
 * @brief            Restoring the data from the backup address to the destination address.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        DestAddress: the destination address of data that need to be restored
 * @param[in]        BackupAddress: the backup address of data to be restored
 * @param[in]        Length: the length of data to be restored
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_360
 */
RAMTST_FUNC static void RamTst_Lld_RestoreData(uint32 *DestAddress, const uint32 *BackupAddress, const uint32 Length)
{
    uint32 *Dst = DestAddress;
    const uint32 *Src = BackupAddress;
    for (uint32 Index = 0; Index < Length; ++Index)
    {
        *Dst = *Src;
        ++Src;
        ++Dst;
    }
}

/**
 * @brief            Filling the data with the fill data to the destination address.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        DestAddress: the destination address of data that need to be filled
 * @param[in]        FillData: the fill data to be filled
 * @param[in]        Length: the length of data to be filled
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_353
 */
RAMTST_FUNC static void RamTst_Lld_FillPattern(uint32 *DestAddress, const uint32 FillData, const uint32 Length)
{
    uint32 *Dst = DestAddress;
    for (uint32 Index = 0; Index < Length; ++Index)
    {
        *Dst = FillData;
        ++Dst;
    }
}

#if (RAMTST_CHECKERBOARD_TEST_SELECTED == STD_ON)
/**
 * @brief            Checkerboard algorithm for RAM test.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        StartAddress: the start address of the RAM to be tested
 * @param[in]        Length: the length of the RAM to be tested
 * @return           Std_ReturnType
 * @retval           E_OK: the test is passed
 * @retval           E_NOT_OK: the test is failed
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_352
 */
RAMTST_FUNC static Std_ReturnType RamTst_Lld_CheckerboardAlgorithm(uint32 *const StartAddress, const uint32 Length)
{
    Std_ReturnType Ret = E_OK;
    uint32 RamInitData = 0xAAAAAAAAU;
    RamTst_NumberOfTestedCellsType Index;
    uint8 TestCycle = 0U;
    /*First pattern is writes 0xAAAAAAAA - 0x55555555 - 0xAAAAAAAA -...*/
    /*Second pattern is writes 0x55555555 - 0xAAAAAAAA - 0x55555555 -...*/
    do
    {
        /*Write the pattern*/
        for (Index = 0U; Index < Length; ++Index)
        {
            if (0U == (Index % 2U))
            {
                /**
                 * MR12 Rule 18.4 VIOLATION: An addition or subtraction operation is being performed on an expression of pointer type.
                 * Pointer operation is more concise for data processing, thus, couldn't adhere to M3CM Rule-18.4.
                 */
                *(StartAddress + Index) = RamInitData; /* PRQA S 0488*/
            }
            else
            {
                *(StartAddress + Index) = ~RamInitData; /* PRQA S 0488*/
            }
        }
        /*Verify the pattern*/
        for (Index = 0U; (Index < Length) && (E_OK == Ret); ++Index)
        {
            if (0U == (Index % 2U))
            {
                if (RamInitData != RamTst_Lld_ReadDataFromRam(StartAddress + Index)) /* PRQA S 0488*/
                {
                    Ret = E_NOT_OK;
                }
            }
            else
            {
                if (~RamInitData != RamTst_Lld_ReadDataFromRam(StartAddress + Index)) /* PRQA S 0488*/
                {
                    Ret = E_NOT_OK;
                }
            }
        }
        TestCycle++;
        /*Change the pattern*/
        RamInitData = ~RamInitData;
    } while ((TestCycle < 2U) && (E_OK == Ret));
    return Ret;
}

#endif

#if (RAMTST_MARCH_TEST_SELECTED == STD_ON)
/**
 * @brief            March algorithm for RAM test.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        StartAddress: the start address of the RAM to be tested
 * @param[in]        Length: the length of the RAM to be tested
 * @return           Std_ReturnType
 * @retval           E_OK: the test is passed
 * @retval           E_NOT_OK: the test is failed
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_359
 */
RAMTST_FUNC static Std_ReturnType RamTst_Lld_MarchAlgorithm(uint32 *const StartAddress, const uint32 Length)
{
    Std_ReturnType Ret = E_OK;
    const uint32 RamInitData = 0U;
    RamTst_NumberOfTestedCellsType Index;
    /* Algorithm March X
    Step1: write 0 with ascending addressing order;
    Step2: read 0 and write 1 with ascending addressing order;
    Step3: read 1 and write 0 with descending addressing order;
    Step4: read 0 with ascending addressing order.
    */
    /* write 0 with ascending addressing order*/
    for (Index = 0U; Index < Length; ++Index)
    {
        *(StartAddress + Index) = RamInitData; /* PRQA S 0488*/
    }
    /* read 0 and write 1 with ascending addressing order*/
    for (Index = 0U; (Index < Length) && (E_OK == Ret); ++Index)
    {
        if (0x0U != RamTst_Lld_ReadDataFromRam(StartAddress + Index)) /* PRQA S 0488*/
        {
            Ret = E_NOT_OK;
        }
        *(StartAddress + Index) = ~RamInitData; /* PRQA S 0488*/
    }
    /* read 1 and write 0 with descending addressing order*/
    for (Index = 0U; (Index < Length) && (E_OK == Ret); ++Index)
    {
        uint32 DescendingIndex = Length - Index - 1U;
        if (0xFFFFFFFFU != RamTst_Lld_ReadDataFromRam(StartAddress + DescendingIndex)) /* PRQA S 0488*/
        {
            Ret = E_NOT_OK;
        }
        *(StartAddress + DescendingIndex) = RamInitData; /* PRQA S 0488*/
    }
    /* read 0 with ascending addressing order*/
    for (Index = 0U; (Index < Length) && (E_OK == Ret); ++Index)
    {
        if (0x0U != RamTst_Lld_ReadDataFromRam(StartAddress + Index)) /* PRQA S 0488*/
        {
            Ret = E_NOT_OK;
        }
    }
    return Ret;
}
#endif

#if (RAMTST_WALK_PATH_TEST_SELECTED == STD_ON)
/**
 * @brief            Walk path algorithm for RAM test.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        StartAddress: the start address of the RAM to be tested
 * @param[in]        Length: the length of the RAM to be tested
 * @return           Std_ReturnType
 * @retval           E_OK: the test is passed
 * @retval           E_NOT_OK: the test is failed
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_363
 */
RAMTST_FUNC static Std_ReturnType RamTst_Lld_WalkPathAlgorithm(uint32 *const StartAddress, const uint32 Length)
{
    Std_ReturnType Ret = E_OK;
    uint32 RamInitData = 0U;
    RamTst_NumberOfTestedCellsType Index;
    uint8 TestCycle = 0U;
    /* First test with Walking 1 pattern*/
    /* Second test with walking 0 pattern*/
    do
    {
        /* Initialization all the RAM that need to be tested*/
        for (Index = 0U; Index < Length; ++Index)
        {
            *(StartAddress + Index) = RamInitData; /* PRQA S 0488*/
        }
        for (Index = 0U; (Index < Length) && (E_OK == Ret); ++Index)
        {
            *(StartAddress + Index) = ~RamInitData; /* PRQA S 0488*/
            for (RamTst_NumberOfTestedCellsType CellNumber = 0U; (CellNumber < Length) && (E_OK == Ret); ++CellNumber)
            {
                if (CellNumber != Index)
                {
                    if (RamInitData != RamTst_Lld_ReadDataFromRam(StartAddress + CellNumber)) /* PRQA S 0488*/
                    {
                        Ret = E_NOT_OK;
                    }
                }
                else
                {
                    if (~RamInitData != RamTst_Lld_ReadDataFromRam(StartAddress + CellNumber)) /* PRQA S 0488*/
                    {
                        Ret = E_NOT_OK;
                    }
                }
            }
            /*Reset the current RAM cell*/
            *(StartAddress + Index) = RamInitData; /* PRQA S 0488*/
        }
        RamInitData = ~RamInitData;
        ++TestCycle;
    } while ((TestCycle < 2U) && (E_OK == Ret));
    return Ret;
}
#endif

#if (RAMTST_GALPAT_TEST_SELECTED == STD_ON)
/**
 * @brief            Galpat algorithm for RAM test.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        StartAddress: the start address of the RAM to be tested
 * @param[in]        Length: the length of the RAM to be tested
 * @return           Std_ReturnType
 * @retval           E_OK: the test is passed
 * @retval           E_NOT_OK: the test is failed
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_354
 */
RAMTST_FUNC static Std_ReturnType RamTst_Lld_GalpatAlgorithm(uint32 *const StartAddress, const uint32 Length)
{
    Std_ReturnType Ret = E_OK;
    const uint32 RamInitData = 0x0U;
    RamTst_NumberOfTestedCellsType Index;
    /* Initialization all the RAM Cells that need to be tested to 0 */
    for (Index = 0U; Index < Length; ++Index)
    {
        *(StartAddress + Index) = RamInitData; /* PRQA S 0488*/
    }
    for (Index = 0U; (Index < Length) && (E_OK == Ret); ++Index)
    {
        *(StartAddress + Index) = ~RamInitData; /* PRQA S 0488*/
        for (RamTst_NumberOfTestedCellsType CellNumber = 0U; (CellNumber < Length) && (E_OK == Ret); ++CellNumber)
        {
            if (CellNumber == Index)
            {
                continue;
            }
            /* Flip cell CellNumber to 1 */
            *(StartAddress + CellNumber) = ~RamInitData; /* PRQA S 0488*/
            /* Verify the Index memory cell is unaffected */
            if (~RamInitData != RamTst_Lld_ReadDataFromRam(StartAddress + Index)) /* PRQA S 0488*/
            {
                Ret = E_NOT_OK;
            }
            else
            {
                /* Flip cell CellNumber back to 0 */
                *(StartAddress + CellNumber) = RamInitData; /* PRQA S 0488*/
                /* Verify the Index memory cell is unaffected */
                if (~RamInitData != RamTst_Lld_ReadDataFromRam(StartAddress + Index)) /* PRQA S 0488*/
                {
                    Ret = E_NOT_OK;
                }
            }
        }
    }
    return Ret;
}
#endif

#if (RAMTST_TRANSP_GALPAT_TEST_SELECTED == STD_ON)
/**
 * @brief            Transparent Galpat algorithm for RAM test.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        StartAddress: the start address of the RAM to be tested
 * @param[in]        Length: the length of the RAM to be tested
 * @return           Std_ReturnType
 * @retval           E_OK: the test is passed
 * @retval           E_NOT_OK: the test is failed
 * @sw_type          sw_detail
 * @private
 *
 */
RAMTST_FUNC static Std_ReturnType RamTst_Lld_TranspGalpatAlgorithm(uint32 *const StartAddress, const uint32 Length)
{
}
#endif

#if (RAMTST_ABRAHAM_TEST_SELECTED == STD_ON)
/**
 * @brief            Abraham algorithm for RAM test.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        StartAddress: the start address of the RAM to be tested
 * @param[in]        Length: the length of the RAM to be tested
 * @return           Std_ReturnType
 * @retval           E_OK: the test is passed
 * @retval           E_NOT_OK: the test is failed
 * @sw_type          sw_detail
 * @private
 *
 */
RAMTST_FUNC static Std_ReturnType RamTst_Lld_AbrahamAlgorithm(uint32 *const StartAddress, const uint32 Length)
{
}
#endif

/**
 * @brief            Algorithm test function for RAM test.
 *
 * @table            @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        Algorithm: the algorithm which used for the RAM test
 * @param[in]        RamTstAlgTst: the RAM test parameters
 * @return           Std_ReturnType
 * @retval           E_OK: the test is passed
 * @retval           E_NOT_OK: the test is failed or the algorithm is not supported
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_351
 */
RAMTST_FUNC static Std_ReturnType RamTst_Lld_AlgorithmTest(RamTst_AlgorithmType Algorithm, RamTst_AlgTstType RamTstAlgTst)
{
    Std_ReturnType Ret = E_OK;
    uint32 PtrTemp;
    uint32 PtrSpNew;
    uint32 ControlData;
    uint32 PtrSp;
    if (NULL_PTR == RamTst_AlgorithmFunction[Algorithm])
    {
        Ret = E_NOT_OK;
    }
    else if (RamTstAlgTst.CurrentBlockRegion == BLOCK_REGION_IS_STACK)
    {
        /* Backup the stack data */
        /**
         * MR12 Rule 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * In this specific case, The integer address must be cast to a pointer type to perform RAM operations. thus, couldn't adhere to M3CM Rule-11.4
         */
        RamTst_Lld_SaveData((uint32 *)RamTstState.CellAddress, (uint32 *)RamTstState.BackupStartAddress, RamTstState.NumExecutedCell); /* PRQA S 0306*/
        ControlData = __get_CONTROL();
        /*Get MSP is active stack point is 0*/
        if (0U == (ControlData & CONTROL_SPSEL_Msk))
        {
            /* MSP is active stack pointer*/
            PtrSp = __get_MSP();
            /*Change the stack pointer to backup area*/
            PtrTemp = PtrSp - RamTstState.CellAddress;
            PtrSpNew = RamTstState.BackupStartAddress + PtrTemp; /*stack backup end Address*/
            __set_MSP(PtrSpNew);
            Ret = RamTst_AlgorithmFunction[Algorithm]((uint32 *)RamTstState.CellAddress, RamTstState.NumExecutedCell); /* PRQA S 0306*/
            /*Restore the stack data*/
            RamTst_Lld_RestoreData((uint32 *)RamTstState.CellAddress, (uint32 *)RamTstState.BackupStartAddress, RamTstState.NumExecutedCell); /* PRQA S 0306*/
            /*Restore the stack pointer*/
            PtrSp = __get_MSP();
            PtrTemp = PtrSp - RamTstState.BackupStartAddress;
            PtrSpNew = RamTstState.CellAddress + PtrTemp; /*stack backup end Address*/
            __set_MSP(PtrSpNew);
        }
        else
        {
            /* PSP is active stack pointer*/
            PtrSp = __get_PSP();
            /*Change the stack pointer to backup area*/
            PtrTemp = PtrSp - RamTstState.CellAddress;
            PtrSpNew = RamTstState.BackupStartAddress + PtrTemp; /*stack backup end Address*/
            __set_PSP(PtrSpNew);
            Ret = RamTst_AlgorithmFunction[Algorithm]((uint32 *)RamTstState.CellAddress, RamTstState.NumExecutedCell); /* PRQA S 0306*/
            /*Restore the stack data*/
            RamTst_Lld_RestoreData((uint32 *)RamTstState.CellAddress, (uint32 *)RamTstState.BackupStartAddress, RamTstState.NumExecutedCell); /* PRQA S 0306*/
            /*Restore the stack pointer*/
            PtrSp = __get_PSP();
            PtrTemp = PtrSp - RamTstState.BackupStartAddress;
            PtrSpNew = RamTstState.CellAddress + PtrTemp; /*stack backup end Address*/
            __set_PSP(PtrSpNew);
        }
    }
    else
    {
        if (RamTstAlgTst.CurrentBlockPolicy == RAMTST_NON_DESTRUCTIVE)
        {
            RamTst_Lld_SaveData((uint32 *)RamTstAlgTst.CellAddress, (uint32 *)RamTstAlgTst.BackupAddress, RamTstAlgTst.NumberCells); /* PRQA S 0306*/
        }
        Ret =  RamTst_AlgorithmFunction[Algorithm]((uint32 *)RamTstAlgTst.CellAddress, RamTstAlgTst.NumberCells); /* PRQA S 0306*/
        if (RamTstAlgTst.CurrentBlockPolicy == RAMTST_NON_DESTRUCTIVE)
        {
            RamTst_Lld_RestoreData((uint32 *)RamTstAlgTst.CellAddress, (uint32 *)RamTstAlgTst.BackupAddress, RamTstAlgTst.NumberCells); /* PRQA S 0306*/
        }
        else
        {
            RamTst_Lld_FillPattern((uint32 *)RamTstAlgTst.CellAddress, RamTstAlgTst.FillData, RamTstAlgTst.NumberCells); /* PRQA S 0306*/
        }
    }
    MCAL_QUA_FAULT_INJECTION(RAMTST_LLD_TEST_FAILED);
    return Ret;
}

/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief           This function initializes the RamTst_Lld module.
 * @details
 *
 * @table            @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 *
 * @param[in]        ConfigPtr Pointer to the configuration set.
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_357
 */
RAMTST_FUNC void RamTst_Lld_Init(const RamTst_ConfigType *ConfigPtr)
{
    RamTstState.ExecStatus = RAMTST_EXECUTION_STOPPED;
    RamTstState.ActiveAlgParamID = ConfigPtr->RamTstDefaultAlgParamsId;
    RamTstState.MinNumTestedCell = ConfigPtr->RamTstMinNumberOfTestedCells;
    RamTstState.RamTstNumOfAlgParamSets = ConfigPtr->RamTstNumberOfAlgParamSets;
    RamTstState.ActiveAlgParamPtr = &ConfigPtr->RamTstAlgConfigPtr[RamTstState.ActiveAlgParamID - 1U];
    RamTstState.NumTestedCell = RamTstState.ActiveAlgParamPtr->RamTstNumberOfTestedCells;
    RamTstState.MaxNumTestedCell = RamTstState.ActiveAlgParamPtr->RamTstMaxNumberOfTestedCells;
    RamTstState.ExtNumTestedCell = RamTstState.ActiveAlgParamPtr->RamTstExtNumberOfTestedCells;
    RamTstState.NumBlocks = RamTstState.ActiveAlgParamPtr->RamTstNumberOfBlocks;
    RamTstState.RamTstAlgorithm = RamTstState.ActiveAlgParamPtr->RamTstAlgorithm;
    RamTstState.BackupStartAddress = RamTstState.ActiveAlgParamPtr->RamTstBackupStartAddress;
    RamTstState.CellAddress = 0U;
    /*init state parameters which used by MainFunction*/
    RamTstState.MainStatePtr = &MainState;
    RamTstState.MainStatePtr->MainActiveAlgParamID = RamTstState.ActiveAlgParamID;
    RamTstState.MainStatePtr->MainBlockTestedIndex = 0U;
    RamTstState.MainStatePtr->MainCellAddress = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[0U].RamTstStartAddress;
    RamTstState.MainStatePtr->MainNumBlocks = RamTstState.NumBlocks;
    for (RamTst_NumberOfBlocksType Block = 0; Block < (RamTst_NumberOfBlocksType)RAMTST_MAX_BLOCK; ++Block)
    {
        RamTstState.RamTstBlockResultBuffer[Block] = RAMTST_RESULT_NOT_TESTED;
    }
    RamTstState.RamTestResult = RAMTST_RESULT_NOT_TESTED;
}

/**
 * @brief           This function de-initializes the RamTst_Lld module.
 * @details
 *
 * @table            @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 *
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_356
 */
RAMTST_FUNC void RamTst_Lld_DeInit(void)
{
    RamTstState.ExecStatus = RAMTST_EXECUTION_UNINIT;
    RamTstState.ActiveAlgParamID = 0U;
    RamTstState.MinNumTestedCell = 0U;
    RamTstState.RamTstNumOfAlgParamSets = 0U;
    RamTstState.ActiveAlgParamPtr = NULL_PTR;
    RamTstState.NumTestedCell = 0U;
    RamTstState.MaxNumTestedCell = 0U;
    RamTstState.ExtNumTestedCell = 0U;
    RamTstState.NumBlocks = 0U;
    RamTstState.RamTstAlgorithm = RAMTST_ALGORITHM_UNDEFINED;
    RamTstState.CellAddress = 0U;
    RamTstState.MainStatePtr->MainActiveAlgParamID = 0U;
    RamTstState.MainStatePtr->MainBlockTestedIndex = 0U;
    for (RamTst_NumberOfBlocksType Block = 0; Block < (RamTst_NumberOfBlocksType)RAMTST_MAX_BLOCK; ++Block)
    {
        RamTstState.RamTstBlockResultBuffer[Block] = RAMTST_RESULT_NOT_TESTED;
    }
    RamTstState.RamTestResult = RAMTST_RESULT_NOT_TESTED;
}

/**
 * @brief           This function is used to perform a specific RAM test based on the current state of the RAM test algorithm.
 * @details
 *
 * @table            @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 *
 * @param[in]        TestedCells Number of cells to be tested.
 * @return           Std_ReturnType
 * @retval           E_OK:     The current RAM cells tested successfully.
 * @retval           E_NOT_OK: The function has not been successfully executed or error occurs during RAM test.
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_355
 */
RAMTST_FUNC Std_ReturnType RamTst_Lld_BlockTest(RamTst_NumberOfTestedCellsType TestedCells)
{
    Std_ReturnType Ret = E_OK;
    RamTst_AlgTstType RamTstAlgTst;
    RamTstAlgTst.CellAddress = RamTstState.CellAddress;
    RamTstAlgTst.NumberCells = TestedCells;
    RamTstAlgTst.CurrentBlockPolicy = RamTstState.CurrentBlockPolicy;
    RamTstAlgTst.CurrentBlockRegion = RamTstState.CurrentBlockRegion;
    RamTstAlgTst.FillData = RamTstState.CurrentBlockFillPattern;
    RamTstAlgTst.BackupAddress = RamTstState.BackupStartAddress;
    Ret = RamTst_Lld_AlgorithmTest(RamTstState.RamTstAlgorithm, RamTstAlgTst);
    return Ret;
}

#define RAMTST_STOP_SEC_CODE
#include "RamTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file RamTst_Lld.c */
