/*
* @file    RamTst.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : RamTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef RAMTST_H
#define RAMTST_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "RamTst_Types.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define RAMTST_VENDOR_ID                       (180)
#define RAMTST_AR_REL_MAJOR_VER                (4)
#define RAMTST_AR_REL_MINOR_VER                (4)
#define RAMTST_AR_REL_REVISION_VER             (0)
#define RAMTST_SW_MAJOR_VER                    (2)
#define RAMTST_SW_MINOR_VER                    (0)
#define RAMTST_SW_PATCH_VER                    (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if RamTst.h and RamTst_Types.h are of the same vendor */
#if (RAMTST_VENDOR_ID != RAMTST_TYPES_VENDOR_ID)
#error "RamTst.h and RamTst_Types.h have different vendor ids"
#endif

/* Check if RamTst.h and RamTst_Types.h are of the same Autosar version */
#if ((RAMTST_AR_REL_MAJOR_VER != RAMTST_TYPES_AR_REL_MAJOR_VER) || \
     (RAMTST_AR_REL_MINOR_VER != RAMTST_TYPES_AR_REL_MINOR_VER) || \
     (RAMTST_AR_REL_REVISION_VER != RAMTST_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of RamTst.h and RamTst_Types.h are different"
#endif

/* Check if RamTst.h and RamTst_Types.h are of the same software version */
#if ((RAMTST_SW_MAJOR_VER != RAMTST_TYPES_SW_MAJOR_VER) || \
     (RAMTST_SW_MINOR_VER != RAMTST_TYPES_SW_MINOR_VER) || \
     (RAMTST_SW_PATCH_VER != RAMTST_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of RamTst.h and RamTst_Types.h are different"
#endif

/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
/**
 * @brief           RamTst instance ID
 * @sw_type         sw_detail
 */
#define RAMTST_INSTANCE_ID                        (0U)

/**
 * @brief           RamTst module ID
 * @sw_type         sw_detail
 */
#define RAMTST_MODULE_ID                          ((uint16)93U)

/**
 * @defgroup        RamTst_SID
 * @brief           Service ID number for all RAMTST driver services
 * @sw_type         sw_detail
 * @{
 */
/**< @brief API Service ID for RamTst_Allow */
#define RAMTST_SID_ALLOW                          (0x03U)
/**< @brief API Service ID for RamTst_ChangeNumberOfTestedCells */
#define RAMTST_SID_CHANGE_NUMBER_OF_TESTED_CELLS  (0x08U)
/**< @brief API Service ID for RamTst_DeInit */
#define RAMTST_SID_DEINIT                         (0x0cU)
/**< @brief API Service ID for RamTst_GetAlgParams */
#define RAMTST_SID_GET_ALG_PARAMS                 (0x12U)
/**< @brief API Service ID for RamTst_GetExecutionStatus */
#define RAMTST_SID_GET_EXECUTION_STATUS           (0x04U)
/**< @brief API Service ID for RamTst_GetNumberOfTestedCells */
#define RAMTST_SID_GET_NUMBER_OF_TESTED_CELLS     (0x09U)
/**< @brief API Service ID for RamTst_GetTestAlgorithm */
#define RAMTST_SID_GET_TEST_ALGORITHM             (0x07U)
/**< @brief API Service ID for RamTst_GetTestResult */
#define RAMTST_SID_GET_TEST_RESULT                (0x05U)
/**< @brief API Service ID for RamTst_GetTestResultPerBlock */
#define RAMTST_SID_GET_TEST_RESULT_PER_BLOCK      (0x06U)
/**< @brief API Service ID for RamTst_GetVersionInfo */
#define RAMTST_SID_GET_VERSION_INFO               (0x0aU)
/**< @brief API Service ID for RamTst_Init */
#define RAMTST_SID_INIT                           (0x00U)
/**< @brief API Service ID for RamTst_MainFunction */
#define RAMTST_SID_MAINFUNCTION                   (0x01U)
/**< @brief API Service ID for RamTst_Resume */
#define RAMTST_SID_RESUME                         (0x0eU)
/**< @brief API Service ID for RamTst_RunFullTest */
#define RAMTST_SID_RUNFULL_TEST                   (0x10U)
/**< @brief API Service ID for RamTst_RunPartialTest */
#define RAMTST_SID_RUNPARTIAL_TEST                (0x11U)
/**< @brief API Service ID for RamTst_SelectAlgParams */
#define RAMTST_SID_SELECT_ALG_PARAMS              (0x0bU)
/**< @brief API Service ID for RamTst_Stop */
#define RAMTST_SID_STOP                           (0x02U)
/**< @brief API Service ID for RamTst_Suspend */
#define RAMTST_SID_SUSPEND                        (0x0dU)
/**< @} */

#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup        RamTst_Dev_Error
 * @brief           RAMTST development error codes define
 * @sw_type         sw_detail
 * @{
 */
#define RAMTST_E_STATUS_FAILURE            (0x01U)     /**< A particular API is called in an unexpected state */
#define RAMTST_E_OUT_OF_RANGE              (0x02U)     /**< API parameter out of specified range */
#define RAMTST_E_UNINIT                    (0x03U)     /**< API service used without module initialization */
#define RAMTST_E_PARAM_POINTER             (0x4U)      /**< RAMTST API service is called using an invalid pointer */
/** @} */
#endif

/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                ENUMS
==================================================================================================*/
/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
#if (RAMTST_ALLOW_API == STD_ON)
/**
 * @brief            Service for continuing the RAM Test after calling 'RamTst_Stop.
 * @details
 *
 * @table            @service_id:       0x03  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   false  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_Allow(void);
#endif

#if (RAMTST_GET_NUMBER_OF_TESTED_CELLS_API == STD_ON)
/**
 * @brief            Service changes the current number of tested cells.
 * @details          This service can be used to change the number of tested cells per MainFunction cycle.
 *
 * @table            @service_id:       0x08  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        NewNumberOfTestedCells The new number of tested cells.
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_ChangeNumberOfTestedCells(RamTst_NumberOfTestedCellsType NewNumberOfTestedCells);
#endif

/**
 * @brief            Service for RAM Test deinitialization.
 * @details
 *
 * @table            @service_id:       0x0c  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_DeInit(void);

#if (RAMTST_GET_ALG_PARAMS_API == STD_ON)
/**
 * @brief            Service returns the ID of the current RAM Test algorithm parameter set.
 * @details
 *
 * @table            @service_id:       0x12  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           RamTst_AlgParamsIdType
 * @sw_type          sw_arch
 */
RamTst_AlgParamsIdType RamTst_GetAlgParams(void);
#endif

#if (RAMTST_GET_EXECUTION_STATUS_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test execution status.
 * @details
 *
 * @table            @service_id:       0x04  \n
 *                   @is_reentrant:     true  \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           RamTst_ExecutionStatusType
 * @sw_type          sw_arch
 */
RamTst_ExecutionStatusType RamTst_GetExecutionStatus(void);
#endif

#if (RAMTST_GET_NUMBER_OF_TESTED_CELLS_API == STD_ON)
/**
 * @brief            Service returns the current number of tested cells per MainFunction cycle.
 * @details
 *
 * @table            @service_id:       0x09  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           RamTst_NumberOfTestedCellsType
 * @sw_type          sw_arch
 */
RamTst_NumberOfTestedCellsType RamTst_GetNumberOfTestedCells(void);
#endif

#if (RAMTST_GET_TEST_ALGORITHM_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test algorithm.
 * @details
 *
 * @table            @service_id:       0x07  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           RamTst_AlgorithmType
 * @sw_type          sw_arch
 */
RamTst_AlgorithmType RamTst_GetTestAlgorithm(void);
#endif

#if (RAMTST_GET_TEST_RESULT_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test result.
 * @details
 *
 * @table            @service_id:       0x05  \n
 *                   @is_reentrant:     true  \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           RamTst_TestResultType
 * @sw_type          sw_arch
 */
RamTst_TestResultType RamTst_GetTestResult(void);
#endif

#if (RAMTST_GET_TEST_RESULT_PER_BLOCK_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test result for the specified block.
 * @details
 *
 * @table            @service_id:       0x06  \n
 *                   @is_reentrant:     true  \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        BlockID Identifies the block
 * @return           RamTst_TestResultType
 * @sw_type          sw_arch
 */
RamTst_TestResultType RamTst_GetTestResultPerBlock(RamTst_NumberOfBlocksType BlockID);
#endif

/**
 * @brief            Service for RAM Test initialization.
 * @details
 *
 * @table            @service_id:       0x00  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        ConfigPtr Pointer to the selected configuration set.
 * @sw_type          sw_arch
 */
void RamTst_Init(const RamTst_ConfigType *ConfigPtr);

/**
 * @brief            Scheduled function for executing the RAM Test in the background.
 * @details
 *
 * @table            @service_id:       0x01  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_MainFunction(void);

#if (RAMTST_RESUME_API == STD_ON)
/**
 * @brief            Service for allowing to continue the background RAM Test at the point it was suspended.
 * @details
 *
 * @table            @service_id:       0x0e  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   false \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_Resume(void);
#endif

#if (RAMTST_RUN_FULL_TEST_API == STD_ON)
/**
 * @brief            Service for executing the full RAM Test in the foreground.
 * @details
 *
 * @table            @service_id:       0x1004  \n
 *                   @is_reentrant:     false   \n
 *                   @is_synchronous:   true    \n
 *                   @autosar_api:      true    \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_RunFullTest(void);
#endif

#if (RAMTST_RUN_PARTIAL_TEST_API == STD_ON)
/**
 * @brief            Service for testing one RAM block in the foreground.
 * @details
 *
 * @table            @service_id:       0x11  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        BlockId Identifies the single RAM block to be tested in the selected set of RamTstAlgParams.
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_RunPartialTest(RamTst_NumberOfBlocksType BlockId);
#endif

#if (RAMTST_SELECT_ALG_PARAMS_API == STD_ON)
/**
 * @brief            Service used to set the test algorithm and its parameter set.
 * @details
 *
 * @table            @service_id:       0x0b  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[in]        NewAlgParamsId Identifies the parameter set to be used.
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_SelectAlgParams(RamTst_AlgParamsIdType NewAlgParamsId);
#endif

#if (RAMTST_STOP_API == STD_ON)
/**
 * @brief            Service for stopping the RAM Test.
 * @details
 *
 * @table            @service_id:       0x02  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   false \n
 *                   @autosar_api:      true  \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_Stop(void);
#endif

#if (RAMTST_SUPSPEND_API == STD_ON)
/**
 * @brief            Service for suspending current operation of background RAM Test, until RESUME is called.
 * @details
 *
 * @table            @service_id:       0x0d   \n
 *                   @is_reentrant:     false  \n
 *                   @is_synchronous:   false  \n
 *                   @autosar_api:      true   \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_Suspend(void);
#endif

/**
 * @brief           The function RamTst_Error shall be called every time when a RAM failure has been detected
 *                  by the selected test algorithm in the background.
 * @details
 *
 * @table            @is_reentrant:     Don't care \n
 *                   @is_synchronous:   --         \n
 *                   @autosar_api:      true       \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_ErrorNotification(void);

/**
 * @brief            The function RamTst_TestCompleted shall be called every time when all RAM blocks of the current test
 *                   configuration have been tested in the background.
 * @details
 *
 * @table            @is_reentrant:     Don't care \n
 *                   @is_synchronous:   --         \n
 *                   @autosar_api:      true       \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_TestCompletedNotification(void);

#if (RAMTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @details
 * @table            @service_id:       0x0a  \n
 *                   @is_reentrant:     true  \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[out]       Versioninfo Pointer to where to store version information of this module
 * @return           void
 * @sw_type          sw_arch
 */
void RamTst_GetVersionInfo(Std_VersionInfoType *Versioninfo);
#endif /* RAMTST_VERSION_INFO_API */

#ifdef __cplusplus
}
#endif

#endif /* End of file RamTst.h */

