/**
* @file    PpuTst_Lld.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 SafLib
*   Platform             : ARM
*   Peripheral           : PpuTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "PpuTst_Lld.h"
#include "PpuTst_Lld_Reg.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define PPUTST_LLD_VENDOR_ID_C                   (180)
#define PPUTST_LLD_AR_REL_MAJOR_VERSION_C        (4)
#define PPUTST_LLD_AR_REL_MINOR_VERSION_C        (4)
#define PPUTST_LLD_AR_REL_REVISION_VERSION_C     (0)
#define PPUTST_LLD_SW_MAJOR_VERSION_C            (2)
#define PPUTST_LLD_SW_MINOR_VERSION_C            (0)
#define PPUTST_LLD_SW_PATCH_VERSION_C            (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and PPUTST_LLD header file are of the same vendor */
#if (PPUTST_LLD_VENDOR_ID_C != PPUTST_LLD_VENDOR_ID)
#error "PpuTst_Lld.c and PpuTst_Lld.h have different vendor ids"
#endif

/* Check if source file and PPUTST_LLD header file are of the same Autosar version */
#if (( PPUTST_LLD_AR_REL_MAJOR_VERSION_C != PPUTST_LLD_AR_REL_MAJOR_VERSION) || \
     ( PPUTST_LLD_AR_REL_MINOR_VERSION_C != PPUTST_LLD_AR_REL_MINOR_VERSION) || \
     ( PPUTST_LLD_AR_REL_REVISION_VERSION_C != PPUTST_LLD_AR_REL_REVISION_VERSION))
#error "AutoSar Version Numbers of PpuTst_Lld.c and PpuTst_Lld.h are different"
#endif

/* Check if source file and PPUTST_LLD header file are of the same Software version */
#if (( PPUTST_LLD_SW_MAJOR_VERSION_C != PPUTST_LLD_SW_MAJOR_VERSION) || \
     ( PPUTST_LLD_SW_MINOR_VERSION_C != PPUTST_LLD_SW_MINOR_VERSION) || \
     ( PPUTST_LLD_SW_PATCH_VERSION_C != PPUTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of PpuTst_Lld.c and PpuTst_Lld.h are different"
#endif

/* Check if source file and PPUTST_LLD_REG header file are of the same vendor */
#if (PPUTST_LLD_VENDOR_ID_C != PPUTST_LLD_REG_VENDOR_ID)
#error "PpuTst_Lld.c and PpuTst_Lld_Reg.h have different vendor ids"
#endif

/* Check if source file and PPUTST_LLD_REG header file are of the same Autosar version */
#if (( PPUTST_LLD_AR_REL_MAJOR_VERSION_C != PPUTST_LLD_REG_AR_REL_MAJOR_VERSION) || \
     ( PPUTST_LLD_AR_REL_MINOR_VERSION_C != PPUTST_LLD_REG_AR_REL_MINOR_VERSION) || \
     ( PPUTST_LLD_AR_REL_REVISION_VERSION_C != PPUTST_LLD_REG_AR_REL_REVISION_VERSION))
#error "AutoSar Version Numbers of PpuTst_Lld.c and PpuTst_Lld_Reg.h are different"
#endif

/* Check if source file and PPUTST_LLD_REG header file are of the same Software version */
#if (( PPUTST_LLD_SW_MAJOR_VERSION_C != PPUTST_LLD_REG_SW_MAJOR_VERSION) || \
     ( PPUTST_LLD_SW_MINOR_VERSION_C != PPUTST_LLD_REG_SW_MINOR_VERSION) || \
     ( PPUTST_LLD_SW_PATCH_VERSION_C != PPUTST_LLD_REG_SW_PATCH_VERSION))
#error "Software Version Numbers of PpuTst_Lld.c and PpuTst_Lld_Reg.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
#define PPUTST_START_SEC_CONST_UNSPECIFIED
#include "PpuTst_MemMap.h"

/**
 * @brief Array of base pointers for all PPU modules.
 * @sw_type sw_detail
 * @private
 */
PPUTST_CONST static volatile PPU_Type *const Ppu_Base[PPU_IP_MODULE_COUNT] = PPU_BASE_PTRS;

#define PPUTST_STOP_SEC_CONST_UNSPECIFIED
#include "PpuTst_MemMap.h"
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
#define PPUTST_START_SEC_CODE
#include "PpuTst_MemMap.h"

/**
 * @brief          This function initializes the soft lock and hard lock protections for the PPU group.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_468
 */
PPUTST_FUNC void PpuTst_Lld_Init(const PpuTst_GroupSLockConfigType GroupConfig)
{
    volatile PPU_Type *const PpuModule = Ppu_Base[GroupConfig.GroupId];
    if (TRUE == GroupConfig.Enable)
    {
        /* Enable the soft lock protection. WREN=1,LOCKBIT=1*/
        PpuModule->SLOCK[(GroupConfig.GroupPathOffset) >> 2U] = (uint8)0x11;
    }
    /* Enable User mode access and disable Hard lock*/
    PpuModule->PROTCFG = (uint32)0x0100;
}

/**
 * @brief          This function unlocks the soft lock bit for a given PPU group configuration.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_882
 */
PPUTST_FUNC void PpuTst_LLd_UnlockSoftLockBit(const PpuTst_GroupSLockConfigType GroupConfig)
{
    volatile PPU_Type *const PpuModule = Ppu_Base[GroupConfig.GroupId];
    /* update the soft lock protection. WREN=1,LOCKBIT=0*/
    PpuModule->SLOCK[(GroupConfig.GroupPathOffset) >> 2U] = (uint8)0x10;
}

/**
 * @brief          Updates the protected register for a given PPU group configuration.
 * @details        The function writes the register to inject the soft lock error.
 *                 Once LOCK_BIT is set, any write access to the corresponding normal register will cause transfer error.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_471
 */
PPUTST_FUNC void PpuTst_Lld_UpdateProtectedRegister(const PpuTst_GroupSLockConfigType GroupConfig)
{
    uint32 Temp;
    volatile PPU_Type *const PpuModule = Ppu_Base[GroupConfig.GroupId];
    Temp = PpuModule->Normal[(GroupConfig.GroupPathOffset) >> 2U];
    /* Write the protected register */
    PpuModule->Normal[(GroupConfig.GroupPathOffset) >> 2U] = Temp;
}

/**
 * @brief          Enable the hard lock bit for a given module group.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_467
 */
PPUTST_FUNC void PpuTst_Lld_EnableHLockBit(const PpuTst_GroupSLockConfigType GroupConfig)
{
    volatile PPU_Type *const PpuModule = Ppu_Base[GroupConfig.GroupId];
    /* Enable the hard lock protection. WREN=1,LOCKBIT=1*/
    PpuModule->PROTCFG |= PPU_PROTCFG_HLOCK_MASK;
}

/**
 * @brief          Updates the soft lock bit for a given PPU group configuration.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_469
 */
PPUTST_FUNC void PpuTst_Lld_UpdateSLockBit(const PpuTst_GroupSLockConfigType GroupConfig)
{
    volatile PPU_Type *const PpuModule = Ppu_Base[GroupConfig.GroupId];
    /* update the soft lock protection. WREN=1,LOCKBIT=0*/
    PpuModule->SLOCK[(GroupConfig.GroupPathOffset) >> 2U] = (uint8)0x10;
}

#define PPUTST_STOP_SEC_CODE
#include "PpuTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file PpuTst_Lld.c */
