/**
* @file    PpuTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : PpuTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 0306 Rule 11.4: A conversion should not be performed between a pointer to object and an integer type.
 * PRQA S 1006 Dir-1.1:  This in-line assembler construct is a language extension. The code has been ignored.
 * PRQA S 3219 Rule-2.1: This particular function is not called from anywhere within the existing translation unit.
 * PRQA S 4342 Rule-10.5: An expression of 'essentially unsigned' type (%1s) is being cast to enum type '%2s'.
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "PpuTst.h"
#include "PpuTst_Lld.h"
#ifdef PPUTST_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif /* PPUTST_ENABLE_USER_MODE_SUPPORT */
#include "Platform.h"

#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* PPUTST_DEV_ERROR_DETECT == STD_ON */

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define PPUTST_VENDOR_ID_C                      (180)
#define PPUTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define PPUTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define PPUTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define PPUTST_SW_MAJOR_VERSION_C               (2)
#define PPUTST_SW_MINOR_VERSION_C               (0)
#define PPUTST_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and PPUTST header file are of the same vendor */
#if (PPUTST_VENDOR_ID_C != PPUTST_VENDOR_ID)
#error "PpuTst.c and PpuTst.h have different vendor ids"
#endif

/* Check if source file and PPUTST header file are of the same Autosar version */
#if (( PPUTST_AR_RELEASE_MAJOR_VERSION_C != PPUTST_AR_RELEASE_MAJOR_VERSION) || \
     ( PPUTST_AR_RELEASE_MINOR_VERSION_C != PPUTST_AR_RELEASE_MINOR_VERSION) || \
     ( PPUTST_AR_RELEASE_REVISION_VERSION_C != PPUTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of PpuTst.c and PpuTst.h are different"
#endif

/* Check if source file and PPUTST header file are of the same Software version */
#if (( PPUTST_SW_MAJOR_VERSION_C != PPUTST_SW_MAJOR_VERSION) || \
     ( PPUTST_SW_MINOR_VERSION_C != PPUTST_SW_MINOR_VERSION) || \
     ( PPUTST_SW_PATCH_VERSION_C != PPUTST_SW_PATCH_VERSION))
#error "Software Version Numbers of PpuTst.c and PpuTst.h are different"
#endif

/* Check if source file and PPUTST_LLD header file are of the same vendor */
#if (PPUTST_VENDOR_ID_C != PPUTST_LLD_VENDOR_ID)
#error "PpuTst.c and PpuTst_Lld.h have different vendor ids"
#endif

/* Check if source file and PPUTST_LLD header file are of the same Autosar version */
#if (( PPUTST_AR_RELEASE_MAJOR_VERSION_C != PPUTST_LLD_AR_REL_MAJOR_VERSION) || \
     ( PPUTST_AR_RELEASE_MINOR_VERSION_C != PPUTST_LLD_AR_REL_MINOR_VERSION) || \
     ( PPUTST_AR_RELEASE_REVISION_VERSION_C != PPUTST_LLD_AR_REL_REVISION_VERSION))
#error "AutoSar Version Numbers of PpuTst.c and PpuTst_Lld.h are different"
#endif

/* Check if source file and PPUTST_LLD header file are of the same Software version */
#if (( PPUTST_SW_MAJOR_VERSION_C != PPUTST_LLD_SW_MAJOR_VERSION) || \
     ( PPUTST_SW_MINOR_VERSION_C != PPUTST_LLD_SW_MINOR_VERSION) || \
     ( PPUTST_SW_PATCH_VERSION_C != PPUTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of PpuTst.c and PpuTst_Lld.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
#define PPUTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "PpuTst_MemMap.h"

/**
 * @brief          Configuration pointer for PpuTst error check inject initialization
 * @details        Data structure containing the set of configuration parameters required for error check inject initialization.
 * @sw_type        sw_detail
 * @private
 */
PPUTST_VAR static const PpuTst_ConfigType *PpuTst_ConfigPtr = NULL_PTR;
#define PPUTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "PpuTst_MemMap.h"

#define PPUTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "PpuTst_MemMap.h"
/**
 * @brief          PpuTst driver execution status
 * @sw_type        sw_detail
 * @private
 */
PPUTST_VAR static PpuTst_StatusType PpuTst_Status;
#define PPUTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "PpuTst_MemMap.h"

#define PPUTST_START_SEC_VAR_INIT_BOOLEAN
#include "PpuTst_MemMap.h"
/**
 * @brief          Flag to indicate if the error injection address is detected
 * @sw_type        sw_detail
 * @private
 */
PPUTST_VAR static boolean ErrorInjectFlag = FALSE;

/**
 * @brief          Flag to indicate if the soft lock is unlocked test result
 * @sw_type        sw_detail
 * @private
 */
PPUTST_VAR static boolean UnlockFlag = FALSE;
#define PPUTST_STOP_SEC_VAR_INIT_BOOLEAN
#include "PpuTst_MemMap.h"

#define PPUTST_START_SEC_CONST_32
#include "PpuTst_MemMap.h"
/**
 * @brief          Array of the soft lock error injection address for the PPU group path
 * @sw_type        sw_detail
 * @private
 */
PPUTST_CONST static const uint32 SoftLockPathInjectAddr[PPUTST_GROUP_COUNT] = SOFT_LOCK_INJECT_ADDRESS;

/**
 * @brief          Array of the hard lock error injection address for the PPU group path
 * @sw_type        sw_detail
 * @private
 */
PPUTST_CONST static const uint32 HardLockPathInjectAddr[PPUTST_GROUP_COUNT] = HARD_LOCK_INJECT_ADDRESS;
#define PPUTST_STOP_SEC_CONST_32
#include "PpuTst_MemMap.h"

#define PPUTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "PpuTst_MemMap.h"
/**
 * @brief          Function pointer used to restore the BusFault_Handler
 * @sw_type        sw_detail
 * @private
 */
PPUTST_VAR static Platform_IrqHandlerType FuncPointer;
#define PPUTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "PpuTst_MemMap.h"

/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                            LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                     EXTERNAL FUNCTIONS PROTOTYPES
==================================================================================================*/
/**
 * @brief          The prototype of the BusFault_Handler
 * @sw_type        sw_detail
 * @private
 */
extern void BusFault_Handler(void);

/*==================================================================================================
 *                                        EXTERNAL VARIATES
==================================================================================================*/
/**
 * @brief          Configuration data used for testing error injection in the PPU module
 * @sw_type        sw_detail
 * @private
 */
extern const PpuTst_ConfigType PpuTst_PreCompileConfig;

/*==================================================================================================
 *                                   LOCAL FUNCTION DECLARATION
==================================================================================================*/

/*==================================================================================================
 *                                        LOCAL FUNCTIONS
==================================================================================================*/
#define PPUTST_START_SEC_CODE
#include "PpuTst_MemMap.h"

/**
 * @brief          Function to initialize the error status of the PPU group path
 * @details
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_388
 */
PPUTST_FUNC static inline void PpuTst_ErrorStatusInit(void)
{
    /** Init Group function path test result state*/
    for (uint8 GroupIndex = 0; GroupIndex < PPUTST_GROUP_COUNT; ++GroupIndex)
    {
        PpuTst_Status.PpuTst_Result[GroupIndex].SLockPathErrorCheckState = PPUTST_PATH_NOT_TESTED;
        PpuTst_Status.PpuTst_Result[GroupIndex].HLockPathErrorCheckState = PPUTST_PATH_NOT_TESTED;
    }
}

/**
 * @brief          Function to set the path test result of the PPU group
 * @details
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupIndex The index of the PPU group
 * @param[in]      FaultAddress The address that triggered the bus fault
 * @param[in]      IsSoftLockPath The type of the path, TRUE for soft lock path, FALSE for hard lock path
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_394
 */
PPUTST_FUNC static inline void PpuTst_SetPathTestResult(PpuTst_GroupIdType GroupIndex, uint32 FaultAddress, boolean IsSoftLockPath)
{
    if (TRUE == IsSoftLockPath)
    {
        if (FaultAddress == SoftLockPathInjectAddr[GroupIndex])
        {
            PpuTst_Status.PpuTst_Result[GroupIndex].SLockPathErrorCheckState = PPUTST_PATH_PASS;
            ErrorInjectFlag = TRUE;
        }
    }
    else
    {
        if (FaultAddress == HardLockPathInjectAddr[GroupIndex])
        {
            PpuTst_Status.PpuTst_Result[GroupIndex].HLockPathErrorCheckState = PPUTST_PATH_PASS;
            ErrorInjectFlag = TRUE;
        }
    }
}

/**
 * @brief          Function to inject the soft lock error for the PPU group path
 * @details        The function writes the mirror register to inject the soft lock error.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      Group The PPU group
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_395
 */
PPUTST_FUNC static void PpuTst_SLock_GroupPathErrorInject(PpuTst_GroupIdType Group)
{
    switch (Group)
    {
        case PPUTST_GROUP_TMR:
            PpuTst_Status.ErrorInject = PPUTST_TMR_SL_ERR_INJECT;
            /*register address is written as 0x40054000 */
            PpuTst_Lld_UpdateProtectedRegister(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_TMR]);
            break;
        case PPUTST_GROUP_SCU:
            PpuTst_Status.ErrorInject = PPUTST_SCU_SL_ERR_INJECT;
            /*register address is written as 0x4007C000 */
            PpuTst_Lld_UpdateProtectedRegister(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_SCU]);
            break;
        case PPUTST_GROUP_PCU:
            PpuTst_Status.ErrorInject = PPUTST_PCU_SL_ERR_INJECT;
            /*register address is written as 0x4007E008 */
            PpuTst_Lld_UpdateProtectedRegister(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_PCU]);
            break;
        case PPUTST_GROUP_RCU:
            PpuTst_Status.ErrorInject = PPUTST_RCU_SL_ERR_INJECT;
            /*register address is written as 0x4007F004 */
            PpuTst_Lld_UpdateProtectedRegister(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_RCU]);
            break;
        case PPUTST_GROUP_EFM:
            PpuTst_Status.ErrorInject = PPUTST_EFM_SL_ERR_INJECT;
            /*register address is written as 0x40010000 */
            PpuTst_Lld_UpdateProtectedRegister(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_EFM]);
            break;
        case PPUTST_GROUP_IPC:
            PpuTst_Status.ErrorInject = PPUTST_IPC_SL_ERR_INJECT;
            /*register address is written as 0x4007D020 */
            PpuTst_Lld_UpdateProtectedRegister(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_IPC]);
            break;
        default:
            /* Do nothing */
            break;
    }
}

/**
 * @brief          Function to inject the hard lock error for the PPU group path
 * @details        The function writes the SLOCK register to inject the hard lock error.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      Group The PPU group
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_391
 */
PPUTST_FUNC static void PpuTst_HLock_GroupPathErrorInject(PpuTst_GroupIdType Group)
{
    switch (Group)
    {
        case PPUTST_GROUP_TMR:
            PpuTst_Status.ErrorInject = PPUTST_TMR_HL_ERR_INJECT;
            /*register address is written as 0x40054C00 SLOCK's address is [offset]/4 */
            PpuTst_Lld_UpdateSLockBit(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_TMR]);
            break;
        case PPUTST_GROUP_SCU:
            PpuTst_Status.ErrorInject = PPUTST_SCU_HL_ERR_INJECT;
            /*register address is written as 0x4007CC00 SLOCK's address is [offset]/4 */
            PpuTst_Lld_UpdateSLockBit(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_SCU]);
            break;
        case PPUTST_GROUP_PCU:
            PpuTst_Status.ErrorInject = PPUTST_PCU_HL_ERR_INJECT;
            /*register address is written as 0x4007EC02 SLOCK's address is [offset]/4 */
            PpuTst_Lld_UpdateSLockBit(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_PCU]);
            break;
        case PPUTST_GROUP_RCU:
            PpuTst_Status.ErrorInject = PPUTST_RCU_HL_ERR_INJECT;
            /*register address is written as 0x4007FC01 SLOCK's address is [offset]/4 */
            PpuTst_Lld_UpdateSLockBit(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_RCU]);
            break;
        case PPUTST_GROUP_EFM:
            PpuTst_Status.ErrorInject = PPUTST_EFM_HL_ERR_INJECT;
            /*register address is written as 0x40010C00 SLOCK's address is [offset]/4 */
            PpuTst_Lld_UpdateSLockBit(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_EFM]);
            break;
        case PPUTST_GROUP_IPC:
            PpuTst_Status.ErrorInject = PPUTST_IPC_HL_ERR_INJECT;
            /*register address is written as 0x4007DC08 SLOCK's address is [offset]/4 */
            PpuTst_Lld_UpdateSLockBit(PpuTst_ConfigPtr->GroupConfigPtr[PPUTST_GROUP_IPC]);
            break;
        default:
            /* Do nothing */
            break;
    }
}

/**
 * @brief          Function to update the timeout status of the PPU group path
 * @details        If the timeout occurs, indicate the path test failed and update the overall result of the PPU test to NOT-OK.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      ErrorState The error state of the PPU group path
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_396
 */
PPUTST_FUNC static inline void PpuTst_TimeoutStatus_Update(PpuTst_PathResultType *const ErrorState)
{
    PpuTst_Status.Result = PPUTST_RESULT_NOT_OK;
    *ErrorState = PPUTST_PATH_FAILED;
}

/**
 * @brief          Function to enable the bus fault exception
 * @details
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      void
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_385
 */
PPUTST_FUNC static inline void PpuTst_EnableBusFaultException(void)
{
    /**
     * MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
     * In this specific case, we need to set the integer value to the base address, thus, we couldn't adhere to M3CM Rule-11.4
     */
    SCB->SHCSR |= SCB_SHCSR_BUSFAULTENA_Msk; /* PRQA S 0306*/
    SCB->CCR |= SCB_CCR_BFHFNMIGN_Msk; /* PRQA S 0306*/
}

/**
 * @brief          This function is used to handle the bus fault exception that used to PPU test.
 * @details        It is designed for precise control over exception handling in embedded systems, where skipping a faulting instruction can allow
                   the system to recover gracefully.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      void
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_PpuTst_899
 */
PPUTST_FUNC static __attribute__((naked)) void PpuTst_BusFault(void)
{
#ifndef TESSY
    /**
     * M3CM-2_2 Dir-1.1 VIOLATION: This line uses the __ASM() macro for inline assembly which is not
     * a part of standard C. In this specific case, We use assembly to ensure the stack position of the PC register is deterministic,
     * hence we couldn't adhere to the Dir-1.1
     */
    __ASM volatile( /* PRQA S 1006*/
#ifndef __GHS_C_YTM32B1MXX
        "PUSH {LR}                      \n" /* push LR to stack */
#endif
        "PUSH {LR}                      \n" /* push LR to stack */
        "bl PpuTst_BusFaultHandler      \n"
        "POP {R1}                       \n" /* pop R1 from stack*/
        "MOV LR, R1                     \n" /* Move R1 to LR*/
        "MOVS R0, #4                    \n" /* Load immediate value 4 into R0*/
        "TST R1, R0                     \n" /* Test if LR[2] is set*/
        "BNE PSP_PROCESS                \n" /* If LR[2] is not zero, branch to PSP_PROCESS*/
        "MRS R1, MSP                    \n" /* Move Main Stack Pointer (MSP) into R1*/
        "B MSP_PROCESS                  \n" /* Branch to MSP_PROCESS*/
        "PSP_PROCESS:                   \n"
        "MRS R1, PSP                    \n" /* Move Process Stack Pointer (PSP) into R1*/
        "MSP_PROCESS:                   \n"
        "LDR R0, [R1,#28]               \n" /* Load the instruction from the address*/
        "ADD R0, R0, #2                 \n" /* Add 2 to the instruction to get the next instruction*/
        "STR R0, [R1,#28]               \n" /* Store the next instruction back to the address*/
        "POP {PC}                       \n" /* Pop the next instruction to LR*/
    );
#endif /* #ifndef TESSY */
}

/**
 * @brief            This function handles bus fault that used to PPU test.
 * @details
 *
 * @table            @is_reentrant:     false  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_PpuTst_881
 */
/* This function is called by the PpuTst_BusFault(), since QAC ignores assembly code, it mistakenly considers this function as unused*/
PPUTST_FUNC static __attribute__((used)) void PpuTst_BusFaultHandler(void) /* PRQA S 3219*/
{
    /**
     * MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
     * In this specific case, we need to obtain the integer value of the base address, thus, we couldn't adhere to M3CM Rule-11.4
     */
    uint32 BusFaultAddress = SCB->BFAR; /* PRQA S 0306 */
    switch (PpuTst_Status.ErrorInject)
    {
        case PPUTST_TMR_SL_ERR_INJECT:
            /*PPU_ITEM_TMR soft lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_TMR, BusFaultAddress, TRUE);
            break;
        case PPUTST_SCU_SL_ERR_INJECT:
            /*PPU_ITEM_SCU soft lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_SCU, BusFaultAddress, TRUE);
            break;
        case PPUTST_PCU_SL_ERR_INJECT:
            /*PPU_ITEM_PCU soft lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_PCU, BusFaultAddress, TRUE);
            break;
        case PPUTST_RCU_SL_ERR_INJECT:
            /*PPU_ITEM_RCU soft lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_RCU, BusFaultAddress, TRUE);
            break;
        case PPUTST_EFM_SL_ERR_INJECT:
            /*PPU_ITEM_EFM soft lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_EFM, BusFaultAddress, TRUE);
            break;
        case PPUTST_IPC_SL_ERR_INJECT:
            /*PPU_ITEM_IPC soft lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_IPC, BusFaultAddress, TRUE);
            break;
        case PPUTST_TMR_HL_ERR_INJECT:
            /*PPU_ITEM_TMR hard lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_TMR, BusFaultAddress, FALSE);
            break;
        case PPUTST_SCU_HL_ERR_INJECT:
            /*PPU_ITEM_SCU hard lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_SCU, BusFaultAddress, FALSE);
            break;
        case PPUTST_PCU_HL_ERR_INJECT:
            /*PPU_ITEM_PCU hard lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_PCU, BusFaultAddress, FALSE);
            break;
        case PPUTST_RCU_HL_ERR_INJECT:
            /*PPU_ITEM_RCU hard lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_RCU, BusFaultAddress, FALSE);
            break;
        case PPUTST_EFM_HL_ERR_INJECT:
            /*PPU_ITEM_EFM hard lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_EFM, BusFaultAddress, FALSE);
            break;
        case PPUTST_IPC_HL_ERR_INJECT:
            /*PPU_ITEM_IPC hard lock path inject*/
            PpuTst_SetPathTestResult(PPUTST_GROUP_IPC, BusFaultAddress, FALSE);
            break;
        default:
            ErrorInjectFlag = FALSE;
            UnlockFlag = FALSE;
            break;
    }
    /*Clear the BusFault exception flag*/
    /**
     * MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
     * In this specific case, we need to set the integer value to the base address, thus, we couldn't adhere to M3CM Rule-11.4
     */
    SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk; /* PRQA S 0306 */
}

/**
 * @brief          Function to inject the soft lock error for the PPU group path
 * @details        The function writes the normal register to test the soft lock and unlock function.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      void
 * @return         void
 * @sw_type        sw_detail
 *
 *  @trace YTSDS_PpuTst_1001
 */
PPUTST_FUNC static void PpuTst_SoftLockTest(void)
{
    uint32 Timeout = 0U;    /* Timeout counter */
    uint8 GroupIndex = 0U;  /* Group index */
    /* Inject error for soft lock path */
    for (GroupIndex = 0; GroupIndex < PPUTST_GROUP_COUNT; ++GroupIndex)
    {
        if (TRUE == (boolean)((PpuTst_ConfigPtr->SLockErrorInjectConfigPtr->Value >> GroupIndex) & 0x01U))
        {
            /**
            * MR12 RULE 10.5 VIOLATION: Here an expression of essentially unsigned type has been cast to an enum type.
            * GroupIndex is always within the valid range of the PpuTst_GroupIdType enumeration (it is used as a loop index from 0 to
            * PPUTST_GROUP_COUNT - 1)., thus, we couldn't adhere to M3CM Rule-10.5
            */
#ifdef PPUTST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(PpuTst_SLock_GroupPathErrorInject, (PpuTst_GroupIdType)GroupIndex); /* PRQA S 4342*/
#else
            PpuTst_SLock_GroupPathErrorInject((PpuTst_GroupIdType)GroupIndex); /* PRQA S 4342*/
#endif /* PPUTST_ENABLE_USER_MODE_SUPPORT*/
            while (FALSE == ErrorInjectFlag)
            {
                /* Wait for the error callback */
                ++Timeout;
                if (Timeout > ERROR_CHECK_TIMEOUT)
                {
                    PpuTst_TimeoutStatus_Update(&PpuTst_Status.PpuTst_Result[GroupIndex].SLockPathErrorCheckState);
#if (PPUTST_ERR_NOTIFICATION_SUPPORT == STD_ON)
                    if (PpuTst_ConfigPtr->PpuTstErrorNotification != NULL_PTR)
                    {
                        PpuTst_ConfigPtr->PpuTstErrorNotification(PPUTST_MODULE_ID);
                    }
#endif /* #if (PPUTST_ERR_NOTIFICATION_SUPPORT == STD_ON) */
                    break;
                }
            }
            Timeout = 0U;
            PpuTst_Status.ErrorInject = PPUTST_INVALID_ERROR_INJECT;
            ErrorInjectFlag = FALSE;
        }
        if (PPUTST_PATH_PASS == PpuTst_Status.PpuTst_Result[GroupIndex].SLockPathErrorCheckState)
        {
            UnlockFlag = TRUE;
#ifdef PPUTST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(PpuTst_LLd_UnlockSoftLockBit, PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
            OsIf_Trusted_Call1param(PpuTst_Lld_UpdateProtectedRegister, PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
#else
            PpuTst_LLd_UnlockSoftLockBit(PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
            PpuTst_Lld_UpdateProtectedRegister(PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
#endif /* PPUTST_ENABLE_USER_MODE_SUPPORT*/
            while (TRUE == UnlockFlag)
            {
                /* Wait for the unlock exception*/
                ++Timeout;
                if (Timeout > ERROR_CHECK_TIMEOUT)
                {
                    break;
                }
            }
            if (FALSE == UnlockFlag)
            {
                PpuTst_TimeoutStatus_Update(&PpuTst_Status.PpuTst_Result[GroupIndex].SLockPathErrorCheckState);
#if (PPUTST_ERR_NOTIFICATION_SUPPORT == STD_ON)
                if (PpuTst_ConfigPtr->PpuTstErrorNotification != NULL_PTR)
                {
                    PpuTst_ConfigPtr->PpuTstErrorNotification(PPUTST_MODULE_ID);
                }
#endif /* #if (PPUTST_ERR_NOTIFICATION_SUPPORT == STD_ON) */
                UnlockFlag = TRUE;
            }
        }
    }
}

/**
 * @brief          Function to inject the hard lock error for the PPU group path
 * @details        The function writes the SLOCK register to inject the hard lock error.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      void
 * @return         void
 * @sw_type        sw_detail
 *
 *  @trace YTSDS_PpuTst_1000
 */
PPUTST_FUNC static void PpuTst_HardLockTest(void)
{
    uint32 Timeout = 0U;    /* Timeout counter */
    for (uint8 GroupIndex = 0; GroupIndex < PPUTST_GROUP_COUNT; ++GroupIndex)
    {
        if (TRUE == (boolean)((PpuTst_ConfigPtr->HLockErrorInjectConfigPtr->Value >> GroupIndex) & 0x01U))
        {
#ifdef PPUTST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(PpuTst_Lld_EnableHLockBit, PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
            /**
            * MR12 RULE 10.5 VIOLATION: Here an expression of essentially unsigned type has been cast to an enum type.
            * GroupIndex is always within the valid range of the PpuTst_GroupIdType enumeration (it is used as a loop index from 0 to
            * PPUTST_GROUP_COUNT - 1)., thus, we couldn't adhere to M3CM Rule-10.5
            */
            OsIf_Trusted_Call1param(PpuTst_HLock_GroupPathErrorInject, (PpuTst_GroupIdType)GroupIndex); /* PRQA S 4342*/
#else
            PpuTst_Lld_EnableHLockBit(PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
            /**
            * MR12 RULE 10.5 VIOLATION: Here an expression of essentially unsigned type has been cast to an enum type.
            * GroupIndex is always within the valid range of the PpuTst_GroupIdType enumeration (it is used as a loop index from 0 to
            * PPUTST_GROUP_COUNT - 1)., thus, we couldn't adhere to M3CM Rule-10.5
            */
            PpuTst_HLock_GroupPathErrorInject((PpuTst_GroupIdType)GroupIndex); /* PRQA S 4342*/
#endif /* PPUTST_ENABLE_USER_MODE_SUPPORT*/
            while (FALSE == ErrorInjectFlag)
            {
                /* Wait for the error callback */
                ++Timeout;
                if (Timeout > ERROR_CHECK_TIMEOUT)
                {
                    PpuTst_TimeoutStatus_Update(&PpuTst_Status.PpuTst_Result[GroupIndex].HLockPathErrorCheckState);
#if (PPUTST_ERR_NOTIFICATION_SUPPORT == STD_ON)
                    if (PpuTst_ConfigPtr->PpuTstErrorNotification != NULL_PTR)
                    {
                        PpuTst_ConfigPtr->PpuTstErrorNotification(PPUTST_MODULE_ID);
                    }
#endif /* PPUTST_ERR_NOTIFICATION_SUPPORT == STD_ON */
                    break;
                }
            }
            Timeout = 0U;
            PpuTst_Status.ErrorInject = PPUTST_INVALID_ERROR_INJECT;
            ErrorInjectFlag = FALSE;
        }
    }
}

/*==================================================================================================
 *                                   GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief            This function is used to initializes the PpuTst driver
 * @details
 *
 * @table            @service_id:       0x01   \n
 *                   @is_reentrant:     false  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        ConfigPtr Pointer to the configuration structure
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_PpuTst_392
 */
PPUTST_FUNC void PpuTst_Init(const PpuTst_ConfigType *ConfigPtr)
{
    if (NULL_PTR != ConfigPtr)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_INIT, PPUTST_E_PARAM_POINTER);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else if (PpuTst_Status.State != PPUTST_STATE_UNINIT)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_INIT, PPUTST_E_INIT_FAILED);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        PpuTst_ConfigPtr = &PpuTst_PreCompileConfig;
        for (uint8 GroupIndex = 0; GroupIndex < PPUTST_GROUP_COUNT; ++GroupIndex)
        {
#ifdef PPUTST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(PpuTst_Lld_Init, PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
#else
            PpuTst_Lld_Init(PpuTst_ConfigPtr->GroupConfigPtr[GroupIndex]);
#endif /* PPUTST_ENABLE_USER_MODE_SUPPORT*/
        }
#ifdef PPUTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(PpuTst_EnableBusFaultException);
#else
        PpuTst_EnableBusFaultException();
#endif /* PPUTST_ENABLE_USER_MODE_SUPPORT */
        PpuTst_ErrorStatusInit();
        PpuTst_Status.ErrorInject = PPUTST_INVALID_ERROR_INJECT;
        PpuTst_Status.State = PPUTST_STATE_INIT;
    }
}

/**
 * @brief            This function resets the PPU module to reset the global variables.
 * @details
 *
 * @table            @service_id:       0x02   \n
 *                   @is_reentrant:     false  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_PpuTst_387
 */
PPUTST_FUNC void PpuTst_DeInit(void)
{
    if (PPUTST_STATE_UNINIT == PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_DEINIT, PPUTST_E_UNINIT);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (PPUTST_STATE_RUNING == PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_DEINIT, PPUTST_E_INVALID_STATE);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        PpuTst_ConfigPtr = NULL_PTR;
        PpuTst_Status.Result = PPUTST_RESULT_NOT_TESTED;
        PpuTst_Status.State = PPUTST_STATE_UNINIT;
    }
}

/**
 * @brief            Service for executing the PPU module test.
 * @details          It performs various error checks on the configured PPU groups and updates the error status accordingly. The function also
 *                   handles timeout scenarios and deinitialize the PPU module if necessary. The overall result of the PPU test is updated based on
 *                   the error status.
 *
 * @table            @service_id:       0x03   \n
 *                   @is_reentrant:     false  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_PpuTst_826
 */
PPUTST_FUNC void PpuTst_Start(void)
{
    if (PPUTST_STATE_UNINIT == PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_START, PPUTST_E_UNINIT);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (PPUTST_STATE_TESTED == PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_START, PPUTST_E_INVALID_STATE);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        PpuTst_Status.State = PPUTST_STATE_RUNING;
        (void)Platform_InstallIrqHandler(BusFault_IRQn, PpuTst_BusFault, &FuncPointer);
        /* Inject error for soft lock path */
        PpuTst_SoftLockTest();
        /* Inject error for hard lock path */
        PpuTst_HardLockTest();
        (void)Platform_InstallIrqHandler(BusFault_IRQn, BusFault_Handler, &FuncPointer);
        if (PpuTst_Status.Result != PPUTST_RESULT_NOT_OK)
        {
            PpuTst_Status.Result = PPUTST_RESULT_OK;
        }
        PpuTst_Status.State = PPUTST_STATE_TESTED;
    }
}

/**
 * @brief            This function returns the result of the PPU test.
 * @details
 *
 * @table            @service_id:       0x04   \n
 *                   @is_reentrant:     true   \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        void
 * @return           PpuTst_ResultType
 * @retval           PPUTST_RESULT_NOT_TESTED: Test is not executed or still in progress.
 * @retval           PPUTST_RESULT_OK: All the test case of injection are passed.
 * @retval           PPUTST_RESULT_NOT_OK: At least one test case of error injection is failed.
 * @sw_type          sw_detail
 *
 * @trace YTSDS_PpuTst_830
 */
PPUTST_FUNC PpuTst_ResultType PpuTst_GetTestResult(void)
{
    PpuTst_ResultType Result = PPUTST_RESULT_NOT_TESTED;
    if (PPUTST_STATE_UNINIT == PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_GET_TEST_RESULT, PPUTST_E_UNINIT);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (PPUTST_STATE_TESTED != PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_GET_TEST_RESULT, PPUTST_E_INVALID_STATE);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        Result = PpuTst_Status.Result;
    }
    return Result;
}

/**
 * @brief            This function is get the test details of the PPU test.
 * @details
 *
 * @table            @service_id:       0x05   \n
 *                   @is_reentrant:     true   \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[out]       PpuTstGetDetail Pointer to where to store the test details.
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_PpuTst_829
 */
PPUTST_FUNC void PpuTst_GetDetailsResult(PpuTst_TestDetailType *PpuTstGetDetail)
{
    if (NULL_PTR == PpuTstGetDetail)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_GET_DETAILS_RESULT, PPUTST_E_PARAM_POINTER);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (PPUTST_STATE_UNINIT == PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_GET_DETAILS_RESULT, PPUTST_E_UNINIT);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (PPUTST_STATE_TESTED != PpuTst_Status.State)
    {
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID,
                              PPUTST_SID_GET_DETAILS_RESULT, PPUTST_E_INVALID_STATE);
#endif /* #if (PPUTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        for (uint8 GroupIndex = 0; GroupIndex < PPUTST_GROUP_COUNT; ++GroupIndex)
        {
            PpuTstGetDetail[GroupIndex] = PpuTst_Status.PpuTst_Result[GroupIndex];
        }
    }
}

#if (PPUTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @details
 *
 * @table            @service_id:       0x06   \n
 *                   @is_reentrant:     true  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[out]       Versioninfo Pointer to where to store version information of this module
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_PpuTst_825
 */
PPUTST_FUNC void PpuTst_GetVersionInfo(Std_VersionInfoType *Versioninfo)
{
    /* Check for DET: PPUTST_E_PARAM_POINTER */
    if (NULL_PTR == Versioninfo)
    {
        /* Report PPUTST_E_PARAM_POINTER DET if service called with NULL_PTR */
#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(PPUTST_MODULE_ID, PPUTST_INSTANCE_ID, PPUTST_SID_GET_VERSION_INFO, PPUTST_E_PARAM_POINTER);
#endif /* PPUTST_DEV_ERROR_DETECT */
    }
    else
    {
        /* Vendor ID information */
        Versioninfo->vendorID = PPUTST_VENDOR_ID;
        /* PpuTst module ID information */
        Versioninfo->moduleID = PPUTST_MODULE_ID;
        /* PpuTst module Software major version information */
        Versioninfo->sw_major_version = (uint8)PPUTST_SW_MAJOR_VERSION;
        /* PpuTst module Software minor version information */
        Versioninfo->sw_minor_version = (uint8)PPUTST_SW_MINOR_VERSION;
        /* PpuTst module Software patch version information */
        Versioninfo->sw_patch_version = (uint8)PPUTST_SW_PATCH_VERSION;
    }
}
#endif /* PPUTST_VERSION_INFO_API == STD_ON */

#define PPUTST_STOP_SEC_CODE
#include "PpuTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file PpuTst.c */
