/*
* @file    PpuTst_Types.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : PpuTst_Types
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef PPUTST_TYPES_H
#define PPUTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 0750 Rule-19.2: A union type specifier has been defined.
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "PpuTst_Cfg.h"

/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define PPUTST_TYPES_VENDOR_ID                      (180)
#define PPUTST_TYPES_AR_RELEASE_MAJOR_VERSION       (4)
#define PPUTST_TYPES_AR_RELEASE_MINOR_VERSION       (4)
#define PPUTST_TYPES_AR_RELEASE_REVISION_VERSION    (0)
#define PPUTST_TYPES_SW_MAJOR_VERSION               (2)
#define PPUTST_TYPES_SW_MINOR_VERSION               (0)
#define PPUTST_TYPES_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and PpuTst configuration header file are of the same vendor */
#if (PPUTST_TYPES_VENDOR_ID != PPUTST_VENDOR_ID_CFG)
#error "PpuTst_Types.h and PpuTst_Cfg.h have different vendor ids"
#endif

/* Check if source file and PpuTst configuration header file are of the same Autosar version */
#if ((PPUTST_TYPES_AR_RELEASE_MAJOR_VERSION != PPUTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (PPUTST_TYPES_AR_RELEASE_MINOR_VERSION != PPUTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (PPUTST_TYPES_AR_RELEASE_REVISION_VERSION != PPUTST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of PpuTst_Types.h and PpuTst_Cfg.h are different"
#endif

/* Check if source file and PpuTst configuration header file are of the same software version */
#if ((PPUTST_TYPES_SW_MAJOR_VERSION != PPUTST_SW_MAJOR_VERSION_CFG) || \
     (PPUTST_TYPES_SW_MINOR_VERSION != PPUTST_SW_MINOR_VERSION_CFG) || \
     (PPUTST_TYPES_SW_PATCH_VERSION != PPUTST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of PpuTst_Types.h and PpuTst_Cfg.h are different"
#endif
/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/*==================================================================================================
 *                                       EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
 *                                             ENUMS
==================================================================================================*/
/**
 * @brief          This enumerated type contains the possible results of all tests.
 * @sw_type        sw_arch
 */
typedef enum
{
    PPUTST_RESULT_NOT_TESTED = (uint8)0x00,  /**< The PPU test has not been executed or is still in progress*/
    PPUTST_RESULT_OK         = (uint8)0x01,  /**< The PPU test result has passed*/
    PPUTST_RESULT_NOT_OK     = (uint8)0x02,  /**< The PPU test result has failed*/
} PpuTst_ResultType;

/**
 * @brief          This enumerated type represents the test result for all group paths.
 * @sw_type        sw_arch
 */
typedef enum
{
    PPUTST_PATH_NOT_TESTED = 0x00U,  /**< The path that doesn't require testing or the test process has not yet been completed*/
    PPUTST_PATH_PASS       = 0x01U,  /**< The path test result has passed */
    PPUTST_PATH_FAILED     = 0x02U   /**< The path test result has failed */
} PpuTst_PathResultType;

/**
 * @brief          This enumerated type contains all error test injections.
 * @sw_type        sw_detail
 */
typedef enum
{
    PPUTST_TMR_SL_ERR_INJECT    = 0U,   /**< Inject a soft lock error for TMR*/
    PPUTST_SCU_SL_ERR_INJECT    = 1U,   /**< Inject a soft lock error for SCU*/
    PPUTST_PCU_SL_ERR_INJECT    = 2U,   /**< Inject a soft lock error for PCU*/
    PPUTST_RCU_SL_ERR_INJECT    = 3U,   /**< Inject a soft lock error for RCU*/
    PPUTST_EFM_SL_ERR_INJECT    = 4U,   /**< Inject a soft lock error for EFM*/
    PPUTST_IPC_SL_ERR_INJECT    = 5U,   /**< Inject a soft lock error for IPC*/
    PPUTST_TMR_HL_ERR_INJECT    = 6U,   /**< Inject a hard lock error for TMR*/
    PPUTST_SCU_HL_ERR_INJECT    = 7U,   /**< Inject a hard lock error for SCU*/
    PPUTST_PCU_HL_ERR_INJECT    = 8U,   /**< Inject a hard lock error for PCU*/
    PPUTST_RCU_HL_ERR_INJECT    = 9U,   /**< Inject a hard lock error for RCU*/
    PPUTST_EFM_HL_ERR_INJECT    = 10U,  /**< Inject a hard lock error for EFM*/
    PPUTST_IPC_HL_ERR_INJECT    = 11U,  /**< Inject a hard lock error for IPC*/
    PPUTST_INVALID_ERROR_INJECT = 12U,  /**< Represents an invalid or undefined error injection case*/
} PpuTst_CheckType;

/**
 * @brief          This enumerated type represents the group ID of the PPU group.
 * @sw_type        sw_detail
 */
typedef enum
{
    PPUTST_GROUP_TMR = 0U,  /**< The group ID of the TMR PPU group*/
    PPUTST_GROUP_SCU = 1U,  /**< The group ID of the SCU PPU group*/
    PPUTST_GROUP_PCU = 2U,  /**< The group ID of the PCU PPU group*/
    PPUTST_GROUP_RCU = 3U,  /**< The group ID of the RCU PPU group*/
    PPUTST_GROUP_EFM = 4U,  /**< The group ID of the EFM PPU group*/
    PPUTST_GROUP_IPC = 5U   /**< The group ID of the IPC PPU group*/
} PpuTst_GroupIdType;

/**
 * @brief          This enumerated type represents the status of the PpuTst driver.
 * @sw_type        sw_arch
 */
typedef enum
{
    PPUTST_STATE_UNINIT = (uint8)0x0,  /**< PpuTst driver state is not initialized */
    PPUTST_STATE_INIT   = (uint8)0x1,  /**< PpuTst driver state is initialized */
    PPUTST_STATE_RUNING = (uint8)0x2,   /**< PpuTst driver state is running */
    PPUTST_STATE_TESTED = (uint8)0x3    /**< PpuTst test execution finished */
} PpuTst_StateType;

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/
/**
 * @brief           This type is used to define the callback function when an error is detected.
 * @sw_type         sw_arch
 */
typedef void(*PpuTst_NotificationType)(uint16 ModuleId);

/**
 * @brief          This structure contains the error check result for each group.
 * @sw_type        sw_arch
 */
typedef struct
{
    PpuTst_PathResultType SLockPathErrorCheckState;  /**< Soft lock function error check state */
    PpuTst_PathResultType HLockPathErrorCheckState;  /**< Hard lock function error check state */
} PpuTst_TestDetailType;

/**
 * @brief          Internal structure type for the PpuTst driver
 * @sw_type        sw_arch
 */
typedef struct
{
    PpuTst_StateType State;                                  /**< Current PpuTst execution state */
    PpuTst_CheckType ErrorInject;                            /**< Used to indicate the error inject type, for judgment in the PpuTst_BusFault Handler */
    PpuTst_TestDetailType PpuTst_Result[PPUTST_GROUP_COUNT]; /**< Array of the PpuTst test result details */
    PpuTst_ResultType Result;                                /**< The overall result of the PpuTst test */
} PpuTst_StatusType;

/**
 * @brief          Internal structure contains the configuration set for PPU groups.
 * @sw_type        sw_arch
 */
/* MR12 Rule-19.2 VIOLATION: In this specific case, Using a union simplifies the test function's complexity while making error injection checks
 * more intuitive, thus, couldn't adhere to M3CM Rule-19.2.
 */
/*PRQA S 0750 ++*/
typedef union
{
    struct
    {
        unsigned int PpuTmrProtectionError : 1;  /**< Enable/Disable TMR protection path error check*/
        unsigned int PpuScuProtectionError : 1;  /**< Enable/Disable SCU protection path error check*/
        unsigned int PpuPcuProtectionError : 1;  /**< Enable/Disable PCU protection path error check*/
        unsigned int PpuRcuProtectionError : 1;  /**< Enable/Disable RCU protection path error check*/
        unsigned int PpuEfmProtectionError : 1;  /**< Enable/Disable EFM protection path error check*/
        unsigned int PpuIpcProtectionError : 1;  /**< Enable/Disable IPC protection path error check*/
    } InjectType;
    unsigned int Value;
} PpuTst_ErrorInjectType;
/*PRQA S 0750 --*/

/**
 * @brief          This structure contains the configuration set for the soft lock of PPU groups.
 * @sw_type        sw_arch
 */
typedef struct
{
    PpuTst_GroupIdType GroupId;  /**< The group ID of the PPU group*/
    boolean Enable;              /**< Enable/Disable the soft lock protection path check*/
    uint32 GroupPathOffset;      /**< The offset of the group path register*/
} PpuTst_GroupSLockConfigType;

/**
 * @brief          This structure contains the configuration set for PPU groups.
 * @sw_type        sw_arch
 */
typedef struct
{
    const PpuTst_GroupSLockConfigType *GroupConfigPtr;        /**< Configuration pointer for PPU groups*/
    const PpuTst_ErrorInjectType *SLockErrorInjectConfigPtr;  /**< Configuration pointer for soft lock protection path check*/
    const PpuTst_ErrorInjectType *HLockErrorInjectConfigPtr;  /**< Configuration pointer for hard lock protection path check*/
    PpuTst_NotificationType PpuTstErrorNotification;          /**< Error callback notification function*/
} PpuTst_ConfigType;

#ifdef __cplusplus
}
#endif

#endif /* End of file PpuTst_Types.h */
