/*
* @file    PpuTst.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : PpuTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef PPUTST_H
#define PPUTST_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "PpuTst_Types.h"

/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define PPUTST_VENDOR_ID                      (180)
#define PPUTST_AR_RELEASE_MAJOR_VERSION       (4)
#define PPUTST_AR_RELEASE_MINOR_VERSION       (4)
#define PPUTST_AR_RELEASE_REVISION_VERSION    (0)
#define PPUTST_SW_MAJOR_VERSION               (2)
#define PPUTST_SW_MINOR_VERSION               (0)
#define PPUTST_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and PPUTST TYPES header file are of the same vendor */
#if (PPUTST_VENDOR_ID != PPUTST_TYPES_VENDOR_ID)
#error "PpuTst.h and PpuTst_Types.h have different vendor ids"
#endif

/* Check if source file and PPUTST TYPES header file are of the same Autosar version */
#if ((PPUTST_AR_RELEASE_MAJOR_VERSION != PPUTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (PPUTST_AR_RELEASE_MINOR_VERSION != PPUTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (PPUTST_AR_RELEASE_REVISION_VERSION != PPUTST_TYPES_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of PpuTst.h and PpuTst_Types.h are different"
#endif

/* Check if source file and PPUTST TYPES header file are of the same software version */
#if ((PPUTST_SW_MAJOR_VERSION != PPUTST_TYPES_SW_MAJOR_VERSION) || \
     (PPUTST_SW_MINOR_VERSION != PPUTST_TYPES_SW_MINOR_VERSION) || \
     (PPUTST_SW_PATCH_VERSION != PPUTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of PpuTst.h and PpuTst_Types.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/**
 * @brief          PpuTst module instance ID
 * @sw_type        sw_detail
 * @private
 */
#define PPUTST_INSTANCE_ID                    ((uint8)0U)

/**
 * @brief          PpuTst module ID
 * @sw_type        sw_detail
 * @private
 */
#define PPUTST_MODULE_ID                      ((uint16)521)

#if (PPUTST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup        PPUTST_DEV_ERROR
 * @brief           PpuTst development errors
 * @sw_type         sw_arch
 * @{
 */
#define PPUTST_E_INIT_FAILED                  ((uint8)0x10)  /**< Initialization was called when the module was already initialized*/
#define PPUTST_E_PARAM_POINTER                ((uint8)0x11)  /**< Error indicating an issue with a parameter pointer (e.g., NULL or invalid pointer)*/
#define PPUTST_E_UNINIT                       ((uint8)0x12)  /**< An API was called without module initialization*/
#define PPUTST_E_INVALID_STATE                ((uint8)0x13)  /**< An API was called in an invalid state*/
/** @} */
#endif

/**
 * @defgroup        PPUTST_SID
 * @brief           PpuTst service ID
 * @sw_type         sw_detail
 * @{
 */
#define PPUTST_SID_INIT                       (0x01U)  /**< Service ID for PpuTst_Init */
#define PPUTST_SID_DEINIT                     (0x02U)  /**< Service ID for PpuTst_DeInit */
#define PPUTST_SID_START                      (0x03U)  /**< Service ID for PpuTst_Start */
#define PPUTST_SID_GET_TEST_RESULT            (0x04U)  /**< Service ID for PpuTst_GetTestResult */
#define PPUTST_SID_GET_DETAILS_RESULT         (0x05U)  /**< Service ID for PpuTst_GetDetailsResult */
#define PPUTST_SID_GET_VERSION_INFO           (0x06U)  /**< Service ID for PpuTst_GetVersionInfo */
/** @} */

/**
 * @brief          The timeout value for the Ppu group test error check
 * @details        For a 120Mhz core clock, the timeout value is approximately 8.33us
 * @sw_type        sw_detail
 */
#define ERROR_CHECK_TIMEOUT                   (1000U)

/**
 * @defgroup        PPUTST_PATH_INJECT_ADDR
 * @brief           The error injection address for the PPU group
 * @sw_type         sw_detail
 * @{
 */
#define TMR_SOFT_LOCK_INJECT_ADDR              ((uint32)0x40054000)  /**< TMR path inject address for soft lock test*/
#define SCU_SOFT_LOCK_INJECT_ADDR              ((uint32)0x4007C000)  /**< SCU path inject address for soft lock test*/
#define PCU_SOFT_LOCK_INJECT_ADDR              ((uint32)0x4007E008)  /**< PCU path inject address for soft lock test*/
#define RCU_SOFT_LOCK_INJECT_ADDR              ((uint32)0x4007F004)  /**< RCU path inject address for soft lock test*/
#define EFM_SOFT_LOCK_INJECT_ADDR              ((uint32)0x40010000)  /**< EFM path inject address for soft lock test*/
#define IPC_SOFT_LOCK_INJECT_ADDR              ((uint32)0x4007D020)  /**< IPC path inject address for soft lock test*/

#define TMR_HARD_LOCK_INJECT_ADDR              ((uint32)0x40054C00)  /**< TMR path inject address for hard lock test*/
#define SCU_HARD_LOCK_INJECT_ADDR              ((uint32)0x4007CC00)  /**< Scu path inject address for hard lock test*/
#define PCU_HARD_LOCK_INJECT_ADDR              ((uint32)0x4007EC02)  /**< PCU path inject address for hard lock test*/
#define RCU_HARD_LOCK_INJECT_ADDR              ((uint32)0x4007FC01)  /**< RCU path inject address for hard lock test*/
#define EFM_HARD_LOCK_INJECT_ADDR              ((uint32)0x40010C00)  /**< EFM path inject address for hard lock test*/
#define IPC_HARD_LOCK_INJECT_ADDR              ((uint32)0x4007DC08)  /**< IPC path inject address for hard lock test*/
/** @} */

/**
 * @brief          The array of the soft lock error injection address for the PPU group
 * @sw_type        sw_detail
 */
#define SOFT_LOCK_INJECT_ADDRESS              { TMR_SOFT_LOCK_INJECT_ADDR, SCU_SOFT_LOCK_INJECT_ADDR,\
                                                PCU_SOFT_LOCK_INJECT_ADDR, RCU_SOFT_LOCK_INJECT_ADDR,\
                                                EFM_SOFT_LOCK_INJECT_ADDR, IPC_SOFT_LOCK_INJECT_ADDR }

/**
 * @brief          The array of the hard lock error injection address for the PPU group
 * @sw_type        sw_detail
 */
#define HARD_LOCK_INJECT_ADDRESS              { TMR_HARD_LOCK_INJECT_ADDR, SCU_HARD_LOCK_INJECT_ADDR,\
                                                PCU_HARD_LOCK_INJECT_ADDR, RCU_HARD_LOCK_INJECT_ADDR,\
                                                EFM_HARD_LOCK_INJECT_ADDR, IPC_HARD_LOCK_INJECT_ADDR }

/**
 * @brief          Length of one instruction, used to skip the instruction that caused the bus fault
 * @sw_type        sw_detail
 */
#define NEXT_PC_OFFSET                        (0x4U)

/*==================================================================================================
 *                                          VARIATES
==================================================================================================*/

/*==================================================================================================
 *                                   EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                   GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief            This function is used to initializes the PpuTst driver
 * @details
 *
 * @table            @service_id:       0x01   \n
 *                   @is_reentrant:     false  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        ConfigPtr Pointer to the configuration structure
 * @return           void
 * @sw_type          sw_arch
 */
void PpuTst_Init(const PpuTst_ConfigType *ConfigPtr);

/**
 * @brief            This function resets the PPU module to reset the global variables.
 * @details
 *
 * @table            @service_id:       0x02   \n
 *                   @is_reentrant:     false  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void PpuTst_DeInit(void);

/**
 * @brief            Service for executing the PPU module test.
 * @details          It performs various error checks on the configured PPU groups and updates the error status accordingly. The function also
 *                   handles timeout scenarios and deinitialize the PPU module if necessary. The overall result of the PPU test is updated based on
 *                   the error status.
 *
 * @table            @service_id:       0x03   \n
 *                   @is_reentrant:     false  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        void
 * @return           void
 * @sw_type          sw_arch
 */
void PpuTst_Start(void);

/**
 * @brief            This function returns the result of the PPU test.
 * @details
 *
 * @table            @service_id:       0x04   \n
 *                   @is_reentrant:     true   \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[in]        void
 * @return           PpuTst_ResultType
 * @retval           PPUTST_RESULT_NOT_TESTED: Test is not executed or still in progress.
 * @retval           PPUTST_RESULT_OK: All the test case of injection are passed.
 * @retval           PPUTST_RESULT_NOT_OK: At least one test case of error injection is failed.
 * @sw_type          sw_arch
 */
PpuTst_ResultType PpuTst_GetTestResult(void);

/**
 * @brief            This function is get the test details of the PPU test.
 * @details
 *
 * @table            @service_id:       0x05   \n
 *                   @is_reentrant:     true   \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[out]       PpuTstGetDetail Pointer to where to store the test details.
 * @return           void
 * @sw_type          sw_arch
 */
void PpuTst_GetDetailsResult(PpuTst_TestDetailType *PpuTstGetDetail);

#if (PPUTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @details
 *
 * @table            @service_id:       0x06   \n
 *                   @is_reentrant:     true  \n
 *                   @is_synchronous:   true   \n
 *                   @autosar_api:      false  \n
 *
 * @param[out]       Versioninfo Pointer to where to store version information of this module
 * @return           void
 * @sw_type          sw_arch
 */
void PpuTst_GetVersionInfo(Std_VersionInfoType *Versioninfo);

#endif /* PPUTST_VERSION_INFO_API == STD_ON */

#ifdef __cplusplus
}
#endif

#endif /* End of file PpuTst.h */
