/**
* @file    IntTst_Lld_Ptmr.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : IntTst_Lld_Ptmr
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/

#include "IntTst_Lld_Ptmr.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define INTTST_LLD_PTMR_VENDOR_ID_C                     (180)
#define INTTST_LLD_PTMR_AR_REL_MAJOR_VER_C              (4)
#define INTTST_LLD_PTMR_AR_REL_MINOR_VER_C              (4)
#define INTTST_LLD_PTMR_AR_REL_REVISION_VER_C           (0)
#define INTTST_LLD_PTMR_SW_MAJOR_VER_C                  (2)
#define INTTST_LLD_PTMR_SW_MINOR_VER_C                  (0)
#define INTTST_LLD_PTMR_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and INTTST_LLD_PTMR header file are of the same vendor */
#if (INTTST_LLD_PTMR_VENDOR_ID_C != INTTST_LLD_PTMR_VENDOR_ID)
#error "IntTst_Lld_Ptmr.c and IntTst_Lld_Ptmr.h have different vendor ids"
#endif

/* Check if source file and INTTST_LLD_PTMR header file are of the same Autosar version */
#if (( INTTST_LLD_PTMR_AR_REL_MAJOR_VER_C != INTTST_LLD_PTMR_AR_REL_MAJOR_VER) || \
      ( INTTST_LLD_PTMR_AR_REL_MINOR_VER_C != INTTST_LLD_PTMR_AR_REL_MINOR_VER) || \
      ( INTTST_LLD_PTMR_AR_REL_REVISION_VER_C != INTTST_LLD_PTMR_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of IntTst_Lld_Ptmr.c and IntTst_Lld_Ptmr.h are different"
#endif

/* Check if source file and INTTST_LLD_PTMR header file are of the same Software version */
#if (( INTTST_LLD_PTMR_SW_MAJOR_VER_C != INTTST_LLD_PTMR_SW_MAJOR_VER) || \
      ( INTTST_LLD_PTMR_SW_MINOR_VER_C != INTTST_LLD_PTMR_SW_MINOR_VER) || \
      ( INTTST_LLD_PTMR_SW_PATCH_VER_C != INTTST_LLD_PTMR_SW_PATCH_VER))
#error "Software Version Numbers of IntTst_Lld_Ptmr.c and IntTst_Lld_Ptmr.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/

#define INTTST_START_SEC_CODE
#include "IntTst_MemMap.h"

/**
 * @brief      Initialize the pTMR module with the pTMR module and freeze mode enabled
 * @param[in]  Base The pTMR module register base address
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_678
 */
INTTST_FUNC INTTST_FUNC void IntTst_LLd_Ptmr_Init(volatile pTMR_Type *Base)
{
    /* enable the pTMR with freeze mode enabled (counter continue to run in the debug mode) */
    Base->MCR = pTMR_MCR_FRZ(1) | pTMR_MCR_EN_MASK;
}

/**
 * @brief Configure the pTMR channel and start it with interrupt enabled
 *
 * @param[in] Base        The pTMR module register base address
 * @param[in] Channel     The pTMR channel number
 * @param[in] StartValue  The start value of the pTMR channel timer counter

 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:       .mcal_text \n
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_679
 */
INTTST_FUNC void IntTst_Lld_Ptmr_StartChannel(volatile pTMR_Type *Base, uint8 Channel, uint32 StartValue)
{
    /* set the start value */
    Base->CH[Channel].TSV = StartValue;
    /* enable the channel with interrupt enabled */
    Base->CH[Channel].TCR = pTMR_CH_TCR_TIE_MASK | pTMR_CH_TCR_TEN_MASK;
}

/**
 * @brief Stop the pTMR channel with channel disabled and interrupt disabled
 *
 * @param[in] Base        The pTMR module register base address
 * @param[in] Channel     The pTMR channel number
 *
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:       .mcal_text \n
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_680
 */
INTTST_FUNC void IntTst_Lld_Ptmr_StopChannel(volatile pTMR_Type *Base, uint8 Channel)
{
    /* set the start value as zero */
    Base->CH[Channel].TSV = 0U;
    /* disable the channel with interrupt enabled */
    Base->CH[Channel].TCR = 0x00U;
}

/**
 * @brief Process the pTMR channel interrupt, double the channel interrupt flag with the channel interrupt enable configuration
 *        clean the interrupt flag via write 1 clear
 *
 * @param[in] Base        The pTMR module register base address
 * @param[in] Channel     The pTMR channel number
 *
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:       .mcal_text \n
* @return void
* @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_676
*/
INTTST_FUNC void IntTst_Lld_Ptmr_ChannelIrqProcess(volatile pTMR_Type *Base, uint8 Channel)
{
    /* read the channel interrupt flag and check with the interrupt enable configuration */
    if ((pTMR_CH_TFR_TIF_MASK == (Base->CH[Channel].TFR & pTMR_CH_TFR_TIF_MASK)) && \
            (pTMR_CH_TCR_TIE_MASK == (Base->CH[Channel].TCR & pTMR_CH_TCR_TIE_MASK)))
    {
        /* clear the interrupt flag */
        Base->CH[Channel].TFR = pTMR_CH_TFR_TIF_MASK;
    }
    else
    {
        /* do nothing */
    }
}

/**
 * @brief Deinitialize the pTMR module
 *
 * @param[in] Base        The pTMR module register base address
 *
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:       .mcal_text \n
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_677
 */
INTTST_FUNC void IntTst_LLd_Ptmr_DeInit(volatile pTMR_Type *Base)
{
    /* disable the pTMR */
    Base->MCR = 0U;
}

#define INTTST_STOP_SEC_CODE
#include "IntTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file IntTst_Lld_Ptmr.c */
