/**
* @file    IntTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : IntTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 4700: Metric value out fo range
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Platform.h"
#include "IntTst.h"

#ifdef INTTST_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif /* INTTST_ENABLE_USER_MODE_SUPPORT */

#include "SchM_IntTst.h"
#include "IntTst_Lld_Intm.h"
#include "IntTst_Lld_Ptmr.h"

#if (INTTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* INTTST_DEV_ERROR_DETECT == STD_ON */

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define INTTST_VENDOR_ID_C                      (180)
#define INTTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define INTTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define INTTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define INTTST_SW_MAJOR_VERSION_C               (2)
#define INTTST_SW_MINOR_VERSION_C               (0)
#define INTTST_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and IntTst header file are of the same vendor */
#if (INTTST_VENDOR_ID_C != INTTST_VENDOR_ID)
#error "IntTst.c and IntTst.h have different vendor ids"
#endif

/* Check if source file and IntTst header file are of the same Autosar version */
#if (( INTTST_AR_RELEASE_MAJOR_VERSION_C != INTTST_AR_RELEASE_MAJOR_VERSION) || \
      ( INTTST_AR_RELEASE_MINOR_VERSION_C != INTTST_AR_RELEASE_MINOR_VERSION) || \
      ( INTTST_AR_RELEASE_REVISION_VERSION_C != INTTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of IntTst.c and IntTst.h are different"
#endif

/* Check if source file and IntTst header file are of the same Software version */
#if (( INTTST_SW_MAJOR_VERSION_C != INTTST_SW_MAJOR_VERSION) || \
      ( INTTST_SW_MINOR_VERSION_C != INTTST_SW_MINOR_VERSION) || \
      ( INTTST_SW_PATCH_VERSION_C != INTTST_SW_PATCH_VERSION))
#error "Software Version Numbers of IntTst.c and IntTst.h are different"
#endif

/* Check if source file and IntTst SchM header file are of the same vendor */
#if (INTTST_VENDOR_ID_C != SCHM_INTTST_VENDOR_ID)
#error "IntTst.c and SchM_IntTst.h have different vendor ids"
#endif

/* Check if source file and SchM IntTst header file are of the same Autosar version*/
#if ((INTTST_AR_RELEASE_MAJOR_VERSION_C != SCHM_INTTST_AR_RELEASE_MAJOR_VERSION) || \
     (INTTST_AR_RELEASE_MINOR_VERSION_C != SCHM_INTTST_AR_RELEASE_MINOR_VERSION))
#error "AutoSar Version Numbers of IntTst.c and SchM_IntTst.h are different"
#endif

/* Check if source file and SchM IntTst header file are of the same Software version*/
#if ((INTTST_SW_MAJOR_VERSION_C != SCHM_INTTST_SW_MAJOR_VERSION) || \
     (INTTST_SW_MINOR_VERSION_C != SCHM_INTTST_SW_MINOR_VERSION) || \
     (INTTST_SW_PATCH_VERSION_C != SCHM_INTTST_SW_PATCH_VERSION))
#error "Software Version Numbers of IntTst.c and SchM_IntTst.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/

/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
 *                                          VARIATES
==================================================================================================*/

#define INTTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "IntTst_MemMap.h"

/**
 * @brief   IntTst module configuration structure pointers, default is NULL
 * @private
*/
INTTST_VAR static const IntTst_ConfigType *IntTstConfigPtr = NULL_PTR;

/**
 * @brief   The IRQHandler for INTM configuration backup, default is NULL
 * @private
*/
INTTST_VAR static Platform_IrqHandlerType BackupIrqHandler = NULL_PTR;

/**
 * @brief   The dummy IRQHandler for INTM configuration recover , default is NULL
 * @private
*/
INTTST_VAR static Platform_IrqHandlerType DummyIrqHandler = NULL_PTR;

#define INTTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "IntTst_MemMap.h"

#define INTTST_START_SEC_VAR_INIT_8
#include "IntTst_MemMap.h"

/**
 * @brief   backup fo the INTM IRQ interrupt priority, default is 0
 * @private
*/
INTTST_VAR static uint8 BackupIntmIrqPriority = 0;

/**
 * @brief   IntTst module state, default is UNINIT
 * @private
*/
INTTST_VAR static IntTst_StateType IntTst_State = INTTST_UNINIT;

/**
* @brief     Global variable to store the error details of the IntTst module test.
* @private
*/
INTTST_VAR static IntTst_ResultType IntTst_Result = INTTST_RESULT_NOT_TESTED;

#define INTTST_STOP_SEC_VAR_INIT_8
#include "IntTst_MemMap.h"

#define INTTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "IntTst_MemMap.h"

/**
* @brief     Global variable to store the error details of the IntTst module test.
* @private
*/
INTTST_VAR static IntTst_TestDetailsType IntTst_TestDetails;

/**
* @brief     Global variable to store the result of the IntTst channel test.
* @private
*/
INTTST_VAR static IntTst_ChannelTestType IntTst_ChannelTestResult[INTTST_CHANNEL_INJECTION_TST_COUNT];

#define INTTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "IntTst_MemMap.h"

/*==================================================================================================
 *                                   LOCAL FUNCTION DECLARATION
==================================================================================================*/

#define INTTST_START_SEC_CODE
#include "IntTst_MemMap.h"

INTTST_FUNC static void IntTst_IntmIrqHandler(void);
INTTST_FUNC static Std_ReturnType IntTst_TestDetailsInit(const IntTst_ConfigType *Config, IntTst_TestDetailsType *TestDetails);

#define INTTST_STOP_SEC_CODE
#include "IntTst_MemMap.h"

/*==================================================================================================
 *                                        LOCAL FUNCTIONS
==================================================================================================*/

#define INTTST_START_SEC_CODE
#include "IntTst_MemMap.h"

/**
 * @brief     INTM IRQ interrupt handler for IntTst
 * @details   This function is the INTM IRQ interrupt handler for IntTst module.
 *            It checks the status of the INTM monitor channel and updates the test result.
 *
 * @table           @service_id:       --    \n
 *                  @is_reentrant:     false \n
 *                  @is_synchronous:   true  \n
 *                  @autosar_api:      false \n
 * @param[in]        void
 * @return           void
 *
 * @sw_type          sw_detail
 *
 * @trace YTSDS_IntTst_849
 */
INTTST_FUNC static void IntTst_IntmIrqHandler(void)
{
    uint8 Channel = 0U;
    for (Channel = 0U; Channel < INTTST_CHANNEL_INJECTION_TST_COUNT; Channel++)
    {
#ifdef INTTST_ENABLE_USER_MODE_SUPPORT
        /* get status of the INTM monitor Channel */
        if (TRUE == (uint8)OsIf_Trusted_Call_Return2param((IntTst_Lld_Intm_GetChannelStatus), INTM, Channel))
#else
        /* get status of the INTM monitor Channel */
        if (TRUE == IntTst_Lld_Intm_GetChannelStatus(INTM, Channel))
#endif /* INTTST_ENABLE_USER_MODE_SUPPORT */
        {
            /* clean the pTMR interrupt flag and disable the interrupt  to avoid repeat enter the INTM interrupt */
#ifdef INTTST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call2params((IntTst_Lld_Ptmr_ChannelIrqProcess), INTTST_PTMR, (IntTstConfigPtr->Monitor[Channel].Channel));
            /* clean the IRQ monitor Channel timer of INTM */
            OsIf_Trusted_Call2params((IntTst_Lld_Intm_AckIrq), INTM, Channel);
#else
            IntTst_Lld_Ptmr_ChannelIrqProcess(INTTST_PTMR, IntTstConfigPtr->Monitor[Channel].Channel);
            /* clean the IRQ monitor Channel timer of INTM */
            IntTst_Lld_Intm_AckIrq(INTM, Channel);
#endif /* INTTST_ENABLE_USER_MODE_SUPPORT */
            SchM_Enter_IntTst_INTTST_EXCLUSIVE_AREA_00();
            /* update the channel test result into IntTst_TestDetails */
            IntTst_TestDetails.IntmChTestDetails[Channel].Test_Result = INTTST_RESULT_OK;
            SchM_Exit_IntTst_INTTST_EXCLUSIVE_AREA_00();
        }
        else
        {
            /* do nothing */
        }
    }
}

/**
 * @brief     This function is used to initialize the test details of the IntTst module.
 * @details   It initializes the test details of the IntTst module with the configuration structure.
 *
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      false \n
 *
 * @param[in] Config - Pointer to the configuration structure of the IntTst module.
 * @param[in] TestDetails - Pointer to the test details structure of the IntTst module.
 * @return    Std_ReturnType
 * @retval    E_OK - The test details are initialized successfully.
 * @retval    E_NOT_OK - The test details are not initialized successfully.
 * @sw_type          sw_detail
 *
 * @trace YTSDS_IntTst_850
 */
INTTST_FUNC static Std_ReturnType IntTst_TestDetailsInit(const IntTst_ConfigType *Config, IntTst_TestDetailsType *TestDetails)
{
    Std_ReturnType RetVal = E_OK;
    uint8 Channel = 0U;
    /* check the input parameters and hardware INTM monitor channel count limitation */
    if ((NULL_PTR == Config) || (NULL_PTR == TestDetails) || (INTM_MON_COUNT < Config->MonitorCount))
    {
        RetVal = E_NOT_OK;  /* return error  without initializing the TestDetails */
    }
    else
    {
        /* initialize the test details */
        TestDetails->IntmChCount = Config->MonitorCount;         /* set the INTM monitor channel count */
        TestDetails->IntmChTestDetails = IntTst_ChannelTestResult; /* set the INTM monitor channel test details */
        /* initialize the INTM monitor channel test details */
        for (Channel = 0; Channel < TestDetails->IntmChCount; Channel++)
        {
            TestDetails->IntmChTestDetails[Channel].Channel = Config->Monitor[Channel].Channel;
            TestDetails->IntmChTestDetails[Channel].InjectError = NONE_ERROR;
            TestDetails->IntmChTestDetails[Channel].Test_Result = INTTST_RESULT_NOT_TESTED;
        }
    }
    return RetVal;
}

/*==================================================================================================
 *                                   GLOBAL FUNCTIONS
==================================================================================================*/
/**
* @brief            Service for initialization of the IntTst driver.
* @details          This function initializes the run-time state structure to provide the INTM channel allocation,
*                   pTMR channel interrupt configuration as trigger sources, and track the state for channels.
*                   It also resets the INTM modules, initializes the module to user-defined settings and default settings.
*
* @table            @service_id:       0x01  \n
*                   @is_reentrant:     false \n
*                   @is_synchronous:   true  \n
*                   @autosar_api:      false \n
* @param[in]        IntTstConfig Pointer to the IntTst configuration structure.
* @return           void
* @sw_type          sw_detail
*
 *
 * @trace YTSDS_IntTst_672
*/
INTTST_FUNC void IntTst_Init(const IntTst_ConfigType *IntTstConfig)
{
    /* check the IntTst module state, and it can only be initialized when it's uinitialized */
    if (INTTST_UNINIT != IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID,
                              INTTST_SID_INIT, INTTST_E_INVALID_STATE);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
#if (STD_ON == INTTST_PRECOMPILE_SUPPORT)
        (void)IntTstConfig;/* avoid compile warning */
        IntTstConfigPtr = &IntTst_PreCompileConfig;
#else
        if (NULL_PTR == IntTstConfig)
        {
#if (STD_ON = INTTST_DEV_ERROR_DETECT)
            (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID,
                                  INTTST_SID_INIT, INTTST_E_PARAM_POINTER);
#endif  /* #if (INTTST_DEV_ERROR_DETECT == STD_ON) */
        }
        else
        {
            /* store the pointer to the configuration structure */
            IntTstConfigPtr = IntTstConfig;
#endif /* INTTST_PRECOMPILE_SUPPORT */
        /* initialize the test details */
        if (E_OK == IntTst_TestDetailsInit(IntTstConfigPtr, &IntTst_TestDetails))
        {
#ifdef INTTST_ENABLE_USER_MODE_SUPPORT
            /* Initialize the INTM controller */
            OsIf_Trusted_Call1param(IntTst_LLd_Intm_Init, INTM);
            /* Initialize the pTMR for the test */
            OsIf_Trusted_Call1param(IntTst_LLd_Ptmr_Init, INTTST_PTMR);
#else
            IntTst_LLd_Intm_Init(INTM);
            /* Initialize the pTMR for the test */
            IntTst_LLd_Ptmr_Init(INTTST_PTMR);
#endif /* INTTST_ENABLE_USER_MODE_SUPPORT */
            /* install the INTM interrupt handler for the IntTst */
            (void)Platform_InstallIrqHandler(INTM_IRQn, IntTst_IntmIrqHandler, &BackupIrqHandler);
            /* get the IRQ interrupt priority of the INTM interrupt */
            (void)Platform_GetIrqPriority(INTM_IRQn, &BackupIntmIrqPriority);
            /* set the IRQ interrupt priority to the highest priority (0) */
            (void)Platform_SetIrqPriority(INTM_IRQn, 0);
            /* enable the INTM IRQ interrupt */
            (void)Platform_SetIrq(INTM_IRQn, TRUE);
            /* Initialize the INTM controller */
            /* set the module test result as NOT_TESTED */
            IntTst_Result = INTTST_RESULT_NOT_TESTED;
            /* set the module state as initialized */
            IntTst_State = INTTST_INIT;
        }
        else /* initialize the test details failed */
        {
            /* set the module test result as NOT_TESTED */
            IntTst_Result = INTTST_RESULT_NOT_TESTED;
            /* set the module state as uninitialized */
            IntTst_State = INTTST_UNINIT;
        }
#if (STD_OFF == INTTST_PRECOMPILE_SUPPORT)
    }
#endif /* INTTST_PRECOMPILE_SUPPORT */
}
}

/**
* @brief           Service for de-initialization of the IntTst driver.
* @details         This function resets the INTM module to reset state and disables the interrupt to the core.

* @table            @service_id:       0x02  \n
*                   @is_reentrant:     false \n
*                   @is_synchronous:   true  \n
*                   @autosar_api:      false \n
* @param[in]        void
* @return           void
* @sw_type          sw_detail
*
 *
 * @trace YTSDS_IntTst_668
*/
INTTST_FUNC void IntTst_DeInit(void)
{
    /* check the module state */
    if (INTTST_UNINIT == IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID,
                              INTTST_SID_DEINIT, INTTST_E_UNINIT);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    /* avoid uinit the IntTst module during the testing */
    else if (INTTST_TESTING == IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID,
                              INTTST_SID_DEINIT, INTTST_E_INVALID_STATE);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
#ifdef INTTST_ENABLE_USER_MODE_SUPPORT
        /* Deinitialize the INTM controller */
        OsIf_Trusted_Call1param(IntTst_LLd_Intm_DeInit, INTM);
        /* Deinitialize the pTMR for the test */
        OsIf_Trusted_Call1param(IntTst_LLd_Ptmr_DeInit, INTTST_PTMR);
#else
        /* Deinitialize the INTM controller */
        IntTst_LLd_Intm_DeInit(INTM);
        /* Deinitialize the pTMR for the test */
        IntTst_LLd_Ptmr_DeInit(INTTST_PTMR);
#endif /* INTTST_ENABLE_USER_MODE_SUPPORT */
        /* recover the INTM interrupt handler for the IntTst */
        (void)Platform_InstallIrqHandler(INTM_IRQn, BackupIrqHandler, &DummyIrqHandler);
        (void)DummyIrqHandler; /* void the dummy handler */
        /* recover the INTM IRQ interrupt priority */
        (void)Platform_SetIrqPriority(INTM_IRQn, BackupIntmIrqPriority);
        /* set the pointer to the configuration structure as NULL */
        IntTstConfigPtr = NULL_PTR;
        /* clean the test details */
        IntTst_TestDetails.IntmChCount = 0U;
        IntTst_TestDetails.IntmChTestDetails = NULL_PTR;
        /* set the module test result as NOT_TESTED */
        IntTst_Result = INTTST_RESULT_NOT_TESTED;
        /* set the module state as uninitialized */
        IntTst_State = INTTST_UNINIT;
    }
}

/**
 * @brief          Service for the startup test function of the IntTst driver.
 * @details        This function is the startup test  function of the INTM test module. It performs various error checks
 *                 on the configured INTM monitor channels and updates the error status accordingly. It also
 *                 handles timeout scenarios and deinitialize the INTM monitor channel and pTMR channel after the test.
 *                 The overall result of the INTM test is updated based on the error status.
 *
 * @table          @service_id:       0x03  \n
 *                 @is_reentrant:     false \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 * @param[in]      void
 * @return         void
 * @sw_type        sw_detail
 *
 *
 * @trace YTSDS_IntTst_674
 */
INTTST_FUNC void IntTst_Start(void) /*PRQA S 4700 */
{
    uint32 Timeout = 0U;    /* Timeout counter */
    uint8 Channel = 0U;     /* Channel index */
    /* check the module state */
    if (INTTST_UNINIT == IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID,
                              INTTST_SID_START, INTTST_E_UNINIT);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (INTTST_INIT != IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID,
                              INTTST_SID_START, INTTST_E_INVALID_STATE);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        /* set the IntTst state to INTTST_TESTING */
        IntTst_State = INTTST_TESTING;
        /* loop through the channels and start the channel test */
        for (Channel = 0; Channel < IntTstConfigPtr->MonitorCount; Channel++)
        {
            /* check whether the channel error injection test is enabled */
            if (IRQ_LATENCY_ERROR == IntTstConfigPtr->Monitor[Channel].InjectError)
            {
#ifdef INTTST_ENABLE_USER_MODE_SUPPORT
                /* set the  INTM Channel interrupt latency and enable monitor */
                OsIf_Trusted_Call4params((IntTst_Lld_Intm_ChannelConfig), INTM, IntTstConfigPtr->Monitor[Channel].Channel, \
                                         IntTstConfigPtr->Monitor[Channel].IrqNumber, INTTST_INTM_CHANNEL_LATENCY_COUNT);
                /* enable the pTMR Channel interrupt and start the Channel for INTM latency timeout error trigger */
                OsIf_Trusted_Call3params((IntTst_Lld_Ptmr_StartChannel), INTTST_PTMR, IntTstConfigPtr->Monitor[Channel].Channel, INTTST_PTMR_CHANNEL_TIMEOUT_VALUE);
#else
                /* set the  INTM Channel interrupt latency and enable monitor */
                IntTst_Lld_Intm_ChannelConfig(INTM, IntTstConfigPtr->Monitor[Channel].Channel, \
                                              IntTstConfigPtr->Monitor[Channel].IrqNumber, INTTST_INTM_CHANNEL_LATENCY_COUNT);
                /* enable the pTMR Channel interrupt and start the Channel for INTM latency timeout error trigger */
                IntTst_Lld_Ptmr_StartChannel(INTTST_PTMR, IntTstConfigPtr->Monitor[Channel].Channel, INTTST_PTMR_CHANNEL_TIMEOUT_VALUE);
#endif /* INTTST_ENABLE_USER_MODE_SUPPORT */
                Timeout = 0U; /* reset the timeout counter */
                while (INTTST_RESULT_NOT_TESTED == IntTst_TestDetails.IntmChTestDetails[Channel].Test_Result)
                {
                    Timeout++; /* increment the timeout counter */
                    MCAL_QUA_FAULT_INJECTION(INTTST_TIMEOUT_ERROR);
                    /* check the timeout */
                    if (Timeout > INTTST_ERROR_CHECK_TIMEOUT)
                    {
                        /* update the channel test result into IntTst_TestDetails */
                        IntTst_TestDetails.IntmChTestDetails[Channel].Test_Result = INTTST_RESULT_NOT_OK;
                        IntTst_TestDetails.IntmChTestDetails[Channel].InjectError = IRQ_LATENCY_ERROR;
                        if (NULL_PTR != IntTstConfigPtr->Notification)
                        {
                            /* call the user notification function */
                            IntTstConfigPtr->Notification(INTTST_MODULE_ID);
                        }
                        /* if any one channel test failed, then the module test failed(NOT_OK) */
                        IntTst_Result = INTTST_RESULT_NOT_OK;
                        break;
                    }
                    else
                    {
                        /* do nothing */
                    }
                }
#ifdef INTTST_ENABLE_USER_MODE_SUPPORT
                /* stop the pTMR Channel for INTM latency timeout error trigger */
                OsIf_Trusted_Call2params((IntTst_Lld_Ptmr_StopChannel), INTTST_PTMR, (IntTstConfigPtr->Monitor[Channel].Channel));
                /* reset the  INTM Channel interrupt latency and disable the monitor */
                OsIf_Trusted_Call2params((IntTst_Lld_Intm_ResetChannelConfig), INTM, IntTstConfigPtr->Monitor[Channel].Channel);
#else
                /* stop the pTMR Channel for INTM latency timeout error trigger */
                IntTst_Lld_Ptmr_StopChannel(INTTST_PTMR, IntTstConfigPtr->Monitor[Channel].Channel);
                /* reset the  INTM Channel interrupt latency and disable the monitor */
                IntTst_Lld_Intm_ResetChannelConfig(INTM, IntTstConfigPtr->Monitor[Channel].Channel);
#endif /* INTTST_ENABLE_USER_MODE_SUPPORT */
            }
            else
            {
                /* do nothing */
            }
        }
        /* finally, check and update the module test result */
        if (INTTST_RESULT_NOT_OK != IntTst_Result)
        {
            IntTst_Result = INTTST_RESULT_OK;
        }
        else
        {
            /* do nothing */
        }
        /* set the IntTst state to INTTST_TESTED */
        IntTst_State = INTTST_TESTED;
    }
}

/**
 * @brief     Service to get the result of the IntTst execution.
 * @details   It returns the result of the IntTst execution with the IntTst module
 *            state check, and only if the IntTst module state is INTTST_TESTED or INTTST_TESTING,
 *            it will return the actual result of the IntTst execution.
 *
 * @table          @service_id:       0x04  \n
 *                 @is_reentrant:     false \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 *
 * @param     void
 * @return    IntTst_ResultType
 * @retval    INTTST_RESULT_NOT_TESTED - Test is not executed.
 * @retval    INTTST_RESULT_OK - All the test case of injection are passed.
 * @retval    INTTST_RESULT_NOT_OK - At least one test case of error injection is failed.
 * @sw_type   sw_detail
 *
 * @trace YTSDS_IntTst_670
 */
INTTST_FUNC IntTst_ResultType IntTst_GetResult(void)
{
    IntTst_ResultType TestResult = INTTST_RESULT_NOT_TESTED;
    if (INTTST_UNINIT == IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID, \
                              INTTST_SID_GET_RESULT, INTTST_E_UNINIT);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (INTTST_INIT == IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID, \
                              INTTST_SID_GET_RESULT, INTTST_E_INVALID_STATE);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        TestResult = IntTst_Result;
    }
    /* retrun the test result */
    return TestResult;
}

/**
 * @brief          Service for get the test details of the IntTst execution
 * @details        This function is used to get the test details of the IntTst execution.
 *
 * @table          @service_id:       0x06  \n
 *                 @is_reentrant:     false \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 *
 * @param[inout]   IntTstDetails Pointer to where to store the test details.
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_IntTst_669
 */
INTTST_FUNC void IntTst_GetDetailsResult(IntTst_TestDetailsType *IntTstDetails)
{
    uint8 Channel = 0U;
    if (INTTST_UNINIT == IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID, \
                              INTTST_SID_GET_DETAILS_RESULT, INTTST_E_UNINIT);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (INTTST_INIT == IntTst_State)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID, \
                              INTTST_SID_GET_DETAILS_RESULT, INTTST_E_INVALID_STATE);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (NULL_PTR == IntTstDetails) /* check the parameter pointer */
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError((uint16)INTTST_MODULE_ID, INTTST_INSTANCE_ID,
                              INTTST_SID_GET_DETAILS_RESULT, INTTST_E_PARAM_POINTER);
#endif /* #if (INTTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        /* copy the test details to the user buffer */
        IntTstDetails->IntmChCount = IntTst_TestDetails.IntmChCount;
        for (Channel = 0; Channel < IntTst_TestDetails.IntmChCount; Channel++)
        {
            /* copy the channel test details */
            IntTstDetails->IntmChTestDetails[Channel].Channel = IntTst_TestDetails.IntmChTestDetails[Channel].Channel;
            IntTstDetails->IntmChTestDetails[Channel].InjectError = IntTst_TestDetails.IntmChTestDetails[Channel].InjectError;
            IntTstDetails->IntmChTestDetails[Channel].Test_Result = IntTst_TestDetails.IntmChTestDetails[Channel].Test_Result;
        }
    }
}

#if (INTTST_VERSION_INFO_API == STD_ON)
/**
 * @brief          Service for get the version information of this module
 * @details        This function is used to get the version information of this module.
 *
 * @table          @service_id:       0x06  \n
 *                 @is_reentrant:     true  \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 * @param[out]     versioninfo Pointer to where to store version information of this module
 * @return         void
 * @sw_type        sw_detail
 *
 * @trace YTSDS_IntTst_671
 */
INTTST_FUNC void IntTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    /* Check for DET: INTTST_E_PARAM_POINTER */
    if ((VersionInfo) == NULL_PTR)
    {
#if (INTTST_DEV_ERROR_DETECT == STD_ON)
        /* Report INTTST_E_PARAM_POINTER DET if service called with NULL_PTR */
        (void)Det_ReportError(
            (uint16)INTTST_MODULE_ID,         \
            INTTST_INSTANCE_ID,               \
            INTTST_SID_GET_VERSION_INFO,      \
            INTTST_E_PARAM_POINTER);
#endif /* INTTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        /* Vendor ID information */
        VersionInfo->vendorID = INTTST_VENDOR_ID;
        /* Adc module ID information */
        VersionInfo->moduleID = INTTST_MODULE_ID;
        /* Adc module Software major version information */
        VersionInfo->sw_major_version = (uint8)INTTST_SW_MAJOR_VERSION;
        /* Adc module Software minor version information */
        VersionInfo->sw_minor_version = (uint8)INTTST_SW_MINOR_VERSION;
        /* Adc module Software patch version information */
        VersionInfo->sw_patch_version = (uint8)INTTST_SW_PATCH_VERSION;
    }
}
#endif /* INTTST_VERSION_INFO_API == STD_ON */

#define INTTST_STOP_SEC_CODE
#include "IntTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file IntTst.c */
