/*
* @file    IntTst_Types.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : IntTst_Type
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef INTTST_TYPES_H
#define INTTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/

#include "Std_Types.h"
#include "IntCtrl_Lld_TypesDef.h"
#include "YTM32B1ME0.h"

/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define INTTST_TYPES_VENDOR_ID                      (180)
#define INTTST_TYPES_AR_RELEASE_MAJOR_VERSION       (4)
#define INTTST_TYPES_AR_RELEASE_MINOR_VERSION       (4)
#define INTTST_TYPES_AR_RELEASE_REVISION_VERSION    (0)
#define INTTST_TYPES_SW_MAJOR_VERSION               (2)
#define INTTST_TYPES_SW_MINOR_VERSION               (0)
#define INTTST_TYPES_SW_PATCH_VERSION               (0)
/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/

/* if the devices is YTM32B1HA01, redefine the pTMR0 as INTTST_PTMR */
#if defined(YTM32B1HA0)
/* remap the pTMR0 to INTTST_PTMR for the IntTst */
#define pTMR_Ch0_IRQn   pTMR0_Ch0_IRQn  /*!< use pTMR0_Ch0_IRQn for the INTM channel 0 test */
#define pTMR_Ch1_IRQn   pTMR0_Ch1_IRQn  /*!< use pTMR0_Ch1_IRQn for the INTM channel 1 test */
#define pTMR_Ch2_IRQn   pTMR0_Ch2_IRQn  /*!< use pTMR0_Ch2_IRQn for the INTM channel 2 test */
#define pTMR_Ch3_IRQn   pTMR0_Ch3_IRQn  /*!< use pTMR0_Ch3_IRQn for the INTM channel 3 test */
#endif

/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/
/*!
 * @brief          Enumerated of the INTM hardware channel
 * @sw_type        sw_arch
 */
typedef enum
{
    INTM_CHANNEL_0           =  0U,     /*!< INTM HW channel 0 */
    INTM_CHANNEL_1           =  1U,     /*!< INTM HW channel 1 */
    INTM_CHANNEL_2           =  2U,     /*!< INTM HW channel 2 */
    INTM_CHANNEL_3           =  3U      /*!< INTM HW channel 3 */
} Intm_HwChannelType;
/**
 * @brief          Enumerated type of the test error injection.
 * @sw_type        sw_arch
 */
typedef enum
{
    NONE_ERROR = 0U,                      /**< INTM none error */
    IRQ_LATENCY_ERROR  = 1U,              /**< INTM IRQ latency timeout error */
} Intm_InjectErrorType;

/**
 * @brief          Enumerated type of the IntTst module work state.
 * @sw_type        sw_detail
 *
 * @trace YTSDS_IntTst_667
 */
typedef enum
{
    INTTST_UNINIT           = 0x00U,  /**< The IntTst module is uninitialized */
    INTTST_INIT             = 0x01U,  /**< The IntTst module is initialized */
    INTTST_TESTING          = 0x02U,  /**< The IntTst module is testing */
    INTTST_TESTED           = 0x03U   /**< The IntTst module is tested, all the required tests are tested */
} IntTst_StateType;

/**
 * @brief          Enumerated result type of the IntTst module execution and INTM monitor channel test result.
 * @sw_type        sw_arch
 */
typedef enum
{
    INTTST_RESULT_NOT_TESTED    = 0x00U,  /**< Test not executed */
    INTTST_RESULT_OK            = 0x01U,  /**< Test passed */
    INTTST_RESULT_NOT_OK        = 0x02U   /**< Test failed */
} IntTst_ResultType;

/**
 * @brief          structure type of a IntTst INTM channel test details.
 * @sw_type        sw_arch
 */
typedef struct
{
    uint8                       Channel;         /**< INTM channel number */
    Intm_InjectErrorType        InjectError;     /**< INTM channel inject error */
    volatile IntTst_ResultType  Test_Result;     /**< INTM channel test result */
} IntTst_ChannelTestType;

/**
 * @brief          Enumerated result type of the test details.
 * @sw_type        sw_arch
 */
typedef struct
{
    uint8                   IntmChCount;        /**< INTM channel number for the test */
    IntTst_ChannelTestType  *IntmChTestDetails; /**< INTM channel test details */
} IntTst_TestDetailsType;

/**
 * @brief          structure type of the IntTst monitor configuration.
 * @sw_type        sw_arch
 */
typedef struct
{
    uint8                Channel;     /**< INTM channel number for the test */
    IRQn_Type            IrqNumber;   /**< INTM channel monitor peripheral IRQ interrupt number */
    Intm_InjectErrorType InjectError; /**< INTM channel inject error */
} IntTst_Monitor_ConfigType;


/**
 * @brief           IntTst user notification function prototype
 * @sw_type         sw_arch
 */
typedef void (*IntTst_NotificationType)(uint16  ModuleId);

/**
 * @brief           IntTst configuration structure
 * @sw_type         sw_arch
 */
typedef struct
{
    const uint8                     MonitorCount;  /**< INTM channel monitor number for the test */
    const IntTst_Monitor_ConfigType *Monitor;      /**< Structure pointer to the INTM channel monitor configuration */
    const IntTst_NotificationType   Notification;  /**< INTM channel monitor notification */
} IntTst_ConfigType;

/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
#ifdef __cplusplus
}
#endif

#endif /* End of file IntTst_Types.h */

