/**
 * @file    FlsTst.h
 * @version
 *
 * @brief   AUTOSAR FlsTst module interface
 * @details API implementation for FLSTST driver
 *
 * @addtogroup FLSTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Adc
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifndef FLSTST_H
#define FLSTST_H

#ifdef __cplusplus
extern "C" {
#endif


/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "FlsTst_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define FLSTST_VENDOR_ID                      (0xB4)
#define FLSTST_AR_RELEASE_MAJOR_VERSION       (4)
#define FLSTST_AR_RELEASE_MINOR_VERSION       (4)
#define FLSTST_AR_RELEASE_REVISION_VERSION    (0)
#define FLSTST_SW_MAJOR_VERSION               (2)
#define FLSTST_SW_MINOR_VERSION               (0)
#define FLSTST_SW_PATCH_VERSION               (0)


/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and FLSTST types define header file are of the same vendor */
#if (FLSTST_VENDOR_ID != FLSTST_TYPES_VENDOR_ID)
#error "FlsTst.h and FlsTst_Types.h have different vendor ids"
#endif

/* Check if source file and FLSTST types define header file are of the same Autosar version */
#if ((FLSTST_AR_RELEASE_MAJOR_VERSION != FLSTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (FLSTST_AR_RELEASE_MINOR_VERSION != FLSTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (FLSTST_AR_RELEASE_REVISION_VERSION != FLSTST_TYPES_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of FlsTst.h and FlsTst_Types.h are different"
#endif

/* Check if source file and FLSTST types define header file are of the same Software version */
#if ((FLSTST_SW_MAJOR_VERSION != FLSTST_TYPES_SW_MAJOR_VERSION) || \
     (FLSTST_SW_MINOR_VERSION != FLSTST_TYPES_SW_MINOR_VERSION) || \
     (FLSTST_SW_PATCH_VERSION != FLSTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of FlsTst.h and FlsTst_Types.h are different"
#endif


/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
 *                                             ENUMS
==================================================================================================*/

/**
 * @brief            This function shall abort Flash test background operation and set the state to FLSTST_ABORTED.
 *
 * @table            @service_id:       0x03  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           None
 * @sw_type sw_arch
 */
void FlsTst_Abort(void);

/**
 * @brief            The function shall de-initialize all Flash Test relevant registers and global variables that were initialized by FlsTst_Init.
 *
 * @table            @service_id:       0x01  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           None
 * @sw_type sw_arch
 */
void FlsTst_DeInit(void);

#if (FLSTST_GET_CURRENT_STATE_API == STD_ON)
/**
 * @brief            The function shall return the current Flash Test execution state.
 *
 * @table            @service_id:       0x06  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           FlsTst_StateType
 * @retval           FLSTST_UNINIT The Flash Test is not initialized or not usable.
 * @retval           FLSTST_INIT The Flash Test is initialized and ready to be started.
 * @retval           FLSTST_RUNNING The Flash Test is currently running.
 * @retval           FLSTST_ABORTED The Flash Test is aborted.
 * @retval           FLSTST_SUSPENDED The Flash Test is waiting to be resumed or is waiting to start forground mode test
 * @sw_type sw_arch
 */
FlsTst_StateType FlsTst_GetCurrentState(void);
#endif  /* (FLSTST_GET_CURRENT_STATE_API == STD_ON) */

#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
/**
 * @brief            The function shall return the error details monitored from the Flash Test driver.
 *
 * @table            @service_id:       0x0B  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           FlsTst_ErrorDetailsType
 * @sw_type sw_arch
 */
FlsTst_ErrorDetailsType FlsTst_GetErrorDetails(void);
#endif  /* (FLSTST_GET_ERROR_DETAILS_API == STD_ON) */

#if (FLSTST_GET_TEST_RESULT_BGND_API == STD_ON)
/**
 * @brief            The function shall return the Flash test result and Test Interval Id of the last background test.
 *
 * @table            @service_id:       0x07  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           FlsTst_TestResultBgndType
 * @sw_type sw_arch
 */
FlsTst_TestResultBgndType FlsTst_GetTestResultBgnd(void);
#endif  /* (FLSTST_GET_TEST_RESULT_BGND_API == STD_ON) */

#if (FLSTST_GET_TEST_RESULT_FGND_API == STD_ON)
/**
 * @brief            The function shall return the Flash test result of the last foreground test.
 *
 * @table            @service_id:       0x0F  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           FlsTst_TestResultFgndType
 * @sw_type sw_arch
 */
FlsTst_TestResultFgndType FlsTst_GetTestResultFgnd(void);
#endif  /* (FLSTST_GET_TEST_RESULT_FGND_API == STD_ON) */

#if (FLSTST_GET_TEST_SIGNATURE_BGND_API == STD_ON)
/**
 * @brief            The function shall return the signature and Test Interval Id of the last background test.
 *
 * @table            @service_id:       0x09  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           FlsTst_TestSignatureBgndType
 * @sw_type sw_arch
 */
FlsTst_TestSignatureBgndType FlsTst_GetTestSignatureBgnd(void);
#endif  /* (FLSTST_GET_TEST_SIGNATURE_BGND_API == STD_ON) */

#if (FLSTST_GET_TEST_SIGNATURE_FGND_API == STD_ON)
/**
 * @brief            The function shall return the signature of the last foreground test.
 *
 * @table            @service_id:       0x0A  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           FlsTst_TestSignatureFgndType
 * @sw_type sw_arch
 */
FlsTst_TestSignatureFgndType FlsTst_GetTestSignatureFgnd(void);
#endif  /* (FLSTST_GET_TEST_SIGNATURE_FGND_API == STD_ON) */

/**
 * @brief            The function shall initialize all Flash Test relevant registers and global variables
 *                   and change the execution state to FLSTST_INIT.
 *
 * @table            @service_id:       0x00  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @param[in]        ConfigPtr Pointer to configuration set
 * @return           None
 * @sw_type sw_arch
 */
void FlsTst_Init(const FlsTst_ConfigType *ConfigPtr);

/**
 * @brief            The function shall test the defined flash blocks in background mode, starting with
 *                    the first flash block in the FlsTstConfigParams.
 *
 * @table            @service_id:       0x0D  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   --    \n
                     @autosar_api:      true  \n
 *
 * @return           None
 * @sw_type sw_arch
 */
void FlsTst_MainFunction(void);

#if (FLSTST_SUSPEND_RESUME_API == STD_ON)
/**
 * @brief            The function shall change the execution state to FLSTST_RUNNING when commanded to
 *                   continue and the current execution state is FLSTST_SUSPENDED.
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           None
 * @sw_type sw_arch
 */
void FlsTst_Resume(void);
#endif  /* (FLSTST_SUSPEND_RESUME_API == STD_ON) */

#if (FLSTST_START_FGND_API == STD_ON)
/**
 * @brief            The function is only applicable for Foreground mode Flash Test operation.
 * @param[in]        FgndBlockId Number of the foreground test to be executed. This is dependent on configuration.
 *
 * @table            @service_id:       0x02  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           Std_ReturnType
 * @retval E_OK     Foreground Flash Test is started successfully
 * @retval E_NOT_OK Foreground Flash Test is not started
 * @sw_type sw_arch
 */
Std_ReturnType FlsTst_StartFgnd(FlsTst_BlockIdFgndType FgndBlockId);
#endif  /* (FLSTST_START_FGND_API == STD_ON) */

#if (FLSTST_SUSPEND_RESUME_API == STD_ON)
/**
 * @brief            The function is only applicable for Background mode Flash Test operation.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           None
 * @sw_type sw_arch
 */
void FlsTst_Suspend(void);
#endif  /* (FLSTST_SUSPEND_RESUME_API == STD_ON) */

#if (FLSTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @param[out]       VersionInfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x08  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
 *
 * @return           None
 * @sw_type sw_arch
 */
void FlsTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif /* (FLSTST_VERSION_INFO_API == STD_ON) */

#ifdef __cplusplus
}
#endif

#endif /* FLSTST_H */

/** @} */

