/**
* @file    EfmTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : EfmTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 3408 Rule-8.4: A compatible declaration shall be visible when an object or function
 *                   with external linkage is defined.
 *
 */

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "EfmTst.h"
#include "EfmTst_Lld.h"
#include "Fls.h"
#if defined (EFMTST_ENABLE_USER_MODE_SUPPORT)
#include "OsIf.h"
#endif /* EFMTST_ENABLE_USER_MODE_SUPPORT */
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define EFMTST_VENDOR_ID_C                      (0xB4)
#define EFMTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define EFMTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define EFMTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define EFMTST_SW_MAJOR_VERSION_C               (2)
#define EFMTST_SW_MINOR_VERSION_C               (0)
#define EFMTST_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and EFMTST header file are of the same vendor */
#if (EFMTST_VENDOR_ID_C != EFMTST_VENDOR_ID)
#error "EfmTst.c and EfmTst.h have different vendor ids"
#endif

/* Check if source file and EFMTST header file are of the same Autosar version */
#if ((EFMTST_AR_RELEASE_MAJOR_VERSION_C != EFMTST_AR_RELEASE_MAJOR_VERSION) || \
     (EFMTST_AR_RELEASE_MINOR_VERSION_C != EFMTST_AR_RELEASE_MINOR_VERSION) || \
     (EFMTST_AR_RELEASE_REVISION_VERSION_C != EFMTST_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of EfmTst.c and EfmTst.h are different"
#endif

/* Check if source file and EFMTST header file are of the same Software version */
#if ((EFMTST_SW_MAJOR_VERSION_C != EFMTST_SW_MAJOR_VERSION) || \
     (EFMTST_SW_MINOR_VERSION_C != EFMTST_SW_MINOR_VERSION) || \
     (EFMTST_SW_PATCH_VERSION_C != EFMTST_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EfmTst.c and EfmTst.h are different"
#endif

/* Check if source file and EFMTST_LLD header file are of the same vendor */
#if (EFMTST_VENDOR_ID_C != EFMTST_LLD_VENDOR_ID)
#error "EfmTst.c and EfmTst_Lld.h have different vendor ids"
#endif

/* Check if source file and EFMTST_LLD header file are of the same Autosar version */
#if (( EFMTST_AR_RELEASE_MAJOR_VERSION_C != EFMTST_LLD_AR_RELEASE_MAJOR_VERSION) || \
      ( EFMTST_AR_RELEASE_MINOR_VERSION_C != EFMTST_LLD_AR_RELEASE_MINOR_VERSION) || \
      ( EFMTST_AR_RELEASE_REVISION_VERSION_C != EFMTST_LLD_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of EfmTst.c and EfmTst_Lld.h are different"
#endif

/* Check if source file and EFMTST_LLD header file are of the same Software version */
#if (( EFMTST_SW_MAJOR_VERSION_C != EFMTST_LLD_SW_MAJOR_VERSION) || \
      ( EFMTST_SW_MINOR_VERSION_C != EFMTST_LLD_SW_MINOR_VERSION) || \
      ( EFMTST_SW_PATCH_VERSION_C != EFMTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of EfmTst.c and EfmTst_Lld.h are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/

/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/
#define EFMTST_START_SEC_CONST_MEMORY_FOR_VALIDATION
#include "EfmTst_MemMap.h"
/* MR12 RULE 8.4 VIOLATION: This constant is used to be injected into the test sector during memory allocation.
 * It is not actually used in the code. If the static keyword is added for restriction, it will cause compilation errors.
 * thus, couldn't adhere to M3CM Rule-8.4.
*/
/**
 * @brief Constant array for Flash EFM test sector data validation.
 * @sw_type sw_detail
 * @private
 */
EFMTST_CONST const EfmTst_SectorDataInjectionType EfmTst_SectorDataInjection[8] =  /*PRQA S 3408*/
{
    0xffU, 0xffU, 0x3fU, 0x4fU,
    0x5aU, 0x5aU, 0xa5U, 0xa5U
};

#define EFMTST_STOP_SEC_CONST_MEMORY_FOR_VALIDATION
/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
#if (EFMTST_PRECOMPILE_SUPPORT == STD_ON)
/**
 * @brief   EfmTst driver pre-compiled configuration data structure
 * @private
*/
extern const EfmTst_ConfigType EfmTst_PreCompileConfig;
#endif /* #if (EFMTST_PRECOMPILE_SUPPORT == STD_ON) */

#define EFMTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "EfmTst_MemMap.h"
/* MR12 RULE 8.4 VIOLATION: The variable will used in integration test and EfmTst_Lld.c file,
* thus, couldn't adhere to M3CM Rule-8.4.
*/
/**
 * @brief The pointer to the configuration structure of the EFM test.
 * @sw_type sw_detail
 * @private
 */
EFMTST_VAR const EfmTst_ConfigType *EfmTst_ConfigPtr = NULL_PTR;  /*PRQA S 3408 */
/**
 * @brief The test detail result of the Efm test.
 * @sw_type sw_detail
 * @private
 */
#if ((EFMTST_ENABLE_WRITE_ENABLE_TEST==STD_ON)&&(EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON))
EFMTST_VAR static EfmTst_TestDetailsType EfmTst_TestResultDetail = {EFMTST_NOT_TESTED, EFMTST_NOT_TESTED, EFMTST_NOT_TESTED};
#elif ((EFMTST_ENABLE_WRITE_ENABLE_TEST==STD_OFF)&&(EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_OFF))
EFMTST_VAR static EfmTst_TestDetailsType EfmTst_TestResultDetail = {EFMTST_NOT_TESTED};
#else
EFMTST_VAR static EfmTst_TestDetailsType EfmTst_TestResultDetail = {EFMTST_NOT_TESTED, EFMTST_NOT_TESTED};
#endif /* #if ((EFMTST_ENABLE_WRITE_ENABLE_TEST==STD_ON)&&(EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON)) */
#define EFMTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "EfmTst_MemMap.h"

#define EFMTST_START_SEC_VAR_INIT_8
#include "EfmTst_MemMap.h"
/**
 * @brief   The EfmTst module running state
 * @private
*/
EFMTST_VAR static EfmTst_StateType EfmTst_Status = EFMTST_UNINIT;
/**
 * @brief The test results of the Efm test.
 * @sw_type sw_detail
 * @private
 */
EFMTST_VAR static EfmTst_TestResultType EfmTst_TestResult = EFMTST_NOT_TESTED;
#define EFMTST_STOP_SEC_VAR_INIT_8
#include "EfmTst_MemMap.h"



/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/


/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/


/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
#define EFMTST_START_SEC_CODE
#include "EfmTst_MemMap.h"
/**
 * @brief  Service for checking sector status.
 * @details This function checks whether the EFM test sector is empty.
 * @param[in] StartAddress The starting address of the EFM test sector.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK: Sector is empty.
 * @retval E_NOT_OK: Sector is not empty.
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_435
 */
EFMTST_FUNC static Std_ReturnType EfmTst_CheckSectorEmpty(uint32 StartAddress)
{
    Std_ReturnType TempRet = E_OK;
    uint32 LoopCnt = 0;
    uint8 SectorData = 0U;
    while ((LoopCnt < EFMTST_TEST_SECTOR_MAX_LENGTH) && (E_OK == TempRet))
    {
        /* Obtain the sector address data */
        SectorData = *(volatile uint8 *)(StartAddress + LoopCnt);
        /* Determine whether the sector address data is 0xff */
        if (0xffU != SectorData)
        {
            TempRet = E_NOT_OK;
        }
        ++LoopCnt;
    }
    return TempRet;
}

/**
 * @brief Service for starting the Write Protection test.
 * @details The function is designed to perform a write-protection test on a protected flash sector.
 *
 * @table            @service_id:       --  \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      false  \n
 *                   @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_747
 */
EFMTST_FUNC static Std_ReturnType EfmTst_WriteProtection_Start(void)
{
    Std_ReturnType TempReturn = E_NOT_OK;
    Std_ReturnType TestReturn = E_NOT_OK;
    boolean EraseTimeout = FALSE;
    uint32 Timeout = 0U;
    /* Run efm test */
    /*protected flash sector test*/
    (void)Fls_Erase(EfmTst_ConfigPtr->ProtectSectorPtr->StartAddr, EfmTst_ConfigPtr->ProtectSectorPtr->Length);
    do
    {
        Fls_MainFunction();
        /* Check whether the erasing operation times out */
        if (Timeout > EFMTST_ERROR_CHECK_TIMEOUT)
        {
            EraseTimeout = TRUE;
            break;
        }
        ++Timeout;
    } while (Fls_GetStatus() != MEMIF_IDLE);
    /* The erasing operation times out */
    if (TRUE == EraseTimeout)
    {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
        if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
        {
            EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
        }
#endif
    }
    else
    {
        /* Check whether the test sector is empty */
        TempReturn = EfmTst_CheckSectorEmpty(EfmTst_ConfigPtr->ProtectSectorPtr->HwStartAddr);
        /* The erasing operation is successful and the test sector is empty (confirm that the data has been erased) */
        if ((Fls_GetJobResult() == MEMIF_JOB_OK) && (E_OK == TempReturn))
        {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
            if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
            {
                EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
            }
#endif
        }
        else
        {
            TestReturn = E_OK;
        }
    }
    return TestReturn;
}

/*==================================================================================================
 *                                       SafMgr FUNCTIONS
==================================================================================================*/

/**
 * @brief  Service for EFM Test initialization.
 * @details The function shall initialize all EFM Test relevant registers and global variables.
 * @param[in] ConfigPtr Pointer to the EFM Test configuration data structure.
 *
 * @table            @service_id:       0x01  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_439
 */
EFMTST_FUNC void EfmTst_Init(EfmTst_ConfigType const *ConfigPtr)
{
    if (NULL_PTR != EfmTst_ConfigPtr)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID,
                              EFMTST_SID_INIT, EFMTST_E_ALREADY_INITIALIZED);
#endif /* (EFMTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
#if (EFMTST_PRECOMPILE_SUPPORT == STD_ON)
        if (NULL_PTR != ConfigPtr)
#else /* (EFMTST_PRECOMPILE_SUPPORT == STD_OFF) */
        if (NULL_PTR == ConfigPtr)
#endif /* (EFMTST_PRECOMPILE_SUPPORT == STD_ON) */
        {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
            /* Invalid pointer */
            (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID,
                                  EFMTST_SID_INIT, EFMTST_E_PARAM_POINTER);
#endif /* (EFMTST_DEV_ERROR_DETECT == STD_ON) */
        }
        else
        {
#if (EFMTST_PRECOMPILE_SUPPORT == STD_ON)
            EfmTst_ConfigPtr = &EfmTst_PreCompileConfig;
#else /* (EFMTST_PRECOMPILE_SUPPORT == STD_OFF) */
            EfmTst_ConfigPtr = ConfigPtr;
#endif /* (EFMTST_PRECOMPILE_SUPPORT == STD_ON) */
            EfmTst_TestResultDetail.ProtectSectorResult = EFMTST_NOT_TESTED;
#if (EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON)
            EfmTst_TestResultDetail.CommandUnlockResult = EFMTST_NOT_TESTED;
#endif
#if (EFMTST_ENABLE_WRITE_ENABLE_TEST==STD_ON)
            EfmTst_TestResultDetail.WirteEnableResult = EFMTST_NOT_TESTED;
#endif
            EfmTst_TestResult = EFMTST_NOT_TESTED;
            /*Init the protect register in EFM*/
#if defined (EFMTST_ENABLE_USER_MODE_SUPPORT)
            OsIf_Trusted_Call1param(EfmTst_Lld_Init, EfmTst_ConfigPtr->SectorProtCfgPtr);
#else
            EfmTst_Lld_Init(EfmTst_ConfigPtr->SectorProtCfgPtr);
#endif /* #if defined (EFMTST_ENABLE_USER_MODE_SUPPORT) */
            EfmTst_Status = EFMTST_INIT;
        }
    }
}


/**
 * @brief  Service for executing EFM Test.
 * @details The function executes all test items based on the configuration of the EFMTst module.
 *
 * @table            @service_id:       0x03  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_541
 */
EFMTST_FUNC void EfmTst_Start(void)
{
    Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
    /*Init or not check*/
    if (EFMTST_UNINIT == EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_START, EFMTST_E_UNINIT);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (EFMTST_TESTED == EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_START, EFMTST_E_INVALID_STATE);
#endif /* #if (EFMTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (EFMTST_TESTING == EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_START, EFMTST_E_INVALID_STATE);
#endif /* #if (EFMTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        EfmTst_Status = EFMTST_TESTING;
#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
#if defined (EFMTST_ENABLE_USER_MODE_SUPPORT)
        RetVal = (Std_ReturnType)OsIf_Trusted_Call_Return(EfmTst_Lld_WriteEnable_Start);
#else
        RetVal = EfmTst_Lld_WriteEnable_Start();
#endif /* #if defined (EFMTST_ENABLE_USER_MODE_SUPPORT) */
        if (E_OK == RetVal)
        {
            EfmTst_TestResultDetail.WirteEnableResult = EFMTST_OK;
        }
        else
        {
            EfmTst_TestResultDetail.WirteEnableResult = EFMTST_NOT_OK;
            EfmTst_TestResult = EFMTST_NOT_OK;
        }
#endif /* #if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)*/
#if (EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON)
#if defined (EFMTST_ENABLE_USER_MODE_SUPPORT)
        RetVal = (Std_ReturnType)OsIf_Trusted_Call_Return(EfmTst_Lld_CommandUnlock_Start);
#else
        RetVal = EfmTst_Lld_CommandUnlock_Start();
#endif /* #if defined (EFMTST_ENABLE_USER_MODE_SUPPORT) */
        if (E_OK == RetVal)
        {
            EfmTst_TestResultDetail.CommandUnlockResult = EFMTST_OK;
        }
        else
        {
            EfmTst_TestResultDetail.CommandUnlockResult = EFMTST_NOT_OK;
            EfmTst_TestResult = EFMTST_NOT_OK;
        }
#endif /* #if (EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON)*/
        RetVal = EfmTst_WriteProtection_Start();
        if (E_OK == RetVal)
        {
            EfmTst_TestResultDetail.ProtectSectorResult = EFMTST_OK;
        }
        else
        {
            EfmTst_TestResultDetail.ProtectSectorResult = EFMTST_NOT_OK;
            EfmTst_TestResult = EFMTST_NOT_OK;
        }
        /* Check the test result */
        if (EfmTst_TestResult != EFMTST_NOT_OK)
        {
            EfmTst_TestResult = EFMTST_OK;
        }
        EfmTst_Status = EFMTST_TESTED;
    }
}

/**
 * @brief  Service for getting EFM Test result.
 * @details The function returns the result of the EFM Test.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return EfmTst_TestResultType
 * @retval EFMTST_NOT_TESTED:The EFM Test is not completed.
 * @retval EFMTST_OK: The EFM Test test result is OK.
 * @retval EFMTST_NOT_OK: The EFM Test result is not OK.
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_707
 */
EFMTST_FUNC EfmTst_TestResultType EfmTst_GetResult(void)
{
    /* Local variable to hold the return value of the function */
    EfmTst_TestResultType RetValue = EFMTST_NOT_TESTED;
    if (EFMTST_UNINIT == EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_GET_RESULT, EFMTST_E_UNINIT);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (EFMTST_TESTED != EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_GET_RESULT, EFMTST_E_INVALID_STATE);
#endif /* #if (EFMTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        /* Get the test result */
        RetValue = EfmTst_TestResult;
    }
    return RetValue;
}

/**
 * @brief Service for EFM Test De-Initialization.
 * @details The function de-initializes all EFM Test relevant registers and global variables.
 *
 * @table            @service_id:       0x02    \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_436
 */
EFMTST_FUNC void EfmTst_DeInit(void)
{
    /* Check if the module is initialized */
    if (EFMTST_UNINIT == EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_DEINIT, EFMTST_E_UNINIT);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (EFMTST_TESTING == EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_DEINIT, EFMTST_E_INVALID_STATE);
#endif /* #if (EFMTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        EfmTst_ConfigPtr = NULL_PTR;
        EfmTst_TestResultDetail.ProtectSectorResult = EFMTST_NOT_TESTED;
#if (EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON)
        EfmTst_TestResultDetail.CommandUnlockResult = EFMTST_NOT_TESTED;
#endif /* #if (EFMTST_ENABLE_CMD_UNLOCK_TEST == STD_ON)*/
#if (EFMTST_ENABLE_WRITE_ENABLE_TEST==STD_ON)
        EfmTst_TestResultDetail.WirteEnableResult = EFMTST_NOT_TESTED;
#endif /* #if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)*/
        EfmTst_TestResult = EFMTST_NOT_TESTED;
        EfmTst_Status = EFMTST_UNINIT;
    }
}

#if (EFMTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @param[out]       VersionInfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x06  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTDT_EfmTst_630
 */
EFMTST_FUNC void EfmTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    if (NULL_PTR == VersionInfo)
    {
        /* Report ECCTST_E_PARAM_POINTER DET if service called with NULL_PTR */
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_GET_VERSION_INFO_SID, EFMTST_E_PARAM_POINTER);
#endif /* #if (ECCTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /* Vendor ID information */
        VersionInfo->vendorID = EFMTST_VENDOR_ID;
        /* EfmTst module ID information */
        VersionInfo->moduleID = EFMTST_MODULE_ID;
        /* EfmTst module Software major version information */
        VersionInfo->sw_major_version = (uint8)EFMTST_SW_MAJOR_VERSION;
        /* EfmTst module Software minor version information */
        VersionInfo->sw_minor_version = (uint8)EFMTST_SW_MINOR_VERSION;
        /* EfmTst module Software patch version information */
        VersionInfo->sw_patch_version = (uint8)EFMTST_SW_PATCH_VERSION;
    }
}
#endif /* #if (EFMTST_VERSION_INFO_API == STD_ON) */

/**
 * @brief Service for getting EFM Test result details.
 * @details The function returns the result details of the EFM Test driver.
 * @param[out] TestDetails  Pointer to the structure of EFMTst result detail data.
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_708
 */
EFMTST_FUNC void EfmTst_GetDetailsResult(EfmTst_TestDetailsType *TestDetails)
{
    if (EFMTST_UNINIT == EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_GET_ERROR_DETAIL, EFMTST_E_UNINIT);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (NULL_PTR == TestDetails)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_GET_ERROR_DETAIL, EFMTST_E_PARAM_POINTER);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (EFMTST_TESTED != EfmTst_Status)
    {
#if (EFMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(EFMTST_MODULE_ID, EFMTST_INSTANCE_ID, EFMTST_SID_GET_ERROR_DETAIL, EFMTST_E_INVALID_STATE);
#endif /* #if (EFMTST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        TestDetails->ProtectSectorResult = EfmTst_TestResultDetail.ProtectSectorResult;
#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
        TestDetails->WirteEnableResult = EfmTst_TestResultDetail.WirteEnableResult;
#endif
#if (EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON)
        TestDetails->CommandUnlockResult = EfmTst_TestResultDetail.CommandUnlockResult;
#endif
    }
}
#define EFMTST_STOP_SEC_CODE
#include "EfmTst_MemMap.h"
#ifdef __cplusplus
}
#endif

/* End of file EfmTst.c */

