/*
* @file    EfmTst_Types.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : EfmTst_Types
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef EFMTST_TYPES_H
#define EFMTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "Std_Types.h"
#include "EfmTst_Cfg.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define EFMTST_TYPES_VENDOR_ID                      (0xB4)
#define EFMTST_TYPES_AR_RELEASE_MAJOR_VERSION       (4)
#define EFMTST_TYPES_AR_RELEASE_MINOR_VERSION       (4)
#define EFMTST_TYPES_AR_RELEASE_REVISION_VERSION    (0)
#define EFMTST_TYPES_SW_MAJOR_VERSION               (2)
#define EFMTST_TYPES_SW_MINOR_VERSION               (0)
#define EFMTST_TYPES_SW_PATCH_VERSION               (0)
/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and EFMTST configuration header file are of the same vendor */
#if (EFMTST_TYPES_VENDOR_ID != EFMTST_VENDOR_ID_CFG)
#error "EfmTst_Types.h and EfmTst_Cfg.h have different vendor ids"
#endif

/* Check if source file and EFMTST configuration header file are of the same Autosar version */
#if ((EFMTST_TYPES_AR_RELEASE_MAJOR_VERSION != EFMTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (EFMTST_TYPES_AR_RELEASE_MINOR_VERSION != EFMTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (EFMTST_TYPES_AR_RELEASE_REVISION_VERSION != EFMTST_AR_RELEASE_REVISION_VERSION_CFG) \
    )
#error "AutoSar Version Numbers of EfmTst_Types.h and EfmTstT_Cfg.h are different"
#endif

/* Check if source file and EFMTST configuration header file are of the same Software version */
#if ((EFMTST_TYPES_SW_MAJOR_VERSION != EFMTST_SW_MAJOR_VERSION_CFG) || \
     (EFMTST_TYPES_SW_MINOR_VERSION != EFMTST_SW_MINOR_VERSION_CFG) || \
     (EFMTST_TYPES_SW_PATCH_VERSION != EFMTST_SW_PATCH_VERSION_CFG) \
    )
#error "Software Version Numbers of EfmTst_Types.h and EfmTst_Cfg.h are different"
#endif
/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/**
 * @defgroup EfmTst_ID_Information_Macro
 * @brief   The ID information list of EfmTst module
 * @sw_type sw_detail
 * @{
 *
 * @trace YTDT_EfmTst_873
 */
#define EFMTST_INSTANCE_ID                     (0U) /**< EfmTst instance ID */
#define EFMTST_MODULE_ID                       ((uint16)518U)  /**< EfmTstTst module ID */

#define EFMTST_SID_INIT                        (0x01U)  /**< API Service ID for EfmTst_Init */
#define EFMTST_SID_DEINIT                      (0x02U)  /**< API Service ID for EfmTst_DeInit */
#define EFMTST_SID_START                       (0x03U)  /**< API Service ID for EfmTst_Start */
#define EFMTST_SID_GET_RESULT                  (0x04U)  /**< API Service ID for EfmTst_GetResult */
#define EFMTST_SID_GET_ERROR_DETAIL            (0x05U)  /**< API Service ID for EfmTst_GetErrorDetail */
#define EFMTST_GET_VERSION_INFO_SID            (0x06U)  /**< API Service ID for EfmTst_GetVersionInfo */
/** @} */
/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                            ENUMS
==================================================================================================*/
/**
 * @brief Type of Flash EFM test result status.
 * @sw_type sw_arch
 *
*/
typedef enum
{
    EFMTST_NOT_TESTED                        = 0x00U, /**< The test is not running,no result to be read */
    EFMTST_OK                                = 0x01U, /**< The test result is OK */
    EFMTST_NOT_OK                            = 0x02U, /**< The test result is not OK */
} EfmTst_TestResultType;

/**
 * @brief  Type of Flash EFM test execution status.
 * @sw_type sw_arch
 *
*/
typedef enum
{
    EFMTST_UNINIT                            = 0x00, /**< The EFM test is uninitialized   */
    EFMTST_INIT                              = 0x01, /**< The EFM test is initialized   */
    EFMTST_TESTING                           = 0x02, /**< The EFM test is running   */
    EFMTST_TESTED                            = 0x03, /**< The EFM test is completed   */
} EfmTst_StateType;

/**
* @brief          EFM test failed notification type.
* @sw_type        sw_arch
 *
*/
typedef void (*EfmTst_FailedNotificationType)(uint16 MoudleId);

/*==================================================================================================
 *                                        TYPEDEF STRUCT
==================================================================================================*/
/**
 * @brief type of Flash EFM test address.
 * @sw_type sw_arch
 *
 */
typedef uint32 EfmTst_AddressType;

/**
 * @brief type of Flash EFM test flash protection configuration data.
 * @sw_type sw_arch
 *
 */
typedef uint32 EfmTst_SectorProtCfgType;

/**
 * @brief type of Flash EFM test sector data injection.
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_869
 */
typedef uint8 EfmTst_SectorDataInjectionType;

/**
 * @brief type of EFM test sector configuration data.
 * @sw_type sw_arch
 *
 */
typedef struct
{
    EfmTst_AddressType StartAddr;       /**< Flash EFM test start address */
    uint32 Length;                      /**< Flash EFM test length */
    EfmTst_AddressType HwStartAddr;     /**< Flash EFM test physical start address */
} EfmTst_SectorType;

/**
 * @brief type of EFM test configuration data.
 * @sw_type sw_arch
 *
 */
typedef struct
{
    EfmTst_SectorType const *ProtectSectorPtr;      /**< Flash EFM test protected sector configuration data pointer */
    EfmTst_SectorProtCfgType const *SectorProtCfgPtr;     /**< Flash EFM test flash protection configuration data pointer */
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
    EfmTst_FailedNotificationType EfmTstErrorNotification; /**< Flash EFM test error notification function pointer */
#endif
#if ((EFMTST_ENABLE_WRITE_ENABLE_TEST==STD_ON)||(EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON))
    EfmTst_AddressType NoneProtectSectorAddr;     /**< Flash EFM test non-protected sector address,used to test the WE and CMD UNLOCK mechanisms*/
#endif
} EfmTst_ConfigType;

/**
 * @brief type of EFM test result detail.
 * @sw_type sw_arch
 *
 */
typedef struct
{
#if (EFMTST_ENABLE_WRITE_ENABLE_TEST==STD_ON)
    EfmTst_TestResultType  WirteEnableResult;   /**< Flash EFM test wirte enable test result */
#endif
#if (EFMTST_ENABLE_CMD_UNLOCK_TEST==STD_ON)
    EfmTst_TestResultType CommandUnlockResult;    /**< Flash EFM test command unlock test result */
#endif
    EfmTst_TestResultType ProtectSectorResult;    /**< Flash EFM test protected sector test result */
} EfmTst_TestDetailsType;


#ifdef __cplusplus
}
#endif

#endif /* End of file EfmTst_Types.h */


