/**
 * @file    EccTst.c
 * @version
 *
 * @brief   AUTOSAR EccTst module interface
 * @details API implementation for ECCTST driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : EccTst
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif
/*
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 0306 Rule 11.4: A conversion should not be performed between a pointer to object and an
 *                    integer type.
 *
 * PRQA S 2985 Rule 2.2: There shall be no dead code.
 *
 * PRQA S 2986 Rule 2.2: There shall be no dead code.
 *
 * PRQA S 2877 Dir  4.1: Run-time failures shall be minimized.
 */

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "EccTst_Lld_Ram.h"
#include "EccTst_Lld_Reg.h"
#include "Platform.h"

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_LLD_RAM_VENDOR_ID_C                      (0xB4)
#define ECCTST_LLD_RAM_AR_RELEASE_MAJOR_VERSION_C       (4)
#define ECCTST_LLD_RAM_AR_RELEASE_MINOR_VERSION_C       (4)
#define ECCTST_LLD_RAM_AR_RELEASE_REVISION_VERSION_C    (0)
#define ECCTST_LLD_RAM_SW_MAJOR_VERSION_C               (2)
#define ECCTST_LLD_RAM_SW_MINOR_VERSION_C               (0)
#define ECCTST_LLD_RAM_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and EccTst_Lld_Ram header file are of the same vendor */
#if (ECCTST_LLD_RAM_VENDOR_ID_C != ECCTST_LLD_RAM_VENDOR_ID)
#error "EccTst_Lld_Ram.c and EccTst_Lld_Ram.h have different vendor ids"
#endif

/* Check if source file and EccTst_Lld_Ram header file are of the same Autosar version */
#if ((ECCTST_LLD_RAM_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_RAM_AR_RELEASE_MAJOR_VERSION) || \
     (ECCTST_LLD_RAM_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_RAM_AR_RELEASE_MINOR_VERSION) || \
     (ECCTST_LLD_RAM_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_RAM_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of EccTst_Lld_Ram.c and EccTst_Lld_Ram.h are different"
#endif

/* Check if source file and EccTst_Lld_Ram header file are of the same Software version */
#if ((ECCTST_LLD_RAM_SW_MAJOR_VERSION_C != ECCTST_LLD_RAM_SW_MAJOR_VERSION) || \
     (ECCTST_LLD_RAM_SW_MINOR_VERSION_C != ECCTST_LLD_RAM_SW_MINOR_VERSION) || \
     (ECCTST_LLD_RAM_SW_PATCH_VERSION_C != ECCTST_LLD_RAM_SW_PATCH_VERSION))
#error "Software Version Numbers of EccTst_Lld_Ram.c and EccTst_Lld_Ram.h are different"
#endif

/* Check if source file and ECCTST register header file are of the same vendor */
#if (ECCTST_LLD_RAM_VENDOR_ID_C != ECCTST_LLD_REG_VENDOR_ID)
#error "EccTst_Lld_Ram.c and EccTst_Lld_Reg.h have different vendor ids"
#endif

/* Check if source file and ECCTST register header file are of the same Autosar version */
#if ((ECCTST_LLD_RAM_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_REG_AR_REL_MAJOR_VER) || \
     (ECCTST_LLD_RAM_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_REG_AR_REL_MINOR_VER) || \
     (ECCTST_LLD_RAM_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_REG_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Ram.c and EccTst_Lld_Reg.h are different"
#endif

/* Check if source file and ECCTST register header file are of the same Software version */
#if ((ECCTST_LLD_RAM_SW_MAJOR_VERSION_C != ECCTST_LLD_REG_SW_MAJOR_VER) || \
     (ECCTST_LLD_RAM_SW_MINOR_VERSION_C != ECCTST_LLD_REG_SW_MINOR_VER) || \
     (ECCTST_LLD_RAM_SW_PATCH_VERSION_C != ECCTST_LLD_REG_SW_PATCH_VER) \
    )
#error "Software Version Numbers of EccTst_Lld_Ram.c and EccTst_Lld_Reg.h are different"
#endif
/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#if defined (CPU_YTM32B1MC0)
#define EMU_BASE_ADDR32                 EMU0_BASE_ADDR32
#define EMU_EICHCR_CH0EN_MASK           EMU_EICHCR_CHEN_MASK
#elif defined (CPU_YTM32B1HA0)
#define EMU_EICHCR_CH0EN_MASK           0x00000001U
#define EMU_DTCM1_ADDR_OFFSET           0x00000004U
#elif defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MD1)
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif

#define RAMECC_TEST_INIT_SIGNATURE  0x5A5AU

#define RAMECC_EMUIRQ_SIGNATURE     0x1285U

#define RAMECC_BUSFAULT_SIGNATURE   0x406AU

#define RAMECC_SINGLEBIT_SIGNATURE  0x48DFU

#define RAMECC_DOUBLEBIT_SIGNATURE  0x08B5U
/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/
/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/
#define ECCTST_START_SEC_CONST_UNSPECIFIED
#include "EccTst_MemMap.h"
/**
 * @brief The register pointer for Emu
 * @sw_type sw_detail
 * @private
 */
ECCTST_CONST static volatile EMU_Type *const Emu_RegBase = EMU_BASE_PTRS;

#define ECCTST_STOP_SEC_CONST_UNSPECIFIED
#include "EccTst_MemMap.h"

#define ECCTST_START_SEC_VAR_INIT_32
#include "EccTst_MemMap.h"
/**
 * @brief RAM ECC test configuration data pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static RamEcc_ConfigType const *RamEcc_ConfigPtr = NULL_PTR;
#define ECCTST_STOP_SEC_VAR_INIT_32
#include "EccTst_MemMap.h"

#define ECCTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "EccTst_MemMap.h"
/**
 * @brief RAM ECC test detail result
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static volatile RamEcc_TestDetailsType RamEcc_DetailResult;
/**
 * @brief RAM ECC test status data
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static volatile RamEcc_RunningStatusType RamEcc_RunningData;
/**
 * @brief RAM ECC error handler pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static Platform_IrqHandlerType RamEcc_FuncPointer;
#define ECCTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "EccTst_MemMap.h"
/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#define ECCTST_START_SEC_CODE
#include "EccTst_MemMap.h"
/**
 * @brief  This function is used to get the SCIF flag
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return boolean
 * @retval TRUE : The SCIF flag is set
 * @retval FALSE : The SCIF flag is not set
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_163
*/
ECCTST_FUNC LOCAL_INLINE boolean EccTst_Lld_Ram_GetSCIF(volatile EMU_Type const *BaseAddr, RamEcc_RegionNameType RegineName)
{
    boolean RetValue;
    /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the left-hand operand.
     * This line of code has been maintained for the sake of code clarity and readability.
     */
    if (0U == ((BaseAddr->ERIF >> (EMU_ERIF_SCIF_SHIFT + (uint8)RegineName)) & 0x01U)) /*PRQA S 2986*/
    {
        RetValue = FALSE;
    }
    else
    {
        RetValue = TRUE;
    }
    return RetValue;
}
/**
 * @brief  This function is used to clear the SCIF flag
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_173
*/
ECCTST_FUNC LOCAL_INLINE void EccTst_Lld_Ram_SetSCIF(volatile EMU_Type *BaseAddr, RamEcc_RegionNameType RegineName)
{
    /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the left-hand operand.
     * This line of code has been maintained for the sake of code clarity and readability.
     */
    BaseAddr->ERIF |= 0x00000001UL << (EMU_ERIF_SCIF_SHIFT + (uint32)RegineName); /*PRQA S 2986*/
}
/**
 * @brief  This function is used to get the NCIF flag
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return boolean
 * @retval TRUE : The NCIF flag is set
 * @retval FALSE : The NCIF flag is not set
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_162
*/
ECCTST_FUNC LOCAL_INLINE boolean EccTst_Lld_Ram_GetNCIF(volatile const EMU_Type *BaseAddr, RamEcc_RegionNameType RegineName)
{
    boolean RetValue;
    if (0U == ((BaseAddr->ERIF >> (EMU_ERIF_NCIF_SHIFT + (uint8)RegineName)) & 0x01U))
    {
        RetValue = FALSE;
    }
    else
    {
        RetValue = TRUE;
    }
    return RetValue;
}
/**
 * @brief This function is used to set the NCIF flag
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_172
*/
ECCTST_FUNC LOCAL_INLINE void EccTst_Lld_Ram_SetNCIF(volatile EMU_Type *BaseAddr, RamEcc_RegionNameType RegineName)
{
    BaseAddr->ERIF |= 0x00000001UL << (EMU_ERIF_NCIF_SHIFT + (uint32)RegineName);
}
/**
 * @brief  This function is used to get the SCIE status
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return boolean
 * @retval TRUE : The SCIE is set
 * @retval FALSE : The SCIE is not set
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_446
*/
ECCTST_FUNC LOCAL_INLINE  boolean EccTst_Lld_Ram_GetSCIE(volatile const EMU_Type *BaseAddr, RamEcc_RegionNameType RegineName)
{
    boolean RetValue;
    /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the left-hand operand.
     * This line of code has been maintained for the sake of code clarity and readability.
     */
    if (0U == ((BaseAddr->ERIE >> (EMU_ERIE_SCIE_SHIFT + (uint8)RegineName)) & 0x01U)) /*PRQA S 2986*/
    {
        RetValue = FALSE;
    }
    else
    {
        RetValue = TRUE;
    }
    return RetValue;
}
/**
 * @brief  This function is used to get the NCIE status
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return boolean
 * @retval TRUE : The NCIE is set
 * @retval FALSE : The NCIE is not set
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_445
*/
ECCTST_FUNC LOCAL_INLINE  boolean EccTst_Lld_Ram_GetNCIE(volatile const EMU_Type *BaseAddr, RamEcc_RegionNameType RegineName)
{
    boolean RetValue;
    if (0U == ((BaseAddr->ERIE >> (EMU_ERIE_NCIE_SHIFT + (uint8)RegineName)) & 0x01U))
    {
        RetValue = FALSE;
    }
    else
    {
        RetValue = TRUE;
    }
    return RetValue;
}
/**
 * @brief This function is used to get the RAM ECC error address
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return uint32
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_161
*/
ECCTST_FUNC LOCAL_INLINE uint32 EccTst_Lld_Ram_GetErrorAddr(volatile const EMU_Type *BaseAddr, RamEcc_RegionNameType RegineName)
{
    return BaseAddr->ERINFO[RegineName].ADDR;
}
/**
 * @brief This function is used to diable RAM ECC inject test
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  RegineName : The test memory name
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_160
*/
ECCTST_FUNC LOCAL_INLINE void EccTst_Lld_Ram_DisableTest(volatile EMU_Type *BaseAddr, RamEcc_RegionNameType RegineName)
{
    /* Disable channel error injection */
    BaseAddr->EICHCR &= ~((uint32)EMU_EICHCR_CH0EN_MASK << (uint32)RegineName);
    /* Disable global error injection */
    BaseAddr->EIGCR &= ~(uint32)EMU_EIGCR_GEN_MASK;
}

/**
 * @brief This function is used to reset the EMU register and RAM ECC test running data
 * @param[in] BaseAddr : The register base address of EMU module
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_164
*/
ECCTST_FUNC static void EccTst_Lld_Ram_InitRegAndData(volatile EMU_Type *BaseAddr)
{
    uint8 Index = 0;
    /* Disable global error injection */
    BaseAddr->EIGCR &= ~(uint32)EMU_EIGCR_GEN_MASK;
    /* Disable all channels register about error injection */
    /* MR12 RULE 2.2 VIOLATION: The loop can only be executed once here in YTM32B1MC0 due to the the limitation of EMU resources.
     * This line of code has been maintained to ensure compatibility with other MCUs.
     */
    for (Index = 0; Index < (uint8)EMU_EICHD_COUNT; ++Index) /*PRQA S 2877*/
    {
        BaseAddr->EICHCR &= ~((uint32)EMU_EICHCR_CH0EN_MASK << (uint32)Index);
        BaseAddr->EICHD[Index].ADDR = 0U;
        BaseAddr->EICHD[Index].CHKBIT = EMU_CHKBIT_DISABLED;
#if defined (CPU_YTM32B1HA0)
        BaseAddr->EICHD[Index].DATAL = EMU_DATABIT_DISABLED;
        BaseAddr->EICHD[Index].DATAH = EMU_DATABIT_DISABLED;
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MC0)
        BaseAddr->EICHD[Index].DATA = EMU_DATABIT_DISABLED;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
    }
    /* Disable interrupts */
    BaseAddr->ERIE &= ~EMU_ERIE_NCIE_MASK;
    BaseAddr->ERIE &= ~EMU_ERIE_SCIE_MASK;
    /* Clear interrupt flags */
    BaseAddr->ERIF |= EMU_ERIF_NCIF_MASK;
    BaseAddr->ERIF |= EMU_ERIF_SCIF_MASK;
    /* MR12 RULE 2.2 VIOLATION: The loop can only be executed once here in YTM32B1MC0 due to the the limitation of EMU resources.
     * This line of code has been maintained to ensure compatibility with other MCUs.
     */
    for (Index = 0; Index < (uint8)EMU_EICHD_COUNT; ++Index) /*PRQA S 2877*/
    {
        /* Clean Correctable Error Count register */
        BaseAddr->ERINFO[Index].CNT = 0U;
    }
    /*Init the test result data*/
    for (Index = 0; Index < RamEcc_ConfigPtr->RamErrorInjectionChannelCount; ++Index)
    {
        /*The RamErrorInjectionChannelCount is limited by YCT config tool when generate,it does not cause
        RamEcc_DetailResult.TestDetailResult access out of range, set NOSONAR*/
        RamEcc_DetailResult.TestDetailResult[Index].ChannelId = EMU_SRAM0;/*NOSONAR*/
        RamEcc_DetailResult.TestDetailResult[Index].TestResult = ECCTST_NOT_TEST;/*NOSONAR*/
    }
    RamEcc_RunningData.SumaryResult = ECCTST_NOT_TEST;
    RamEcc_RunningData.RunningIndex = 0U;
    RamEcc_RunningData.RunningMode = RAMECC_NONE_TEST;
}
/**
 * @brief This function is the ISR of system exception when RAM ECC test running.
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_159
*/
ECCTST_FUNC static void EccTst_Lld_Ram_BusFaultIsr(void)
{
    uint32 TempData = 0U;
    /* Disable channel error injection */
    /* MR12 RULE 2.2 VIOLATION: The loop can only be executed once here in YTM32B1MC0 due to the the limitation of EMU resources.
     * This line of code has been maintained to ensure compatibility with other MCUs.
     */
    for (uint8 Index = 0; Index < (uint8)EMU_EICHD_COUNT; ++Index) /*PRQA S 2877*/
    {
        Emu_RegBase->EICHCR &= ~((uint32)EMU_EICHCR_CH0EN_MASK << (uint32)Index);
    }
    /* Disable global error injection */
    Emu_RegBase->EIGCR &= ~(uint32)EMU_EIGCR_GEN_MASK;
    /*Clear the BusFault exception flag*/
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to clear EMU error flags.
    * couldn't adhere to M3CM Rule-11.4
    */
    SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk | SCB_CFSR_PRECISERR_Msk; /*PRQA S 0306*/
    TempData = RamEcc_RunningData.RunningSignature;
    RamEcc_RunningData.RunningSignature = (TempData & 0xFFFF0000U) + \
                                          ((TempData ^ ((uint32)RAMECC_BUSFAULT_SIGNATURE)) & 0x0000FFFFU);
}
/**
 * @brief This function is used to read the RAM ECC test address
 * @param[in] Address The address to read
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return uint32
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_167
*/
ECCTST_FUNC static uint32 EccTst_Lld_Ram_ReadAddress(uint32 Address)
{
    volatile uint32 TempData;
#if (PLATFORM_ENABLE_CACHE == STD_ON)
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to clear the DCACHE for ECC error injection,
    * and couldn't adhere to M3CM Rule-11.4.
    */
    SCB_InvalidateDCache_by_Addr((uint32 *)Address, 32); /*PRQA S 0306*/
#endif
    __ASM("ISB");
    __ASM("DSB");
    TempData = *(volatile uint32 *)Address;
    TempData += 1U;
    /*The purpose of this code is to prevent the compiler from optimizing the previous two lines of code,
        especially the GHS.*/
    if (TempData != 1U)
    {
        TempData = 0U;
    }
    return TempData;
}
/**
 * @brief This function is used to enable or disable the interrupt for RAM ECC test
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in] TestRegion The test memory name
 * @param[in] ErrorMode The RAM ECC test mode
 * @param[in] IntrEnable Enable or disable the interrupt
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_171
*/
ECCTST_FUNC static void EccTst_Lld_Ram_SetInterrupts(volatile EMU_Type *BaseAddr,
        RamEcc_RegionNameType TestRegion,
        RamEcc_InjectErrorModeType ErrorMode,
        boolean IntrEnable)
{
    if (RAMECC_DATA_CORRECTABLE_TEST == ErrorMode)
    {
        /* Clean interrupt flag */
        /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the left-hand operand.
        * This line of code has been maintained for the sake of code clarity and readability.
        */
        BaseAddr->ERIF |= 0x00000001UL << (EMU_ERIF_SCIF_SHIFT + (uint32)TestRegion); /*PRQA S 2986*/
        /* Config interrupt enable bit*/
        if (TRUE == IntrEnable)
        {
            BaseAddr->ERIE |= 0x00000001UL << (EMU_ERIE_SCIE_SHIFT + (uint32)TestRegion); /*PRQA S 2986*/
        }
        else
        {
            BaseAddr->ERIE &= ~(0x00000001UL << (EMU_ERIE_SCIE_SHIFT + (uint32)TestRegion)); /*PRQA S 2986*/
        }
    }
    else
    {
        /* Clean interrupt flag */
        BaseAddr->ERIF |= 0x00000001UL << (EMU_ERIF_NCIF_SHIFT + (uint32)TestRegion);
        /* Config interrupt enable bit*/
        if (TRUE == IntrEnable)
        {
            BaseAddr->ERIE |= 0x00000001UL << (EMU_ERIE_NCIE_SHIFT + (uint32)TestRegion);
        }
        else
        {
            BaseAddr->ERIE &= ~(0x00000001UL << (EMU_ERIE_NCIE_SHIFT + (uint32)TestRegion));
        }
    }
}
/**
 * @brief  This function is used to config test data and address for RAM ECC test,and enbale the test
 * @param[in] BaseAddr : The register base address of EMU module
 * @param[in]  ErrorMode The RAM ECC test mode
 * @param[in]  ChannelData The RAM ECC test channel configuration data
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_165
*/

ECCTST_FUNC static void EccTst_Lld_Ram_InjectData(volatile EMU_Type *BaseAddr,
        RamEcc_InjectErrorModeType ErrorMode,
        RamEcc_ChannelConfigType const *ChannelData)
{
    RamEcc_RegionNameType TestRegion = ChannelData->RamErrorInjectionChannelId;
    if (RAMECC_CHKBIT_TEST == ErrorMode)
    {
        /* Set multiple error injection check bits.*/
        BaseAddr->EICHD[TestRegion].CHKBIT = ChannelData->RamErrInjectionChkBit;
#if defined (CPU_YTM32B1HA0)
        BaseAddr->EICHD[TestRegion].DATAL = 0x00000000U;
        BaseAddr->EICHD[TestRegion].DATAH = 0x00000000U;
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MC0)
        BaseAddr->EICHD[TestRegion].DATA = 0x00000000U;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
    }
    else if (RAMECC_DATA_CORRECTABLE_TEST == ErrorMode)
    {
        /* Set multiple error injection data bits.*/
#if defined (CPU_YTM32B1HA0)
        BaseAddr->EICHD[TestRegion].DATAL = ChannelData->RamErrInjectionDataBit0L;
        BaseAddr->EICHD[TestRegion].DATAH = ChannelData->RamErrInjectionDataBit0H;
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MC0)
        BaseAddr->EICHD[TestRegion].DATA = ChannelData->RamErrInjectionDataBit0L;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
        BaseAddr->EICHD[TestRegion].CHKBIT = 0x00000000U;
    }
    else
    {
        /* Set multiple error injection data bits.*/
#if defined (CPU_YTM32B1HA0)
        BaseAddr->EICHD[TestRegion].DATAL = ChannelData->RamErrInjectionDataBit0L | ChannelData->RamErrInjectionDataBit1L;
        BaseAddr->EICHD[TestRegion].DATAH = ChannelData->RamErrInjectionDataBit0H | ChannelData->RamErrInjectionDataBit1H;
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MC0)
        BaseAddr->EICHD[TestRegion].DATA = ChannelData->RamErrInjectionDataBit0L | ChannelData->RamErrInjectionDataBit1L;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
        BaseAddr->EICHD[TestRegion].CHKBIT = 0x00000000U;
    }
    /* Set error injection address.*/
#if defined (CPU_YTM32B1HA0)
    switch (TestRegion)
    {
        case EMU_ITCM:
            /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the left-hand operand.
            * This line of code has been maintained for the sake of code clarity and readability.
            */
            BaseAddr->EICHD[TestRegion].ADDR = ChannelData->RamErrorInjectionAddr - ECCTST_EMU_ITCM_BASE_ADDR; /*PRQA S 2985*/
            break;
        case EMU_DTCM0:
            BaseAddr->EICHD[TestRegion].ADDR = ChannelData->RamErrorInjectionAddr - ECCTST_EMU_DTCM0_BASE_ADDR;
            break;
        case EMU_DTCM1:
            BaseAddr->EICHD[TestRegion].ADDR = ChannelData->RamErrorInjectionAddr - ECCTST_EMU_DTCM1_BASE_ADDR - EMU_DTCM1_ADDR_OFFSET;
            break;
        default:
            BaseAddr->EICHD[TestRegion].ADDR = ChannelData->RamErrorInjectionAddr;
            break;
    }
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MC0)
    BaseAddr->EICHD[TestRegion].ADDR = ChannelData->RamErrorInjectionAddr;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
    /* Enable channel error injection */
    BaseAddr->EICHCR |= (uint32)EMU_EICHCR_CH0EN_MASK << ((uint32)TestRegion);
    /* Enable global error injection */
#if defined (YTM32B1HA0)||defined (CPU_YTM32B1MC0)
    BaseAddr->EIGCR = 0x89ABCDEFU;
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)
    BaseAddr->EIGCR |= EMU_EIGCR_GEN_MASK;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}
/**
 * @brief  This function is used to judge the result of ECC test
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_169
*/
ECCTST_FUNC static void EccTst_Lld_Ram_ResultJudged(void)
{
    uint32 ErrorAddress;
    boolean ScifFlag;
    boolean NcifFlag;
    uint8  TempIndex = RamEcc_RunningData.RunningIndex;
    RamEcc_RegionNameType TestRegion = RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId;
    ErrorAddress = EccTst_Lld_Ram_GetErrorAddr(Emu_RegBase, TestRegion);
    /*The ITCM and DTCM error address in register is a offset address*/
#if defined (CPU_YTM32B1HA0)
    switch (TestRegion)
    {
        case EMU_ITCM:
            /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the left-hand operand.
            * This line of code has been maintained for the sake of code clarity and readability.
            */
            ErrorAddress = ErrorAddress + ECCTST_EMU_ITCM_BASE_ADDR; /*PRQA S 2985*/
            break;
        case EMU_DTCM0:
            ErrorAddress = ErrorAddress + ECCTST_EMU_DTCM0_BASE_ADDR;
            break;
        case EMU_DTCM1:
            ErrorAddress = ErrorAddress + ECCTST_EMU_DTCM1_BASE_ADDR + EMU_DTCM1_ADDR_OFFSET;
            break;
        default:
            /*Nothing to do*/
            break;
    }
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MC0)
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
    if (RAMECC_DATA_CORRECTABLE_TEST == RamEcc_RunningData.RunningMode)
    {
        ScifFlag = EccTst_Lld_Ram_GetSCIF(Emu_RegBase, TestRegion);
        if ((TRUE == ScifFlag) && \
                (ErrorAddress == RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionAddr))
        {
            /*The single correctable error is firstly test , when test ok ,write E_OK, the none correctable error will not write e_ok even
                the result is ok. the result 'ok' will keep test end, when no 'not ok' in the rest test*/
            /*The RamEcc_RunningData.RunningIndex is limited in EccTst_Lld_RamInjectMain,it does not cause
                RamEcc_DetailResult.RamEccRst access out of range, set NOSONAR*/
            RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult = ECCTST_OK;/*NOSONAR*/
        }
        else
        {
            RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult = ECCTST_NOT_OK;/*NOSONAR*/
            RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (RamEcc_ConfigPtr->RamEccTestFailedNotification != NULL_PTR)
            {
                RamEcc_ConfigPtr->RamEccTestFailedNotification(ECCTST_MODULE_ID);
            }
        }
        EccTst_Lld_Ram_SetSCIF(Emu_RegBase, TestRegion);
    }
    else if (RAMECC_DATA_NONE_CORRECTABLE_TEST == RamEcc_RunningData.RunningMode)
    {
        NcifFlag = EccTst_Lld_Ram_GetNCIF(Emu_RegBase, TestRegion);
        if ((FALSE == NcifFlag) || \
                (ErrorAddress != RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionAddr))
        {
            /*When error flag or error address is mismatched, set result as not ok*/
            RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult = ECCTST_NOT_OK;
            RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (RamEcc_ConfigPtr->RamEccTestFailedNotification != NULL_PTR)
            {
                RamEcc_ConfigPtr->RamEccTestFailedNotification(ECCTST_MODULE_ID);
            }
        }
        EccTst_Lld_Ram_SetNCIF(Emu_RegBase, TestRegion);
    }
    else
    {
        ScifFlag = EccTst_Lld_Ram_GetSCIF(Emu_RegBase, TestRegion);
        NcifFlag = EccTst_Lld_Ram_GetNCIF(Emu_RegBase, TestRegion);
        if (((FALSE == ScifFlag) && \
                (FALSE == NcifFlag)) || \
                (ErrorAddress != RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionAddr))
        {
            /*When error flag or error address is mismatched, set result as not ok*/
            RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult = ECCTST_NOT_OK;
            RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (RamEcc_ConfigPtr->RamEccTestFailedNotification != NULL_PTR)
            {
                RamEcc_ConfigPtr->RamEccTestFailedNotification(ECCTST_MODULE_ID);
            }
        }
        EccTst_Lld_Ram_SetSCIF(Emu_RegBase, TestRegion);
        EccTst_Lld_Ram_SetNCIF(Emu_RegBase, TestRegion);
    }
}
/**
 * @brief Check the signature of RAM ECC test
 * @param[in]  Index: The index of the ongoing test region
 * @param[in]  RunningMode: The running mode of the ongoing test region
 * @param[in]  TempSignature: The signature of the ongoing test region
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_371
*/
ECCTST_FUNC  static Std_ReturnType EccTst_Lld_Ram_SignatureCheck(uint8 Index, RamEcc_InjectErrorModeType RunningMode, uint32 TempSignature)
{
    Std_ReturnType RetValue = E_NOT_OK;
    uint32 TempInex = (((uint32)Index) << 16) + \
                      (((uint32)RunningMode) << 24);
    if ((RAMECC_DATA_CORRECTABLE_TEST == RunningMode) || (RAMECC_CHKBIT_TEST == RunningMode))
    {
        TempInex += RAMECC_SINGLEBIT_SIGNATURE;
    }
    else
    {
        TempInex += RAMECC_DOUBLEBIT_SIGNATURE;
    }
    /*The ramecc test excute EMU irq*/
    if (TempInex == TempSignature)
    {
        RetValue = E_OK;
    }
    return RetValue;
}

/**
 * @brief This function is used to check the pre-condition of RAM ECC test
 * @param[in] TempIndex : The index of test memory in the config
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_995
*/
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Ram_PreCheck(uint8 TempIndex)
{
    Std_ReturnType RetValue = E_OK;
    boolean ScifFlag;
    boolean NcifFlag;
    /*[YTSWS_EccTst_300]The EccTst module shall support pre-condition check before the commencement of testing.*/
    ScifFlag = EccTst_Lld_Ram_GetSCIF(Emu_RegBase,
                                      RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
    NcifFlag = EccTst_Lld_Ram_GetNCIF(Emu_RegBase,
                                      RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
    MCAL_QUA_FAULT_INJECTION(ECCTST_LLD_RAM_ECCERR_FLAG);
    if ((TRUE == ScifFlag) || (TRUE == NcifFlag))
    {
        EccTst_Lld_Ram_SetSCIF(Emu_RegBase, RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
        EccTst_Lld_Ram_SetNCIF(Emu_RegBase, RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
        ScifFlag = EccTst_Lld_Ram_GetSCIF(Emu_RegBase,
                                          RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
        NcifFlag = EccTst_Lld_Ram_GetNCIF(Emu_RegBase,
                                          RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
        MCAL_QUA_FAULT_INJECTION(ECCTST_LLD_RAM_ECCERR_FLAG);
        if ((TRUE == ScifFlag) || (TRUE == NcifFlag))
        {
            RamEcc_DetailResult.TestDetailResult[TempIndex].ChannelId = \
                    RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId;
            RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult = ECCTST_NOT_OK;
            RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (RamEcc_ConfigPtr->RamEccTestFailedNotification != NULL_PTR)
            {
                RamEcc_ConfigPtr->RamEccTestFailedNotification(ECCTST_MODULE_ID);
            }
            RetValue = E_NOT_OK;
        }
    }
    return RetValue;
}

/**
 * @brief This function is used to perform RAM ECC tests
 * @param[in] TempIndex : The index of test memory in the config
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_166
*/
ECCTST_FUNC static void EccTst_Lld_Ram_InjectError(uint8 TempIndex)
{
    Std_ReturnType TempValue = E_OK;
    TempValue = EccTst_Lld_Ram_PreCheck(TempIndex);
    if (E_OK == TempValue)
    {
        /*RAM ECC inject test*/
        /*Reset the running status data*/
        RamEcc_RunningData.RunningIndex = TempIndex;
        RamEcc_RunningData.RunningMode = RAMECC_NONE_TEST;
        RamEcc_DetailResult.TestDetailResult[TempIndex].ChannelId = \
                RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId;
        /*Init the interrupt*/
        EccTst_Lld_Ram_SetInterrupts(Emu_RegBase, RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId, \
                                     RAMECC_DATA_CORRECTABLE_TEST, \
                                     TRUE);
        EccTst_Lld_Ram_SetInterrupts(Emu_RegBase, RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId, \
                                     RAMECC_DATA_NONE_CORRECTABLE_TEST, \
                                     TRUE);
        /*******************************************************************************************************************/
        /*RAM ECC one bit inject test*/
        RamEcc_RunningData.RunningMode = RAMECC_DATA_CORRECTABLE_TEST;
        RamEcc_RunningData.RunningSignature = (((uint32)TempIndex) << 16) + \
                                              (((uint32)RAMECC_DATA_CORRECTABLE_TEST) << 24) + \
                                              RAMECC_TEST_INIT_SIGNATURE;
        EccTst_Lld_Ram_InjectData(Emu_RegBase, \
                                  RAMECC_DATA_CORRECTABLE_TEST, \
                                  & (RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex]));
        /* Read memory to inject error */
        (void)EccTst_Lld_Ram_ReadAddress(RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionAddr);
        /*Check the result,the result should be OK or NOT_OK, otherwaise no interrupt flag is generated and the test fails */
        /*The RamEcc_RunningData.RunningIndex is limited in EccTst_Lld_RamInjectMain, it does not cause
                    RamEcc_DetailResult.RamEccRst access out of range, set NOSONAR*/
        if (ECCTST_NOT_TEST == RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult)  /*NOSONAR*/
        {
            RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult  = ECCTST_NOT_OK;
            RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
        }
        if (E_NOT_OK == EccTst_Lld_Ram_SignatureCheck(TempIndex, RAMECC_DATA_CORRECTABLE_TEST, RamEcc_RunningData.RunningSignature))
        {
            RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult  = ECCTST_NOT_OK;
            RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
        }
        /*disable test*/
        EccTst_Lld_Ram_DisableTest(Emu_RegBase, RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
        /*******************************************************************************************************************/
        /*RAM ECC two bit data inject test*/
#if defined (CPU_YTM32B1HA0)
        if ((RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrInjectionDataBit1L != EMU_DATABIT_DISABLED) || \
                (RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrInjectionDataBit1H != EMU_DATABIT_DISABLED))
        {
            RamEcc_RunningData.RunningMode = RAMECC_DATA_NONE_CORRECTABLE_TEST;
            RamEcc_RunningData.RunningSignature = (((uint32)TempIndex) << 16) + \
                                                  (((uint32)RAMECC_DATA_NONE_CORRECTABLE_TEST) << 24) + \
                                                  RAMECC_TEST_INIT_SIGNATURE;
            EccTst_Lld_Ram_InjectData(Emu_RegBase, \
                                      RAMECC_DATA_NONE_CORRECTABLE_TEST, \
                                      & (RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex]));
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1MC0)
        if (RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrInjectionDataBit1L != EMU_DATABIT_DISABLED)
        {
            RamEcc_RunningData.RunningMode = RAMECC_DATA_NONE_CORRECTABLE_TEST;
            RamEcc_RunningData.RunningSignature = (((uint32)TempIndex) << 16) + \
                                                  (((uint32)RAMECC_DATA_NONE_CORRECTABLE_TEST) << 24) + \
                                                  RAMECC_TEST_INIT_SIGNATURE;
            EccTst_Lld_Ram_InjectData(Emu_RegBase, \
                                      RAMECC_DATA_NONE_CORRECTABLE_TEST, \
                                      & (RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex]));
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
            /* Read memory to inject error */
            (void)EccTst_Lld_Ram_ReadAddress(RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionAddr);
            if (E_NOT_OK == EccTst_Lld_Ram_SignatureCheck(TempIndex,
                    RAMECC_DATA_NONE_CORRECTABLE_TEST,
                    RamEcc_RunningData.RunningSignature))
            {
                RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult  = ECCTST_NOT_OK;
                RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            }
            /*disable test*/
            EccTst_Lld_Ram_DisableTest(Emu_RegBase,
                                       RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
        }
        /*******************************************************************************************************************/
        /*RAM ECC chk bit inject test*/
        if (RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrInjectionChkBit != EMU_CHKBIT_DISABLED)
        {
            RamEcc_RunningData.RunningMode = RAMECC_CHKBIT_TEST;
            RamEcc_RunningData.RunningSignature = (((uint32)TempIndex) << 16) + \
                                                  (((uint32)RAMECC_CHKBIT_TEST) << 24) + \
                                                  RAMECC_TEST_INIT_SIGNATURE;
            EccTst_Lld_Ram_InjectData(Emu_RegBase, \
                                      RAMECC_CHKBIT_TEST, \
                                      & (RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex]));
            /* Read memory to inject error */
            (void)EccTst_Lld_Ram_ReadAddress(RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionAddr);
            if (E_NOT_OK == EccTst_Lld_Ram_SignatureCheck(TempIndex, RAMECC_CHKBIT_TEST, RamEcc_RunningData.RunningSignature))
            {
                RamEcc_DetailResult.TestDetailResult[TempIndex].TestResult  = ECCTST_NOT_OK;
                RamEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            }
            /*disable test*/
            EccTst_Lld_Ram_DisableTest(Emu_RegBase, RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId);
        }
        /*The resgion test end ,close the interrupt*/
        EccTst_Lld_Ram_SetInterrupts(Emu_RegBase, \
                                     RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId, \
                                     RAMECC_DATA_CORRECTABLE_TEST, \
                                     FALSE);
        EccTst_Lld_Ram_SetInterrupts(Emu_RegBase, \
                                     RamEcc_ConfigPtr->RamErrorInjectionChannel[TempIndex].RamErrorInjectionChannelId, \
                                     RAMECC_DATA_NONE_CORRECTABLE_TEST, \
                                     FALSE);
    }
}
/**
 * @brief  This function is used to set exception handler.
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_170
*/
ECCTST_FUNC static void EccTst_Lld_Ram_SetExpHandler(void)
{
    /*************************************************************************************************/
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to clear EMU error flags.
    * couldn't adhere to M3CM Rule-11.4
    */
    /*Enable usageFault exception*/
    SCB->SHCSR |= SCB_SHCSR_BUSFAULTENA_Msk;/*PRQA S 0306*/
    SCB->CCR |= SCB_CCR_BFHFNMIGN_Msk;/*PRQA S 0306*/
    (void)Platform_InstallIrqHandler(BusFault_IRQn, EccTst_Lld_Ram_BusFaultIsr, &RamEcc_FuncPointer);
}

/**
 * @brief  This function is used to recover exception handler
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_168
*/
ECCTST_FUNC static void EccTst_Lld_Ram_RecoveredExpHandler(void)
{
    Platform_IrqHandlerType TempBusFaultHandler;
    (void)Platform_InstallIrqHandler(BusFault_IRQn, RamEcc_FuncPointer, &TempBusFaultHandler);
}


/**
 * @brief RAM ECC test error interrupt process.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_158
 */
ECCTST_FUNC static void EccTst_Lld_Ram_IrqProcess(void)
{
    uint32 TempData = 0U;
    RamEcc_RegionNameType TempChannelId = RamEcc_ConfigPtr->RamErrorInjectionChannel[RamEcc_RunningData.RunningIndex].RamErrorInjectionChannelId;
    /*Read and check result*/
    EccTst_Lld_Ram_ResultJudged();
    /*Disable Emu channel*/
    EccTst_Lld_Ram_DisableTest(Emu_RegBase, TempChannelId);
    TempData = RamEcc_RunningData.RunningSignature;
    RamEcc_RunningData.RunningSignature = (TempData & 0xFFFF0000U) + \
                                          ((TempData ^ ((uint32)RAMECC_EMUIRQ_SIGNATURE)) & 0x0000FFFFU);
}

#if defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1HA0)
/**
* @brief      The RAM ECC error interrupt ISR.
*
* @table            @service_id:       --  \n
                    @is_reentrant:     false  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \
                    @memory_map:       .mcal_text \n
*
* @return None
* @sw_type sw_detail
*
*/
ECCTST_FUNC static void EccTst_Lld_Ram_EmuIsr(void)
{
    boolean ScifFlag = FALSE;
    boolean NcifFlag = FALSE;
    boolean ScieFlag = FALSE;
    boolean NcieFlag = FALSE;
    RamEcc_RegionNameType TempChannelId = RamEcc_ConfigPtr->RamErrorInjectionChannel[RamEcc_RunningData.RunningIndex].RamErrorInjectionChannelId;
    ScieFlag = EccTst_Lld_Ram_GetSCIE(Emu_RegBase, TempChannelId);
    ScifFlag = EccTst_Lld_Ram_GetSCIF(Emu_RegBase, TempChannelId);
    NcieFlag = EccTst_Lld_Ram_GetNCIE(Emu_RegBase, TempChannelId);
    NcifFlag = EccTst_Lld_Ram_GetNCIF(Emu_RegBase, TempChannelId);
    if (((TRUE == ScifFlag) && (TRUE == ScieFlag)) || \
            ((TRUE == NcieFlag) && (TRUE == NcifFlag)))
    {
        EccTst_Lld_Ram_IrqProcess();
    }
    else
    {
        /*Nothing to do */
    }
}
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1MC0)
/**
* @brief      The RAM ECC error interrupt ISR.
*
* @table            @service_id:       --  \n
                    @is_reentrant:     false  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
*
* @return None
* @sw_type sw_detail
* @private
*/
ECCTST_FUNC static void EccTst_Lld_Ram_EmuSBIsr(void)
{
    boolean ScifFlag = FALSE;
    boolean ScieFlag = FALSE;
    RamEcc_RegionNameType TempChannelId = RamEcc_ConfigPtr->RamErrorInjectionChannel[RamEcc_RunningData.RunningIndex].RamErrorInjectionChannelId;
    ScifFlag = EccTst_Lld_Ram_GetSCIF(Emu_RegBase, TempChannelId);
    ScieFlag = EccTst_Lld_Ram_GetSCIE(Emu_RegBase, TempChannelId);
    if ((TRUE == ScifFlag) && (TRUE == ScieFlag))
    {
        EccTst_Lld_Ram_IrqProcess();
    }
    else
    {
        /*Nothing to do */
    }
}
/**
* @brief      The RAM ECC error interrupt ISR.
*
* @table            @service_id:       --  \n
                    @is_reentrant:     false  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
*
* @return None
* @sw_type sw_detail
* @private
*/
ECCTST_FUNC static void EccTst_Lld_Ram_EmuDBIsr(void)
{
    boolean NcifFlag = FALSE;
    boolean NcieFlag = FALSE;
    RamEcc_RegionNameType TempChannelId = RamEcc_ConfigPtr->RamErrorInjectionChannel[RamEcc_RunningData.RunningIndex].RamErrorInjectionChannelId;
    NcifFlag = EccTst_Lld_Ram_GetNCIF(Emu_RegBase, TempChannelId);
    NcieFlag = EccTst_Lld_Ram_GetNCIE(Emu_RegBase, TempChannelId);
    if ((TRUE == NcifFlag) && (TRUE == NcieFlag))
    {
        EccTst_Lld_Ram_IrqProcess();
    }
    else
    {
        /*Nothing to do */
    }
}
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief Service for RAM ECC test initialization.
 * @details The function shall initialize all RAM ECC Test relevant registers and global variables.
 * @param[in] FlsEccTstConfigPtr  RAM ECC test configuration data pointer
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK: RAM ECC test initialization success
 * @retval E_NOT_OK: RAM ECC test initialization failed
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_154
 */
ECCTST_FUNC Std_ReturnType EccTst_Lld_Ram_Init(const RamEcc_ConfigType *ConfigPtr)
{
    /* Local variable to hold the return value of the function */
    Std_ReturnType ReturnValue = E_NOT_OK;
    if (NULL_PTR == ConfigPtr)
    {
        /* If the configuration pointer is NULL, report an error and return E_NOT_OK */
        ReturnValue = E_NOT_OK;
    }
    else
    {
        RamEcc_ConfigPtr = ConfigPtr;
        EccTst_Lld_Ram_InitRegAndData(Emu_RegBase);
        ReturnValue = E_OK;
    }
    return ReturnValue;
}

/**
 * @brief Service for RAM ECC Test De-Initialization.
 * @details The function de-initializes all RAM ECC Test relevant registers and global variables.
 *
 * @table            @service_id:       --      \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_152
 */
ECCTST_FUNC void EccTst_Lld_Ram_DeInit(void)
{
    /*Reset register and variable*/
    EccTst_Lld_Ram_InitRegAndData(Emu_RegBase);
    RamEcc_ConfigPtr = NULL_PTR;
}

/**
 * @brief Service for RAM ECC test executing.
 * @details The function executes all RAM ECC test items based on the configuration of the EccTst module.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK: RAM ECC test execution success
 * @retval E_NOT_OK: RAM ECC test execution failed
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_156
 */
ECCTST_FUNC Std_ReturnType EccTst_Lld_Ram_Start(void)
{
    /* Local variable to hold the return value of the function */
    Std_ReturnType ReturnValue = E_NOT_OK;
    Platform_IrqHandlerType TempEccIrqPointer;
#if defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1HA0)
    Platform_IrqHandlerType TempEmuIrqHandler;
    uint8 TempEmuIrqPriority;
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1MC0)
    Platform_IrqHandlerType TempEmuSBIrqHandler;
    Platform_IrqHandlerType TempEmuDBIrqHandler;
    uint8 TempEmuSBIrqPriority;
    uint8 TempEmuDBIrqPriority;
#else
#endif
    if (NULL_PTR == RamEcc_ConfigPtr)
    {
        /* If the configuration pointer is NULL, report an error and return E_NOT_OK */
        ReturnValue = E_NOT_OK;
    }
    else
    {
        EccTst_Lld_Ram_SetExpHandler();
        /*Enable EMU interrupt in NVIC*/
#if defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1HA0)
        (void)Platform_SetIrq(EMU_IRQn, TRUE);
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1MC0)
        (void)Platform_SetIrq(EMU_SB_IRQn, TRUE);
        (void)Platform_SetIrq(EMU_DB_IRQn, TRUE);
#else
#endif
        /*Overloading EMU ECC irq handlers*/
#if defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1HA0)
        (void)Platform_InstallIrqHandler(EMU_IRQn, EccTst_Lld_Ram_EmuIsr, &TempEmuIrqHandler);
        (void)Platform_GetIrqPriority(EMU_IRQn, &TempEmuIrqPriority);
        (void)Platform_SetIrqPriority(EMU_IRQn, 0u);
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1MC0)
        (void)Platform_InstallIrqHandler(EMU_DB_IRQn, EccTst_Lld_Ram_EmuDBIsr, &TempEmuDBIrqHandler);
        (void)Platform_InstallIrqHandler(EMU_SB_IRQn, EccTst_Lld_Ram_EmuSBIsr, &TempEmuSBIrqHandler);
        (void)Platform_GetIrqPriority(EMU_DB_IRQn, &TempEmuDBIrqPriority);
        (void)Platform_SetIrqPriority(EMU_DB_IRQn, 0u);
        (void)Platform_GetIrqPriority(EMU_SB_IRQn, &TempEmuSBIrqPriority);
        (void)Platform_SetIrqPriority(EMU_SB_IRQn, 0u);
#else
#endif
        for (uint8 Index = 0; Index < RamEcc_ConfigPtr->RamErrorInjectionChannelCount; ++Index)
        {
            EccTst_Lld_Ram_InjectError(Index);
        }
        /*After all test blocks have been executed, if the test results remain as "NOT_TEST,"
        it indicates that the test was successful, and the result should be updated to "ECCTST_OK."*/
        if (ECCTST_NOT_TEST == RamEcc_RunningData.SumaryResult)
        {
            RamEcc_RunningData.SumaryResult = ECCTST_OK;
        }
        EccTst_Lld_Ram_RecoveredExpHandler();
        /*Restore EMU ECC irq handlers*/
#if defined (CPU_YTM32B1ME0)||defined (CPU_YTM32B1HA0)
        (void)Platform_InstallIrqHandler(EMU_IRQn, TempEmuIrqHandler, &TempEccIrqPointer);
        (void)Platform_SetIrqPriority(EMU_IRQn, TempEmuIrqPriority);
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1MC0)
        (void)Platform_InstallIrqHandler(EMU_DB_IRQn, TempEmuDBIrqHandler, &TempEccIrqPointer);
        (void)Platform_InstallIrqHandler(EMU_SB_IRQn, TempEmuSBIrqHandler, &TempEccIrqPointer);
        (void)Platform_SetIrqPriority(EMU_DB_IRQn, TempEmuDBIrqPriority);
        (void)Platform_SetIrqPriority(EMU_SB_IRQn, TempEmuSBIrqPriority);
#else
#endif
        ReturnValue = E_OK;
    }
    return ReturnValue;
}

/**
 * @brief Service for getting RAM ECC test result.
 * @details The function returns the result of the RAM ECC test.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return FlsEcc_ResultStatusType
 * @retval ECCTST_NOT_TEST: RAM ECC test is not running
 * @retval ECCTST_OK: RAM ECC test result is OK
 * @retval ECCTST_NOT_OK: RAM ECC test result is not OK
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_153
 */
ECCTST_FUNC EccTst_TestResultType EccTst_Lld_Ram_GetInjectRes(void)
{
    return RamEcc_RunningData.SumaryResult;
}

/**
 * @brief Service for getting RAM ECC test error detail.
 * @details The function returns the error details of the RAM ECC Test driver.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @param[out] ErrorDetailsPtr   RAM ECC test error details pointer
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_155
 */
ECCTST_FUNC void EccTst_Lld_Ram_InjectGetErr(RamEcc_TestDetailsType *InjectErrDetail)
{
    if (NULL_PTR == InjectErrDetail)
    {
        /*Nothing to do when the pointer is NULL*/
    }
    else
    {
        *InjectErrDetail = RamEcc_DetailResult;
    }
}


#define ECCTST_STOP_SEC_CODE
#include "EccTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

