/**
 * @file    EccTst_Lld_Can.c
 * @version
 *
 * @brief   AUTOSAR EccTst module interface
 * @details API implementation for ECCTST driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : EccTst
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifdef __cplusplus
extern "C" {
#endif
/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "EccTst_Lld_Can.h"
#include "EccTst_Lld_Reg.h"
#include "Platform.h"

/**
 *  @page misra_violations MISRA-C:2012 violations
 *  PRQA S 0306 Rule-11.4: A conversion should not be performed between a pointer to object and
 *  an integer type
 *
 *  PRQA S 4700: Metric value out fo range
 */

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_LLD_CAN_VENDOR_ID_C                      (0xB4)
#define ECCTST_LLD_CAN_AR_RELEASE_MAJOR_VERSION_C       (4)
#define ECCTST_LLD_CAN_AR_RELEASE_MINOR_VERSION_C       (4)
#define ECCTST_LLD_CAN_AR_RELEASE_REVISION_VERSION_C    (0)
#define ECCTST_LLD_CAN_SW_MAJOR_VERSION_C               (2)
#define ECCTST_LLD_CAN_SW_MINOR_VERSION_C               (0)
#define ECCTST_LLD_CAN_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and EccTst_Lld_Can header file are of the same vendor */
#if (ECCTST_LLD_CAN_VENDOR_ID_C != ECCTST_LLD_CAN_VENDOR_ID)
#error "EccTst_Lld_Can.c and EccTst_Lld_Can.h have different vendor ids"
#endif

/* Check if source file and EccTst_Lld_Can header file are of the same Autosar version */
#if ((ECCTST_LLD_CAN_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_CAN_AR_RELEASE_MAJOR_VERSION) || \
     (ECCTST_LLD_CAN_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_CAN_AR_RELEASE_MINOR_VERSION) || \
     (ECCTST_LLD_CAN_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_CAN_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Can.c and EccTst_Lld_Can.h are different"
#endif

/* Check if source file and EccTst_Lld_Can header file are of the same Software version */
#if ((ECCTST_LLD_CAN_SW_MAJOR_VERSION_C != ECCTST_LLD_CAN_SW_MAJOR_VERSION) || \
     (ECCTST_LLD_CAN_SW_MINOR_VERSION_C != ECCTST_LLD_CAN_SW_MINOR_VERSION) || \
     (ECCTST_LLD_CAN_SW_PATCH_VERSION_C != ECCTST_LLD_CAN_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EccTst_Lld_Can.c and EccTst_Lld_Can.h are different"
#endif

/* Check if source file and ECCTST register header file are of the same vendor */
#if (ECCTST_LLD_CAN_VENDOR_ID_C != ECCTST_LLD_REG_VENDOR_ID)
#error "EccTst_Lld_Can.c and EccTst_Lld_Reg.h have different vendor ids"
#endif

/* Check if source file and ECCTST register header file are of the same Autosar version */
#if ((ECCTST_LLD_CAN_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_REG_AR_REL_MAJOR_VER) || \
     (ECCTST_LLD_CAN_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_REG_AR_REL_MINOR_VER) || \
     (ECCTST_LLD_CAN_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_REG_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Can.c and EccTst_Lld_Reg.h are different"
#endif

/* Check if source file and ECCTST register header file are of the same Software version */
#if ((ECCTST_LLD_CAN_SW_MAJOR_VERSION_C != ECCTST_LLD_REG_SW_MAJOR_VER) || \
     (ECCTST_LLD_CAN_SW_MINOR_VERSION_C != ECCTST_LLD_REG_SW_MINOR_VER) || \
     (ECCTST_LLD_CAN_SW_PATCH_VERSION_C != ECCTST_LLD_REG_SW_PATCH_VER) \
    )
#error "Software Version Numbers of EccTst_Lld_Can.c and EccTst_Lld_Reg.h are different"
#endif
/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#define CANECC_TIMEOUT_COUNT 0xFFU

#define CANECC_RERRAR_CPU_CORRECTABLE_MASK 0x00040000U
#define CANECC_RERRAR_CPU_NONCORRECTABLE_MASK 0x01040000U

#define CANECC_ERRSR_FLAG_MASK 0x000D000DU
#define CANECC_MECR_INTC_MASK  0x0000E000U
#define CANECC_MECR_ENABLE_MASK 0x000DE080U

#define CANECC_TEST_INIT_SIGNATURE  0x5A5AU
#define CANECC_ERRORIRQ_SIGNATURE   0x576EU
#define CANECC_BUSFAULT_SIGNATURE   0xC90DU
#define CANECC_SINGLEBIT_SIGNATURE  0x0D34U
#define CANECC_DOUBLEBIT_SIGNATURE  0xC439U

#define ECCTST_CAN32_WRITE(address, value) ((*(volatile uint32 *)(address)) = (value))
#define ECCTST_CAN32_READ(address) (*(volatile uint32 *)(address))
#define ECCTST_CAN8_WRITE(address, value) ((*(volatile uint8 *)(address)) = (value))
#define ECCTST_CAN8_READ(address) (*(volatile uint8 *)(address))

#define CAN_CONST_BYTENUM_PER_WORD                      (4U)
#define CAN_CONST_BITS_NUM_PER_BYTE                     (8U)
#define CAN_CONST_ECC_DIAG_CODE_9C                      (0x9CU)
#define CAN_CONST_ECC_DIAG_CODE_96                      (0x96U)
#define CAN_CONST_ECC_DIAG_CODE_93                      (0x93U)
#define CAN_CONST_ECC_DIAG_CODE_99                      (0x99U)
#define CAN_CONST_ECC_DIAG_CODE_9A                      (0x9AU)
#define CAN_CONST_ECC_DIAG_CODE_87                      (0x87U)
#define CAN_CONST_ECC_DIAG_CODE_95                      (0x95U)
#define CAN_CONST_ECC_DIAG_CODE_8E                      (0x8EU)
#define CAN_CONST_ECC_DIAG_CODE_81                      (0x81U)
#define CAN_CONST_ECC_DIAG_CODE_82                      (0x82U)
#define CAN_CONST_ECC_DIAG_CODE_84                      (0x84U)
#define CAN_CONST_ECC_DIAG_CODE_88                      (0x88U)
#define CAN_CONST_ECC_DIAG_CODE_90                      (0x90U)
#define CAN_CONST_BIT0_TRUE_OF_A_BYTE                   (0x01U)
#define CAN_CONST_BIT1_TRUE_OF_A_BYTE                   (0x02U)
#define CAN_CONST_BIT2_TRUE_OF_A_BYTE                   (0x04U)
#define CAN_CONST_BIT3_TRUE_OF_A_BYTE                   (0x08U)
#define CAN_CONST_BIT4_TRUE_OF_A_BYTE                   (0x10U)
#define CAN_CONST_BIT5_TRUE_OF_A_BYTE                   (0x20U)
#define CAN_CONST_BIT6_TRUE_OF_A_BYTE                   (0x40U)
#define CAN_CONST_BIT7_TRUE_OF_A_BYTE                   (0x80U)

#define CAN_CONST_OFFADDR_OF_MAILBOX                    (0x00000080U)
#define CAN_CONST_OFFADDR_OF_RXIMR                      (0x00000880U)
#define CAN_CONST_OFFADDR_OF_RXTIMESTAMP0               (0x00000C20U)
#define CAN_CONST_OFFADDR_OF_RXTIMESTAMP1               (0x00000C24U)
#define CAN_CONST_OFFADDR_OF_HRTIMESTAMP                (0x00000C30U)
#define CAN_CONST_OFFADDR_OF_RXFIR                      (0x00000A80U)
#define CAN_CONST_OFFADDR_OF_MASK                       (0x00000AA0U)
#define CAN_CONST_OFFADDR_OF_FDSMB                      (0x00000F28U)
#define CAN_CONST_OFFADDR_OF_SMB                        (0x00000AB0U)
#define CAN_CONST_OFFADDR_OF_TIMESTAMP                  (0x00001000U)
#define CAN_CONST_OFFADDR_OF_ENHFIFO                    (0x00002000U)
#define CAN_CONST_OFFADDR_OF_ENHFIFO_FILTER             (0x00003000U)

#define CAN_CONST_RAM_WORDNUM_OF_RXFIR                  (6U)
#define CAN_CONST_RAM_WORDNUM_OF_MASK                   (4U)
#define CAN_CONST_RAM_WORDNUM_OF_FDSMB                  (54U)
#define CAN_CONST_RAM_WORDNUM_OF_SMB                    (12U)
#define CAN_CONST_RAM_WORDNUM_OF_TIMESTAMP              (512U)

/* 20U * 32U(max Enhanced fifo nums) */
#define CAN_CONST_RAM_WORDNUM_OF_ENHFIFO                (20U * 32U)
#define CAN_CONST_RAM_WORDNUM_OF_ENHFIFO_FILTER         (128U)
/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/


/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   GLOBAL FUNCTIONS PROTOTYPES
==================================================================================================*/
/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/
#define ECCTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "EccTst_MemMap.h"
/**
 * @brief CAN ECC test detail result
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static volatile CanEcc_TestDetailsType CanEcc_DetailResult;
/**
 * @brief CAN ECC test status
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static volatile CanEcc_RunningStatusType CanEcc_RunningData;
/**
 * @brief CAN ECC error handler pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static Platform_IrqHandlerType CanEcc_ErrFuncPointer;
/**
 * @brief CAN ECC busfault handler pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static Platform_IrqHandlerType CanEcc_BusFaultFuncPointer;
/**
 * @brief CAN ECC test ongoing item information
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static CanEcc_InjctionOutputType CanEcc_RunningTestResult;

#define ECCTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "EccTst_MemMap.h"

#define ECCTST_START_SEC_VAR_INIT_32
#include "EccTst_MemMap.h"
/**
 * @brief CAN ECC test configuration data pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static CanEcc_ConfigType  const *CanEcc_ConfigPtr = NULL_PTR;
#define ECCTST_STOP_SEC_VAR_INIT_32
#include "EccTst_MemMap.h"

#define ECCTST_START_SEC_CONST_UNSPECIFIED
#include "EccTst_MemMap.h"
/**
 * @brief CAN moudle register base address
 * @sw_type sw_detail
 * @private
 */
ECCTST_CONST static volatile CAN_Type *const Can_RegBase[CAN_INSTANCE_COUNT] = CAN_BASE_PTRS;
/**
 * @brief CAN moudle IRQ number list
 * @sw_type sw_detail
 * @private
 */
ECCTST_CONST static const IRQn_Type Can_IrqNumber[CAN_INSTANCE_COUNT] = CAN_IRQ_NUM_LIST;
#define ECCTST_STOP_SEC_CONST_UNSPECIFIED
#include "EccTst_MemMap.h"

/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/
/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#define ECCTST_START_SEC_CODE
#include "EccTst_MemMap.h"
/**
 * @brief Get the CAN ECC error interrupt flag
 * @param[in]  CanInstanceId  The CAN instance ID
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     uint32
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_380
 */
ECCTST_FUNC static uint32 EccTst_Lld_Can_GetIntcFlag(uint8 CanInstanceId)
{
    return (Can_RegBase[CanInstanceId]->ERRSR & CANECC_ERRSR_FLAG_MASK);
}

/**
 * @brief Clear the CAN ECC error interrupt flag
 * @param[in]  CanInstanceId  The CAN instance ID
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_379
*/
ECCTST_FUNC static void EccTst_Lld_Can_ClearIntcFlag(uint8 CanInstanceId)
{
    Can_RegBase[CanInstanceId]->ERRSR = CANECC_ERRSR_FLAG_MASK;
}

/**
 * @brief Get the CAN ECC error interrupt enable status
 * @param[in]  CanInstanceId  The CAN instance ID
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     uint32
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_449
 */
ECCTST_FUNC static uint32 EccTst_Lld_Can_GetIntcStatus(uint8 CanInstanceId)
{
    return (Can_RegBase[CanInstanceId]->MECR & CANECC_MECR_INTC_MASK);
}

/**
 * @brief      Disable the CAN ECC error injection.
 * @param[in]  CanInstanceId  The CAN instance ID
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_EccTst_203
 */
ECCTST_FUNC static void EccTst_Lld_Can_DisInjection(uint8 CanInstanceId)
{
    /*
    *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in
    *  code applications, it needs to be converted into a constant
    */
    Can_RegBase[CanInstanceId]->ERRIAR = 0x00000000U;
    Can_RegBase[CanInstanceId]->ERRIDPR = 0x00000000U;
    Can_RegBase[CanInstanceId]->ERRIPPR = 0x00000000U;
}

/**
 * @brief  This function set injection error register and trigger the error.
 * @param[in]  CanInstanceId: CAN instance ID
 * @param[in]  InjPtr: CAN injection error data
 * @param[in]  TestMode: CAN ECC test mode
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return  uint8
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_204
 */
ECCTST_FUNC static uint8 EccTst_Lld_Can_ErrInjection(uint8 CanInstanceId, const CanEcc_InjctionInputType *InjPtr, CanEcc_InjectErrorModeType TestMode)
{
    uint8 RetVal;
    uint32 BaseAddr;
    BaseAddr = (uint32)Can_RegBase[CanInstanceId];
    ECCTST_CAN8_WRITE(BaseAddr + InjPtr->TestAddr, InjPtr->InitData);
    Can_RegBase[CanInstanceId]->ERRIAR = InjPtr->InjAddr;
    if (CANECC_CHKBIT_TEST == TestMode)
    {
        Can_RegBase[CanInstanceId]->ERRIDPR = 0x00000000U;
        Can_RegBase[CanInstanceId]->ERRIPPR = InjPtr->InjData;
    }
    else
    {
        /*Write init data and inject error for corretable or non-corretable test*/
        /*(uint32 InjAddr, uint32 InjData, uint32 InjChecksum)
        *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in
        *  code applications, it needs to be converted into a constant
        */
        Can_RegBase[CanInstanceId]->ERRIDPR = InjPtr->InjData;
        Can_RegBase[CanInstanceId]->ERRIPPR = 0x00000000U;
    }
    /*Read data to trigger the CAN ECC error*/
#if (PLATFORM_ENABLE_CACHE == STD_ON)
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to clear the DCACHE for ECC error injection,
    * and couldn't adhere to M3CM Rule-11.4.
    */
    SCB_InvalidateDCache_by_Addr((uint32 *)(BaseAddr + InjPtr->TestAddr), 32); /*PRQA S 0306*/
#endif
    RetVal = ECCTST_CAN8_READ(BaseAddr + InjPtr->TestAddr);
    return RetVal;
}

/**
 * @brief  This function is used to read the test result from register.
 * @param[in]  CanInstanceId: CAN instance ID
 * @param[out]  OutputPtr: The pointer to the result data
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return  void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_226
 */
ECCTST_FUNC static void EccTst_Lld_Can_ReadResult(uint8 CanInstanceId, CanEcc_InjctionOutputType *OutputPtr)
{
    uint32 BaseAddr;
    BaseAddr = (uint32)Can_RegBase[CanInstanceId];
    OutputPtr->ErrorReg = ECCTST_CAN32_READ(BaseAddr + CAN_ERRSR_OFFSET32);
    OutputPtr->ErrorAddr = ECCTST_CAN32_READ(BaseAddr + CAN_RERRAR_OFFSET32);
    OutputPtr->ErrorData = ECCTST_CAN32_READ(BaseAddr + CAN_RERRDR_OFFSET32);
    OutputPtr->ErrorSyndrome = ECCTST_CAN32_READ(BaseAddr + CAN_RERRSYNR_OFFSET32);
    Can_RegBase[CanInstanceId]->ERRSR = CANECC_ERRSR_FLAG_MASK;
}

/**
 * @brief  This function is used to Initialize the data.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_225
 */
ECCTST_FUNC static void EccTst_Lld_Can_ReadDataInit(void)
{
    CanEcc_RunningTestResult.DataResult = 0U;
    CanEcc_RunningTestResult.ErrorData = 0U;
    CanEcc_RunningTestResult.ErrorAddr = 0U;
    CanEcc_RunningTestResult.ErrorReg = 0U;
    CanEcc_RunningTestResult.ErrorSyndrome = 0U;
}
/**
 * @brief  This function is used to get one bit ECC syndrome.
 * @param[in]  TempData: The data to get ECC syndrome.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_206
 */
ECCTST_FUNC LOCAL_INLINE uint8 EccTst_Lld_Can_GetOneBitEccSyndrome(uint8 TempData)
{
    uint8 TempSyndrome = 0;
    switch (TempData)
    {
        case CAN_CONST_BIT0_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_9C;
            break;
        }
        case CAN_CONST_BIT1_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_96;
            break;
        }
        case CAN_CONST_BIT2_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_93;
            break;
        }
        case CAN_CONST_BIT3_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_99;
            break;
        }
        case CAN_CONST_BIT4_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_9A;
            break;
        }
        case CAN_CONST_BIT5_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_87;
            break;
        }
        case CAN_CONST_BIT6_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_95;
            break;
        }
        case CAN_CONST_BIT7_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_8E;
            break;
        }
        default:
        {
            TempSyndrome = 0x00U;
            break;
        }
    }
    return TempSyndrome;
}
/**
 * @brief   This function is used to get check bit ECC syndrome.
 * @param[in]  TempData: The data to get ECC syndrome.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_205
 */
ECCTST_FUNC LOCAL_INLINE uint8 EccTst_Lld_Can_GetChkBitEccSyndrome(uint8 TempData)
{
    uint8 TempSyndrome = 0;
    switch (TempData)
    {
        case CAN_CONST_BIT0_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_81;
            break;
        }
        case CAN_CONST_BIT1_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_82;
            break;
        }
        case CAN_CONST_BIT2_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_84;
            break;
        }
        case CAN_CONST_BIT3_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_88;
            break;
        }
        case CAN_CONST_BIT4_TRUE_OF_A_BYTE:
        {
            TempSyndrome = CAN_CONST_ECC_DIAG_CODE_90;
            break;
        }
        default:
        {
            TempSyndrome = 0x00U;
            break;
        }
    }
    return TempSyndrome;
}
/**
 * @brief  This function is used to judge the result of the test.
 * @param[in]  TestMode: CAN ECC test mode
 * @param[in]  InjPtr: CAN injection error data
 * @param[in]  OutputPtr: CAN ECC test result data
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_227
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_ResultJudge(CanEcc_InjectErrorModeType TestMode,
        CanEcc_InjctionInputType const *InjPtr,
        CanEcc_InjctionOutputType const *OutputPtr)
{
    Std_ReturnType RetVal = E_NOT_OK;
    uint8 TempSyndrome = 0;
    uint8 TempData = 0;
    uint8 TempData2 = 0;
    TempData = (uint8)(InjPtr->InjData >> (CAN_CONST_BITS_NUM_PER_BYTE * (InjPtr->TestAddr & 0x00000003U)));
    TempData2 = (uint8)(OutputPtr->ErrorSyndrome >> (CAN_CONST_BITS_NUM_PER_BYTE * (InjPtr->TestAddr & 0x00000003U)));
    if (CANECC_DATA_CORRENTABLE_TEST == TestMode)
    {
        TempSyndrome = EccTst_Lld_Can_GetOneBitEccSyndrome(TempData);
        if ((OutputPtr->ErrorAddr == (InjPtr->InjAddr | CANECC_RERRAR_CPU_CORRECTABLE_MASK)) && \
                (CAN_ERRSR_CEIF_MASK == (OutputPtr->ErrorReg & CAN_ERRSR_CEIF_MASK)) && \
                (TempData2 == TempSyndrome))
        {
            RetVal = E_OK;
        }
    }
    else if (CANECC_CHKBIT_TEST == TestMode)
    {
        TempSyndrome = EccTst_Lld_Can_GetChkBitEccSyndrome(TempData);
        if ((OutputPtr->ErrorAddr == (InjPtr->InjAddr | CANECC_RERRAR_CPU_CORRECTABLE_MASK)) && \
                (CAN_ERRSR_CEIF_MASK == (OutputPtr->ErrorReg & CAN_ERRSR_CEIF_MASK)) && \
                (TempData2 == TempSyndrome))
        {
            RetVal = E_OK;
        }
    }
    else
    {
        if ((OutputPtr->ErrorAddr == (InjPtr->InjAddr | CANECC_RERRAR_CPU_NONCORRECTABLE_MASK)) && \
                (CAN_ERRSR_HANCEIF_MASK == (OutputPtr->ErrorReg & CAN_ERRSR_HANCEIF_MASK)) && \
                (OutputPtr->ErrorSyndrome != 0U))
        {
            RetVal = E_OK;
        }
    }
    return RetVal;
}

/**
 * @brief  This function is used to disable the memory ECC
 * @param  CanInstanceId: CAN instance ID
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @retval void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_209
 */
ECCTST_FUNC static void EccTst_Lld_Can_LocDisableMemEcc(uint8 CanInstanceId)
{
    Can_RegBase[CanInstanceId]->CTRL2 |= CAN_CTRL2_ECRWRE_MASK;
    Can_RegBase[CanInstanceId]->MECR  &= ~(CAN_MECR_ECRWRDIS_MASK | CAN_MECR_HANCEI_MSK_MASK | CAN_MECR_FANCEI_MSK_MASK | CAN_MECR_NCEFAFRZ_MASK);
    Can_RegBase[CanInstanceId]->MECR  |= CAN_MECR_ECRWRDIS_MASK;
    Can_RegBase[CanInstanceId]->CTRL2 &= ~CAN_CTRL2_ECRWRE_MASK;
}
/**
 * @brief  This function enables the memory ECC.
 * @param  CanInstanceId: CAN instance ID.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @retval void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_211
 */
ECCTST_FUNC static void EccTst_Lld_Can_LocEnableMemEcc(uint8 CanInstanceId)
{
    Can_RegBase[CanInstanceId]->CTRL2 |= CAN_CTRL2_ECRWRE_MASK;
    Can_RegBase[CanInstanceId]->MECR  &= ~CAN_MECR_ECRWRDIS_MASK;
    Can_RegBase[CanInstanceId]->MECR  = CANECC_MECR_ENABLE_MASK;
    Can_RegBase[CanInstanceId]->MECR  |= CAN_MECR_ECRWRDIS_MASK;
    Can_RegBase[CanInstanceId]->CTRL2 &= ~CAN_CTRL2_ECRWRE_MASK;
}

/**
 * @brief  This function is used to enable or disable the ram access.
 * @param  CanInstanceId: The ID of the CAN instance.
 * @param  CanRamAccess: The status of the ram access.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @retval void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_212
 */
ECCTST_FUNC static void EccTst_Lld_Can_LocEnableRamAccess(uint8 CanInstanceId, boolean Access)
{
    if (TRUE == Access)
    {
        Can_RegBase[CanInstanceId]->CTRL2 |= CAN_CTRL2_WRMFRZ_MASK;
    }
    else
    {
        Can_RegBase[CanInstanceId]->CTRL2 &= ~CAN_CTRL2_WRMFRZ_MASK;
    }
}
/**
 * @brief  This function is used to enable/disable  a CAN instance.
 * @param  CanInstanceId: CAN instance id.
 * @param  Enable: Enable/Disable a CAN instance.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @retval void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_210
 */
ECCTST_FUNC static void EccTst_Lld_Can_LocEnableCanChn(uint8 CanInstanceId, boolean Enable)
{
    if (TRUE == Enable)
    {
        Can_RegBase[CanInstanceId]->MCR &= ~CAN_MCR_MDIS_MASK;
        Can_RegBase[CanInstanceId]->MCR &= ~CAN_MCR_RFEN_MASK;
    }
    else
    {
        uint8 TimeOutCntTemp = 0U;
        /* Reset the FlexCan by software reset */
        Can_RegBase[CanInstanceId]->MCR &= ~(CAN_MCR_FRZ_MASK | CAN_MCR_HALT_MASK);
        Can_RegBase[CanInstanceId]->MCR |= CAN_MCR_SOFTRST_MASK;
        /* Wait for reset cycle to complete */
        while ((CAN_MCR_SOFTRST_MASK == (Can_RegBase[CanInstanceId]->MCR & CAN_MCR_SOFTRST_MASK)) && (TimeOutCntTemp < CANECC_TIMEOUT_COUNT))
        {
            ++TimeOutCntTemp;
        }
        /*disable CAN*/
        Can_RegBase[CanInstanceId]->MCR |= CAN_MCR_MDIS_MASK;
    }
}

/**
 * @brief  This function is used to write the init data to CAN internal ram for CAN ECC test.
 * @param  StartAddress: The start address of the data.
 * @param  WordLength: The length of the data counting in word.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @retval void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_999
 */
ECCTST_FUNC static void EccTst_Lld_Can_WriteInitData(uint32 StartAddress, uint32 WordLength)
{
    uint32 ChnRamAddr32 = StartAddress;
    for (uint32 Addr32Index = 0U; Addr32Index < WordLength; ++Addr32Index)
    {
        ECCTST_CAN32_WRITE(ChnRamAddr32, CANECC_TEST_INIT_SIGNATURE);
        ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
    }
}
/**
 * @brief  This function is used to initialize the CAN ram.
 * @param  CanInstanceId: CAN instance ID.
 * @param  CanInstPtr:   Pointer to the CAN instance.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @retval void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_213
 */
ECCTST_FUNC static void EccTst_Lld_Can_LocInitRam(uint8 CanInstanceId, const CanEcc_CanInstType *CanInstPtr)
{
    uint32 BaseAddr = (uint32)Can_RegBase[CanInstanceId];
    uint32 ChnRamAddr32;
    uint32 MailBoxMaxNum = CanInstPtr->CanMbNumber;
    /* Set CAN_CTRL2.WRMFRZ to enable write Access to all memory positions that require init */
    EccTst_Lld_Can_LocEnableRamAccess(CanInstanceId, TRUE);
    /* Clear the FlexCAN message buffer */
    ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_MAILBOX;
    EccTst_Lld_Can_WriteInitData(ChnRamAddr32, (MailBoxMaxNum * CAN_CONST_BYTENUM_PER_WORD));
    /* Clear RXIMR */
    ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXIMR;
    EccTst_Lld_Can_WriteInitData(ChnRamAddr32, MailBoxMaxNum);
    if (TRUE == CanInstPtr->CanHrTimeStampValid)
    {
        /* Clear time stamp region */
        ECCTST_CAN32_WRITE(BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXTIMESTAMP0, 0x0U);
        ECCTST_CAN32_WRITE(BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXTIMESTAMP1, 0x0U);
        ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_HRTIMESTAMP;
        EccTst_Lld_Can_WriteInitData(ChnRamAddr32, MailBoxMaxNum);
    }
    ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXFIR;
    EccTst_Lld_Can_WriteInitData(ChnRamAddr32, CAN_CONST_RAM_WORDNUM_OF_RXFIR);
    /* Clear Masks*/
    ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_MASK;
    EccTst_Lld_Can_WriteInitData(ChnRamAddr32, CAN_CONST_RAM_WORDNUM_OF_MASK);
    if (TRUE == CanInstPtr->CanFDEnable)
    {
        ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_FDSMB;
        EccTst_Lld_Can_WriteInitData(ChnRamAddr32, CAN_CONST_RAM_WORDNUM_OF_FDSMB);
    }
    else
    {
        /* Clear SMB of Classical CAN */
        ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_SMB;
        EccTst_Lld_Can_WriteInitData(ChnRamAddr32, CAN_CONST_RAM_WORDNUM_OF_SMB);
    }
#if (CAN_ENHANCED_RX_FIFO_TESTED == STD_ON)
    if (TRUE == CanInstPtr->CanEnhancedFifoEnable)
    {
        /* Clear enhance fifo */
        ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_ENHFIFO;
        EccTst_Lld_Can_WriteInitData(ChnRamAddr32, CAN_CONST_RAM_WORDNUM_OF_ENHFIFO);
        /* Clear filter elements of enhance fifo */
        ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_ENHFIFO_FILTER;
        EccTst_Lld_Can_WriteInitData(ChnRamAddr32, CAN_CONST_RAM_WORDNUM_OF_ENHFIFO_FILTER);
    }
#endif
    /* Clear RXMGMASK located in RAM */
    ECCTST_CAN32_WRITE(BaseAddr + CAN_RXMGMASK_OFFSET32, 0x0U);
    /* Clear RX14MASK located in RAM */
    ECCTST_CAN32_WRITE(BaseAddr + CAN_RX14MASK_OFFSET32, 0x0U);
    /* Clear RX15MASK located in RAM */
    ECCTST_CAN32_WRITE(BaseAddr + CAN_RX15MASK_OFFSET32, 0x0U);
    /* Clear RXFGMASK located in RAM */
    ECCTST_CAN32_WRITE(BaseAddr + CAN_RXFGMASK_OFFSET32, 0x0U);
    /* Clear CAN_CTRL2.WRMFRZ  to restrict write Access to memory */
    EccTst_Lld_Can_LocEnableRamAccess(CanInstanceId, FALSE);
}

/**
 * @brief  This function is used to initialize the CAN driver for CAN ECC test.
 * @param  CanInstanceId: CAN instance ID.
 * @param  CanInstPtr: Pointer to the CAN ECC configuration structure.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_202
 */
ECCTST_FUNC static void EccTst_Lld_Can_ChnInit(uint8 ChnHwId, const CanEcc_CanInstType *CanInstPtr)
{
    /*Disable CAN and reset the channel*/
    volatile CAN_Type *const CanRegPtr = Can_RegBase[ChnHwId];
    uint16 TimeOutCntTemp = 0U;
    uint32 Temp = 0U;
    /* Enable CAN module */
    if ((uint32)CAN_MCR_MDIS_MASK == (CanRegPtr->MCR & CAN_MCR_MDIS_MASK))
    {
        CanRegPtr->MCR &= ~CAN_MCR_MDIS_MASK;
    }
    /* Reset the FlexCan by software reset */
    CanRegPtr->MCR &= ~(CAN_MCR_FRZ_MASK | CAN_MCR_HALT_MASK);
    CanRegPtr->MCR |= CAN_MCR_SOFTRST_MASK;
    /* Wait for reset cycle to complete */
    /*In unit test ,the CAN_MCR[SOFTRST] CAN not be cleared by hardware,so the unit can't test all branch*/
    while ((CAN_MCR_SOFTRST_MASK == (CanRegPtr->MCR & CAN_MCR_SOFTRST_MASK)) && (TimeOutCntTemp < CANECC_TIMEOUT_COUNT))
    {
        ++TimeOutCntTemp;
    }
    /*Disable MemEcc*/
    EccTst_Lld_Can_LocDisableMemEcc(ChnHwId);
    /* Set module to freeze mode */
    /* Set to freeze mode */
    CanRegPtr->MCR &= ~CAN_MCR_RFEN_MASK;
    CanRegPtr->MCR |= CAN_MCR_FRZ_MASK | CAN_MCR_HALT_MASK;
    /* Wait for freeze mode acknowledge is set(FRZACK == 1) or timeout */
    while (((uint32)CAN_MCR_FRZACK_MASK != (CanRegPtr->MCR & CAN_MCR_FRZACK_MASK)) && (TimeOutCntTemp < CANECC_TIMEOUT_COUNT))
    {
        ++TimeOutCntTemp;
    }
    EccTst_Lld_Can_LocDisableMemEcc(ChnHwId);
    /*set mode*/
    Temp = CanRegPtr->MCR;
    Temp |= CAN_MCR_SRXDIS_MASK | CAN_MCR_WRNEN_MASK;
    Temp &= ~CAN_MCR_LPRIOEN_MASK;
    if (TRUE == CanInstPtr->CanFDEnable)
    {
        Temp |= CAN_MCR_FDEN_MASK;
    }
    else
    {
        Temp &= ~CAN_MCR_FDEN_MASK;
    }
    Temp |= CAN_MCR_IRMQ_MASK;
    CanRegPtr->MCR  = Temp;
    CanRegPtr->CTRL1 &= ~CAN_CTRL1_LPB_MASK;
    CanRegPtr->CTRL2 |= CAN_CTRL2_RRS_MASK | CAN_CTRL2_EACEN_MASK | CAN_CTRL2_BTE_MASK;
    EccTst_Lld_Can_LocInitRam(ChnHwId, CanInstPtr);
    EccTst_Lld_Can_LocEnableRamAccess(ChnHwId, TRUE);
    EccTst_Lld_Can_LocEnableMemEcc(ChnHwId);
}

/**
 * @brief Check the signature of CAN ECC test
 * @param[in]  Index: The index of the ongoing test region
 * @param[in]  RunningMode: The running mode of the ongoing test region
 * @param[in]  TestRegion: The region name of the ongoing test region
 * @param[in]  TempSignature: The signature of the ongoing test region
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_370
*/
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_SignatureCheck(uint8 HwChannelId, CanEcc_InjectErrorModeType RunningMode, uint32 TempSignature)
{
    Std_ReturnType RetValue = E_NOT_OK;
    uint32 TempInex = (((uint32)HwChannelId) << 16U) + \
                      (((uint32)RunningMode) << 24U);
    if ((CANECC_DATA_CORRENTABLE_TEST == RunningMode) || (CANECC_CHKBIT_TEST == RunningMode))
    {
        TempInex += CANECC_SINGLEBIT_SIGNATURE;
    }
    else
    {
        TempInex += CANECC_DOUBLEBIT_SIGNATURE;
    }
    /*The CAN ECC test excute ECC error irq*/
    if (TempInex == TempSignature)
    {
        RetValue = E_OK;
    }
    return RetValue;
}
/**
 * @brief  This function is used to execute the ECC test for a item.
 * @param  Index: Channel index in CAN ECC configuration table.
 * @param  ParamPtr: Pointer to the configuration structure.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_207
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocBlockTest(uint8 Index, const CanEcc_CanBlockType *ParamPtr)
{
    CanEcc_InjctionInputType CanInjectionData;
    Std_ReturnType TempResult = E_OK;
    uint8 TempChannel = 0;
    uint8 TempAddrL = (uint8)(ParamPtr->InjectAddress & 0x0003U);
    /*Prepare the injection data and addr*/
    CanInjectionData.TestAddr = (uint32)ParamPtr->TestAddress;
    CanInjectionData.InitData = ParamPtr->TestInitialData;
    CanInjectionData.InjAddr = ((uint32)(ParamPtr->InjectAddress)) & 0xFFFCU;
    TempChannel = CanEcc_RunningData.RunningChn;
    EccTst_Lld_Can_ReadDataInit();
    /*Correntable ECC test*/
    CanEcc_RunningData.TestMode = CANECC_DATA_CORRENTABLE_TEST;
    CanEcc_RunningData.RunningSignature = (((uint32)TempChannel) << 16U) + \
                                          (((uint32)CANECC_DATA_CORRENTABLE_TEST) << 24U) + \
                                          CANECC_TEST_INIT_SIGNATURE;
    CanInjectionData.InjData = ((uint32)ParamPtr->InjectionCorrentaleData) << (CAN_CONST_BITS_NUM_PER_BYTE * TempAddrL);
    CanEcc_RunningTestResult.DataResult = EccTst_Lld_Can_ErrInjection(TempChannel, \
                                          &CanInjectionData, CANECC_DATA_CORRENTABLE_TEST);
    TempResult = EccTst_Lld_Can_ResultJudge(CANECC_DATA_CORRENTABLE_TEST, &CanInjectionData, &CanEcc_RunningTestResult);
    TempResult |= EccTst_Lld_Can_SignatureCheck(TempChannel,
                  CANECC_DATA_CORRENTABLE_TEST,
                  CanEcc_RunningData.RunningSignature);
    /*Check the result of the test, and set detail result*/
    if (TempResult != E_OK)
    {
        CanEcc_DetailResult.CanEccResult[Index].CanEccDetailResult = ECCTST_NOT_OK;
        CanEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
        if (CanEcc_ConfigPtr->CanEccTestFailedNotification != NULL_PTR)
        {
            CanEcc_ConfigPtr->CanEccTestFailedNotification(ECCTST_MODULE_ID);
        }
    }
    /*Non-Correntable ECC test*/
    if (ParamPtr->InjectionNonCorrentaleData != 0x00U)
    {
        CanEcc_RunningData.TestMode = CANECC_DATA_NONE_CORRENTABLE_TEST;
        CanEcc_RunningData.RunningSignature = (((uint32)TempChannel) << 16U) + \
                                              (((uint32)CANECC_DATA_NONE_CORRENTABLE_TEST) << 24U) + \
                                              CANECC_TEST_INIT_SIGNATURE;
        CanInjectionData.InjData = ((uint32)ParamPtr->InjectionNonCorrentaleData) << (CAN_CONST_BITS_NUM_PER_BYTE * TempAddrL);
        CanEcc_RunningTestResult.DataResult = EccTst_Lld_Can_ErrInjection(TempChannel, \
                                              &CanInjectionData, CANECC_DATA_NONE_CORRENTABLE_TEST);
        TempResult = EccTst_Lld_Can_ResultJudge(CANECC_DATA_NONE_CORRENTABLE_TEST,
                                                &CanInjectionData,
                                                &CanEcc_RunningTestResult);
        TempResult |= EccTst_Lld_Can_SignatureCheck(TempChannel,
                      CANECC_DATA_NONE_CORRENTABLE_TEST,
                      CanEcc_RunningData.RunningSignature);
        /*Check the result of the test, and update the detail result only when the test result is NOT OK*/
        if (TempResult != E_OK)
        {
            CanEcc_DetailResult.CanEccResult[Index].CanEccDetailResult = ECCTST_NOT_OK;
            CanEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (CanEcc_ConfigPtr->CanEccTestFailedNotification != NULL_PTR)
            {
                CanEcc_ConfigPtr->CanEccTestFailedNotification(ECCTST_MODULE_ID);
            }
        }
    }
    /*Checksum ECC test*/
    if (ParamPtr->InjectionChecksum != 0x00U)
    {
        CanEcc_RunningData.TestMode = CANECC_CHKBIT_TEST;
        CanEcc_RunningData.RunningSignature = (((uint32)TempChannel) << 16U) + \
                                              (((uint32)CANECC_CHKBIT_TEST) << 24U) + \
                                              CANECC_TEST_INIT_SIGNATURE;
        CanInjectionData.InjData = ((uint32)ParamPtr->InjectionChecksum) << (CAN_CONST_BITS_NUM_PER_BYTE * TempAddrL);
        CanEcc_RunningTestResult.DataResult = EccTst_Lld_Can_ErrInjection(TempChannel,
                                              &CanInjectionData,
                                              CANECC_CHKBIT_TEST);
        TempResult = EccTst_Lld_Can_ResultJudge(CANECC_CHKBIT_TEST, &CanInjectionData, &CanEcc_RunningTestResult);
        TempResult |= EccTst_Lld_Can_SignatureCheck(TempChannel,
                      CANECC_CHKBIT_TEST,
                      CanEcc_RunningData.RunningSignature);
        /*Check the result of the test, and update the detail result only when the test result is NOT OK*/
        if (TempResult != E_OK)
        {
            CanEcc_DetailResult.CanEccResult[Index].CanEccDetailResult = ECCTST_NOT_OK;
            CanEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (CanEcc_ConfigPtr->CanEccTestFailedNotification != NULL_PTR)
            {
                CanEcc_ConfigPtr->CanEccTestFailedNotification(ECCTST_MODULE_ID);
            }
        }
    }
    return E_OK;
}
/**
 * @brief  This function is used to clear the ECC error status.
 * @param  Inst: The instance of the CAN controller in configuration table.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_208
 */
ECCTST_FUNC static void EccTst_Lld_Can_LocClearEccErrStatus(uint8 Inst)
{
    CanEcc_DetailResult.CanEccResult[Inst].CanEccDetailResult = ECCTST_NOT_TEST;
    CanEcc_DetailResult.CanEccResult[Inst].CanInstanceId = 0xFFU;
}

/**
 * @brief  This function is used to execute the ECC test for Mbs.
 * @param  Inst: The instance of the CAN controller in configuration table.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_218
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockMbs(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockMessageBuffersPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
/**
 * @brief This function is used to test the message buffers of RxImrs.
 * @param Inst: The instance of the CAN driver.
 * @param TmpParamPtr: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return EccTst_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_221
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockRxImrs(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockRxIndividualMasksPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}

/**
 * @brief  This function is used to test the message buffers of RxFirs.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_219
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockRxFirs(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockRxFilterIdhitsPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
/**
 * @brief  This function is used to test the message buffers of RxGlobalMasks.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_220
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockRxGlobalMasks(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockRxFilterGlobalMasksPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
/**
 * @brief  This function is used to test the message buffers of TxSmb.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_224
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockTxSmb(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockTxSmbPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
/**
 * @brief  This function is used to test the message buffers of RxSmb.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_222
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockRxSmbs(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockRxSmbsPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
/**
 * @brief  This function is used to test the message buffers of RxSmbTimeStamps.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_223
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockRxSmbTimeStamps(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockRxSmbTimeStampsPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
/**
 * @brief  This function is used to test the message buffers of HrTimeStamp.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_217
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockHrTimeStamp(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockHrTimeStampPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
#if (CAN_ENHANCED_RX_FIFO_TESTED == STD_ON)
/**
 * @brief  This function is used to test the message buffers of enhance RxFifo.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_215
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockEnhanceRxFifo(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockEnhancedRxFifoPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
/**
 * @brief  This function is used to test the message buffers of enhance RxFifoFilter.
 * @param  Inst: The instance of the CAN module.
 * @param  EccTstCanEccConfig: The pointer to the ECC test configuration for the test channel.
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_216
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestBlockEnhanceRxFifoFilter(uint8 Inst, \
        const CanEcc_CanInstType *EccTstCanEccConfig)
{
    Std_ReturnType RetVal;
    const CanEcc_CanBlockType *TmpParamPtr = EccTstCanEccConfig->EccTstCanBlockEnhancedRxFifoFiltersPtr;
    RetVal = EccTst_Lld_Can_LocBlockTest(Inst, TmpParamPtr);
    return RetVal;
}
#endif


/**
 * @brief      The bus fault interrupt handler used in CAN ECC test.
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_996
 */
ECCTST_FUNC static void EccTst_Lld_Can_BusFaultIrq(void)
{
    uint32 TempData = 0U;
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to clear BusFault error flags cuased by CAN ECC error injection,
    *  couldn't adhere to M3CM Rule-11.4
    */
    SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk | SCB_CFSR_PRECISERR_Msk; /*PRQA S 0306*/
    EccTst_Lld_Can_DisInjection(CanEcc_RunningData.RunningChn);
    TempData = CanEcc_RunningData.RunningSignature;
    CanEcc_RunningData.RunningSignature = (TempData & 0xFFFF0000U) + \
                                          ((TempData ^ ((uint32)CANECC_BUSFAULT_SIGNATURE)) & 0x0000FFFFU);
}

/**
 * @brief      The ECC error interrupt handler used in CAN ECC test
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_997
 */
ECCTST_FUNC static void EccTst_Lld_Can_EccErrIrq(void)
{
    uint8 TempChan = CanEcc_RunningData.RunningChn;
    uint32 CanEccIntcFlag = FALSE;
    uint32 CanEccIntcStatus = FALSE;
    uint32 TempData = 0U;
    CanEccIntcStatus = EccTst_Lld_Can_GetIntcStatus(TempChan);
    CanEccIntcFlag = EccTst_Lld_Can_GetIntcFlag(TempChan);
    if ((CanEccIntcStatus != 0U) && (CanEccIntcFlag != 0U))
    {
        EccTst_Lld_Can_ReadResult(TempChan, &CanEcc_RunningTestResult);
        EccTst_Lld_Can_ClearIntcFlag(TempChan);
        TempData = CanEcc_RunningData.RunningSignature;
        CanEcc_RunningData.RunningSignature = (TempData & 0xFFFF0000U) + \
                                              ((TempData ^ ((uint32)CANECC_ERRORIRQ_SIGNATURE)) & 0x0000FFFFU);
    }
    else
    {
        /*Nothing to do*/
    }
}

/**
 * @brief This function is used to check the pre-condition before the CAN ECC test.
 * @param CanEccIndex: Index of the CAN instance
 * @param EccTstCanEccConfig : Pointer to the configuration structure
 * @param NotifyPtr: Pointer to the test failed notification
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_998
 */
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_PreCheck(uint8 CanEccIndex, uint8 HwChannelId)
{
    Std_ReturnType RetVal = E_OK;
    uint32 CanEccIntcFlag = FALSE;
    /*[YTSWS_EccTst_300]The EccTst module shall support pre-condition check before the commencement of testing.*/
    CanEccIntcFlag = EccTst_Lld_Can_GetIntcFlag(HwChannelId);
    MCAL_QUA_FAULT_INJECTION(ECCTST_LLD_CAN_ECCERR_FLAG);
    if (CanEccIntcFlag != 0U)
    {
        EccTst_Lld_Can_ClearIntcFlag(HwChannelId);
        CanEccIntcFlag = EccTst_Lld_Can_GetIntcFlag(HwChannelId);
        MCAL_QUA_FAULT_INJECTION(ECCTST_LLD_CAN_ECCERR_FLAG);
        if (CanEccIntcFlag != 0U)
        {
            /*Set the CANx test result to NOT OK*/
            CanEcc_DetailResult.CanEccResult[CanEccIndex].CanInstanceId = HwChannelId;
            CanEcc_DetailResult.CanEccResult[CanEccIndex].CanEccDetailResult = ECCTST_NOT_OK;
            /*Set CAN ECC summer result to not ok*/
            CanEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            RetVal = E_NOT_OK;
        }
    }
    return RetVal;
}

/**
 * @brief This function is used to execute ECC test for a CAN instance.
 * @param CanEccIndex: Index of the CAN instance
 * @param EccTstCanEccConfig : Pointer to the configuration structure
 * @param NotifyPtr: Pointer to the test failed notification
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_214
 */
/* M3CM 4700 VIOLATION:This function governs the execution flow of the CAN ECC test. It needs to make judgments on the test contents and
 * call the corresponding test functions accordingly. Therefore, both STPTH and STCAL cannot follow the standard.
*/
/*PRQA S 4700 ++*/
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Can_LocTestAllBlock(uint8 CanEccIndex,
        const CanEcc_CanInstType *EccTstCanEccConfig,
        const EccTst_FailedNotificationType NotifyPtr) /* PRQA S 4700 --*/
{
    Std_ReturnType RetVal = (Std_ReturnType)E_OK;
    Std_ReturnType IntcFlagCheck = E_OK;
    IntcFlagCheck = EccTst_Lld_Can_PreCheck(CanEccIndex, EccTstCanEccConfig->EccTstCanHwId);
    if (E_OK == IntcFlagCheck)
    {
        Platform_IrqHandlerType TempEccIrqPointer;
        uint8 TempEmuIrqPriority;
        EccTst_Lld_Can_LocClearEccErrStatus(CanEccIndex);
        CanEcc_RunningData.RunningChn = EccTstCanEccConfig->EccTstCanHwId;
        CanEcc_RunningData.RunningIndex = CanEccIndex;
        /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to dispose BusFault cuased by CAN ECC error injection,
        * couldn't adhere to M3CM Rule-11.4
        */
        SCB->SHCSR |= SCB_SHCSR_BUSFAULTENA_Msk; /*PRQA S 0306*/
        SCB->CCR |= SCB_CCR_BFHFNMIGN_Msk; /*PRQA S 0306*/
        (void)Platform_InstallIrqHandler(BusFault_IRQn, EccTst_Lld_Can_BusFaultIrq, &CanEcc_BusFaultFuncPointer);
        /*Enable EMU interrupt in NVIC*/
        (void)Platform_SetIrq(Can_IrqNumber[EccTstCanEccConfig->EccTstCanHwId], TRUE);
        /*Overloading CAN ECC irq handlers*/
        (void)Platform_InstallIrqHandler(Can_IrqNumber[EccTstCanEccConfig->EccTstCanHwId], \
                                         EccTst_Lld_Can_EccErrIrq, &CanEcc_ErrFuncPointer);
        (void)Platform_GetIrqPriority(Can_IrqNumber[EccTstCanEccConfig->EccTstCanHwId], &TempEmuIrqPriority);
        (void)Platform_SetIrqPriority(Can_IrqNumber[EccTstCanEccConfig->EccTstCanHwId], 0u);
        CanEcc_DetailResult.CanEccResult[CanEccIndex].CanInstanceId = EccTstCanEccConfig->EccTstCanHwId;
        CanEcc_DetailResult.CanEccResult[CanEccIndex].CanEccDetailResult = ECCTST_OK;
        if (TRUE == EccTstCanEccConfig->EccTstCanBlockMessageBuffersPtr->InjectionTestEnable)
        {
            RetVal = EccTst_Lld_Can_LocTestBlockMbs(CanEccIndex, EccTstCanEccConfig);
        }
        if (TRUE == EccTstCanEccConfig->EccTstCanBlockRxIndividualMasksPtr->InjectionTestEnable)
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockRxImrs(CanEccIndex, EccTstCanEccConfig);
        }
        if (TRUE == EccTstCanEccConfig->EccTstCanBlockRxFilterIdhitsPtr->InjectionTestEnable)
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockRxFirs(CanEccIndex, EccTstCanEccConfig);
        }
        if (TRUE == EccTstCanEccConfig->EccTstCanBlockRxFilterGlobalMasksPtr->InjectionTestEnable)
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockRxGlobalMasks(CanEccIndex, EccTstCanEccConfig);
        }
        if (TRUE == EccTstCanEccConfig->EccTstCanBlockTxSmbPtr->InjectionTestEnable)
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockTxSmb(CanEccIndex, EccTstCanEccConfig);
        }
        if (TRUE == EccTstCanEccConfig->EccTstCanBlockRxSmbsPtr->InjectionTestEnable)
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockRxSmbs(CanEccIndex, EccTstCanEccConfig);
        }
        if (TRUE == EccTstCanEccConfig->EccTstCanBlockRxSmbTimeStampsPtr->InjectionTestEnable)
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockRxSmbTimeStamps(CanEccIndex, EccTstCanEccConfig);
        }
        if ((TRUE == EccTstCanEccConfig->CanHrTimeStampValid) && \
                (TRUE == EccTstCanEccConfig->EccTstCanBlockHrTimeStampPtr->InjectionTestEnable))
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockHrTimeStamp(CanEccIndex, EccTstCanEccConfig);
        }
#if (CAN_ENHANCED_RX_FIFO_TESTED == STD_ON)
        if ((EccTstCanEccConfig->EccTstCanBlockEnhancedRxFifoPtr != NULL_PTR) && \
                (TRUE == EccTstCanEccConfig->EccTstCanBlockEnhancedRxFifoPtr->InjectionTestEnable))
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockEnhanceRxFifo(CanEccIndex, EccTstCanEccConfig);
        }
        if ((EccTstCanEccConfig->EccTstCanBlockEnhancedRxFifoFiltersPtr != NULL_PTR) && \
                (TRUE == EccTstCanEccConfig->EccTstCanBlockEnhancedRxFifoFiltersPtr->InjectionTestEnable))
        {
            RetVal |= EccTst_Lld_Can_LocTestBlockEnhanceRxFifoFilter(CanEccIndex, EccTstCanEccConfig);
        }
#endif
        (void)Platform_InstallIrqHandler(BusFault_IRQn, CanEcc_BusFaultFuncPointer, &TempEccIrqPointer);
        (void)Platform_InstallIrqHandler(Can_IrqNumber[EccTstCanEccConfig->EccTstCanHwId],
                                         CanEcc_ErrFuncPointer,
                                         &TempEccIrqPointer);
        (void)Platform_SetIrqPriority(Can_IrqNumber[EccTstCanEccConfig->EccTstCanHwId], TempEmuIrqPriority);
    }
    else
    {
        if (NotifyPtr != NULL_PTR)
        {
            NotifyPtr(ECCTST_MODULE_ID);
        }
    }
    return RetVal;
}

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
* @brief      Service for CAN ECC test initialization.
* @details The function shall initialize all CAN ECC Test relevant registers and global variables.
* @param[in]  CanEccTstConfigPtr  Pointer to CAN ECC test configuration.
*
* @table            @service_id:       --  \n
                    @is_reentrant:     false \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
*
* @return     Std_ReturnType
* @retval E_OK: CAN ECC test all instance init success
* @retval E_NOT_OK: CAN ECC test at least one instance init failed
* @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_231
*/
ECCTST_FUNC Std_ReturnType EccTst_Lld_Can_Init(CanEcc_ConfigType const *CanEccTstConfigPtr)
{
    Std_ReturnType Ret = E_OK;
    if (NULL_PTR == CanEccTstConfigPtr)
    {
        /*Return E_NOT_OK if the CanEccTstConfigPtr is NULL*/
        Ret = E_NOT_OK;
    }
    else
    {
        for (uint8 LootCnt = 0; LootCnt < CanEccTstConfigPtr->EccTstCanInstNum; LootCnt++)
        {
            EccTst_Lld_Can_ChnInit(CanEccTstConfigPtr->EccTstCanInstPtr[LootCnt].EccTstCanHwId,
                                   &CanEccTstConfigPtr->EccTstCanInstPtr[LootCnt]);
            CanEcc_DetailResult.CanEccResult[LootCnt].CanInstanceId = 0U;
            CanEcc_DetailResult.CanEccResult[LootCnt].CanEccDetailResult = ECCTST_NOT_TEST;
        }
        CanEcc_RunningData.SumaryResult = ECCTST_NOT_TEST;
        CanEcc_ConfigPtr = CanEccTstConfigPtr;
    }
    return Ret;
}

/**
 * @brief      Main function for CAN ECC test.
 * @details    This function serves as the main entry point for the CAN ECC test.
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_232
 */
ECCTST_FUNC Std_ReturnType EccTst_Lld_Can_Start(void)
{
    Std_ReturnType RetVal = (Std_ReturnType)E_OK;
    if (NULL_PTR == CanEcc_ConfigPtr)
    {
        /*Return E_NOT_OK if the EccTst is not initialized*/
        RetVal = (Std_ReturnType)E_NOT_OK;
    }
    else
    {
        for (uint8 Index = 0U; Index < CanEcc_ConfigPtr->EccTstCanInstNum; ++Index)
        {
            RetVal |= EccTst_Lld_Can_LocTestAllBlock(Index,
                      &CanEcc_ConfigPtr->EccTstCanInstPtr[Index],
                      CanEcc_ConfigPtr->CanEccTestFailedNotification);
        }
        /*After all test blocks have been executed, if the test results remain as "NOT_TEST,"
        it indicates that the test was successful, and the result should be updated to "TEST_OK." */
        if (ECCTST_NOT_TEST == CanEcc_RunningData.SumaryResult)
        {
            CanEcc_RunningData.SumaryResult = ECCTST_OK;
        }
    }
    return RetVal;
}
/**
 * @brief      Get the result of the CAN ECC test.
 * @details    This function retrieves the result status of the CAN ECC test.
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     EccTst_TestResultType
 * @retval ECCTST_NOT_TEST: CAN ECC test has not been executed
 * @retval ECCTST_OK: CAN ECC test result is OK
 * @retval ECCTST_NOT_OK: CAN ECC test result is NOT OK
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_230
 */
ECCTST_FUNC EccTst_TestResultType EccTst_Lld_Can_GetTestResult(void)
{
    return CanEcc_RunningData.SumaryResult;
}

/**
 * @brief      Get detailed error information for the CAN ECC test.
 * @details    This function retrieves detailed error information for the CAN ECC test.
 * @param[out] InjectErrDetail  Pointer to the CAN ECC test error details
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_229
 */
ECCTST_FUNC void EccTst_Lld_Can_GetErrorDetail(CanEcc_TestDetailsType *InjectErrDetail)
{
    if (NULL_PTR == InjectErrDetail)
    {
        /* Nothing to do */
    }
    else
    {
        *InjectErrDetail = CanEcc_DetailResult;
    }
}

/**
 * @brief      De-initialize the CAN ECC test.
 * @details    This function de-initializes all instances of the CAN ECC test.
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_228
 */
ECCTST_FUNC void EccTst_Lld_Can_DeInit(void)
{
    if (NULL_PTR == CanEcc_ConfigPtr)
    {
        /* Nothing to do */
    }
    else
    {
        /* set controllermode to Stopped */
        for (uint8 Index = 0U; Index < CanEcc_ConfigPtr->EccTstCanInstNum; ++Index)
        {
            EccTst_Lld_Can_LocEnableCanChn(CanEcc_ConfigPtr->EccTstCanInstPtr[Index].EccTstCanHwId, FALSE);
            CanEcc_DetailResult.CanEccResult[Index].CanEccDetailResult = ECCTST_NOT_TEST;
            CanEcc_DetailResult.CanEccResult[Index].CanInstanceId = 0U;
        }
        EccTst_Lld_Can_ReadDataInit();
        CanEcc_RunningData.SumaryResult = ECCTST_NOT_TEST;
        CanEcc_ConfigPtr = NULL_PTR;
    }
}


#define ECCTST_STOP_SEC_CODE
#include "EccTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

