/**
 * @file    EccTst_Lld_Ram.h
 * @version
 *
 * @brief   AUTOSAR EccTst module interface
 * @details API implementation for ECCTST driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Adc
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef ECCTST_LLD_RAM_H
#define ECCTST_LLD_RAM_H

#ifdef __cplusplus
extern "C" {
#endif

/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "EccTst_Types.h"


/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_LLD_RAM_VENDOR_ID                      (0xB4)
#define ECCTST_LLD_RAM_AR_RELEASE_MAJOR_VERSION       (4)
#define ECCTST_LLD_RAM_AR_RELEASE_MINOR_VERSION       (4)
#define ECCTST_LLD_RAM_AR_RELEASE_REVISION_VERSION    (0)
#define ECCTST_LLD_RAM_SW_MAJOR_VERSION               (2)
#define ECCTST_LLD_RAM_SW_MINOR_VERSION               (0)
#define ECCTST_LLD_RAM_SW_PATCH_VERSION               (0)


/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ECCTST types header file are of the same vendor */
#if (ECCTST_LLD_RAM_VENDOR_ID != ECCTST_TYPES_VENDOR_ID)
#error "EccTst_Lld_Ram.h and EccTst_Types.h have different vendor ids"
#endif

/* Check if source file and ECCTST types header file are of the same Autosar version */
#if ((ECCTST_LLD_RAM_AR_RELEASE_MAJOR_VERSION != ECCTST_TYPES_AR_REL_MAJOR_VER) || \
     (ECCTST_LLD_RAM_AR_RELEASE_MINOR_VERSION != ECCTST_TYPES_AR_REL_MINOR_VER) || \
     (ECCTST_LLD_RAM_AR_RELEASE_REVISION_VERSION != ECCTST_TYPES_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Ram.h and EccTst_Types.h are different"
#endif

/* Check if source file and ECCTST types header file are of the same software version */
#if ((ECCTST_LLD_RAM_SW_MAJOR_VERSION != ECCTST_TYPES_SW_MAJOR_VER) || \
     (ECCTST_LLD_RAM_SW_MINOR_VERSION != ECCTST_TYPES_SW_MINOR_VER) || \
     (ECCTST_LLD_RAM_SW_PATCH_VERSION != ECCTST_TYPES_SW_PATCH_VER) \
    )
#error "Software Version Numbers of EccTst_Lld_Ram.h and EccTst_Types.h are different"
#endif


/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/** @brief EMU MODULE INSTANCE ID */
#define EMU_INSTANCE_ID ((uint8)0U)

/** @brief EMU MODULE ID */
#define EMU_MODULE_ID ((uint8)255U)

/**
 * @brief           Service ID number for all Emu driver services
 */
/** @{ */
/** @brief API Service ID for EMU_Init */
#define EMU_SID_INIT (0x00U)
/** @brief API Service ID for EMU_DeInit */
#define EMU_SID_DEINIT (0x01U)
/** @brief API Service ID for EMU_GetEccError */
#define EMU_SID_GET_ECC_ERROR (0x02U)
/** @brief API Service ID for EMU_InjectError */
#define EMU_SID_INJECT_ERROR (0x03)

/** @} */
#define EMU_SRAM_L_END_ADDRESS (0x1fffffffU) /**SRAM_L end address */

/** @brief The data mask for one bit error injection into data bits */
#define EMU_INJECT_ONEBIT_DATA_MASK  (0x00010000U)
/** @brief The data mask for one bit error injection into check bits*/
#define EMU_INJECT_ONEBIT_CHK_MASK  (0x02U)
/** @brief The data mask for two bit error injection into data bits*/
#define EMU_INJECT_TWOBIT_DATA_MASK (0x00020002U)
/** @brief The data mask for two bit error injection into check bits*/
#define EMU_INJECT_TWOBIT_CHK_MASK (0x11U)

/** @brief  RAM ECC no data inject */
#define EMU_DATABIT_DISABLED   0x00000000U
/** @brief RAM ECC no chkbit inject */
#define EMU_CHKBIT_DISABLED    0x00U

/*==================================================================================================
 *                                             ENUMS
==================================================================================================*/
/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                        GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                       FUNCTION PROTOTYPES
==================================================================================================*/
/**
 * @brief      Service for RAM ECC test initialization
 * @details    The function shall initialize all RAM ECC Test relevant registers and global variables.
 *
 * @param[in]  ConfigPtr  Pointer to the configuration structure.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK    ECC test initialization is successful.
 * @retval E_NOT_OK ECC test initialization is not successful.
 * @sw_type sw_detail
 *
*/
Std_ReturnType EccTst_Lld_Ram_Init(const RamEcc_ConfigType *ConfigPtr);

/**
 * @brief      Service for RAM ECC Test De-Initialization.
 * @details The function de-initializes all RAM ECC Test relevant registers and global variables.
 *
 * @table            @service_id:       --    \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 */
void EccTst_Lld_Ram_DeInit(void);

/**
 * @brief      Service for executing RAM ECC test.
 * @details The function executes all RAM ECC test items based on the configuration of the EccTst module.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK    RAM ECC test completed.
 * @retval E_NOT_OK RAM ECC test not completed.
 * @sw_type sw_detail
 *
*/
Std_ReturnType EccTst_Lld_Ram_Start(void);

/**
 * @brief     Service for getting RAM ECC test result.
 * @details   The function returns the result of the RAM ECC Test.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     EccTst_TestResultType
 * @retval ECCTST_NOT_TESTED       RAM ECC test not completed.
 * @retval ECCTST_OK               RAM ECC test successful.
 * @retval ECCTST_NOT_OK           RAM ECC test failed.
 * @sw_type sw_detail
 *
 */
EccTst_TestResultType EccTst_Lld_Ram_GetInjectRes(void);

/**
 * @brief      Service for getting RAM ECC test error details.
 * @details The function returns the error details of the RAM ECC test driver.
 * @param[out] InjectErrDetail Pointer to the structure of RAM ECC test result detail data.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 */
void EccTst_Lld_Ram_InjectGetErr(RamEcc_TestDetailsType *InjectErrDetail);

#ifdef __cplusplus
}
#endif

#endif /* ECCTST_H */

/** @} */

