/**
 * @file    EccTst.h
 * @version
 *
 * @brief   AUTOSAR EccTst module interface
 * @details API implementation for ECCTST driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Adc
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef ECCTST_H
#define ECCTST_H

#ifdef __cplusplus
extern "C" {
#endif


/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "EccTst_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_VENDOR_ID                      (0xB4)
#define ECCTST_AR_RELEASE_MAJOR_VERSION       (4)
#define ECCTST_AR_RELEASE_MINOR_VERSION       (4)
#define ECCTST_AR_RELEASE_REVISION_VERSION    (0)
#define ECCTST_SW_MAJOR_VERSION               (2)
#define ECCTST_SW_MINOR_VERSION               (0)
#define ECCTST_SW_PATCH_VERSION               (0)


/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ECCTST types header file are of the same vendor */
#if (ECCTST_VENDOR_ID != ECCTST_TYPES_VENDOR_ID)
#error "EccTst.h and EccTst_Types.h have different vendor ids"
#endif

/* Check if source file and ECCTST types header file are of the same Autosar version */
#if ((ECCTST_AR_RELEASE_MAJOR_VERSION != ECCTST_TYPES_AR_REL_MAJOR_VER) || \
     (ECCTST_AR_RELEASE_MINOR_VERSION != ECCTST_TYPES_AR_REL_MINOR_VER) || \
     (ECCTST_AR_RELEASE_REVISION_VERSION != ECCTST_TYPES_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of EccTst.h and EccTst_Types.h are different"
#endif

/* Check if source file and ECCTST types header file are of the same software version */
#if ((ECCTST_SW_MAJOR_VERSION != ECCTST_TYPES_SW_MAJOR_VER) || \
     (ECCTST_SW_MINOR_VERSION != ECCTST_TYPES_SW_MINOR_VER) || \
     (ECCTST_SW_PATCH_VERSION != ECCTST_TYPES_SW_PATCH_VER) \
    )
#error "Software Version Numbers of EccTst.h and EccTst_Types.h are different"
#endif


/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                            ENUMS
==================================================================================================*/


/*==================================================================================================
 *                                        TYPEDEF STRUCT
==================================================================================================*/


/*==================================================================================================
 *                                             FUNCTION
==================================================================================================*/
/**
 * @brief  Service for the EccTst module initialization.
 * @param[in] ConfigPtr Pointer to the ECC Test configuration data structure.
 *
 * @table            @service_id:       0x01  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return None
 * @sw_type sw_arch
 */
void EccTst_Init(EccTst_ConfigType const *ConfigPtr);

/**
 * @brief Service for executing ECC test.
 *
 * @table            @service_id:       0x03  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return None
 * @sw_type sw_arch
 */
void EccTst_Start(void);
/**
 * @brief  Service for getting result of the EccTst module.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return EccTst_TestResultType
 * @retval ECCTST_NOT_TEST:The ECC Test is not completed.
 * @retval ECCTST_OK: The ECC Test test result is OK.
 * @retval ECCTST_NOT_OK: The ECC Test result is not OK.
 * @sw_type sw_arch
 */
EccTst_TestResultType EccTst_GetResult(void);

/**
 * @brief Service for getting detail results of  the EccTst module.
 * @param[out] EccErrorDetailPtr  Pointer to the structure of EccTst result detail data.
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return None
 * @sw_type sw_arch
 */
void EccTst_GetDetailsResult(EccTst_TestDetailsType *EccErrorDetailPtr);

/**
 * @brief Service for the EccTst module De-Initialization.
 *
 * @table            @service_id:       0x02    \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
 *
 * @return None
 * @sw_type sw_arch
 */
void EccTst_DeInit(void);

#if (ECCTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Service for get the version information of the EccTst module.
 * @param[out]       VersionInfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x06  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return           void
 * @sw_type sw_arch
 */
void EccTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif /* (ECCTST_VERSION_INFO_API == STD_ON) */

#ifdef __cplusplus
}
#endif

#endif /* ECCTST_H */

/** @} */

