/*
 * @file    DmaTst_Types.h
 *==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : DmaTst_Types
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
 *==================================================================================================
 */

#ifndef DMATST_TYPES_H
#define DMATST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @page misra_violations MISRA-C:2012 violations List
 * PRQA S 0750 Rule 11.4: A union type specifier has been defined.
 * PRQA S 0635 Rule 1.2:  A bit-field shall be declared with a type explicitly supported by the C90 standard to avoid security vulnerabilities.
 */
/*==================================================================================================
 *                                            INCLUDE FILES
 *==================================================================================================*/
#include "Std_Types.h"
#include "DmaTst_Cfg.h"
/*==================================================================================================
 *                                HEADER FILE VERSION INFORMATION
 *==================================================================================================*/
#define DMATST_TYPES_VENDOR_ID                    (180)
#define DMATST_TYPES_AR_REL_MAJOR_VER              (4)
#define DMATST_TYPES_AR_REL_MINOR_VER              (4)
#define DMATST_TYPES_AR_REL_REVISION_VER           (0)
#define DMATST_TYPES_SW_MAJOR_VER                  (2)
#define DMATST_TYPES_SW_MINOR_VER                  (0)
#define DMATST_TYPES_SW_PATCH_VER                  (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and DMA configuration header file are of the same vendor */
#if (DMATST_TYPES_VENDOR_ID != DMATST_VENDOR_ID_CFG)
#error "DmaTst.h and DmaTst_Cfg.h have different vendor ids"
#endif

/* Check if source file and DMA configuration header file are of the same Autosar version */
#if ((DMATST_TYPES_AR_REL_MAJOR_VER != DMATST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (DMATST_TYPES_AR_REL_MINOR_VER != DMATST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (DMATST_TYPES_AR_REL_REVISION_VER != DMATST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of DmaTst.h and DmaTst_Cfg.h are different"
#endif

/* Check if source file and DMA configuration header file are of the same software version */
#if ((DMATST_TYPES_SW_MAJOR_VER != DMATST_SW_MAJOR_VERSION_CFG) || \
     (DMATST_TYPES_SW_MINOR_VER != DMATST_SW_MINOR_VERSION_CFG) || \
     (DMATST_TYPES_SW_PATCH_VER != DMATST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of DmaTst.h and DmaTst_Cfg.h are different"
#endif
/*==================================================================================================
 *                                             ENUMS
 *==================================================================================================*/
/**
 * @brief This enumerated type is contains all error test injection definitions
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_593
 */
typedef enum
{
    TRANSFER_CANCEL_ERROR = 0U, /**< Transfer cancellation verification test item */
    SOURCE_ADDRESS_ERROR  = 1U, /**< Source address alignment check test item */
    SOURCE_OFFSET_ERROR   = 2U, /**< Source offset configuration test item */
    DEST_ADDRESS_ERROR    = 3U, /**< Destination address alignment check test item */
    DEST_OFFSET_ERROR     = 4U, /**< Destination offset configuration test item */
    TRANSFER_COUNT_ERROR  = 5U, /**< Transfer count overflow/underflow test item */
    RAM_RELOAD_ERROR      = 6U, /**< RAM reload descriptor validation test item */
    SOURCE_BUS_ERROR      = 7U, /**< Source bus access verification test item */
    DEST_BUS_ERROR        = 8U, /**< Destination bus access verification test item */
    TRANSFER_NORMAL       = 9U  /**< Normal transfer validation test item */
} DmaTst_CheckType;

/**
 * @brief This enumerated type will contain the possible results of all tests
 * @sw_type sw_arch
 *
 * @trace YTSAS_DmaTst_625
 */
typedef enum
{
    DMATST_RESULT_NOT_TESTED    = 0x00U, /**< Test not executed */
    DMATST_RESULT_OK            = 0x01U, /**< Test passed */
    DMATST_RESULT_NOT_OK        = 0x02U  /**< Test failed */
} DmaTst_ResultType;

/**
 * @brief This enumerated type is test result for all items
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_602
 */
typedef enum
{
    DMATST_ITEM_NOT_TESTED      = 0x00U, /**< Items that do not need testing */
    DMATST_ITEM_PASS            = 0x01U, /**< Items test passed */
    DMATST_ITEM_FAILED          = 0x02U  /**< Items test failed */
} DmaTst_ItemResultType;

/**
 * @brief This enumerated type is DmaTst driver status
 * @sw_type sw_arch
 *
 * @trace YTSAS_DmaTst_627
 */
typedef enum
{
    DMATST_UNINIT        = 0x0U, /**< DmaTst driver status is not initialization */
    DMATST_INIT          = 0X1U, /**< DmaTst driver status is initialization */
    DMATST_TESTING       = 0X2U, /**< DmaTst driver status is testing */
    DMATST_TESTED        = 0X3U  /**< DmaTst driver status is tested */
} DmaTst_StatusType;

/**
 * @brief Specifies the DMA channel number
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_1010
 */
typedef enum
{
    DMA_CHANNEL_0 = 0U, /**< DMA channel 0 */
    DMA_CHANNEL_1 = 1U, /**< DMA channel 1 */
    DMA_CHANNEL_2 = 2U, /**< DMA channel 2 */
    DMA_CHANNEL_3 = 3U, /**< DMA channel 3 */
    DMA_CHANNEL_4 = 4U, /**< DMA channel 4 */
    DMA_CHANNEL_5 = 5U, /**< DMA channel 5 */
    DMA_CHANNEL_6 = 6U, /**< DMA channel 6 */
    DMA_CHANNEL_7 = 7U, /**< DMA channel 7 */
    DMA_CHANNEL_8 = 8U, /**< DMA channel 8 */
    DMA_CHANNEL_9 = 9U, /**< DMA channel 9 */
    DMA_CHANNEL_10 = 10U, /**< DMA channel 10 */
    DMA_CHANNEL_11 = 11U, /**< DMA channel 11 */
    DMA_CHANNEL_12 = 12U, /**< DMA channel 12 */
    DMA_CHANNEL_13 = 13U, /**< DMA channel 13 */
    DMA_CHANNEL_14 = 14U, /**< DMA channel 14 */
    DMA_CHANNEL_15 = 15U, /**< DMA channel 15 */
    DMA_CHANNEL_16 = 16U, /**< DMA channel 16 */
    DMA_CHANNEL_17 = 17U, /**< DMA channel 17 */
    DMA_CHANNEL_18 = 18U, /**< DMA channel 18 */
    DMA_CHANNEL_19 = 19U, /**< DMA channel 19 */
    DMA_CHANNEL_20 = 20U, /**< DMA channel 20 */
    DMA_CHANNEL_21 = 21U, /**< DMA channel 21 */
    DMA_CHANNEL_22 = 22U, /**< DMA channel 22 */
    DMA_CHANNEL_23 = 23U, /**< DMA channel 23 */
    DMA_CHANNEL_24 = 24U, /**< DMA channel 24 */
    DMA_CHANNEL_25 = 25U, /**< DMA channel 25 */
    DMA_CHANNEL_26 = 26U, /**< DMA channel 26 */
    DMA_CHANNEL_27 = 27U, /**< DMA channel 27 */
    DMA_CHANNEL_28 = 28U, /**< DMA channel 28 */
    DMA_CHANNEL_29 = 29U, /**< DMA channel 29 */
    DMA_CHANNEL_30 = 30U, /**< DMA channel 30 */
    DMA_CHANNEL_31 = 31U  /**< DMA channel 31 */
} DmaTst_ChannelHwType;
/*==================================================================================================
 *                                             TYPEDEF
 *==================================================================================================*/
/**
 * @brief DMA channel identifier
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_592
 */
typedef uint8 DmaTst_ChannelType;

/**
 * @brief Callback function pointer type
 * @details Defines the prototype for DMA test callback functions
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_591
 */
typedef void (*DmaTst_CallbackType)(DmaTst_ChannelType Channel);

/**
 * @brief DMA transfer address data type
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_607
 */
typedef uint32 DmaTst_TransferAddressType;

/**
 * @brief DMA disable request on transfer completion flag
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_595
 */
typedef boolean DmaTst_DisReqOnCompletionType;

/**
 * @brief Error check function pointer type for DMA channel validation
 * @details Defines prototype for channel-specific error check routines used during test execution.
 *          Each function implements specific error injection scenario validation.
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_597
 */
typedef Std_ReturnType(*DmaTst_ErrorCheckFunctionType)(DmaTst_ChannelType Channel);

/**
 * @brief   Test failed notification function
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_841
 */
typedef void (*DmaTst_FailedNotificationType)(uint16 ModuleId);

/**
 * @brief   This structure contains the configuration set for each DMA channel.
 * @sw_type sw_detail
 */
/**
 * MR12 RULE 11.4 VIOLATION: A union type has been defined. In this specific case,
 * type-punning is required for hardware/protocol data access, thus, couldn't adhere to M3CM Rule-11.4.
 *
 * @trace YTSDS_DmaTst_599
 */
typedef union /*NOSONAR*/
{ /*PRQA S 0750*/
    struct
    {
        /**
         * MR12 RULE 1.2 VIOLATION: A bit-field uses a non-C90 standard type.
         * In this specific case, alignment with hardware register layout or protocol bit-level requirements
         * necessitates the use of an explicitly sized type (e.g., uint16_t),thus,couldn't c adhere to M3CM Rule-1.2.
         */
        uint16 TransferCancelError : 1;        /**< Enable/Disable DMA transfer cancel error */ /*PRQA S 0635*/
        uint16 SourceAddressError : 1;         /**< Enable/Disable DMA source address error */ /*PRQA S 0635*/
        uint16 SourceOffsetError : 1;          /**< Enable/Disable DMA source offset error */ /*PRQA S 0635*/
        uint16 DestinationAddressError : 1;    /**< Enable/Disable DMA destination address error */ /*PRQA S 0635*/
        uint16 DestinationOffsetError : 1;     /**< Enable/Disable DMA destination offset error */ /*PRQA S 0635*/
        uint16 TransferCountError : 1;         /**< Enable/Disable DMA transfer bytes or trigger count error */ /*PRQA S 0635*/
        uint16 RamReloadError : 1;             /**< Enable/Disable DMA ram reload error */ /*PRQA S 0635*/
        uint16 SourceBusError : 1;             /**< Enable/Disable DMA source bus error */ /*PRQA S 0635*/
        uint16 DestinationBusError : 1;        /**< Enable/Disable DMA destination bus error */ /*PRQA S 0635*/
    } InjectType;
    uint16 Value;
} DmaTst_ErrorInjectType;

/**
 * @brief   This structure contains the configuration set for Runtime test.
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_1011
 */
typedef struct
{
    boolean NormalTransferCheck;          /**<  Enable/Disable NormalTransfer check */
    boolean ChannelLinkCheck;             /**<  Enable/Disable ChannelLink check */
    boolean RamReloadCheck;               /**<  Enable/Disable RamReload check */
    uint8   TestChannel;                  /**<  Test Channel  */
    uint8   LinkNextChannel;              /**<  LinkNext Channel  */
} DmaTst_RuntimeConfigType;

/**
 * @brief Typedef for DEM error management implemented by MCAL drivers
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_840
 */
typedef struct
{
    uint8 State;   /**< enabling/disabling the DEM error: Active=STD_ON/ Inactive=STD_OFF */
    uint16 Id ;     /**< ID of DEM error (0 if STD_OFF) */
} DmaTst_DemErrorType;

/**
 * @brief Main DMA test module configuration container
 * @details Aggregates channel number and error check configurations
 * @sw_type sw_arch
 *
 * @trace YTSAS_DmaTst_945
 */
typedef struct
{
    uint8 ChannelCount; /**< Logical channel identifier for testing */
    DmaTst_ErrorInjectType *ErrorTypeConfigPtr; /**< Associated error check configurations */
    DmaTst_FailedNotificationType FailedNotification; /**< Callback function for error handling */
    DmaTst_RuntimeConfigType *RuntimeConfigPtr; /**< Runtime configuration */
#if (DMATST_ENABLE_DEM_REPORT_ERROR_STATUS == STD_ON)
    const DmaTst_DemErrorType DmaTstErrorHardwareCfg;                   /**< @brief DmaTst Driver DEM Error. */
#endif
} DmaTst_ConfigType;

/**
 * @brief Error check result storage structure
 * @details Record test outcomes for 9 error categories and normal transfer validation
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_598
 */
typedef struct
{
    DmaTst_ItemResultType   TransferCancelErrorState;    /**< Transfer cancellation error status */
    DmaTst_ItemResultType   SourceAddressErrorState;     /**< Source address misalignment status */
    DmaTst_ItemResultType   SourceOffsetErrorState;      /**< Source offset configuration error status */
    DmaTst_ItemResultType   DestinationAddressErrorState;/**< Destination address misalignment status */
    DmaTst_ItemResultType   DestinationOffsetErrorState; /**< Destination offset configuration error status */
    DmaTst_ItemResultType   TransferCountErrorState;     /**< Transfer count overflow/underflow status */
    DmaTst_ItemResultType   RamReloadErrorState;         /**< RAM reload descriptor error status */
    DmaTst_ItemResultType   SourceBusErrorState;         /**< Source bus access failure status */
    DmaTst_ItemResultType   DestinationBusErrorState;    /**< Destination bus access failure status */
    DmaTst_ItemResultType   TransferNormalState;         /**< Normal transfer validation status */
} DmaTst_ErrorCheckReturnType;

/**
 * @brief Channel-specific error result container
 * @details Associates physical channel number with complete error state results
 * @sw_type sw_arch
 *
 * @trace YTSAS_DmaTst_618
 */
typedef struct
{
    uint8 Channel;                        /**< Physical channel identifier (0-n) */
    DmaTst_ErrorCheckReturnType ErrorCheckState; /**< Complete error check results for the channel */
} DmaTst_GetErrorType;

/**
 * @brief Channel-specific error result container
 * @details Associates physical channel number with complete error state results
 * @sw_type sw_arch
 *
 * @trace YTSAS_DmaTst_1089
 */
typedef struct
{
    DmaTst_ResultType Result; /**< Test result */
    DmaTst_StatusType Status; /**< Test status */
    DmaTst_GetErrorType ErrorDetail[DMATST_CHANNEL_ERROR_TST_COUNT + DMATST_ENABLE_RUNTIMETEST]; /**< Error check results */
} DmaTst_InformationType;

/**
 * @brief DMA CTS
 * DMA CTS structure, reflects DMA CTS configuration registers.
 * @sw_type sw_detail
 *
 * @trace YTSDS_DmaTst_605
 */
typedef struct
{
    uint32 SADDR;              /*!< Source address */
    sint16 SOFF;               /*!< Source offset */
    uint16 ATTR;               /*!< Transfer attributes */
    uint32 BCNT;               /*!< One loop transfer byte count */
    sint32 STO;                /*!< Source trigger loop offset */
    uint32 DADDR;              /*!< Destination address */
    sint16 DOFF;               /*!< Destination offset */
    uint16 TCNT;               /*!< Trigger loop transfer count */
    sint32 RamLoadAddr;      /*!< RAM load address */
    uint16 CSR;                /*!< Control/status register */
    uint16 TCNTRV;             /*!< Trigger loop counter value */
} DmaTst_SoftwareCtsType;

#ifdef __cplusplus
}
#endif
#endif




