/*
* @file    DmaTst.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : DmaTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifndef DMATST_H
#define DMATST_H

#ifdef __cplusplus
extern "C" {
#endif
/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "DmaTst_Types.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define DMATST_VENDOR_ID                            (180)
#define DMATST_AR_RELEASE_MAJOR_VERSION              (4)
#define DMATST_AR_RELEASE_MINOR_VERSION              (4)
#define DMATST_AR_RELEASE_REVISION_VERSION           (0)
#define DMATST_SW_MAJOR_VERSION                      (2)
#define DMATST_SW_MINOR_VERSION                      (0)
#define DMATST_SW_PATCH_VERSION                      (0)
/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/**
 * @file     DmaTst.h
 */
/* Check if source file and DMATST_TYPES header file are of the same vendor */
#if (DMATST_VENDOR_ID != DMATST_TYPES_VENDOR_ID)
#error "DmaTst.h and DmaTst_Types.h have different vendor ids"
#endif

/* Check if source file and DMATST_TYPES header file are of the same Autosar version */
#if ((DMATST_AR_RELEASE_MAJOR_VERSION != DMATST_TYPES_AR_REL_MAJOR_VER) || \
     (DMATST_AR_RELEASE_MAJOR_VERSION != DMATST_TYPES_AR_REL_MINOR_VER) || \
     (DMATST_AR_RELEASE_REVISION_VERSION != DMATST_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of DmaTst.h and DmaTst_Types.h are different"
#endif

/* Check if source file and DMATST_TYPES header file are of the same software version */
#if ((DMATST_SW_MAJOR_VERSION != DMATST_TYPES_SW_MAJOR_VER) || \
     (DMATST_SW_MINOR_VERSION != DMATST_TYPES_SW_MINOR_VER) || \
     (DMATST_SW_PATCH_VERSION != DMATST_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of DmaTst.h and DmaTst_Types.h are different"
#endif
/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

/** @brief Instance ID for DmaTst module */
#define DMATST_INSTANCE_ID ((uint8)0U)
/** @brief Module ID for DmaTst module */
#define DMATST_MODULE_ID   ((uint16)516U)

#if (DMATST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup        DmaTst_DetErrorGroup
 * @brief           DmaTst development errors
 * @sw_type         sw_detail
 * @{
 *
 * @trace YTSDS_DmaTst_624
 */
/** @brief Development error: Invalid state */
#define DMATST_E_INVALID_STATE                 ((uint8)0x10)
/** @brief Development error: Initialization failed */
#define DMATST_E_INIT_FAILED                   ((uint8)0x11)
/** @brief Development error: Channel not initialized */
#define DMATST_E_NOT_INIT_FAILED               ((uint8)0x12)
/** @brief Development error: Invalid parameter pointer */
#define DMATST_E_PARAM_POINTER                 ((uint8)0x13)
/** @brief Development error: Module uninitialized */
#define DMATST_E_UNINIT                        ((uint8)0x05)
/** @} */
#endif

/**
 * @defgroup        DmaTst_ApiGroup
 * @brief           DmaTst service ID
 * @sw_type         sw_detail
 * @{
 *
 * @trace YTSDS_DmaTst_623
 */
/** @brief Service ID for DmaTst_Init */
#define DMATST_SID_INIT                        (0x01U)
/** @brief Service ID for DmaTst_DeInit */
#define DMATST_SID_DEINIT                      (0x02U)
/** @brief Service ID for DmaTst_Start */
#define DMATST_SID_START                       (0x03U)
/** @brief Service ID for DmaTst_GetResult */
#define DMATST_SID_GET_RESULT                  (0x04U)
/** @brief Service ID for DmaTst_GetResultDetail */
#define DMATST_SID_GET_DETAILS_RESULT           (0x05U)
/** @brief Service ID for DmaTst_GetVersionInfo */
#define DMATST_SID_GET_VERSION_INFO            (0x06U)
/** @brief Service ID for DmaTst_MainFunction */
#define DMATST_SID_MAINFUNCTION                 (0x07U)
/** @} */

/**
 * @defgroup       DmaTst_ErrorStatusRegisterMasks
 * @brief          DMA channel error status register bit masks
 * @sw_type        sw_detail
 * @{
 *
 * @trace YTSDS_DmaTst_625
 */
/** @brief DMA channel error status register, ECX bit */
#define DMATST_ERS_ECX_TRIGGER                 (0x40000U)
/** @brief DMA channel error status register, SAE bit */
#define DMATST_ERS_SAE_TRIGGER                 (0x08000U)
/** @brief DMA channel error status register, SOE bit */
#define DMATST_ERS_SOE_TRIGGER                 (0x04000U)
/** @brief DMA channel error status register, DAE bit */
#define DMATST_ERS_DAE_TRIGGER                 (0x02000U)
/** @brief DMA channel error status register, DOE bit */
#define DMATST_ERS_DOE_TRIGGER                 (0x01000U)
/** @brief DMA channel error status register, BTCE bit */
#define DMATST_ERS_BTCE_TRIGGER                (0x00800U)
/** @brief DMA channel error status register, RLDE bit */
#define DMATST_ERS_RLDE_TRIGGER                (0x00400U)
/** @brief DMA channel error status register, SBE bit */
#define DMATST_ERS_SBE_TRIGGER                 (0x00200U)
/** @brief DMA channel error status register, DBE bit */
#define DMATST_ERS_DBE_TRIGGER                 (0x00100U)
/** @} */

/** @brief DMA channel error inject count */
#define DMATST_ERROR_INJECT_COUNT              (0x09U)
/** @brief DMA channel invalid source address */
#define DMATST_INVALIDED_SRC_ADDR              (uint32)0x50000000U
/** @brief DMA channel invalid destination address */
#define DMATST_INVALIDED_DEST_ADDR             (uint32)0x50000000U
/** @brief CTS offset used for RAM reload address alignment test */
#define DMATST_CTS_ADDR_OFFSET                 (uint32)0x4U
/** @brief Total number of DMA test scenarios */
#define DMATST_TOTAL_SCENARIOS                 (DMATST_CHANNEL_ERROR_TST_COUNT + DMATST_ENABLE_RUNTIMETEST)
/** @brief Index for normal transfer test */
#define DMATST_NORMAL_IDX                      0U
/** @brief Index for channel link test */
#define DMATST_CHANNEL_LINK_IDX                1U
/** @brief Index for RAM reload test */
#define DMATST_RAMRELOAD_IDX                   2U
/** @brief Size of the test array */
#define DAMTST_TEST_ARRAY_SIZE                 3U
/*==================================================================================================
 *                                          VARIATES
==================================================================================================*/

/*==================================================================================================
 *                                   EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
 *                                   GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief            This function initializes the run-time state structure to provide the DMA channel allocation
 *                   release, protect, and track the state for channels. This function also resets the DMA modules,
 *                   initializes the module to user-defined settings and default settings.
 * @details
 * @table            @service_id:       0x01        \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_arch
 * @param[in]        ConfigPtr Pointer to DmaTst module configuration structure.
 * @return           void
 *
 * @trace YTSAS_DmaTst_620
 */
void DmaTst_Init(const DmaTst_ConfigType *ConfigPtr);

/**
 * @brief            De-initializes the DMA Test module and resets all configurations.
 * @details          Clears configuration pointers, resets module status to uninitialized state, and invalidates test results.
 *                   Performs error detection through the Development Error Tracer (DET) if invoked when already uninitialized.
 * @table            @service_id:       0x02        \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_arch
 * @return           void
 *
 * @trace YTSAS_DmaTst_611
 */
void DmaTst_DeInit(void);

/**
 * @brief            Executes comprehensive DMA error detection test sequence.
 * @details          Performs channel initialization verification, error injection tests (source/destination address errors,
 *                   transfer count errors, etc.), and timeout monitoring. Manages error callback registration and DMA channel
 *                   state transitions according to AUTOSAR specifications.
 * @table            @service_id:       0x03        \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_arch
 * @return           Std_ReturnType
 * @retval E_OK      Test sequence completed (aggregate result available via GetResult)
 * @retval E_NOT_OK  Invalid module state or configuration error
 *
 * @trace YTSAS_DmaTst_947
 */
Std_ReturnType DmaTst_Start(void);

/**
 * @brief            Retrieves aggregated test execution status.
 * @details          Provides final test verdict combining all channel test results. Maintains state until next test cycle
 *                   initialization.
 * @table            @service_id:       0x04        \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_arch
 * @return           DmaTst_ResultType
 * @retval DMATST_RESULT_NOT_TESTED  Pre-test state
 * @retval DMATST_RESULT_OK          All configured tests passed
 * @retval DMATST_RESULT_NOT_OK      One or more tests failed
 *
 * @trace YTSAS_DmaTst_619
 */
DmaTst_ResultType DmaTst_GetResult(void);

/**
 * @brief            Retrieves per-channel detailed error status.
 * @details          Populates test result structure with individual error type statuses (9 error categories per channel)
 *                   including address errors, transfer count errors, and bus faults. Valid only after test execution.
 * @table            @service_id:       0x05        \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_arch
 * @param[out]       DmaTstGetError  Pointer to result array (size DMATST_CHANNEL_ERROR_TST_COUNT)
 * @return           void
 *
 * @trace YTSAS_DmaTst_986
 */
void DmaTst_GetDetailsResult(DmaTst_GetErrorType *DmaTstGetError);

#if DMATST_ENABLE_RUNTIMETEST == STD_ON
/**
 * @brief           Main function for the DMA test.
 * @details         In Runtime, this channel is tested by calling the function DmaTst_MainFunction(), focusing only on the normal transfer test.
 * @table           @service_id:       0x07        \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type         sw_arch
 * @return          Std_ReturnType
 * @retval E_OK     Test sequence completed (aggregate result available via GetResult)
 * @retval E_NOT_OK Invalid module state or configuration error
 *
 * @trace YTSAS_DmaTst_1137
 */
Std_ReturnType DmaTst_MainFunction(void);
#endif

#if (DMATST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @details
 * @table           @service_id:       0x06        \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 *
 * @param[out]       versioninfo Pointer to where to store version information of this module
 * @return           void
 *
 */
void DmaTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif /* DMATST_VERSION_INFO_API == STD_ON */

#ifdef __cplusplus
}
#endif

#endif /* End of file DmaTst.h */

