/**
 * @file    CrcTst.c
 * @brief   Safety library CrcTst module interface
 * @details API implementation for the CRC Test (CRCTST) driver.
 * @note    This module provides functionality for CRC testing in both foreground and background modes.
 */
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 SafLib
*   Platform             : ARM
*   Peripheral           : CrcTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 2986 Rule 2.2: There shall be no dead code. The operation here is redundant as the result's value is always that of the left-hand operand.
 */
/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/
#include "Compiler.h"
#if defined(CPU_YTM32B1HA0)
#include "CrcTst_Lld_Pcrc.h"
#else
#include "CrcTst_Lld_Crc.h"
#endif
#include "CrcTst.h"
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif
#if (CRCTST_DEM_EVENT_REPORT == STD_ON)
#include "Dem.h"
#endif
#include "SchM_CrcTst.h"

/*==================================================================================================
*                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CRCTST_VENDOR_ID_C                      (180)
#define CRCTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CRCTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define CRCTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define CRCTST_SW_MAJOR_VERSION_C               (2)
#define CRCTST_SW_MINOR_VERSION_C               (0)
#define CRCTST_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
*                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CRCTST header file are of the same vendor */
#if (CRCTST_VENDOR_ID_C != CRCTST_VENDOR_ID)
#error "CrcTst.c and CrcTst.h have different vendor ids"
#endif

/* Check if source file and CRCTST header file are of the same Autosar version */
#if (( CRCTST_AR_RELEASE_MAJOR_VERSION_C != CRCTST_AR_RELEASE_MAJOR_VERSION) || \
      ( CRCTST_AR_RELEASE_MINOR_VERSION_C != CRCTST_AR_RELEASE_MINOR_VERSION) || \
      ( CRCTST_AR_RELEASE_REVISION_VERSION_C != CRCTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CrcTst.c and CrcTst.h are different"
#endif

/* Check if source file and CRCTST header file are of the same Software version */
#if ((CRCTST_SW_MAJOR_VERSION_C != CRCTST_SW_MAJOR_VERSION) || \
     (CRCTST_SW_MINOR_VERSION_C != CRCTST_SW_MINOR_VERSION) || \
     (CRCTST_SW_PATCH_VERSION_C != CRCTST_SW_PATCH_VERSION))
#error "Software Version Numbers of CrcTst.c and CrcTst.h are different"
#endif

#if defined(CPU_YTM32B1HA0)

/* Check if source file and CrcTst_Lld_Pcrc.h file are of the same vendor */
#if (CRCTST_VENDOR_ID_C != CRCTST_LLD_PCRC_VENDOR_ID)
#error "CrcTst.c and CrcTst_Lld_Pcrc.h have different vendor ids"
#endif

/* Check if source file and CrcTst_Lld_Pcrc file are of the same Autosar version */
#if (( CRCTST_AR_RELEASE_MAJOR_VERSION_C != CRCTST_LLD_PCRC_AR_RELEASE_MAJOR_VERSION) || \
      ( CRCTST_AR_RELEASE_MINOR_VERSION_C != CRCTST_LLD_PCRC_AR_RELEASE_MINOR_VERSION) || \
      ( CRCTST_AR_RELEASE_REVISION_VERSION_C != CRCTST_LLD_PCRC_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CrcTst.c and CrcTst_Lld_Pcrc.h are different"
#endif

/* Check if source file and CrcTst_Lld_Pcrc.h file are of the same Software version */
#if ((CRCTST_SW_MAJOR_VERSION_C != CRCTST_LLD_PCRC_SW_MAJOR_VERSION) || \
     (CRCTST_SW_MINOR_VERSION_C != CRCTST_LLD_PCRC_SW_MINOR_VERSION) || \
     (CRCTST_SW_PATCH_VERSION_C != CRCTST_LLD_PCRC_SW_PATCH_VERSION))
#error "Software Version Numbers of CrcTst.c and CrcTst_Lld_Pcrc.h are different"
#endif

#else
/* Check if source file and CrcTst_Lld_Crc.h file are of the same vendor */
#if (CRCTST_VENDOR_ID_C != CRCTST_LLD_CRC_VENDOR_ID)
#error "CrcTst.c and CrcTst_Lld_Crc.h have different vendor ids"
#endif

/* Check if source file and CrcTst_Lld_Crc file are of the same Autosar version */
#if (( CRCTST_AR_RELEASE_MAJOR_VERSION_C != CRCTST_LLD_CRC_AR_RELEASE_MAJOR_VERSION) || \
      ( CRCTST_AR_RELEASE_MINOR_VERSION_C != CRCTST_LLD_CRC_AR_RELEASE_MINOR_VERSION) || \
      ( CRCTST_AR_RELEASE_REVISION_VERSION_C != CRCTST_LLD_CRC_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CrcTst.c and CrcTst_Lld_Crc.h are different"
#endif

/* Check if source file and CrcTst_Lld_Crc.h file are of the same Software version */
#if ((CRCTST_SW_MAJOR_VERSION_C != CRCTST_LLD_CRC_SW_MAJOR_VERSION) || \
     (CRCTST_SW_MINOR_VERSION_C != CRCTST_LLD_CRC_SW_MINOR_VERSION) || \
     (CRCTST_SW_PATCH_VERSION_C != CRCTST_LLD_CRC_SW_PATCH_VERSION))
#error "Software Version Numbers of CrcTst.c and CrcTst_Lld_Crc.h are different"
#endif
#endif

/* Check if source file and CRCTST Schm header file are of the same vendor */
#if (CRCTST_VENDOR_ID_C != SCHM_CRCTST_VENDOR_ID)
#error "CrcTst.c and SchM_CrcTst.h have different vendor ids"
#endif

/* Check if source file and SchM_CrcTst file are of the same Autosar version */
#if (( CRCTST_AR_RELEASE_MAJOR_VERSION_C != SCHM_CRCTST_AR_RELEASE_MAJOR_VERSION) || \
      ( CRCTST_AR_RELEASE_MINOR_VERSION_C != SCHM_CRCTST_AR_RELEASE_MINOR_VERSION) || \
      ( CRCTST_AR_RELEASE_REVISION_VERSION_C != SCHM_CRCTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CrcTst.c and SchM_CrcTst.h are different"
#endif

/* Check if source file and CRCTST header file are of the same Software version */
#if ((CRCTST_SW_MAJOR_VERSION_C != SCHM_CRCTST_SW_MAJOR_VERSION) || \
     (CRCTST_SW_MINOR_VERSION_C != SCHM_CRCTST_SW_MINOR_VERSION) || \
     (CRCTST_SW_PATCH_VERSION_C != SCHM_CRCTST_SW_PATCH_VERSION))
#error "Software Version Numbers of CrcTst.c and SchM_CrcTst.h are different"
#endif

/*==================================================================================================
*                                       LOCAL MACROS
==================================================================================================*/
/**
 * @defgroup CRCPolynomial
 * @brief    CRC polynomial for CRC calculation.
 * @sw_type  sw_detail
 *
 * @trace YTSDS_CrcTst_657
 * @{
 */
/** @brief CRC polynomial for 8-bit CRC calculation. */
#define CRCTST_POLYNOMIAL_8                (0x1DU)
/** @brief CRC polynomial for 16-bit CRC calculation. */
#define CRCTST_POLYNOMIAL_16               (0x1021U)
/** @brief CRC polynomial for 32-bit CRC calculation. */
#define CRCTST_POLYNOMIAL_32               (0x04C11DB7U)
/** @} */

/** @brief Number of bytes in a 32-bit value. */
#define CRCTST_BYTESOFU32                  (0x04U)

/** @brief Index for the first byte in a 32-bit value. */
#define CRCTST_INDEX_FIRST                 (0x01U)
/** @brief Index for the second byte in a 32-bit value. */
#define CRCTST_INDEX_SECOND                (0x02U)
/** @brief Index for the third byte in a 32-bit value. */
#define CRCTST_INDEX_THIRD                 (0x03U)

#if defined(CPU_YTM32B1ME0) || defined(CPU_YTM32B1MD1) || defined(CPU_YTM32B1MC0)
/** @brief Hardware mode for 16-bit CRC calculation. */
#define CRCTST_HARDMODE_16BITS             (0x01U)
/** @brief Hardware mode for 32-bit CRC calculation. */
#define CRCTST_HARDMODE_32BITS             (0x02U)
#endif

/**
 * @defgroup CRCLenth
 * @brief    CRC lenth for CRC8 CRC16 CRC32
 * @sw_type  sw_detail
 *
 * @trace YTSDS_CrcTst_654
 * @{
 */
/** @brief Length of 8-bit CRC. */
#define CRCTST_LENGTH_8BITS                (0x8U)
/** @brief Length of 16-bit CRC. */
#define CRCTST_LENGTH_16BITS               (0x10U)
/** @brief Length of 32-bit CRC. */
#define CRCTST_LENGTH_32BITS               (0x20U)
/** @} */

#define CRCTST_TEST_DATA_GROUP_NUM         (0x07U)

#define CRCTST_VALUE_24                    (24U)
#define CRCTST_VALUE_16                    (16U)
#define CRCTST_VALUE_8                     (8U)

#define CRCTST_VALUE_4                     (4U)

#define CRCTST_CRC8_INIT_VALUE             (0xFFU)
#define CRCTST_CRC16_INIT_VALUE            (0xFFFFU)
#define CRCTST_CRC32_INIT_VALUE            (0xFFFFFFFFU)
#define CRCTST_CRC8_XOR_VALUE              (0xFFU)
#define CRCTST_CRC16_XOR_VALUE             (0x0000U)
#define CRCTST_CRC32_XOR_VALUE             (0xFFFFFFFFU)

/*==================================================================================================
*                                      LOCAL CONSTANTS
==================================================================================================*/
#define CRCTST_START_SEC_CONST_8
#include "CrcTst_MemMap.h"

#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
/**
 * @brief   Test data for CRC functionality testing.
 * @details This array contains predefined test data groups used for validating CRC calculations.
 *          The data is derived from AUTOSARSAR_SWS_CRCLibrary document.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_814
 */
CRCTST_CONST static const uint8 CrcTst_TestData[CRCTST_TEST_DATA_GROUP_NUM][9] =
{
    {0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U},  /* Test data group 1 (4B data + 5B padding) */
    {0xF2U, 0x01U, 0x83U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U},  /* Test data group 2 (3B data + 6B padding) */
    {0x0FU, 0xAAU, 0x00U, 0x55U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U},  /* Test data group 3 (4B data + 5B padding) */
    {0x00U, 0xFFU, 0x55U, 0x11U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U},  /* Test data group 4 (4B data + 5B padding) */
    {0x33U, 0x22U, 0x55U, 0xAAU, 0xBBU, 0xCCU, 0xDDU, 0xEEU, 0xFFU},  /* Test data group 5 (9B data, full length) */
    {0x92U, 0x6BU, 0x55U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U},  /* Test data group 6 (3B data + 6B padding) */
    {0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U}   /* Test data group 7 (4B data + 5B padding) */
};

/**
 * @brief   Length of each test data group.
 * @details This array specifies the number of bytes in each test data group.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_815
 */
CRCTST_CONST static const uint8 CrcTst_TestDataLength[CRCTST_TEST_DATA_GROUP_NUM] = {4U, 3U, 4U, 4U, 9U, 3U, 4U};
#endif

#if (CRCTST_CRC8_TEST_ENABLE == STD_ON)
/**
 * @brief   Expected CRC8 results for the test data groups.
 * @details This array contains the expected CRC8 values for each test data group.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_812
 */
CRCTST_CONST static const uint8 CrcTst_ExpectedCrc8Results[CRCTST_TEST_DATA_GROUP_NUM] =
{
    0x59U, 0x37U, 0x79U, 0xB8U, 0xCBU, 0x8CU, 0x74U
};
#endif

#define CRCTST_STOP_SEC_CONST_8
#include "CrcTst_MemMap.h"

#define CRCTST_START_SEC_CONST_16
#include "CrcTst_MemMap.h"

#if (CRCTST_CRC16_TEST_ENABLE == STD_ON)
/**
 * @brief   Expected CRC16 results for the test data groups.
 * @details This array contains the expected CRC16 values for each test data group.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_810
 */
CRCTST_CONST static const uint16 CrcTst_ExpectedCrc16Results[CRCTST_TEST_DATA_GROUP_NUM] =
{
    0x84C0, 0xD374, 0x2023, 0xB8F9, 0xF53F, 0x0745, 0x1D0F
};
#endif

#define CRCTST_STOP_SEC_CONST_16
#include "CrcTst_MemMap.h"

#define CRCTST_START_SEC_CONST_32
#include "CrcTst_MemMap.h"

#if (CRCTST_CRC32_TEST_ENABLE == STD_ON)
/**
 * @brief   Expected CRC32 results for the test data groups.
 * @details This array contains the expected CRC32 values for each test data group.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_811
 */
CRCTST_CONST static const uint32 CrcTst_ExpectedCrc32Results[CRCTST_TEST_DATA_GROUP_NUM] =
{
    0x2144DF1CU, 0x24AB9D77U, 0xB6C9B287U, 0x32A06212U, 0xB0AE863DU, 0x9CDEA29BU, 0xFFFFFFFFU
};
#endif

#define CRCTST_STOP_SEC_CONST_32
#include "CrcTst_MemMap.h"
/*==================================================================================================
*                                      GLOBAL VARIABLES
==================================================================================================*/
#define CRCTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "CrcTst_MemMap.h"
#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
CRCTST_VAR static CrcTst_TestResultType CrcTst_BgndResults[CRCTST_REGISTER_INTEGRITY_TEST_NUM];
#endif

#define CRCTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "CrcTst_MemMap.h"

#define CRCTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "CrcTst_MemMap.h"

#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
CRCTST_VAR static CrcTst_CrcFuncTestDetailsType CrcTst_CrcFuncTestDetails =
{
    .Crc8Result = CRCTST_E_NOT_TESTED,
    .Crc16Result = CRCTST_E_NOT_TESTED,
    .Crc32Result = CRCTST_E_NOT_TESTED,
};
#endif

/**
 * @brief        CRC Test Driver State Variable
 * @details      This variable stores the configuration and state information for the CRC test driver.
 * @private
 */
CRCTST_VAR static CrcTst_DriverStateType CrcTst_DriverState =
{
    .CrcTstConfig = NULL_PTR,
    .State = CRCTST_UNINIT,
#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
    .CrcFuncTestResult = CRCTST_E_NOT_TESTED,
    .CrcFuncTestDetails = &CrcTst_CrcFuncTestDetails,
#endif
#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
    .TestId = 0U,
    .LastRegisterTestResult = CRCTST_E_NOT_TESTED,
    .RegisterTestResult = CRCTST_E_NOT_TESTED,
    .RegisterValueErrorCnt = 0U,
    .BgndResults = CrcTst_BgndResults,
#endif
};
#define CRCTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "CrcTst_MemMap.h"

/*==================================================================================================
                                                EXTERNAL VARIABLES
==================================================================================================*/
#if (CRCTST_PRECOMPILE_SUPPORT == STD_ON)
/**
 * @brief   CrcTst driver configuration structure
 * @private
 */
extern const CrcTst_ConfigType CrcTst_PreCompileConfig;
#endif

/*==================================================================================================
*                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
LOCAL_INLINE uint32 CrcTst_Hardware(const uint8 *CrcTstDataPtr, uint32 CrcTstLength, const CrcTst_PolConfigType *TempCrcTstConfig);
#if (CRCTST_CRC8_TEST_ENABLE == STD_ON)
#if defined(CPU_YTM32B1HA0)
static uint8 CrcTst_CalculateCRC8(const uint8 *CrcTstDataPtr, uint32 CrcTstLength);
#endif
#endif
#if (CRCTST_CRC16_TEST_ENABLE == STD_ON)
static uint16 CrcTst_CalculateCRC16(const uint8 *CrcTstDataPtr, uint32 CrcTstLength);
#endif
#if ((CRCTST_CRC32_TEST_ENABLE == STD_ON) || (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON))
static uint32 CrcTst_CalculateCRC32(const uint8 *CrcTstDataPtr, uint32 CrcTstLength);
#endif

/*==================================================================================================
*                                       LOCAL FUNCTIONS
==================================================================================================*/
#define CRCTST_START_SEC_CODE
#include "CrcTst_MemMap.h"

/**
 * @brief        Converts a sequence of bytes from the input array into a 32-bit unsigned integer.
 * @details      This function reads 4 bytes from the input array (`CrcTstDataPtr`) starting at the specified index (`Index`),
 *               and combines them into a single 32-bit unsigned integer. The bytes are assumed to be in big-endian order.
 *               This function is called by 'CrcTst_Hardware' function.
 * @param[in]    Index The index of the 32-bit value in the input array. The actual byte offset is calculated as
 *              `Index * CRCTST_BYTESOFU32`, where `CRCTST_BYTESOFU32` is the number of bytes per 32-bit value (4).
 * @param [in]   CrcTstDataPtr Pointer to the input array containing the byte data.
 * @return       uint32 The 32-bit unsigned integer formed by combining the 4 bytes starting at the specified index.
 * @private
 */
CRCTST_FUNC LOCAL_INLINE uint32 CrcTst_DataConvert(uint32 Index, const uint8 *CrcTstDataPtr)
{
    uint32 Result = 0;
    /* Temporary variable to hold each byte value */
    uint32 ByteValue = 0;
    /* Read the first byte (MSB) and shift it left by 24 bits */
    ByteValue = CrcTstDataPtr[Index * CRCTST_BYTESOFU32];
    Result = ByteValue << CRCTST_VALUE_24;
    /* Read the second byte, shift it left by 16 bits, and combine it with the result */
    ByteValue = CrcTstDataPtr[(Index * CRCTST_BYTESOFU32) + CRCTST_INDEX_FIRST];
    Result |= ByteValue << CRCTST_VALUE_16;
    /* Read the third byte, shift it left by 8 bits, and combine it with the result */
    ByteValue = CrcTstDataPtr[(Index * CRCTST_BYTESOFU32) + CRCTST_INDEX_SECOND];
    Result |= ByteValue << CRCTST_VALUE_8;
    /* Read the fourth byte (LSB) and directly combine it with the result */
    Result |= (CrcTstDataPtr[(Index * CRCTST_BYTESOFU32) + CRCTST_INDEX_THIRD]);
    /* Return the final 32-bit value */
    return Result;
}

#if defined(CPU_YTM32B1HA0)
/**
 * @brief        Calculate CRC value.
 * @details      This function performs CRC calculation using hardware registers.
 *               It supports different CRC widths (8, 16, 32 bits) and configurations.
 * @param[in]    CrcTstDataPtr  Pointer to the start address of the data block.
 * @param[in]    CrcTstLength   Length of the data block in bytes.
 * @param[in]    TempCrcTstConfig Pointer to the CRC configuration structure.
 * @return       uint32 CRC result.
 * @private
 */
CRCTST_FUNC LOCAL_INLINE uint32 CrcTst_Hardware(const uint8 *CrcTstDataPtr, uint32 CrcTstLength, const CrcTst_PolConfigType *TempCrcTstConfig)
{
    uint32 TempCtrl = 0;
    uint32 Result = 0;
    uint32 Index = 0;
    uint32 WordCount = 0;
    uint32 RemainByteCount = 0;
    uint32 TempData = 0;
    uint8 PolyModeSet = 0;
    /* Get CrcWidth */
    PolyModeSet = TempCrcTstConfig->CrcWidth - 1U;
    SchM_Enter_CrcTst_CRCTST_EXCLUSIVE_AREA_00();
    TempCtrl =  CrcTst_Lld_PcrcGetCtlCmd(TempCrcTstConfig, PolyModeSet);
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
    /* Initial CRC module with TempCtrl. */
    OsIf_Trusted_Call2params(CrcTst_Lld_PcrcWriteCtlCmd, PCRC, TempCtrl);
    /* Load CRC Polynomials */
    OsIf_Trusted_Call2params(CrcTst_Lld_PcrcWritePolyNomials, PCRC, ((uint32)(TempCrcTstConfig->Polynomial)));
    /* Load CRC initialValue. */
    OsIf_Trusted_Call2params(CrcTst_Lld_PcrcWriteInitialValue, PCRC, ((uint32)(TempCrcTstConfig->InitialValue)));
#else
    /* Configure the control register to initialize CRC module with TempCtrl. */
    CrcTst_Lld_PcrcWriteCtlCmd(PCRC, TempCtrl);
    /* Load CRC Polynomials */
    CrcTst_Lld_PcrcWritePolyNomials(PCRC, TempCrcTstConfig->Polynomial);
    /* Load CRC initialValue. */
    CrcTst_Lld_PcrcWriteInitialValue(PCRC, (uint32)TempCrcTstConfig->InitialValue);
#endif
    /* Load CRC data. First write as uint32. */
    WordCount = CrcTstLength / CRCTST_BYTESOFU32;
    RemainByteCount = CrcTstLength % CRCTST_BYTESOFU32;
    for (Index = 0; Index < WordCount; ++Index)
    {
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * The pointer address is needed to determine whether privileged mode is entered. Thus, couldn't adhere to M3CM Rule-11.4.
         */
        if ((CrcTstDataPtr >= (uint8 *)0x40000000U) && (CrcTstDataPtr <= (uint8 *)0x401FFFFFU)) /* PRQA S 0306*/
        {
            /* CrcTstDataPtr may be the register address */
            TempData = (uint32)OsIf_Trusted_Call_Return2param((CrcTst_DataConvert), (Index), (CrcTstDataPtr));
        }
        else
        {
            TempData = CrcTst_DataConvert(Index, CrcTstDataPtr);
        }
#else
        TempData = CrcTst_DataConvert(Index, CrcTstDataPtr);
#endif
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(CrcTst_Lld_PcrcWriteDataU32, PCRC, TempData);
#else
        CrcTst_Lld_PcrcWriteDataU32(PCRC, TempData);
#endif
    }
    /* Load CRC data. Second write as a byte. */
    for (Index = 0; Index < RemainByteCount; ++Index)
    {
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(CrcTst_Lld_PcrcWriteDataU8, PCRC, CrcTstDataPtr[(WordCount * CRCTST_BYTESOFU32) + Index]);
#else
        CrcTst_Lld_PcrcWriteDataU8(PCRC, CrcTstDataPtr[(WordCount * CRCTST_BYTESOFU32) + Index]);
#endif
    }
    /* Read out CRC value. */
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
    Result = (uint32)OsIf_Trusted_Call_Return1param(CrcTst_Lld_PcrcReadCrcResult, PCRC);
#else
    Result = CrcTst_Lld_PcrcReadCrcResult(PCRC);
#endif
    Result = (uint32)(Result ^ (TempCrcTstConfig->XorValue));
    SchM_Exit_CrcTst_CRCTST_EXCLUSIVE_AREA_00();
    return Result;
}
#else
CRCTST_FUNC LOCAL_INLINE uint32 CrcTst_Hardware(const uint8 *CrcTstDataPtr, uint32 CrcTstLength, const CrcTst_PolConfigType *TempCrcTstConfig)
{
    /* Control register value for CRC hardware */
    uint32 Ctrl;
    /* Final CRC result */
    uint32 Result;
    /* Number of 32-bit words in the input data */
    uint32 WordCount;
    /* Number of remaining bytes after processing 32-bit words */
    uint32 RemainByteCount;
    /* Temporary variable to hold converted 32-bit data */
    uint32 TempData;
    /* Polynomial mode (16-bit or 32-bit) */
    uint8 PolyModeSet = 0;
    uint32 Index;
    /* Enter exclusive area to ensure thread-safe access to hardware */
    SchM_Enter_CrcTst_CRCTST_EXCLUSIVE_AREA_00();
    /* Set polynomial mode based on configuration */
    if (CRCTST_POLYNOMIAL_16 == TempCrcTstConfig->Polynomial)
    {
        PolyModeSet = CRCTST_HARDMODE_16BITS;
    }
    else if (CRCTST_POLYNOMIAL_32 == TempCrcTstConfig->Polynomial)
    {
        PolyModeSet = CRCTST_HARDMODE_32BITS;
    }
    else
    {
        /* No action required for unsupported polynomial */
    }
    /* Get control command based on configuration and polynomial mode */
    Ctrl = CrcTst_Lld_CrcGetCtlCmd(TempCrcTstConfig, PolyModeSet);
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
    /* Initialize CRC module with control command in user mode */
    OsIf_Trusted_Call2params(CrcTst_Lld_CrcWriteCtlCmd, CRC0, Ctrl);
    /* Load initial value into CRC module in user mode */
    OsIf_Trusted_Call2params(CrcTst_Lld_CrcWriteInitialValue, CRC0, (TempCrcTstConfig->InitialValue));
#else
    /* Initialize CRC module with control command in privileged mode */
    CrcTst_Lld_CrcWriteCtlCmd(CRC0, Ctrl);
    /* Load initial value into CRC module in privileged mode */
    CrcTst_Lld_CrcWriteInitialValue(CRC0, (TempCrcTstConfig->InitialValue));
#endif
    /* Calculate number of 32-bit words and remaining bytes */
    WordCount = CrcTstLength / CRCTST_BYTESOFU32;
    RemainByteCount = CrcTstLength % CRCTST_BYTESOFU32;
    /* Load data in 32-bit chunks */
    for (Index = 0; Index < WordCount; ++Index)
    {
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * The pointer address is needed to determine whether privileged mode is entered. Thus, couldn't adhere to M3CM Rule-11.4.
         */
        if ((CrcTstDataPtr >= (uint8 *)(0x40000000)) && (CrcTstDataPtr <= (uint8 *)(0x4007FFFF))) /* PRQA S 0306*/
        {
            TempData = (uint32)OsIf_Trusted_Call_Return2param(CrcTst_DataConvert, Index, CrcTstDataPtr);
        }
        else
        {
            TempData = CrcTst_DataConvert(Index, CrcTstDataPtr);
        }
#else
        /* Convert data to 32-bit format in privileged mode */
        TempData = CrcTst_DataConvert(Index, CrcTstDataPtr);
#endif
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
        /* Write 32-bit data to CRC module in user mode */
        OsIf_Trusted_Call2params(CrcTst_Lld_CrcWriteDataU32, CRC0, TempData);
#else
        /* Write 32-bit data to CRC module in privileged mode */
        CrcTst_Lld_CrcWriteDataU32(CRC0, TempData);
#endif
    }
    /* Load remaining bytes */
    for (Index = 0; Index < RemainByteCount; ++Index)
    {
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
        /* Write remaining bytes to CRC module in user mode */
        OsIf_Trusted_Call2params(CrcTst_Lld_CrcWriteDataU8, CRC0, CrcTstDataPtr[(WordCount * CRCTST_BYTESOFU32) + Index]);
#else
        /* Write remaining bytes to CRC module in privileged mode */
        CrcTst_Lld_CrcWriteDataU8(CRC0, CrcTstDataPtr[(WordCount * CRCTST_BYTESOFU32) + Index]);
#endif
    }
    /* Read CRC result */
#ifdef CRCTST_ENABLE_USER_MODE_SUPPORT
    Result = (uint32)OsIf_Trusted_Call_Return1param(CrcTst_Lld_CrcReadCrcResult, CRC0);
#else
    Result = CrcTst_Lld_CrcReadCrcResult(CRC0);
#endif
    /* Apply XOR value to the result as per configuration */
    Result = (Result ^ (TempCrcTstConfig->XorValue));
    SchM_Exit_CrcTst_CRCTST_EXCLUSIVE_AREA_00();
    return Result;
}
#endif

#if defined(CPU_YTM32B1HA0)
#if (CRCTST_CRC8_TEST_ENABLE == STD_ON)
/**
 * @brief         Implements CRC8 calculation using a polynomial defined in AUTOSAR_SWS_CRCLibrary.
 * @details       This function calculates the CRC8 checksum for a given data block using a predefined polynomial.
 *                The function initializes the CRC value to 0xFF and configures the CRC calculation parameters
 *                such as polynomial, initial value, and reflection settings. The actual CRC calculation is
 *                performed by a hardware function.
 * @param[in]     CrcTstDataPtr Pointer to the start address of data block to be calculated.
 * @param[in]     CrcTstLength Length of the data block to be calculated in bytes.
 * @return        uint8 The calculated CRC8 checksum.
 * @sw_type       sw_detail
 */
CRCTST_FUNC static uint8 CrcTst_CalculateCRC8(const uint8 *CrcTstDataPtr, uint32 CrcTstLength)
{
    uint8 Result = 0xFFU;
    CrcTst_PolConfigType PolConfig;
    PolConfig.CrcWidth      = CRCTST_LENGTH_8BITS;
    PolConfig.Polynomial    = CRCTST_POLYNOMIAL_8;
    PolConfig.InitialValue  = CRCTST_CRC8_INIT_VALUE;
    PolConfig.InputReflect  = FALSE;
    PolConfig.OutputReflect = FALSE;
    PolConfig.XorValue      = CRCTST_CRC8_XOR_VALUE;
    Result = (uint8)CrcTst_Hardware(CrcTstDataPtr, CrcTstLength, &PolConfig);
    return Result;
}
#endif /* #if (CRCTST_CRC8_TEST_ENABLE == STD_ON) */
#endif /* #if defined(CPU_YTM32B1HA0) */

#if (CRCTST_CRC16_TEST_ENABLE == STD_ON)
/**
 * @brief         Implements CRC16 calculation using a polynomial defined in AUTOSAR_SWS_CRCLibrary.
 * @details       This function calculates the CRC16 checksum for a given data block using a predefined polynomial.
 *                The function initializes the CRC value to 0xFF and configures the CRC calculation parameters
 *                such as polynomial, initial value, and reflection settings. The actual CRC calculation is
 *                performed by a hardware function.
 * @param[in]     CrcTstDataPtr Pointer to the start address of data block to be calculated.
 * @param[in]     CrcTstLength Length of the data block to be calculated in bytes.
 * @return        uint8 The calculated CRC8 checksum.
 * @sw_type       sw_detail
 *
 * @trace YTSDS_CrcTst_252
 */
CRCTST_FUNC static uint16 CrcTst_CalculateCRC16(const uint8 *CrcTstDataPtr, uint32 CrcTstLength)
{
    uint16 Result = 0xFFFFU;
    CrcTst_PolConfigType PolConfig;
    PolConfig.CrcWidth      = CRCTST_LENGTH_16BITS;
    PolConfig.Polynomial    = CRCTST_POLYNOMIAL_16;
    PolConfig.InitialValue  = CRCTST_CRC16_INIT_VALUE;
    PolConfig.InputReflect  = FALSE;
    PolConfig.OutputReflect = FALSE;
    PolConfig.XorValue      = CRCTST_CRC16_XOR_VALUE;
    Result = (uint16)CrcTst_Hardware(CrcTstDataPtr, CrcTstLength, &PolConfig);
    return Result;
}
#endif

#if ((CRCTST_CRC32_TEST_ENABLE == STD_ON) || (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON))
/**
 * @brief         Implements CRC16 calculation using a polynomial defined in AUTOSAR_SWS_CRCLibrary.
 * @details       This function calculates the CRC8 checksum for a given data block using a predefined polynomial.
 *                The function initializes the CRC value to 0xFF and configures the CRC calculation parameters
 *                such as polynomial, initial value, and reflection settings. The actual CRC calculation is
 *                performed by a hardware function.
 * @param[in]     CrcTstDataPtr Pointer to the start address of data block to be calculated.
 * @param[in]     CrcTstLength Length of the data block to be calculated in bytes.
 * @return        uint8 The calculated CRC8 checksum.
 * @sw_type       sw_detail
 *
 * @trace YTSDS_CrcTst_253
 */
CRCTST_FUNC static uint32 CrcTst_CalculateCRC32(const uint8 *CrcTstDataPtr, uint32 CrcTstLength)
{
    uint32 Result = 0xFFFFFFFFU;
    CrcTst_PolConfigType PolConfig;
    PolConfig.CrcWidth      = CRCTST_LENGTH_32BITS;
    PolConfig.Polynomial    = CRCTST_POLYNOMIAL_32;
    PolConfig.InitialValue  = CRCTST_CRC32_INIT_VALUE;
    PolConfig.InputReflect  = TRUE;
    PolConfig.OutputReflect = TRUE;
    PolConfig.XorValue      = CRCTST_CRC32_XOR_VALUE;
    Result = CrcTst_Hardware(CrcTstDataPtr, CrcTstLength, &PolConfig);
    return Result;
}
#endif

#if defined(CPU_YTM32B1HA0)
#if (CRCTST_CRC8_TEST_ENABLE == STD_ON)
CRCTST_FUNC LOCAL_INLINE Std_ReturnType CrcTst_PerformCrc8Tests(void)
{
    Std_ReturnType ReturnStatus = E_OK;
    uint8 Crc8Result = 0;
    for (uint8 Index = 0; Index < CRCTST_TEST_DATA_GROUP_NUM; ++Index)
    {
        Crc8Result = CrcTst_CalculateCRC8((const uint8 *)&CrcTst_TestData[Index], CrcTst_TestDataLength[Index]);
        if (Crc8Result != CrcTst_ExpectedCrc8Results[Index])
        {
            CrcTst_DriverState.CrcFuncTestDetails->Crc8Result = CRCTST_E_NOT_OK;
            CrcTst_DriverState.CrcFuncTestResult = CRCTST_E_NOT_OK;
#if (CRCTST_FGND_ERROR_NOTIFICATION == STD_ON)
            (*(CrcTst_DriverState.CrcTstConfig->FgndErrNotification))(CRCTST_MODULE_ID);
#endif /* #if (CRCTST_FGND_ERROR_NOTIFICATION == STD_ON) */
            ReturnStatus = E_NOT_OK;
            break;
        }
    }
    if (E_OK == ReturnStatus)
    {
        CrcTst_DriverState.CrcFuncTestDetails->Crc8Result = CRCTST_E_OKAY;
    }
    return ReturnStatus;
}
#endif /* #if (CRCTST_CRC8_TEST_ENABLE == STD_ON) */
#endif /* #if defined(CPU_YTM32B1HA0) */

#if (CRCTST_CRC16_TEST_ENABLE == STD_ON)
CRCTST_FUNC LOCAL_INLINE Std_ReturnType CrcTst_PerformCrc16Tests(void)
{
    Std_ReturnType ReturnStatus = E_OK;
    uint16 Crc16Result = 0;
    for (uint8 Index = 0; Index < CRCTST_TEST_DATA_GROUP_NUM; ++Index)
    {
        Crc16Result = CrcTst_CalculateCRC16((const uint8 *)&CrcTst_TestData[Index], CrcTst_TestDataLength[Index]);
        if (Crc16Result != CrcTst_ExpectedCrc16Results[Index])
        {
            CrcTst_DriverState.CrcFuncTestDetails->Crc16Result = CRCTST_E_NOT_OK;
            CrcTst_DriverState.CrcFuncTestResult = CRCTST_E_NOT_OK;
#if (CRCTST_FGND_ERROR_NOTIFICATION == STD_ON)
            (*(CrcTst_DriverState.CrcTstConfig->FgndErrNotification))(CRCTST_MODULE_ID);
#endif /* #if (CRCTST_FGND_ERROR_NOTIFICATION == STD_ON) */
            ReturnStatus = E_NOT_OK;
            break;
        }
    }
    if (E_OK == ReturnStatus)
    {
        CrcTst_DriverState.CrcFuncTestDetails->Crc16Result = CRCTST_E_OKAY;
    }
    return ReturnStatus;
}
#endif /* #if (CRCTST_CRC16_TEST_ENABLE == STD_ON) */

#if (CRCTST_CRC32_TEST_ENABLE == STD_ON)
CRCTST_FUNC LOCAL_INLINE Std_ReturnType CrcTst_PerformCrc32Tests(void)
{
    Std_ReturnType ReturnStatus = E_OK;
    uint32 Crc32Result = 0;
    for (uint8 Index = 0; Index < CRCTST_TEST_DATA_GROUP_NUM; ++Index)
    {
        Crc32Result = CrcTst_CalculateCRC32((const uint8 *)&CrcTst_TestData[Index], CrcTst_TestDataLength[Index]);
        if (Crc32Result != CrcTst_ExpectedCrc32Results[Index])
        {
            CrcTst_DriverState.CrcFuncTestDetails->Crc32Result = CRCTST_E_NOT_OK;
            CrcTst_DriverState.CrcFuncTestResult = CRCTST_E_NOT_OK;
#if (CRCTST_FGND_ERROR_NOTIFICATION == STD_ON)
            (*(CrcTst_DriverState.CrcTstConfig->FgndErrNotification))(CRCTST_MODULE_ID);
#endif /* #if (CRCTST_FGND_ERROR_NOTIFICATION == STD_ON) */
            ReturnStatus = E_NOT_OK;
            break;
        }
    }
    if (E_OK == ReturnStatus)
    {
        CrcTst_DriverState.CrcFuncTestDetails->Crc32Result = CRCTST_E_OKAY;
    }
    return ReturnStatus;
}
#endif /* #if (CRCTST_CRC32_TEST_ENABLE == STD_ON) */

/*==================================================================================================
                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief        Initialize and change the state of the CRC Test module.
 * @details      This function initializes the CRC Test module by setting up the configuration
 *               and resetting the internal state variables. It also validates the input configuration.
 * @param[in]    ConfigPtr Pointer to the configuration structure.
 * @table        @service_id:       0x00        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_265
 */
CRCTST_FUNC void CrcTst_Init(const CrcTst_ConfigType *ConfigPtr)
{
    switch (CrcTst_DriverState.State)
    {
        case CRCTST_INIT:
        case CRCTST_RUNNING_BGND:
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CRCTST_MODULE_ID, CRCTST_INSTANCE_ID,
                                  CRCTST_SID_INIT, CRCTST_E_ALREADY_INITIALIZED);
#endif
            break;
        /* SONAR VIOLATION: This case is number of lines more than 5, but wrapping it in a function would make it less readable */
        case CRCTST_UNINIT:   /*NOSONAR*/
#if (CRCTST_PRECOMPILE_SUPPORT == STD_ON)
            CrcTst_DriverState.CrcTstConfig = &CrcTst_PreCompileConfig;
            (void)ConfigPtr;
#else
            if (NULL_PTR == ConfigPtr)
            {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
                (void)Det_ReportError
                (
                    CRCTST_MODULE_ID,
                    CRCTST_INSTANCE_ID,
                    CRCTST_SID_INIT,
                    CRCTST_E_PARAM_POINTER
                );
#endif /* #if (CRCTST_PRECOMPILE_SUPPORT == STD_ON) */
            }
            else
            {
                CrcTst_DriverState.CrcTstConfig = ConfigPtr;
            }
#endif  /* #if (CRCTST_PRECOMPILE_SUPPORT == STD_ON)*/
#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
            CrcTst_DriverState.CrcFuncTestResult = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.CrcFuncTestDetails->Crc8Result = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.CrcFuncTestDetails->Crc16Result = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.CrcFuncTestDetails->Crc32Result = CRCTST_E_NOT_TESTED;
#endif
#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
            CrcTst_DriverState.TestId = 0U;
            for (uint32 AtomicTestId = 0; AtomicTestId < CrcTst_DriverState.CrcTstConfig->BgndConfigNum; ++AtomicTestId)
            {
                CrcTst_DriverState.BgndResults[AtomicTestId] = CRCTST_E_NOT_TESTED;
            }
            CrcTst_DriverState.LastRegisterTestResult = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.RegisterTestResult = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.RegisterValueErrorCnt = 0U;
#endif
            CrcTst_DriverState.State = CRCTST_INIT;
            break;
        default:
            CrcTst_DriverState.State = CRCTST_UNINIT;
            break;
    }
}

/**
 * @brief        De-initialize the CRC Test module.
 * @details      De-initialize the structure, variables, change the state from or CRCTST_INIT to CRCTST_UNINIT.
 * @table        @service_id:       0x01        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @return       void
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_256
 */
CRCTST_FUNC void CrcTst_DeInit(void)
{
    switch (CrcTst_DriverState.State)
    {
        case CRCTST_RUNNING_BGND:
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CRCTST_MODULE_ID, CRCTST_INSTANCE_ID,
                                  CRCTST_SID_DEINIT, CRCTST_E_STATUS_FAILURE);
#endif
            break;
        /* SONAR VIOLATION: This case is number of lines more than 5, but wrapping it in a function would make it less readable */
        case CRCTST_INIT: /*NOSONAR*/
#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
            CrcTst_DriverState.CrcFuncTestResult = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.CrcFuncTestDetails->Crc8Result = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.CrcFuncTestDetails->Crc16Result = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.CrcFuncTestDetails->Crc32Result = CRCTST_E_NOT_TESTED;
#endif
#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
            CrcTst_DriverState.TestId = 0U;
            for (uint32 AtomicTestId = 0; AtomicTestId < CrcTst_DriverState.CrcTstConfig->BgndConfigNum; ++AtomicTestId)
            {
                CrcTst_DriverState.BgndResults[AtomicTestId] = CRCTST_E_NOT_TESTED;
            }
            CrcTst_DriverState.LastRegisterTestResult = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.RegisterTestResult = CRCTST_E_NOT_TESTED;
            CrcTst_DriverState.RegisterValueErrorCnt = 0U;
#endif
            CrcTst_DriverState.State = CRCTST_UNINIT;
            CrcTst_DriverState.CrcTstConfig = NULL_PTR;
            break;
        case CRCTST_UNINIT:
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CRCTST_MODULE_ID, CRCTST_INSTANCE_ID,
                                  CRCTST_SID_DEINIT, CRCTST_E_UNINIT);
#endif
            break;
        default:
            CrcTst_DriverState.State = CRCTST_UNINIT;
            break;
    }
}

/**
 * @brief        Get the current state of the CRC Test module.
 * @details      This function returns the current state of the CRC Test module, which is stored in the
 *               global state variable `CrcTst_DriverState.State`.
 * @return       CrcTst_StateType The current state of the CrcTst module.
 * @table        @service_id:       0x02        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_263
 */
CRCTST_FUNC CrcTst_StateType CrcTst_GetState(void)
{
    return CrcTst_DriverState.State;
}

#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
/**
 * @brief        Main function of the CRC Test module to process the foreground test.
 * @details      This function performs the foreground CRC test for the configured CRC types (CRC8, CRC16, CRC32).
 *               It iterates through the test data groups, calculates the CRC for each group, and compares the result
 *               with the expected CRC value. If any mismatch is detected, an error notification is triggered, and the
 *               test result is marked as failed. The function also handles error detection and reporting if the module
 *               is uninitialized.
 * @return       Std_ReturnType
 * @retval       E_OK: Foreground test completed successfully.
 * @retval       E_NOT_OK: Foreground test failed or an error occurred during execution.
 *
 * @table        @service_id:       0x03        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_517
 */
CRCTST_FUNC Std_ReturnType CrcTst_StartFgnd(void)
{
    Std_ReturnType ReturnStatus = E_OK;
    if (CRCTST_UNINIT == CrcTst_DriverState.State)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_START_FGND,
            CRCTST_E_UNINIT
        );
#endif
        ReturnStatus = E_NOT_OK;
    }
    else
    {
#if defined(CPU_YTM32B1HA0)
#if (CRCTST_CRC8_TEST_ENABLE == STD_ON)
        ReturnStatus = CrcTst_PerformCrc8Tests();
#endif /* #if (CRCTST_CRC8_TEST_ENABLE == STD_ON) */
#endif /* #if defined(CPU_YTM32B1HA0) */
#if (CRCTST_CRC16_TEST_ENABLE == STD_ON)
        /* MR12 RULE 2.2 VIOLATION: The '|=' operation is redundant when the 'CRCTST_CRC8_TEST_ENABLE' is 'STD_OFF',
         * and the value of the result is always that of the right-hand operand, because the 'ReturnStatus' is
         * initialized to 'E_OK' before the '|=' operation. But when the 'CRCTST_CRC8_TEST_ENABLE' is 'STD_ON',
         * the '|=' operation is valid. */
        ReturnStatus |= CrcTst_PerformCrc16Tests();  /*PRQA S 2986*/
#endif /* #if (CRCTST_CRC16_TEST_ENABLE == STD_ON) */
#if (CRCTST_CRC32_TEST_ENABLE == STD_ON)
        ReturnStatus |= CrcTst_PerformCrc32Tests();
#endif /* #if (CRCTST_CRC32_TEST_ENABLE == STD_ON) */
        if (CrcTst_DriverState.CrcFuncTestResult != CRCTST_E_NOT_OK)
        {
            CrcTst_DriverState.CrcFuncTestResult = CRCTST_E_OKAY;
        }
    }
    return ReturnStatus;
}

/**
 * @brief        Get the last result of the foreground test.
 * @details      This function retrieves the result of the last executed foreground CRC test.
 *               It checks the module's state and returns the appropriate result.
 * @return       CrcTst_TestResultType Last result of the foreground test.
 * @retval       CRCTST_E_NOT_TESTED Not tested yet.
 * @retval       CRCTST_E_OKAY No errors detected.
 * @retval       CRCTST_E_NOT_OK At least one error detected.
 * @table        @service_id:       0x04        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_261
 */
CRCTST_FUNC CrcTst_TestResultType CrcTst_GetFgndResult(void)
{
    CrcTst_TestResultType Result = CRCTST_E_NOT_TESTED;
    if (CRCTST_UNINIT == CrcTst_DriverState.State)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_GET_FGND_RESULT,
            CRCTST_E_UNINIT
        );
#endif
    }
    else
    {
        Result = CrcTst_DriverState.CrcFuncTestResult;
    }
    return Result;
}


/**
 * @brief        Get the details of the foreground test results.
 * @details      This function retrieves the detailed results of the foreground CRC test.
 * @param[out]   FgndResults Pointer to store the foreground test results.
 * @table        @service_id:       0x05        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @return       void
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_652
 */
CRCTST_FUNC void CrcTst_GetFgndDetailsResult(CrcTst_CrcFuncTestDetailsType *FgndResults)
{
    /* Check if the module is initialized */
    if (CRCTST_UNINIT == CrcTst_DriverState.State)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_GET_FGND_RESULT_DETAILS,
            CRCTST_E_UNINIT
        );
#endif /* #if (CRCTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        FgndResults->Crc8Result = CrcTst_DriverState.CrcFuncTestDetails->Crc8Result;
        FgndResults->Crc16Result = CrcTst_DriverState.CrcFuncTestDetails->Crc16Result;
        FgndResults->Crc32Result = CrcTst_DriverState.CrcFuncTestDetails->Crc32Result;
    }
}

#endif  /* #if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON) */

#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
/**
 * @brief        Get the register signature.
 *
 * @details      This function calculates and retrieves the register signature for the CRC Test module.
 *               The signature is computed based on the background configuration data provided in the module's configuration.
 *               If the signature is stored in RAM (not in Flash), the function calculates the CRC32 value for each
 *               background configuration data block and stores the result in the designated signature array.
 *
 * @return       Std_ReturnType
 * @retval       E_OK: Function executed successfully.
 * @retval       E_NOT_OK: Function failed due to invalid parameters or module state.
 *
 * @table        @service_id:       0x06        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_516
 */
CRCTST_FUNC Std_ReturnType CrcTst_GetRegisterSignature(void)
{
    Std_ReturnType ReturnStatus = E_NOT_OK;
#if (CRCTST_SIGNATURE_IN_FLASH_SUPPORT == STD_OFF)
    uint32 AtomicTestCrcResult = 0U;
    const uint8 *AtomicDataPtr = NULL_PTR;
#endif
    if (CrcTst_DriverState.State == CRCTST_UNINIT)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_GET_REG_SIGN,
            CRCTST_E_UNINIT
        );
#endif
    }
    else if (CrcTst_DriverState.State == CRCTST_RUNNING_BGND)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_GET_REG_SIGN,
            CRCTST_E_STATUS_FAILURE
        );
#endif
    }
    else
    {
        /* Only when the signature is stored in RAM can it be calculated */
#if (CRCTST_SIGNATURE_IN_FLASH_SUPPORT == STD_OFF)
        /*Calculate register CRC in back ground configure*/
        for (uint32 TestId = 0U; TestId < CrcTst_DriverState.CrcTstConfig->BgndConfigNum; ++TestId)
        {
            AtomicDataPtr = CrcTst_DriverState.CrcTstConfig->BgndConfig[TestId].CalcData;
            /* Calculate CRC result */
            AtomicTestCrcResult = CrcTst_CalculateCRC32(AtomicDataPtr, CRCTST_VALUE_4);
            /* store the signature */
            CrcTst_DriverState.CrcTstConfig->BgndRegisterSignatureCaculate[TestId] = AtomicTestCrcResult;
        }
        ReturnStatus = E_OK;
#endif
    }
    return ReturnStatus;
}

/**
* @brief        Service to start the background test.
*
* @details      This function starts the background CRC test process. It checks the module's state
*               and ensures that the module is properly initialized before starting the test. If the module is
*               uninitialized, an error is reported. Otherwise, the function initializes the test state variables,
*               including the test results array, and sets the module state to `CRCTST_RUNNING_BGND` to indicate
*               that the background test is in progress.
*
* @param[in]    void
* @return       Std_ReturnType
* @retval       E_OK: Background test started successfully.
* @retval       E_NOT_OK: Background test failed to start due to invalid state or configuration.
* @table        @service_id:       0x07        \n
*               @is_reentrant:     false       \n
*               @is_synchronous:   true        \n
*               @autosar_api:      false       \n
*               @memory_map:       .mcal_text  \n
* @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_822
*/
CRCTST_FUNC Std_ReturnType CrcTst_StartBgnd(void)
{
    Std_ReturnType ReturnStatus = E_NOT_OK;
    if (CRCTST_UNINIT == CrcTst_DriverState.State)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_START_BGND,
            CRCTST_E_UNINIT
        );
#endif
    }
    else if (CRCTST_RUNNING_BGND == CrcTst_DriverState.State)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_START_BGND,
            CRCTST_E_STATUS_FAILURE
        );
#endif
    }
    else
    {
        CrcTst_DriverState.TestId = 0U;
        for (uint32 TestId = 0; TestId < CrcTst_DriverState.CrcTstConfig->BgndConfigNum; ++TestId)
        {
            CrcTst_DriverState.BgndResults[TestId] = CRCTST_E_NOT_TESTED;
        }
        CrcTst_DriverState.LastRegisterTestResult = CRCTST_E_NOT_TESTED;
        CrcTst_DriverState.RegisterTestResult = CRCTST_E_NOT_TESTED;
        CrcTst_DriverState.RegisterValueErrorCnt = 0U;
        CrcTst_DriverState.State = CRCTST_RUNNING_BGND;
        ReturnStatus = E_OK;
    }
    return ReturnStatus;
}

/**
 * @brief        Main function of the CRC Test module to process the background test.
 * @details      Cyclically called by scheduler to perform processing of atomic testcase in background mode.
 * @param[in]    void
 * @table        @service_id:       0x08        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @return       Std_ReturnType
 * @retval       E_OK function is executed successfully
 * @retval       E_NOT_OK parameter or state is not valid
 * @return       Std_ReturnType E_OK if successful, E_NOT_OK otherwise.
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_267
 */
CRCTST_FUNC Std_ReturnType CrcTst_MainFunction(void)
{
    Std_ReturnType ReturnStatus = E_NOT_OK;
    uint32 TestCrcResult = 0U;
    uint32 AtomicSignature = 0U;
    const uint8 *AtomicDataPtr = NULL_PTR;
    switch (CrcTst_DriverState.State)
    {
        /* SONAR VIOLATION: This case is number of lines more than 5, but wrapping it in a function would make it less readable */
        case CRCTST_RUNNING_BGND:   /*NOSONAR*/
            if (0U == CrcTst_DriverState.TestId)
            {
                CrcTst_DriverState.RegisterTestResult = CRCTST_E_NOT_TESTED;
                CrcTst_DriverState.RegisterValueErrorCnt = 0U;
            }
            /* Execute an atomic test here. */
            AtomicDataPtr = CrcTst_DriverState.CrcTstConfig->BgndConfig[CrcTst_DriverState.TestId].CalcData;
            AtomicSignature = *(CrcTst_DriverState.CrcTstConfig->BgndConfig[CrcTst_DriverState.TestId].ExpectedResult);
            TestCrcResult = CrcTst_CalculateCRC32(AtomicDataPtr, CRCTST_VALUE_4);
            /* After the atomic test. */
            /* Calculate the background accumulated signature. */
            if (AtomicSignature != TestCrcResult)
            {
                /* Store atomic test results */
                CrcTst_DriverState.BgndResults[CrcTst_DriverState.TestId] = CRCTST_E_NOT_OK;
                ++CrcTst_DriverState.RegisterValueErrorCnt;
                CrcTst_DriverState.RegisterTestResult = CRCTST_E_NOT_OK;
#if (CRCTST_DEM_EVENT_REPORT == STD_ON)
                (void)Dem_SetEventStatus(CRCTST_E_FAILURE_DEM_EVENT_ID, DEM_EVENT_STATUS_PREFAILED);
#endif
            }
            else
            {
                /* Store atomic test results */
                CrcTst_DriverState.BgndResults[CrcTst_DriverState.TestId] = CRCTST_E_OKAY;
            }
            /* Start a new atomic test.  */
            ++CrcTst_DriverState.TestId;
            /* Completed all the tests of the current background configuration. */
            if (CrcTst_DriverState.TestId >= CrcTst_DriverState.CrcTstConfig->BgndConfigNum)
            {
                CrcTst_DriverState.TestId = 0U;
                /* Completed all the tests of all the background configurations. */
                if (0U == CrcTst_DriverState.RegisterValueErrorCnt)
                {
                    CrcTst_DriverState.LastRegisterTestResult = CRCTST_E_OKAY;
                    CrcTst_DriverState.RegisterTestResult = CRCTST_E_OKAY;
#if (CRCTST_DEM_EVENT_REPORT == STD_ON)
                    (void)Dem_SetEventStatus(CRCTST_E_FAILURE_DEM_EVENT_ID, DEM_EVENT_STATUS_PREPASSED);
#endif
                }
                else
                {
                    CrcTst_DriverState.LastRegisterTestResult = CRCTST_E_NOT_OK;
                    CrcTst_DriverState.RegisterTestResult = CRCTST_E_NOT_OK;
                }
#if (CRCTST_BKND_COMPLETE_CALLBACK_ENABLE == STD_ON)
                (*(CrcTst_DriverState.CrcTstConfig->BgndCompleteCallBack))();
#endif /* #if (CRCTST_BKND_COMPLETE_CALLBACK_ENABLE == STD_ON) */
            }
            ReturnStatus = E_OK;
            break;
        case CRCTST_INIT:
        case CRCTST_UNINIT:
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CRCTST_MODULE_ID, CRCTST_INSTANCE_ID,
                                  CRCTST_SID_MAIN_FUNCTION, CRCTST_E_STATUS_FAILURE);
#endif
            break;
        default:
            CrcTst_DriverState.State = CRCTST_UNINIT;
            break;
    }
    return ReturnStatus;
}

/**
 * @brief        Abort the CRC Test in background test.
 *
 * @details      This function aborts the ongoing CRC test by changing the module state from `CRCTST_RUNNING_BGND`
 *               to `CRCTST_INIT`. If the module is in the `CRCTST_UNINIT` state, an error is reported
 *               using the Development Error Tracer (DET).
 * @table        @service_id:       0x09        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @return       void
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_250
 */
CRCTST_FUNC void CrcTst_Abort(void)
{
    switch (CrcTst_DriverState.State)
    {
        case CRCTST_RUNNING_BGND:
            CrcTst_DriverState.State = CRCTST_INIT;
            break;
        case CRCTST_UNINIT:
        case CRCTST_INIT:
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CRCTST_MODULE_ID, CRCTST_INSTANCE_ID,
                                  CRCTST_SID_ABORT, CRCTST_E_STATUS_FAILURE);
#endif
            break;
        default:
            CrcTst_DriverState.State = CRCTST_UNINIT;
            break;
    }
}

/**
 * @brief        Get the result of the background test.
 * @param[in]    void
 * @table        @service_id:       0x0A        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @return       CrcTst_TestResultType
 * @retval       CRCTST_E_NOT_TESTED Not tested yet
 * @retval       CRCTST_E_OKAY No errors detected
 * @retval       CRCTST_E_NOT_OK At least one error detected
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_260
 */
CRCTST_FUNC  CrcTst_TestResultType CrcTst_GetBgndResult(void)
{
    CrcTst_TestResultType RetResult = CRCTST_E_NOT_TESTED;
    if (CRCTST_UNINIT == CrcTst_DriverState.State)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_GET_BGND_RESULT,
            CRCTST_E_UNINIT
        );
#endif
    }
    else
    {
        RetResult = CrcTst_DriverState.LastRegisterTestResult;
    }
    return RetResult;
}

/**
 * @brief        Get the details of the background test results.
 * @details      This function retrieves the detailed results of the background CRC test.
 * @param[out]   BgndResults Pointer to store the background test results.
 * @table        @service_id:       0x0B        \n
 *               @is_reentrant:     false       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @return       void
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_651
 */
CRCTST_FUNC void CrcTst_GetBgndDetailsResult(CrcTst_TestResultType *BgndResults)
{
    /* Check if the module is initialized */
    if (CRCTST_UNINIT == CrcTst_DriverState.State)
    {
#if (CRCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            CRCTST_MODULE_ID,
            CRCTST_INSTANCE_ID,
            CRCTST_SID_GET_BGND_RESULT_DETAILS,
            CRCTST_E_UNINIT
        );
#endif /* #if (CRCTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        for (uint32 TestId = 0; TestId < CrcTst_DriverState.CrcTstConfig->BgndConfigNum; ++TestId)
        {
            BgndResults[TestId] =  CrcTst_DriverState.BgndResults[TestId];
        }
    }
}

#endif  /* #if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON) */

/**
 * @brief        This service returns the version information of this module
 * @param[out]   Versioninfo Pointer to where to store the version information of this module.
 * @table        @service_id:       0x0C        \n
 *               @is_reentrant:     true       \n
 *               @is_synchronous:   true        \n
 *               @autosar_api:      false       \n
 *               @memory_map:       .mcal_text  \n
 * @sw_type      sw_detail
 *
 * @trace YTSDS_CrcTst_653
 */
#if (((CRCTST_VERSION_INFO_API == STD_ON) && (CRCTST_DEV_ERROR_DETECT == STD_ON)))
void CrcTst_GetVersionInfo(Std_VersionInfoType *Versioninfo)
{
    if (NULL_PTR == Versioninfo)
    {
        (void)Det_ReportError((uint16)CRCTST_MODULE_ID,
                              CRCTST_INSTANCE_ID,
                              CRCTST_SID_GET_VERSION_INFO,
                              CRCTST_E_PARAM_POINTER);
    }
    else
    {
        /* Vendor ID information */
        Versioninfo->vendorID = CRCTST_VENDOR_ID;
        /* Crc module ID information */
        Versioninfo->moduleID = CRCTST_MODULE_ID;
        /* Crc module Software major version information */
        Versioninfo->sw_major_version = (uint8)CRCTST_SW_MAJOR_VERSION;
        /* Crc module Software minor version information */
        Versioninfo->sw_minor_version = (uint8)CRCTST_SW_MINOR_VERSION;
        /* Crc module Software patch version information */
        Versioninfo->sw_patch_version = (uint8)CRCTST_SW_PATCH_VERSION;
    }
}
#elif (((CRCTST_VERSION_INFO_API == STD_ON) && (CRCTST_DEV_ERROR_DETECT == STD_OFF)))
void CrcTst_GetVersionInfo(Std_VersionInfoType *Versioninfo)
{
    /* Vendor ID information */
    ((Std_VersionInfoType *)(Versioninfo))->vendorID = CRCTST_VENDOR_ID;
    /* Crc module ID information */
    ((Std_VersionInfoType *)(Versioninfo))->moduleID = CRCTST_MODULE_ID;
    /* Crc module Software major version information */
    ((Std_VersionInfoType *)(Versioninfo))->sw_major_version = (uint8)CRCTST_SW_MAJOR_VERSION;
    /* Crc module Software minor version information */
    ((Std_VersionInfoType *)(Versioninfo))->sw_minor_version = (uint8)CRCTST_SW_MINOR_VERSION;
    /* Crc module Software patch version information */
    ((Std_VersionInfoType *)(Versioninfo))->sw_patch_version = (uint8)CRCTST_SW_PATCH_VERSION;
}
#endif


#define CRCTST_STOP_SEC_CODE
#include "CrcTst_MemMap.h"

#ifdef __cplusplus
}
#endif

