/**
 * @file    CrcTst_Types.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR CrcTst module interface
 * @details API implementation for CrcTst driver
 *
 * @addtogroup CRCTST_MODULE
 * @{
 */

/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : CrcTst
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
 *=================================================================================================*/

#ifndef CRCTST_TYPES_H
#define CRCTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
 *=================================================================================================*/
#include "Std_Types.h"
#include "CrcTst_Cfg.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CRCTST_TYPES_VENDOR_ID                      (180)
#define CRCTST_TYPES_AR_RELEASE_MAJOR_VERSION       (4)
#define CRCTST_TYPES_AR_RELEASE_MINOR_VERSION       (4)
#define CRCTST_TYPES_AR_RELEASE_REVISION_VERSION    (0)
#define CRCTST_TYPES_SW_MAJOR_VERSION               (2)
#define CRCTST_TYPES_SW_MINOR_VERSION               (0)
#define CRCTST_TYPES_SW_PATCH_VERSION               (0)


/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CRCTST configuration header file are of the same vendor */
#if (CRCTST_TYPES_VENDOR_ID != CRCTST_VENDOR_ID_CFG)
#error "CrcTst_Types.h and CrcTst_Cfg.h have different vendor ids"
#endif

/* Check if source file and CRCTST configuration header file are of the same Autosar version */
#if ((CRCTST_TYPES_AR_RELEASE_MAJOR_VERSION != CRCTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (CRCTST_TYPES_AR_RELEASE_MINOR_VERSION != CRCTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (CRCTST_TYPES_AR_RELEASE_REVISION_VERSION != CRCTST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of CrcTst_Types.h and CrcTst_Cfg.h are different"
#endif

/* Check if source file and CRCTST configuration header file are of the same software version */
#if ((CRCTST_TYPES_SW_MAJOR_VERSION != CRCTST_SW_MAJOR_VERSION_CFG) || \
     (CRCTST_TYPES_SW_MINOR_VERSION != CRCTST_SW_MINOR_VERSION_CFG) || \
     (CRCTST_TYPES_SW_PATCH_VERSION != CRCTST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of CrcTst_Types.h and CrcTst_Cfg.h are different"
#endif

/*==================================================================================================
 *                            LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
 *=================================================================================================*/
/**
 * @brief   Configuration structure for CRC polynomial and parameters.
 * @details This structure defines the parameters required for CRC calculation,
 *          including polynomial, initial value, and reflection options.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_499
 */
typedef struct
{
    uint8 CrcWidth;                 /**< CRC result width */
    uint32 Polynomial;              /**< CRC Polynomial */
    uint32 InitialValue;            /**< CRC Initial Value */
    boolean InputReflect;           /**< Input data reflect */
    boolean OutputReflect;          /**< Output data reflect */
    uint32 XorValue;                /**< XOR value */
} CrcTst_PolConfigType;

/**
 * @brief   Enumeration for CRC Test states.
 * @details This enumeration defines the possible states of the CRC Test module,
 *          such as uninitialized, initialized, and running in background mode.
 * @sw_type sw_arch
 */
typedef enum
{
    CRCTST_INIT         = 0x00U, /**< The Crc Test is initialized and can be started. */
    CRCTST_UNINIT       = 0x01U, /**< The Crc Test can be initialized. */
    CRCTST_RUNNING_BGND = 0x02U  /**< The Crc Test is currently executed. */
} CrcTst_StateType;

/**
 * @brief   Enumeration for Crc Test results.
 * @sw_type sw_arch
 */
typedef enum
{
    CRCTST_E_NOT_OK     = 0x00U, /**< The Crc Test detected at least one single test errors. */
    CRCTST_E_OKAY       = 0x01U, /**< The Crc Test passed without errors. */
    CRCTST_E_NOT_TESTED = 0x02U  /**< There is no Crc Test result available (default). */
} CrcTst_TestResultType;

/**
 * @brief   Structure for CRC function test details.
 * @details This structure contains the results of CRC8, CRC16, and CRC32 tests,
 *          providing detailed information about the test outcomes.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_823
 */
typedef struct
{
    CrcTst_TestResultType Crc8Result;  /**< Crc8 Test result */
    CrcTst_TestResultType Crc16Result; /**< Crc16 Test result */
    CrcTst_TestResultType Crc32Result; /**< Crc32 Test result */
} CrcTst_CrcFuncTestDetailsType;

/*==================================================================================================
 *                                             TYPEDEF
 *=================================================================================================*/

#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
/**
 * @brief   Function pointer type for CRC test completion callback.
 * @sw_type sw_arch
 */
typedef void (*CrcTst_CompleteCallBackType)(void);
#endif

#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
/**
 * @brief   Function pointer type for CrcTst fore ground test error notification.
 * @sw_type sw_arch
 */
typedef void (*CrcTst_CrcFailNotificationType)(uint16 ModuleId);
#endif

/**
 * @brief   Configuration structure for atomic CRC test.
 * @details This structure defines the data and expected results for an atomic CRC test.
 * @sw_type sw_detail
 *
 * @trace YTSDS_CrcTst_277
 */
typedef struct
{
    const uint8 *CalcData;                      /**< Pointer to the data using signature verification */
    const uint32 *ExpectedResult;                /**< Pointer to the signature */
} CrcTst_AtomicConfigType;


/**
 * @brief   Configuration structure for the CrcTst module.
 * @details This structure defines the configuration parameters for the CRC Test module.
 *          It includes settings for both foreground and background tests, such as error notification functions,
 *          test configurations, and callback functions.
 *
 * @sw_type sw_arch
 */
typedef struct
{
#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
    CrcTst_CrcFailNotificationType FgndErrNotification;  /**< Pointer to the foreground error notification function, invoked when a CRC error is detected in foreground mode. */
#endif

#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
    const uint32 BgndConfigNum;                          /**< Number of background test configurations. */
    const CrcTst_AtomicConfigType *BgndConfig;           /**< Pointer to the background test configuration array, which defines the data and parameters for each atomic test. */
    uint32 *BgndRegisterSignatureCaculate;               /**< Pointer to the calculated register integrity signature, used to store the CRC results for background tests. */
    CrcTst_CompleteCallBackType BgndCompleteCallBack;    /**< Pointer to the background test completion callback function. */
#endif
} CrcTst_ConfigType;



/**
 * @brief      State structure for the CrcTst driver.
 * @details    This structure holds the configuration and state information for the CrcTst driver.\n
 *             The conditional members are only included if the corresponding feature is enabled
 *             (e.g., `CRCTST_CRC_FUNCTION_TEST_ENABLE` or `CRCTST_REGISTER_INTEGRITY_TEST_ENABLE`).
 * @sw_type    sw_detail
 *
 * @trace YTSDS_CrcTst_898
 */
typedef struct
{
    const CrcTst_ConfigType *CrcTstConfig;      /**< Pointer to the configuration data for the CRC Test module. */
    CrcTst_StateType State;                     /**< Current execution state of the CRC Test module. */

#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
    CrcTst_TestResultType CrcFuncTestResult;    /**< Last test result for the foreground CRC function test. */
    CrcTst_CrcFuncTestDetailsType *CrcFuncTestDetails; /**< Pointer to detailed results of the foreground CRC function test. */
#endif

#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
    uint32 TestId;                              /**< Current test ID for the background register integrity test. */
    uint32 RegisterValueErrorCnt;               /**< Counter for signature errors detected during the background test. */
    CrcTst_TestResultType LastRegisterTestResult; /**< Last result of the register integrity test in background mode. */
    CrcTst_TestResultType RegisterTestResult;   /**< Current test result of the register integrity test in background mode. */
    CrcTst_TestResultType *BgndResults;         /**< Pointer to an array storing results of all atomic tests in background mode. */
#endif
} CrcTst_DriverStateType;


#ifdef __cplusplus
}
#endif

#endif /* CRCTST_TYPES_H */

/** @} */
