/**
 * @file    CorTst_Types.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR CorTst module interface
 * @details API implementation for CorTst driver
 *
 * @addtogroup CORTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Core
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifndef CORTST_TYPES_H
#define CORTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "CorTst_M33_Cfg.h"
#include "CorTst_Cfg.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CORTST_TYPES_VENDOR_ID                      (180)
#define CORTST_TYPES_AR_RELEASE_MAJOR_VERSION       (4)
#define CORTST_TYPES_AR_RELEASE_MINOR_VERSION       (4)
#define CORTST_TYPES_AR_RELEASE_REVISION_VERSION    (0)
#define CORTST_TYPES_SW_MAJOR_VERSION               (2)
#define CORTST_TYPES_SW_MINOR_VERSION               (0)
#define CORTST_TYPES_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* CorTst_Cfg.h version check */
#if (CORTST_TYPES_VENDOR_ID != CORTST_VENDOR_ID_CFG)
#error "CorTst.c and CorTst_Cfg.h have different vendor ids"
#endif
#if ((CORTST_TYPES_AR_RELEASE_MAJOR_VERSION != CORTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (CORTST_TYPES_AR_RELEASE_MINOR_VERSION != CORTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (CORTST_TYPES_AR_RELEASE_REVISION_VERSION != CORTST_AR_RELEASE_REVISION_VERSION_CFG) \
    )
#error "AutoSar Version Numbers of CorTst.c and CorTst_Cfg.h are different"
#endif
#if ((CORTST_TYPES_SW_MAJOR_VERSION != CORTST_SW_MAJOR_VERSION_CFG) || \
     (CORTST_TYPES_SW_MINOR_VERSION != CORTST_SW_MINOR_VERSION_CFG) || \
     (CORTST_TYPES_SW_PATCH_VERSION != CORTST_SW_PATCH_VERSION_CFG) \
    )
#error "Software Version Numbers of CorTst.c and CorTst_Cfg.h are different"
#endif

/* CorTst_M33_Cfg.h version check */
#if (CORTST_TYPES_VENDOR_ID != CORTST_M33_VENDOR_ID_CFG)
#error "CorTst.c and CorTst_M33_Cfg.h have different vendor ids"
#endif
#if ((CORTST_TYPES_AR_RELEASE_MAJOR_VERSION != CORTST_M33_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (CORTST_TYPES_AR_RELEASE_MINOR_VERSION != CORTST_M33_AR_RELEASE_MINOR_VERSION_CFG) || \
     (CORTST_TYPES_AR_RELEASE_REVISION_VERSION != CORTST_M33_AR_RELEASE_REVISION_VERSION_CFG) \
    )
#error "AutoSar Version Numbers of CorTst.c and CorTst_M33_Cfg.h are different"
#endif
#if ((CORTST_TYPES_SW_MAJOR_VERSION != CORTST_M33_SW_MAJOR_VERSION_CFG) || \
     (CORTST_TYPES_SW_MINOR_VERSION != CORTST_M33_SW_MINOR_VERSION_CFG) || \
     (CORTST_TYPES_SW_PATCH_VERSION != CORTST_M33_SW_PATCH_VERSION_CFG) \
    )
#error "Software Version Numbers of CorTst.c and CorTst_M33_Cfg.h are different"
#endif
/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/**
 * @brief       Service ID number for all CORTST driver services
 * @defgroup    CorTst_ID_Info_Macro
 * @sw_type     sw_detail
 *
 * @trace       YTDT_CorTst_872
 * @{
 */
/** @brief API Service ID for CorTst_Init */
#define CORTST_SID_INIT                 (0x00U)
/** @brief API Service ID for CorTst_DeInit */
#define CORTST_SID_DEINIT               (0x01U)
/** @brief API Service ID for CorTst_Abort */
#define CORTST_SID_ABORT                (0x02U)
/** @brief API Service ID for CorTst_GetState */
#define CORTST_SID_GET_STATE            (0x03U)
/** @brief API Service ID for CorTst_GetCurrentStatus */
#define CORTST_SID_GET_CURRENT_STATUS   (0x04U)
/** @brief API Service ID for CorTst_GetSignature */
#define CORTST_SID_GET_SIGNATURE        (0x05U)
/** @brief API Service ID for CorTst_GetFgndSignature */
#define CORTST_SID_GET_FGND_SIGNATURE   (0x06U)
/** @brief API Service ID for CorTst_Start */
#define CORTST_SID_START                (0x07U)
/** @brief API Service ID for CorTst_GetVersionInfo */
#define CORTST_SID_GET_VERSION_INFO     (0x08U)
/** @brief API Service ID for CorTst_MainFunction */
#define CORTST_SID_MAIN_FUNCTION        (0x0BU)
/** @brief API Service ID for CorTst_GetFgndResult */
#define CORTST_SID_GET_FGND_RESULT      (0x0DU)
/** @brief API Service ID for CorTst_GetDetailsResult */
#define CORTST_SID_GET_DETAILS_RESULT   (0x0EU)

/** @} */

/**
* @brief       CORTST development errors
* @defgroup    CorTst_Dev_Error_Macro
* @sw_type     sw_detail
*
* @trace       YTDT_CorTst_871
* @{
*/
/** @brief API CorTst_Init service called with wrong parameter. */
#define CORTST_E_INIT_FAILED            (0x00U)
/** @brief A particular API is called in an unexpected state. */
#define CORTST_E_STATUS_FAILURE         (0x01U)
/** @brief API service called with wrong parameter range. */
#define CORTST_E_PARAM_INVALID          (0x11U)
/** @brief API called without Core Test initialization. */
#define CORTST_E_UNINIT                 (0x20U)
/** @brief API service CorTst_Init() called again without a CorTst_DeInit() in between. */
#define CORTST_E_ALREADY_INITIALIZED    (0x23U)
/** @brief API service called with a NULL pointer. */
#define CORTST_E_PARAM_POINTER          (0x24U)
/** @} */

/*==================================================================================================
 *                            LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/
/**
 * @brief       This is a status value returned by the API CorTst_GetState().
 * @sw_type     sw_arch
 */
typedef enum
{
    CORTST_ABORT        = 0x00U, /**< The Core Test has been cancelled by API CorTst_Abort(). */
    CORTST_INIT         = 0x01U, /**< The Core Test is initialized and can be started. */
    CORTST_UNINIT       = 0x02U, /**< The Core Test can be initialized. */
    CORTST_RUNNING_BGND = 0x03U  /**< The Core Test is currently executed. */
} CorTst_StateType;

/**
 * @brief       This is the type of the Core Test test return if a status is returned from API to
 *              the caller of the API.
 * @sw_type     sw_arch
 */
typedef enum
{
    CORTST_E_NOT_OK     = 0x00U, /**< The Core Test detected at least one single test errors. */
    CORTST_E_OKAY       = 0x01U, /**< The Core Test passed without errors. */
    CORTST_E_NOT_TESTED = 0x02U  /**< There is no Core Test result available (default). */
} CorTst_ResultType;

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/
/**
 * @brief       This is the type of the Core Test return value if a checksum/signature is returned
 *              from API to the caller of the API.
 * @sw_type     sw_arch
 */
typedef uint32 CorTst_CsumSignatureType;

/**
 * @brief       This is the type of the parameter (Id) used for a specific test configuration to run.
 * @sw_type     sw_detail
 *
 * @trace       YTDT_CorTst_857
 */
typedef uint32 CorTst_TestIdType;

/**
 * @brief       This is the type of the parameter (Id) used for a specific foreground test
 *              configuration to run.
 * @sw_type     sw_arch
 */
typedef uint32 CorTst_TestIdFgndType;

/**
 * @brief       This is the type of the Test Interval Id.
 * @sw_type     sw_detail
 *
 * @trace       YTDT_CorTst_858
 */
typedef uint32 CorTst_TestIntervalIdType;

/**
 * @brief       This is the type of the parameter (Id) used for a specific atomic test to run.
 * @sw_type     sw_detail
 *
 * @trace       YTDT_CorTst_856
 */
typedef uint32 CorTst_TestIdAtomicType;

/**
 * @brief       This structure contains all the needed data to configure a foreground test.
 * @sw_type     sw_detail
 *
 * @trace       YTDT_CorTst_855
 */
typedef struct
{
    boolean AtomicTestEn[CORTST_ATOMIC_TEST_NUM];   /**< Array of atomic test enable flags. */
} CorTst_FgndConfigType;

/**
 * @brief       This structure contains all the needed data to configure a background test.
 * @sw_type     sw_detail
 *
 * @trace       YTDT_CorTst_853
 */
typedef struct
{
    boolean AtomicTestEn[CORTST_ATOMIC_TEST_NUM];   /**< Array of atomic test enable flags. */
} CorTst_BgndConfigType;

/**
 * @brief       Configuration data structure of the CorTst module.
 * @sw_type     sw_arch
 */
typedef struct
{
    CorTst_TestIdFgndType FgndConfigNum;        /**< Number of foreground test configurations. */
    CorTst_TestIdType BgndConfigNum;            /**< Number of background test configurations. */
    const CorTst_FgndConfigType *FgndConfig;    /**< Pointer to the foreground test configuration. */
    const CorTst_BgndConfigType *BgndConfig;    /**< Pointer to the background test configuration. */
} CorTst_ConfigType;

/**
 * @brief       Type for test signature in background mode.
 * @sw_type     sw_arch
 */
typedef struct
{
    CorTst_TestIntervalIdType TestIntervalId;   /**< The background test interval ID. */
    CorTst_CsumSignatureType Signature;         /**< The background test signature. */
} CorTst_CsumSignatureBgndType;

/**
 * @brief       This is the type of the Core Test test return if a status is returned from API to
 *              the caller of the API.
 * @sw_type     sw_arch
 */
typedef struct
{
    CorTst_TestIntervalIdType TestIntervalId;   /**< The test interval ID. */
    CorTst_ResultType Result;                   /**< The test result. */
} CorTst_ErrOkType;

/**
 * @brief       State structure for the CorTst driver
 * @sw_type     sw_detail
 *
 * @trace       YTDT_CorTst_854
 */
typedef struct
{
    const CorTst_ConfigType *CorTstConfig;      /**< Pointer to the configuration data */
    CorTst_StateType State;                     /**< Current Core Test execution state */
    CorTst_TestIdType TestId;                   /**< Last ID of the test configuration to be executed */
    CorTst_TestIdAtomicType AtomicTestId;       /**< Last atomic test ID */
    CorTst_TestIntervalIdType TestIntervalId;   /**< Last test interval ID in background mode */
    CorTst_ErrOkType ErrOk;                     /**< Last interval ID and test result in background mode */
    CorTst_ResultType AtomicResults[CORTST_ATOMIC_TEST_NUM];    /**< All atomic test results */
    CorTst_TestIdFgndType FgndSignErrCnt;       /**< Signature error counters in foreground mode */
    CorTst_CsumSignatureType FgndAccumSign;     /**< Accumulate signature of the last executed Core Test in foreground mode */
    CorTst_ResultType BgndResult;               /**< Last test result in background mode */
    CorTst_TestIdType BgndSignErrCnt;           /**< Signature error counters in background mode */
    CorTst_CsumSignatureType BgndAccumSign;     /**< Accumulate signature of the last executed Core Test in background mode */
    CorTst_CsumSignatureBgndType BgndSign;      /**< Signature of the last executed Core Test in background mode */
} CorTst_DriverStateType;

#ifdef __cplusplus
}
#endif

#endif /* CORTST_TYPES_H */

/** @} */

