/**
 * @file    CorTst_Compiler.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR CorTst module compiler
 * @details API implementation for CorTst driver
 *
 * @addtogroup CORTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Core
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifndef CORTST_COMPILER_H
#define CORTST_COMPILER_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 0341 Rule-20.10    : Using the stringify operator '#'.
 */

/**
 * MR12 Rule-13.5 VIOLATION: The '#' preprocessing operator (sometimes referred to as the stringify operator) has been used in the definition of a macro.
 */

/* PRQA S 0341 EOF */
/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CORTST_COMPILER_VENDOR_ID                      (180)
#define CORTST_COMPILER_AR_RELEASE_MAJOR_VERSION       (4)
#define CORTST_COMPILER_AR_RELEASE_MINOR_VERSION       (4)
#define CORTST_COMPILER_AR_RELEASE_REVISION_VERSION    (0)
#define CORTST_COMPILER_SW_MAJOR_VERSION               (2)
#define CORTST_COMPILER_SW_MINOR_VERSION               (0)
#define CORTST_COMPILER_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6100100)
#define CST_THUMB .thumb
#define CST_THUMB2 .thumb; .syntax unified
#define CST_TYPE(name, ident) .type name, %ident
#define CST_OPCODE_START /*opcode start*/
#define CST_OPCODE_END /*opcode end*/
#define CST_LABEL
#define CST_CODE
#define CST_WORD .word
#define CST_SHORT .hword
#define CST_EXPORT .global
#define CST_EXTERN .global
/* The ABI for the Arm Architecture suggests that code maintains
 * eight-byte stack alignment at its interfaces. A warning may occur:
 * Warning: L6306W: '~PRES8' section foo.o(.text) should not use the address
 * of 'REQ8' function foobar. So try to omit this warning. */
#define CST_PRES8 .eabi_attribute Tag_ABI_align_preserved, 1
#define CST_ALIGN_POWER_OF_TWO(POWER) .balign (1 << POWER)
#define CST_LTORG .ltorg
#define CST_ALLOC_BYTES(N) .space N
/* has to be at the beginning of a line */
#define CST_SET(NAME,VALUE) .equ NAME, VALUE
/* has to be at the beginning of a line */
#define CST_DEFINE(NAME,VALUE) .equ NAME, VALUE
#define CST_SECTION_EXEC(SECTION_NAME) .section .SECTION_NAME,"ax"
#define CST_SECTION_DATA(SECTION_NAME) .section .SECTION_NAME,"aw"
#define CST_SECTION_CONST(SECTION_NAME) .section .SECTION_NAME,"a"
#define CST_SECTION_EXEC_W(SECTION_NAME) .section .SECTION_NAME,"awx"
#define CST_SECTION_DATA_UNINIT(SECTION_NAME) .section .SECTION_NAME,"aw"
#define CST_SECTION_EXEC_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC(SECTION_NAME)
#define CST_SECTION_DATA_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA(SECTION_NAME)
#define CST_SECTION_CONST_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_CONST(SECTION_NAME)
#define CST_SECTION_EXEC_W_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC_W(SECTION_NAME)
#define CST_SECTION_DATA_UNINIT_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA_UNINIT(SECTION_NAME)
#define CST_FILE_END
/* C/C++ compiler macros for placing a variable in an assembly defined section */
#define CST_PLACE_IN_SECTION_HELPER(SECTION_NAME) __attribute__ (( section(#SECTION_NAME) ))
#define CST_PLACE_IN_SECTION(SECTION_NAME) CST_PLACE_IN_SECTION_HELPER(.SECTION_NAME)
#elif defined (__GNUC__)
#define CST_THUMB .thumb
#define CST_THUMB2 .thumb; .syntax unified
#define CST_TYPE(name, ident) .type name, %ident
#define CST_OPCODE_START /*opcode start*/
#define CST_OPCODE_END /*opcode end*/
#define CST_LABEL
#define CST_CODE
#define CST_WORD .long
#define CST_SHORT .short
#define CST_EXPORT .global
#define CST_EXTERN .global
/* The ABI for the Arm Architecture suggests that code maintains
 * eight-byte stack alignment at its interfaces. A warning may occur:
 * Warning: L6306W: '~PRES8' section foo.o(.text) should not use the address
 * of 'REQ8' function foobar. So try to omit this warning. */
#define CST_PRES8 .eabi_attribute Tag_ABI_align_preserved, 1
#define CST_ALIGN_POWER_OF_TWO(POWER) .balign (1 << POWER)
#define CST_LTORG .ltorg
#define CST_ALLOC_BYTES(N) .space N
/* has to be at the beginning of a line */
#define CST_SET(NAME,VALUE) .set NAME, VALUE
/* has to be at the beginning of a line */
#define CST_DEFINE(NAME,VALUE) .set NAME, VALUE
#define CST_SECTION_EXEC(SECTION_NAME) .section .SECTION_NAME,"ax"
#define CST_SECTION_DATA(SECTION_NAME) .section .SECTION_NAME,"aw"
#define CST_SECTION_CONST(SECTION_NAME) .section .SECTION_NAME,"a"
#define CST_SECTION_EXEC_W(SECTION_NAME) .section .SECTION_NAME,"awx"
#define CST_SECTION_DATA_UNINIT(SECTION_NAME) .section .SECTION_NAME,"aw"
#define CST_SECTION_EXEC_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC(SECTION_NAME)
#define CST_SECTION_DATA_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA(SECTION_NAME)
#define CST_SECTION_CONST_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_CONST(SECTION_NAME)
#define CST_SECTION_EXEC_W_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC_W(SECTION_NAME)
#define CST_SECTION_DATA_UNINIT_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA_UNINIT(SECTION_NAME)
#define CST_FILE_END
/* C/C++ compiler macros for placing a variable in an assembly defined section */
#define CST_PLACE_IN_SECTION_HELPER(SECTION_NAME) __attribute__ (( section(#SECTION_NAME) ))
#define CST_PLACE_IN_SECTION(SECTION_NAME) CST_PLACE_IN_SECTION_HELPER(.SECTION_NAME)
#elif defined (__IAR_SYSTEMS_ASM__) || defined (__IAR_SYSTEMS_ICC__)
#define CST_THUMB CODE16
#define CST_THUMB2 THUMB
#define CST_TYPE(name, ident) /*not needed*/
#define CST_OPCODE_START DATA
#define CST_OPCODE_END THUMB
#define CST_LABEL DATA
#define CST_CODE THUMB
#define CST_WORD DC32
#define CST_SHORT DC16
#define CST_EXPORT EXPORT
#define CST_EXTERN EXTERN
/* The ABI for the Arm Architecture suggests that code maintains
 * eight-byte stack alignment at its interfaces. A warning may occur:
 * Warning: L6306W: '~PRES8' section foo.o(.text) should not use the address
 * of 'REQ8' function foobar. So try to omit this warning. */
#define CST_PRES8
#define CST_ALIGN_POWER_OF_TWO(POWER) ALIGNROM POWER
#define CST_LTORG LTORG
#define CST_ALLOC_BYTES(N) DS8 N
#define CST_SET(NAME,VALUE) NAME: SET VALUE
#define CST_DEFINE(NAME,VALUE) NAME: DEFINE VALUE
#define CST_SECTION_EXEC(SECTION_NAME) section .SECTION_NAME :CODE
#define CST_SECTION_DATA(SECTION_NAME) section .SECTION_NAME :DATA
#define CST_SECTION_CONST(SECTION_NAME) section .SECTION_NAME :CONST
#define CST_SECTION_EXEC_W(SECTION_NAME) section .SECTION_NAME :DATA
#define CST_SECTION_DATA_UNINIT(SECTION_NAME) section .SECTION_NAME :DATA
#define CST_SECTION_EXEC_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC(SECTION_NAME) (ALIGNMENT)
#define CST_SECTION_DATA_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA(SECTION_NAME) (ALIGNMENT)
#define CST_SECTION_CONST_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_CONST(SECTION_NAME) (ALIGNMENT)
#define CST_SECTION_EXEC_W_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC_W(SECTION_NAME) (ALIGNMENT)
#define CST_SECTION_DATA_UNINIT_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA_UNINIT(SECTION_NAME) (ALIGNMENT)
#define CST_FILE_END END
#define APSR_nzcvq APSR
/* C/C++ compiler macros for placing a variable in an assembly defined section */
#define CST_PLACE_IN_SECTION_HELPER_0(X) #X
#define CST_PLACE_IN_SECTION_HELPER_1(SECTION_NAME) CST_PLACE_IN_SECTION_HELPER_0(location = #SECTION_NAME)
#define CST_PLACE_IN_SECTION_HELPER_2(SECTION_NAME) CST_PLACE_IN_SECTION_HELPER_1(.SECTION_NAME)
#define CST_PLACE_IN_SECTION(SECTION_NAME) _Pragma(CST_PLACE_IN_SECTION_HELPER_2(SECTION_NAME))
#elif defined (__ghs__)
#define function $function
#define object $object
#define CST_THUMB .thumb
#define CST_THUMB2 .thumb2
#define CST_TYPE(name, ident) .type name, ident
#define CST_OPCODE_START /*opcode start*/
#define CST_OPCODE_END /*opcode end*/
#define CST_LABEL
#define CST_CODE
#define CST_WORD .long
#define CST_SHORT .short
#define CST_EXPORT .global
#define CST_EXTERN .global
/* The ABI for the Arm Architecture suggests that code maintains
 * eight-byte stack alignment at its interfaces. A warning may occur:
 * Warning: L6306W: '~PRES8' section foo.o(.text) should not use the address
 * of 'REQ8' function foobar. So try to omit this warning. */
#define CST_PRES8
#define CST_ALIGN_POWER_OF_TWO(POWER) .align (1 << POWER)
#define CST_LTORG .ltorg
#define CST_ALLOC_BYTES(N) .space N
/* has to be at the beginning of a line */
#define CST_SET(NAME,VALUE) .set NAME, VALUE
/* has to be at the beginning of a line */
#define CST_DEFINE(NAME,VALUE) .set NAME, VALUE
#define CST_SECTION_EXEC(SECTION_NAME) .section .SECTION_NAME,"ax"
#define CST_SECTION_DATA(SECTION_NAME) .section .SECTION_NAME,"aw"
#define CST_SECTION_CONST(SECTION_NAME) .section .SECTION_NAME,"a"
#define CST_SECTION_EXEC_W(SECTION_NAME) .section .SECTION_NAME,"awx"
#define CST_SECTION_DATA_UNINIT(SECTION_NAME) .section .SECTION_NAME,"awb"
#define CST_SECTION_EXEC_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC(SECTION_NAME)
#define CST_SECTION_DATA_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA(SECTION_NAME)
#define CST_SECTION_CONST_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_CONST(SECTION_NAME)
#define CST_SECTION_EXEC_W_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_EXEC_W(SECTION_NAME)
#define CST_SECTION_DATA_UNINIT_ALIGN(SECTION_NAME, ALIGNMENT) CST_SECTION_DATA_UNINIT(SECTION_NAME)
#define CST_FILE_END
/* C/C++ compiler macros for placing a variable in an assembly defined section */
#define CST_PLACE_IN_SECTION_HELPER(SECTION_NAME) __attribute__ (( section(#SECTION_NAME) ))
#define CST_PLACE_IN_SECTION(SECTION_NAME) CST_PLACE_IN_SECTION_HELPER(.SECTION_NAME)
#define APSR_nzcvq APSR
#else
#error "Unknown compiler!"
#endif

#define CST_BYTES_2     1
#define CST_BYTES_4     2
#define CST_BYTES_8     3
#define CST_BYTES_16    4
#define CST_BYTES_32    5
#define CST_BYTES_64    6
#define CST_BYTES_128   7
#define CST_ALIGN_BYTES_2   CST_ALIGN_POWER_OF_TWO(1)
#define CST_ALIGN_BYTES_4   CST_ALIGN_POWER_OF_TWO(2)
#define CST_ALIGN_BYTES_8   CST_ALIGN_POWER_OF_TWO(3)
#define CST_ALIGN_BYTES_16  CST_ALIGN_POWER_OF_TWO(4)
#define CST_ALIGN_BYTES_32  CST_ALIGN_POWER_OF_TWO(5)
#define CST_ALIGN_BYTES_64  CST_ALIGN_POWER_OF_TWO(6)
#define CST_ALIGN_BYTES_128 CST_ALIGN_POWER_OF_TWO(7)

#ifdef CST_ENABLE_QUA_FAULT_INJECTION
#include "CorTst_TestMacro.h"
#else
#define CST_QUA_FAULT_INJECTION(x)
#endif

/**
* use this to write 32bit opcode. This is one (HIGH) is the first half and it should go first.
*/
#define CST_OPCODE32_HIGH(DATA) CST_SHORT DATA
/**
* use this to write 32bit opcode. This is one (LOW) is the second half and it should go after CST_OPCODE32_HIGH.
*/
#define CST_OPCODE32_LOW(DATA) CST_SHORT DATA
/**
* use this to write 16bit opcode.
*/
#define CST_OPCODE16(DATA) CST_SHORT DATA
#define CST_PREPARE_PIPELINE \
    CPSID i;\
    NOP;\
    ISB;
#define CST_FLUSH_PIPELINE \
    NOP.W;\
    ISB;

#ifdef __cplusplus
}
#endif

#endif /* CORTST_COMPILER_H */

/** @} */

