/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
* Checks floating-point numbers rounding to floating-point integral numbers 
* provided by cm33fpu_sp_round.v and cm33fpu_sp_cvt.v modules. Also checks
* absolute value calculation.
*
* In addition, the following functionality is tested:
* - handling special values (+-sNaN, +-qNaN, +-Inf, +-0)
* - handling signed values
* - handling denormalized input
* - producing the FPSCR[IXC, IOC, IDC] flags
*
* SP FPU Decoder coverage:
* ------------------------
*   - VRINT{A,N,P,M}.F32 
*   - VRINTZ.F32
*   - VRINTR.F32
*   - VRINTX.F32
*   - VABS.F32
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FPU_ENABLE==1)

    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_SpfpuRoundingTest
    
    /* Symbols defined outside but used within current module */
    CST_EXTERN m33_cst_test_tail_fpu_end
    CST_EXTERN m33_cst_sum_fpu_s10_s31_to_r1_destr_r2
    CST_EXTERN m33_cst_sum_fpu_s20_s31_to_r1_destr_r2
    
    /* Pre-signature */
    CST_SET(PRESIGNATURE_SEED_0, 0x7049)
    CST_SET(PRESIGNATURE_SEED_1, 0x3184)
    
    /* Values stored in registers:
       - R1  ... accumulated signature
       - R12 ... FPSCR register
    */

   /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
   /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_SpfpuRoundingTest" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_TYPE(M33_Cst_SpfpuRoundingTest, function)
    CST_THUMB2
M33_Cst_SpfpuRoundingTest:

    PUSH    {R4-R12,R14}
    MRS     R1,CONTROL  /* Store CONTROL prior first FPU instruction */
    PUSH    {R1}
    VPUSH   {S16-S31}

    /*-----------------------------------------------------------------------*/
    /* Test - preparation                                                    */
    /*-----------------------------------------------------------------------*/
    MOV     R1,#PRESIGNATURE_SEED_0
    MOVT    R1,#PRESIGNATURE_SEED_1
        
    /*-----------------------------------------------------------------------*/
    /* Test - start                                                          */
    /*-----------------------------------------------------------------------*/
    VMRS        R12,FPSCR   /* Store FPSCR register */
    
    
    /*************************************************************************/
    /* Round a floating-point value to an integral floating-point            */
    /*************************************************************************/
    /* Let assume Floating-point numbers with exponent exp between 0 and 23 
       which have the following precision limitations when rounded.
     
      NumA |exp | mantissa1       |            NumB |exp | mantissa3      | 
     ------|----|-----------------|-----      ------|----|----------------|-------
     Run1 :| 22 |  0b0000...000/1 | inexact   Run1: | 22 | 0b1111...111/1 | inexact
     Run2 :| 21 |  0b0000...00/10 | inexact   Run2: | 21 | 0b1111...11/01 | inexact
     Run3 :| 20 |  0b0000...0/100 | inexact   Run3: | 20 | 0b1111...1/011 | inexact
     Run4 :| 19 |  0b0000.../1000 | inexact   Run4: | 19 | 0b1111.../0111 | inexact
                                                                           
     Run20:|  3 |  0x000/1...0000 | inexact   Run20:|  3 | 0x111/0...1111 | inexact
     Run21:|  2 |  0b00/10...0000 | inexact   Run21:|  2 | 0x11/01...1111 | inexact
     Run22:|  1 |  0b0/100...0000 | inexact   Run22:|  1 | 0b1/011...1111 | inexact
     Run23:|  0 |  0b/1000...0000 | inexact   Run23:|  0 | 0b/0111...1111 | inexact      
     */  
    
    MOV     R10,#((23+127)<<23)     /* 23(exp)+127(bias) */
    MOV     R11,#(1)                /* fraction bit */
    MOV     R5,#(1<<23)             /* R5=0x00800000 */
    SUB     R5,R5,R11               /* R5=0x007FFFFF Set fraction bits to 1 */
    MOV     R6,R5                   /* R6=0x007FFFFF Set fraction bits to 1 */
m33_cst_spfpu_rounding_test_loop1:

    /* Prepare NumA & NumB for rounding */    
    SUB     R10,R10,#(1<<23)        /* exp = (exp - 1) */
    
    ORR     R3,R10,R11              /* +NumA -> S0 */
    ORR     R4,R3,#(1<<31)          /* -NumA -> S1 */
    ORR     R7,R10,R5               /* +NumB -> S2 */
    ORR     R8,R7,#(1<<31)          /* -NumB -> S3 */
    
    VMOV    D0,R3,R4
    VMOV    D1,R7,R8 
    
    /* Floating-point to Integral */
    VRINTA.F32  S31,S0      /* Round to Nearest with Ties to Away */
    VRINTA.F32  S30,S1      /* Round to Nearest with Ties to Away */
    VRINTN.F32  S29,S0      /* Round to Nearest with Ties to Even */
    VRINTN.F32  S28,S1      /* Round to Nearest with Ties to Even */
    VRINTP.F32  S27,S0      /* Round toward +Infinity */
    VRINTP.F32  S26,S1      /* Round toward +Infinity */
    VRINTM.F32  S25,S0      /* Round toward -Infinity */
    VRINTM.F32  S24,S1      /* Round toward -Infinity */
    VRINTZ.F32  S23,S0      /* Round toward Zero */
    VRINTZ.F32  S22,S1      /* Round toward Zero */
    
    VRINTA.F32  S21,S2      /* Round to Nearest with Ties to Away */
    VRINTA.F32  S20,S3      /* Round to Nearest with Ties to Away */
    VRINTN.F32  S19,S2      /* Round to Nearest with Ties to Even */
    VRINTN.F32  S18,S3      /* Round to Nearest with Ties to Even */
    VRINTP.F32  S17,S2      /* Round toward +Infinity */
    VRINTP.F32  S16,S3      /* Round toward +Infinity */
    VRINTM.F32  S15,S2      /* Round toward -Infinity */
    VRINTM.F32  S14,S3      /* Round toward -Infinity */
    VRINTZ.F32  S13,S2      /* Round toward Zero */
    VRINTZ.F32  S12,S3      /* Round toward Zero */
    VABS.F32    S11,S2      /* Get absolute value */
    VABS.F32    S10,S3      /* Get absolute value */
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s10_s31_to_r1_destr_r2
    
    /* Prepare new mantissas for next run */
    BIC     R6,R6,R11           
    SUB     R5,R6,#1            /* mantissa2 */
    LSL     R11,R11,#1          /* mantissa1 - move fraction bit */
    
    CMP     R11,#(1<<22)
    BLS     m33_cst_spfpu_rounding_test_loop1   /* loop 23x */
    
    
    /* We will additionally check rounding for the following modes 
      (RMode bits [22:23]) in the FPSCR
        Run1: 0b00 Round to Nearest             (RN) mode 
        Run1: 0b01 Round towards Plus Infinity  (RP) mode
        Run2: 0b10 Round towards Minus Infinity (RM) mode
              0b11 Round towards Zero           (RZ) mode (tested above)
     */
    BIC     R3,R10,#(1<<23)     /* R3=0x3F000000 (+0.5) */
    ORR     R4,R3,#(1<<31)      /* R4=0xBF000000 (-0.5) */
    SUB     R7,R3,#1            /* R7=0x3EFFFFFF (+0.49999..) */
    ORR     R8,R7,#(1<<31)      /* R8=0xBEFFFFFF (-0.49999..) */
    
    VMOV    D0,R3,R4
    VMOV    D1,R7,R8
    
    MOV     R0,#0   /* Initialize RMode to Round to Nearest  */
m33_cst_spfpu_rounding_test_loop_fpscr_rm_mode:
    
    VMSR    FPSCR,R0            /* Write RMode and clear FPSCR */
    
    VRINTR.F32  S31,S0          /* No signal -> FPSCR.IXC=0 */
    VRINTR.F32  S30,S1          /* No signal -> FPSCR.IXC=0 */
    VRINTR.F32  S29,S2          /* No signal -> FPSCR.IXC=0 */
    VRINTR.F32  S28,S3          /* No signal -> FPSCR.IXC=0 */
    
    VRINTZ.F32  S27,S0          /* Round toward Zero No signal -> FPSCR.IXC=0 */
    VRINTZ.F32  S26,S1          /* Round toward Zero No signal -> FPSCR.IXC=0 */
    VRINTZ.F32  S25,S2          /* Round toward Zero No signal -> FPSCR.IXC=0 */
    VRINTZ.F32  S24,S3          /* Round toward Zero No signal -> FPSCR.IXC=0 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    VRINTX.F32  S23,S0          /* Signal inexact -> FPSCR.IXC=1 */
    VRINTX.F32  S22,S1          /* Signal inexact -> FPSCR.IXC=1 */
    VRINTX.F32  S21,S2          /* Signal inexact -> FPSCR.IXC=1 */
    VRINTX.F32  S20,S3          /* Signal inexact -> FPSCR.IXC=1 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s20_s31_to_r1_destr_r2
    
    ADD     R0,R0,#(1<<22)      /* Prepare new RMode bits */
    CMP     R0,#(3<<22)
    BLS     m33_cst_spfpu_rounding_test_loop_fpscr_rm_mode
    
    
    /* We will test handling of:
      +0, -0, +Inf, -Inf, +qNaN, -qNaN, +sNaN, -sNaN, +Denorm, -Denorm */
    MOV     R3,#0x7f800000  /* +Inf */
    ORR     R4,R3,#(1<<31)  /* -Inf */
    ORR     R7,R3,#(1<<22)  /* +qNaN */
    ORR     R8,R4,#(1<<22)  /* -qNaN */
    ORR     R9,R3,#(1<<21)  /* +sNaN */
    ORR     R10,R4,#(1<<1)  /* -sNaN */
    MOV     R5,#0           /* +0 */
    ORR     R6,R5,#(1<<31)  /* -0 */
    MOV     R0,#(1<<22)     /* + Denorm */
    ORR     R2,R6,#(1<<0)   /* - Denorm */
    
    VMOV    D5,R0,R2
    VMOV    D6,R3,R4
    VMOV    D7,R5,R6 
    VMOV    D8,R7,R8
    VMOV    D9,R9,R10
    
    VMSR    FPSCR,R5            /* Write RMode and clear FPSCR */
    
    VRINTA.F32  S31,S13
    VRINTA.F32  S30,S12
    VRINTZ.F32  S29,S15
    VRINTZ.F32  S28,S14
    VRINTR.F32  S27,S17
    VRINTR.F32  S26,S16
    VRINTX.F32  S25,S19          /* Signal invalid -> FPSCR.IOC=1 */
    VRINTX.F32  S24,S18          /* Signal invalid -> FPSCR.IOC=1 */
    /* Denormalized input */
    VRINTM.F32  S23,S11
    VRINTP.F32  S22,S10
    
    VMRS    R5,FPSCR 
    ORR     R5,R5,#(1<<24)      /* Set Flush-to-zero */
    VMSR    FPSCR,R5            /* Write RMode and clear FPSCR */
    
    VRINTM.F32  S21,S11         /* Signal input denormal -> FPSCR.IDC=1 */
    VRINTP.F32  S20,S10         /* Signal input denormal -> FPSCR.IDC=1 */
    
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            /* Load FPSCR register */
    EOR     R1,R2,R1,ROR #1
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s20_s31_to_r1_destr_r2
    
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
m33_cst_spfpu_rounding_test_end:
    VMSR        FPSCR,R12   /* Restore FPSCR register */
    /* Test result is returned in R0, according to the conventions */
    MOV         R0,R1 
    B           m33_cst_test_tail_fpu_end
    

    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_FPU_ENABLE */

    CST_FILE_END

