/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
*
* Covers FPU load/store operation type:
*   - Load/Store {register,immediate,literal}
*   - Moves between ARM core registers and FPU registers
*
* Load/store operations are done to/from the following address locations:
*  - CST_RAM_TARGET0
*  - CST_RAM_TARGET1
*  - Stack
*
* SP FPU Decoder coverage:
* ------------------------
*  - VLDR
*  - VLDR (literal)
*  - VSTR
*  - VSTM{mode}  (VSTMIA,VSTMDB,VPUSH)
*  - VLDM{mode}  (VLDMIA,VLDMDB,VPOP)
*  - VMOV (between two ARM core registers and two single-precision registers)   
*  - VMOV (between two ARM core registers and a doubleword register)
*  - VMOV (between ARM core register and single-precision register)
*  - VMOV (scalar to ARM core register)
*  - VMOV (ARM core register to scalar)
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FPU_ENABLE==1)

    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_SpfpuLoadstoreTest
    
    /* Symbols defined outside but used within current module */
    CST_EXTERN CST_RAM_TARGET0
    CST_EXTERN CST_RAM_TARGET1
    CST_EXTERN m33_cst_test_tail_fpu_end
    CST_EXTERN m33_cst_write_fpu_regs_from_r0
    CST_EXTERN m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    CST_SET(PRESIGNATURE_SEED_0, 0x1485)
    CST_SET(PRESIGNATURE_SEED_1, 0xFC0A)
    
    /* Values stored in registers:
       - R0  ... 0
       - R1  ... accumulated signature
    */

    /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
    /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_SpfpuLoadstoreTest" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_TYPE(M33_Cst_SpfpuLoadstoreTest, function)
    CST_THUMB2
M33_Cst_SpfpuLoadstoreTest:

    PUSH    {R4-R12,R14}
    MRS     R1,CONTROL  /* Store CONTROL prior first FPU instruction */
    PUSH    {R1}
    VPUSH   {S16-S31}

    /*-----------------------------------------------------------------------*/
    /* Test - preparation                                                    */
    /*-----------------------------------------------------------------------*/
    MOV     R1,#PRESIGNATURE_SEED_0
    MOVT    R1,#PRESIGNATURE_SEED_1
    
        
    /*-----------------------------------------------------------------------*/
    /* Test - start                                                          */
    /*-----------------------------------------------------------------------*/
    /**************************************************************************
    * Instructions: 
    *   - VSTR   
    *   - VLDR   
    ***************************************************************************/
    /* Clear target memories */
    MOV     R0,#0
    LDR     R8,=CST_RAM_TARGET0
    BL      m33_cst_spfpu_loadstore_test_clear_target_memory
    ADD     R7,R8,#0    /* R7 -> pointer to the beginning of CST_RAM_TARGET0 */
    LDR     R8,=CST_RAM_TARGET1
    BL      m33_cst_spfpu_loadstore_test_clear_target_memory
    ADD     R8,R8,#60   /* R8 -> pointer to the end of CST_RAM_TARGET1 */
    
    
    /* Initialize S0-S31 register */
    BL      m33_cst_spfpu_loadstore_test_init_s0_s31
    
    
    /* Store S31-S24 to CST_RAM_TARGET0 */
    ADD     R6,R7,#(1020 + 8)   /* Data is stored in the CST_RAM_TARGET0 
                                    memory as follows:  */
    VSTR    D15,[R7,#0]         /* [Rn,#0]  - S30-S31  */
    VSTR    S29,[R6,#-1020]     /* [Rn,#8]  - S29      */
    VSTR    S28,[R7,#12]        /* [Rn,#12] - S28      */
    VSTR    D13,[R6,#-1012]     /* [Rn,#16] - S26-S27  */
    VSTR    S25,[R7,#24]        /* [Rn,#24] - S25      */
    VSTR    S24,[R6,#-1000]     /* [Rn,#28] - S24      */
    
    /* Store S7-S0 to CST_RAM_TARGET1 */
    SUB     R9,R8,#(1020+36)    /* Data is stored in the CST_RAM_TARGET1 
                                   memory as follows:  */
    VSTR    S7,[R8,#-60]        /* [Rn,#0]  - S7-S6    */     
    VSTR    S6,[R9,#1000]       /* [Rn,#4]  - S6       */
    VSTR    D2,[R8,#-52]        /* [Rn,#8]  - S5-S4    */
    VSTR    S3,[R9,#1012]       /* [Rn,#16] - S3       */
    VSTR    S2,[R8,#-40]        /* [Rn,#20] - S2       */
    VSTR    D0,[R9,#1020]       /* [Rn,#24] - S1-S0    */
    
    /* Store S15-S8 to CST_RAM_TARGET0 */
    ADD     R6,R7,#(340 + 32)   /* Data is stored in the CST_RAM_TARGET0 
                                   memory as follows:  */
    VSTR    D7, [R6,#-340]      /* [Rn,#32] - S15-S14  */
    VSTR    S13,[R7,#40]        /* [Rn,#40] - S13      */
    VSTR    S12,[R6,#-328]      /* [Rn,#44] - S12      */
    VSTR    D5, [R7,#48]        /* [Rn,#48] - S11-S10  */
    VSTR    S9, [R6,#-316]      /* [Rn,#56] - S9       */
    VSTR    S8, [R7,#60]        /* [Rn,#60] - S8       */
    
    /* Store S15-S8 to CST_RAM_TARGET1 */
    SUB     R9,R8,#(340 + 4)    /* Data is stored in the CST_RAM_TARGET1
                                   memory as follows:   */
    VSTR    S23,[R8,#-28]       /* [Rn,#32] - S23      */
    VSTR    S22,[R9,#320]       /* [Rn,#36] - S22      */
    VSTR    D10,[R8,#-20]       /* [Rn,#40] - S21-S20  */
    VSTR    S19,[R9,#332]       /* [Rn,#48] - S19      */
    VSTR    S18,[R8,#-8]        /* [Rn,#52] - S18      */
    VSTR    D8,[R9,#340]        /* [Rn,#56] - S17-S16  */
    
    /* Clear all FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    
    /* Load S0-S7 from CST_RAM_TARGET0 */
    ADD     R6,R7,#(680)        /* Data is loaded from the CST_RAM_TARGET0
                                   memory as follows:  */
    VLDR    S0,[R7,#4]          /* [Rn,#4]  - S0       */     
    VLDR    S1,[R6,#-680]       /* [Rn,#0]  - S1       */
    VLDR    D1,[R7,#8]          /* [Rn,#8]  - S3-S2    */
    VLDR    S4,[R6,#-660]       /* [Rn,#20] - S4       */
    VLDR    S5,[R7,#16]         /* [Rn,#16] - S5       */
    VLDR    D3,[R6,#-656]       /* [Rn,#24] - S7-S6    */
    
    /* Load S24-S31 from CST_RAM_TARGET1 */
    SUB     R9,R8,#(680 + 60)   /* Data is loaded from the CST_RAM_TARGET1
                                   memory as follows:  */
    VLDR    D12,[R9,#680]       /* [Rn,#0]  - S25-S24  */
    VLDR    S26,[R8,#-48]       /* [Rn,#12] - S26      */    
    VLDR    S27,[R9,#688]       /* [Rn,#8]  - S27      */
    VLDR    D14,[R8,#-44]       /* [Rn,#16] - S29-S28  */    
    VLDR    S30,[R9,#708]       /* [Rn,#28] - S30      */    
    VLDR    S31,[R8,#-36]       /* [Rn,#24] - S31      */  

    /* Load S23-S16 from CST_RAM_TARGET0 */
    ADD     R6,R7,#(816 + 32)   /* Data is loaded from the CST_RAM_TARGET0
                                   memory as follows:  */
    VLDR    S16,[R7,#36]        /* [Rn,#36] - S16      */
    VLDR    S17,[R6,#-816]      /* [Rn,#32] - S17      */
    VLDR    D9, [R7,#40]        /* [Rn,#40] - S15-S14  */
    VLDR    S20,[R6,#-796]      /* [Rn,#52] - S20      */
    VLDR    S21,[R7,#48]        /* [Rn,#48] - S21      */
    VLDR    D11,[R6,#-792]      /* [Rn,#56] - S23-S22  */
    
    /* Load S15-S8 from CST_RAM_TARGET1 */
    SUB     R9,R8,#(204 + 28)    /* Data is loaded from the CST_RAM_TARGET0
                                    memory as follows:  */
    VLDR    D4, [R9,#204]        /* [Rn,#32] - S9-S8   */   
    VLDR    S10,[R8,#-16]        /* [Rn,#44] - S10     */   
    VLDR    S11,[R9,#212]        /* [Rn,#40] - S11     */   
    VLDR    D6, [R8,#-12]        /* [Rn,#48] - S13-S12 */  
    VLDR    S14,[R9,#232]        /* [Rn,#60] - S14     */  
    VLDR    S15,[R8,#-4]         /* [Rn,#56] - S15     */
    
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    
    /**************************************************************************
    * Instructions: 
    *   - VMOV (between two ARM core registers and two single-precision registers)   
    *   - VMOV (between two ARM core registers and a doubleword register)
    *   - VMOV (between ARM core register and single-precision register)
    *   - VMOV (scalar to ARM core register)
    *   - VMOV (ARM core register to scalar)
    ***************************************************************************/
    /* The following code swaps values between two single-precision 
       registers by using various VMOV instructions as follows: 
       S0  <-> S1
       S2  <-> S3
           ...
       S28 <-> S29
       S30 <-> S31 
     */
    VMOV    R2,R3,S0,S1     
    VMOV    R4,R5,D1
    CST_OPCODE_START
    /* VMOV    R6,D2[0] GHS compiler fix */    
    CST_OPCODE32_HIGH(0xEE12)    
    CST_OPCODE32_LOW(0x6B10) 
    CST_OPCODE_END    
    VMOV    R9,R10,S5,S6    
    CST_OPCODE_START
    /* VMOV    R11,D3[1] GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE33)
    CST_OPCODE32_LOW(0xBB10)
    CST_OPCODE_END
    VMOV    S0,S1,R3,R2     
    VMOV    D1,R5,R4
    CST_OPCODE_START
    /* VMOV    D2[0],R9 GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE02)
    CST_OPCODE32_LOW(0x9B10)
    CST_OPCODE_END
    VMOV    S5,S6,R6,R11    
    CST_OPCODE_START
    /* VMOV    D3[1],R10 GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE23)
    CST_OPCODE32_LOW(0xAB10)
    CST_OPCODE_END
    
    VMOV    R2,R3,D4
    CST_OPCODE_START
    /* VMOV    R4,D5[0] GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE15)
    CST_OPCODE32_LOW(0x4B10)
    CST_OPCODE_END
    VMOV    R5,R6,S11,S12
    VMOV    R9,S13
    VMOV    R10,R11,D7
    VMOV    D4,R3,R2
    CST_OPCODE_START
    /* VMOV    D5[0],R5 GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE05)
    CST_OPCODE32_LOW(0x5B10)
    CST_OPCODE_END
    VMOV    S11,S12,R4,R9
    VMOV    S13,S14,R6,R11
    CST_OPCODE_START
    /* VMOV    D7[1],R10 GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE27)
    CST_OPCODE32_LOW(0xAB10)
    CST_OPCODE_END
    
    VMOV    R2,R3,S16,S17
    VMOV    R4,R5,D9
    VMOV    R6,S20
    CST_OPCODE_START
    /* VMOV    R9,D10[1] GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE3A)
    CST_OPCODE32_LOW(0x9B10)
    CST_OPCODE_END
    VMOV    R10,R11,D11
    VMOV    D8,R3,R2
    CST_OPCODE_START
    /* VMOV    D9[0],R5 GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE09)
    CST_OPCODE32_LOW(0x5B10)
    CST_OPCODE_END
    VMOV    S19,R4
    VMOV    S20,S21,R9,R6
    VMOV    D11,R11,R10
    
    VMOV    R2,R3,D12
    CST_OPCODE_START
    /* VMOV    R4,D13[0] GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE1D)
    CST_OPCODE32_LOW(0x4B10)
    CST_OPCODE_END
    VMOV    R5,R6,S27,S28
    VMOV    R9,S29
    VMOV    R10,R11,D15
    CST_OPCODE_START
    /* VMOV    D12[0],R3 GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE0C)
    CST_OPCODE32_LOW(0x3B10)
    CST_OPCODE_END
    VMOV    S25,S26,R2,R5 
    CST_OPCODE_START
    /* VMOV    D13[1],R4 GHS compiler fix */
    CST_OPCODE32_HIGH(0xEE2D)
    CST_OPCODE32_LOW(0x4B10)
    CST_OPCODE_END
    VMOV    D14,R9,R6
    VMOV    S30,R11  
    VMOV    S31,R10
   
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
   
   
    /**************************************************************************
    * Instructions: 
    *   - VSTM{mode}  (VSTMIA,VSTMDB,VPUSH)
    *   - VLDM{mode}  (VLDMIA,VLDMDB,VPOP)
    ***************************************************************************/
    /**************************************************************************/
    /* Test with CST_RAM_TARGET0 and Stack                                   */
    /**************************************************************************/
    LDR     R8,=CST_RAM_TARGET0
    BL      m33_cst_spfpu_loadstore_test_clear_target_memory

    /* Initialize S0-S31 registers  */
    BL      m33_cst_spfpu_loadstore_test_init_s0_s31
       
    ADD     R5,R8,#128
    /* Store data and decrement address */
    VSTMDB  R5!,{S0-S31}    /* D-VD=0 imm8=32 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S31}     /* D-VD=0 imm8=32 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#124
    /* Store data and decrement address */
    VSTMDB  R5!,{S1-S31}    /* D-VD=1 imm8=31  */
    VSTMDB  R13!,{S0}       /* D-VD=0 imm8=1    VPUSH {S0} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S30}     /* D-VD=0 imm8=31  */
    VLDMIA  R13!,{S31}      /* D-VD=31 imm8=1   VPOP {S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#116
    /* Store data and decrement address */
    VSTMDB  R5!,{S3-S31}    /* D-VD=3 imm8=29  */
    VSTMDB  R13!,{S0-S2}    /* D-VD=0 imm8=3    VPUSH {S0-S2} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S28}     /* D-VD=0 imm8=29  */
    VLDMIA  R13!,{S29-S31}  /* D-VD=29 imm8=3   VPOP {S29-S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#108
    /* Store data and decrement address */
    VSTMDB  R5!,{S5-S31}    /* D-VD=5 imm8=27  */
    VSTMDB  R13!,{S0-S4}    /* D-VD=0 imm8=5    VPUSH {S0-S4} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S26}     /* D-VD=0  imm8=27 */
    VLDMIA  R13!,{S27-S31}  /* D-VD=27 imm8=5   VPOP {S27-S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#100
    /* Store data and decrement address */
    VSTMDB  R5!,{S7-S31}    /* D-VD=7 imm8=25  */
    VSTMDB  R13!,{S0-S6}    /* D-VD=0 imm8=7    VPUSH {S0-S6} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S24}     /* D-VD=0 imm8=25  */
    VLDMIA  R13!,{S25-S31}  /* D-VD=25 imm8=7   VPOP {S25-S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#92
    /* Store data and decrement address */
    VSTMDB  R5!,{S9-S31}    /* D-VD=9 imm8=23  */
    VSTMDB  R13!,{S0-S8}    /* D-VD=0 imm8=9    VPUSH {S0-S8} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S22}     /* D-VD=0  imm8=23 */
    VLDMIA  R13!,{S23-S31}  /* D-VD=23 imm8=9   VPOP {S23-S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#84
    /* Store data and decrement address */
    VSTMDB  R5!,{S11-S31}   /* D-VD=11 imm8=21 */
    VSTMDB  R13!,{S0-S10}   /* D-VD=0  imm8=11  VPUSH {S0-S10} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S20}     /* D-VD=0  imm8=21 */
    VLDMIA  R13!,{S21-S31}  /* D-VD=21 imm8=11  VPOP {S21-S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#76
    /* Store data and decrement address */
    VSTMDB  R5!,{S13-S31}   /* D-VD=13 imm8=19 */
    VSTMDB  R13!,{S0-S12}   /* D-VD=0  imm8=13  VPUSH {S0-S12} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S18}     /* D-VD=0  imm8=19 */
    VLDMIA  R13!,{S19-S31}  /* D-VD=19 imm8=13  VPOP {S19-S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R5,R8,#68
    /* Store data and decrement address */
    VSTMDB  R5!,{S15-S31}   /* D-VD=15 imm8=17 */
    VSTMDB  R13!,{S0-S14}   /* D-VD=0  imm8=15  VPUSH {S0-S14} Encoding T2 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data */
    VLDMIA  R8,{S0-S16}     /* D-VD=0  imm8=17 */
    VLDMIA  R13!,{S17-S31}  /* D-VD=17 imm8=15  VPOP {S17-S31} Encoding T2 */
    /* We need to check R5 was updated !! */
    CMP     R5,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    
    /**************************************************************************/
    /* Test with CST_RAM_TARGET1 and Stack */
    /**************************************************************************/
    LDR     R8,=CST_RAM_TARGET1
    BL      m33_cst_spfpu_loadstore_test_clear_target_memory
    
    /* --- Prepare data to be stored and loaded ---  */
    BL      m33_cst_spfpu_loadstore_test_init_s0_s31
    
    MOV     R10,R8
    /* Store data and increment address */
    VSTMIA  R10!,{D0-D15}    /* D-VD=0 imm8=32 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Load data and decrement address */
    VLDMDB  R10!,{D0-D15}    /* D-VD=0 imm8=32 */
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R10,#120
    VSTMIA  R8,{D1-D15}     /* D-VD=1  imm8=30 */
    VSTMDB  R13!,{D0}       /* D-VD=0  imm8=2   VPUSH {D0} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D14}   /* D-VD=0  imm8=30 */
    VLDMIA  R13!,{D15}      /* D-VD=30 imm8=2   VPOP {D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R8,#112
    VSTMIA  R8,{D2-D15}     /* D-VD=4  imm8=28 */
    VSTMDB  R13!,{D0-D1}    /* D-VD=0  imm8=4   VPUSH {D0-D1} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D13}   /* D-VD=0  imm8=28 */
    VLDMIA  R13!,{D14-D15}  /* D-VD=28 imm8=4   VPOP {D14-D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R8,#104
    VSTMIA  R8,{D3-D15}     /* D-VD=6  imm8=26 */
    VSTMDB  R13!,{D0-D2}    /* D-VD=0  imm8=6   VPUSH {D0-D2} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D12}   /* D-VD=0  imm8=26 */
    VLDMIA  R13!,{D13-D15}  /* D-VD=26 imm8=6   VPOP {D13-D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R8,#96
    VSTMIA  R8,{D4-D15}     /* D-VD=8 imm8=24  */
    VSTMDB  R13!,{D0-D3}    /* D-VD=0 imm8=8    VPUSH {D0-D3} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D11}   /* D-VD=0  imm8=24 */
    VLDMIA  R13!,{D12-D15}  /* D-VD=24 imm8=8   VPOP {D12-D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R8,#88
    VSTMIA  R8,{D5-D15}     /* D-VD=10 imm8=22 */
    VSTMDB  R13!,{D0-D4}    /* D-VD=0  imm8=10  VPUSH {D0-D4} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D10}   /* D-VD=0  imm8=22 */
    VLDMIA  R13!,{D11-D15}  /* D-VD=22 imm8=10  VPOP {D11-D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R8,#80
    VSTMIA  R8,{D6-D15}     /* D-VD=12 imm8=20 */
    VSTMDB  R13!,{D0-D5}    /* D-VD=0  imm8=12  VPUSH {D0-D5} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D9}    /* D-VD=0  imm8=20 */
    VLDMIA  R13!,{D10-D15}  /* D-VD=20 imm8=12  VPOP {D0-D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R8,#72
    VSTMIA  R8,{D7-D15}     /* D-VD=14 imm8=18 */
    VSTMDB  R13!,{D0-D6}    /* D-VD=0  imm8=14  VPUSH {D0-D6} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D8}    /* D-VD=0  imm8=18 */
    VLDMIA  R13!,{D9-D15}   /* D-VD=18 imm8=16  VPOP {D9-D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    ADD     R10,R8,#64
    VSTMIA  R8,{D8-D15}     /* D-VD=16 imm8=16 */
    VSTMDB  R13!,{D0-D7}    /* D-VD=0  imm8=16  VPUSH {D0-D7} Encoding T1 */
    /* Clear FPU registers */
    BL      m33_cst_write_fpu_regs_from_r0
    VLDMDB  R10!,{D0-D7}    /* D-VD=0  imm8=16 */
    VLDMIA  R13!,{D8-D15}   /* D-VD=16 imm8=16  VPOP {D8-D15} Encoding T1 */
    /* We need to check R10 was updated !! */
    CMP     R10,R8
    BNE     m33_cst_spfpu_loadstore_test_end
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
m33_cst_spfpu_loadstore_test_end:
    /* Test result is returned in R0, according to the conventions */
    MOV     R0,R1 
    B       m33_cst_test_tail_fpu_end    
    
    
    /*------------------------------------------------------------------------*/
    /* Internal test routines                                                 */
    /*------------------------------------------------------------------------*/
m33_cst_spfpu_loadstore_test_clear_target_memory:   
    MOV     R11,#0  /* Initialize counter */
m33_cst_spfpu_loadstore_test_clear_loop:
    STR     R0,[R8,R11]
    ADD     R11,R11,#4
    CMP     R11,#128
    BNE     m33_cst_spfpu_loadstore_test_clear_loop
    DSB 
    
    BX      LR
    
    CST_ALIGN_BYTES_4
    CST_LABEL
m33_cst_spfpu_loadstore_data1:
    CST_WORD 0x00000000
    CST_WORD 0x00111111
    CST_WORD 0x00222222
    CST_WORD 0x00333333
    CST_WORD 0x00444444
    CST_WORD 0x00555555
    CST_WORD 0x00666666
    CST_WORD 0x00777777
    CST_WORD 0x00888888
    CST_WORD 0x00999999
    CST_WORD 0x00AAAAAA
    CST_WORD 0x00BBBBBB
    CST_WORD 0x00CCCCCC
    CST_WORD 0x00DDDDDD
    CST_WORD 0x00EEEEEE
    CST_WORD 0x00FFFFFF   

    /**********************************************************************************************
    * Instruction: 
    * - VLDR (literal)
    *  Case1: Offset negative vs PC (m33_cst_spfpu_loadstore_data1: label)
    *  Case2: Offset positive vs PC (m33_cst_spfpu_loadstore_data2: label)
    **********************************************************************************************/
    CST_CODE
m33_cst_spfpu_loadstore_test_init_s0_s31:
    VLDR    S0,m33_cst_spfpu_loadstore_data1
m33_cst_spfpu_loadstore_test_init_s1_s31:
    VLDR    S1,m33_cst_spfpu_loadstore_data2 
m33_cst_spfpu_loadstore_test_init_s2_s31:
    VLDR    S2,m33_cst_spfpu_loadstore_data1 + 4
m33_cst_spfpu_loadstore_test_init_s3_s31:
    VLDR    S3,m33_cst_spfpu_loadstore_data2 + 4 
m33_cst_spfpu_loadstore_test_init_s4_s31:
    VLDR    S4,m33_cst_spfpu_loadstore_data1 + 8
m33_cst_spfpu_loadstore_test_init_s5_s31:
    VLDR    S5,m33_cst_spfpu_loadstore_data2 + 8
m33_cst_spfpu_loadstore_test_init_s6_s31:
    VLDR    S6,m33_cst_spfpu_loadstore_data1 + 12 
m33_cst_spfpu_loadstore_test_init_s7_s31:
    VLDR    S7,m33_cst_spfpu_loadstore_data2 + 12 
m33_cst_spfpu_loadstore_test_init_s8_s31:
    VLDR    S8,m33_cst_spfpu_loadstore_data1 + 16
m33_cst_spfpu_loadstore_test_init_s9_s31:
    VLDR    S9,m33_cst_spfpu_loadstore_data2 + 16
m33_cst_spfpu_loadstore_test_init_s10_s31:
    VLDR    S10,m33_cst_spfpu_loadstore_data1 + 20
m33_cst_spfpu_loadstore_test_init_s11_s31:
    VLDR    S11,m33_cst_spfpu_loadstore_data2 + 20
m33_cst_spfpu_loadstore_test_init_s12_s31:
    VLDR    S12,m33_cst_spfpu_loadstore_data1 + 24
m33_cst_spfpu_loadstore_test_init_s13_s31:
    VLDR    S13,m33_cst_spfpu_loadstore_data2 + 24
m33_cst_spfpu_loadstore_test_init_s14_s31:
    VLDR    S14,m33_cst_spfpu_loadstore_data1 + 28
m33_cst_spfpu_loadstore_test_init_s15_s31:          
    VLDR    S15,m33_cst_spfpu_loadstore_data2 + 28
m33_cst_spfpu_loadstore_test_init_s16_s31:
    VLDR    S16,m33_cst_spfpu_loadstore_data1 + 32
m33_cst_spfpu_loadstore_test_init_s17_s31:
    VLDR    S17,m33_cst_spfpu_loadstore_data2 + 32
m33_cst_spfpu_loadstore_test_init_s18_s31:
    VLDR    S18,m33_cst_spfpu_loadstore_data1 + 36
m33_cst_spfpu_loadstore_test_init_s19_s31:
    VLDR    S19,m33_cst_spfpu_loadstore_data2 + 36
m33_cst_spfpu_loadstore_test_init_s20_s31:
    VLDR    S20,m33_cst_spfpu_loadstore_data1 + 40
m33_cst_spfpu_loadstore_test_init_s21_s31:
    VLDR    S21,m33_cst_spfpu_loadstore_data2 + 40
m33_cst_spfpu_loadstore_test_init_s22_s31:
    VLDR    S22,m33_cst_spfpu_loadstore_data1 + 44
m33_cst_spfpu_loadstore_test_init_s23_s31:    
    VLDR    S23,m33_cst_spfpu_loadstore_data2 + 44
m33_cst_spfpu_loadstore_test_init_s24_s31:
    VLDR    S24,m33_cst_spfpu_loadstore_data1 + 48
m33_cst_spfpu_loadstore_test_init_s25_s31:
    VLDR    S25,m33_cst_spfpu_loadstore_data2 + 48
m33_cst_spfpu_loadstore_test_init_s26_s31:
    VLDR    S26,m33_cst_spfpu_loadstore_data1 + 52
m33_cst_spfpu_loadstore_test_init_s27_s31:
    VLDR    S27,m33_cst_spfpu_loadstore_data2 + 52
m33_cst_spfpu_loadstore_test_init_s28_s31:
    VLDR    S28,m33_cst_spfpu_loadstore_data1 + 56
m33_cst_spfpu_loadstore_test_init_s29_s31:
    VLDR    S29,m33_cst_spfpu_loadstore_data2 + 56
m33_cst_spfpu_loadstore_test_init_s30_s31:
    VLDR    S30,m33_cst_spfpu_loadstore_data1 + 60
m33_cst_spfpu_loadstore_test_init_s31_s31:
    VLDR    S31,m33_cst_spfpu_loadstore_data2 + 60
    
    BX      LR
    
    
    CST_ALIGN_BYTES_4
    CST_LABEL
m33_cst_spfpu_loadstore_data2:
    CST_WORD 0x80000000
    CST_WORD 0x80111111
    CST_WORD 0x80222222
    CST_WORD 0x80333333
    CST_WORD 0x80444444
    CST_WORD 0x80555555
    CST_WORD 0x80666666
    CST_WORD 0x80777777
    CST_WORD 0x80888888
    CST_WORD 0x80999999
    CST_WORD 0x80AAAAAA
    CST_WORD 0x80BBBBBB
    CST_WORD 0x80CCCCCC
    CST_WORD 0x80DDDDDD
    CST_WORD 0x80EEEEEE
    CST_WORD 0x80FFFFFF
     
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_FPU_ENABLE */

    CST_FILE_END

