/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
* - Test exception entry/exit with Extended stack for
*   a) lazy stacking enabled,
*   b) lazy stacking disabled.
* - Test exception exit based on LR (EXC_RETURN) value
* - Test FPU data correct on exception return.
* - Test CONTROL.FPCA behavior when FCCR.ASPEN==1.
* - Test CONTROL.FPCA behavior on exception entry/exit.
* - Test FPU undefined instructions encoding.
*
* Note: We do not test whether FPU data is physically written to Extended 
*       stack immediatelly or after execution of FPU instruction when lazy
*       stacking is enabled. Such fault can be considered as Performance 
*       faults. We test only that FPU registers were always correctly restored
*       on exception return.
******************************************************************************/
    
#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FPU_ENABLE==1)

    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_SpfpuExcStackTest
       
    
    /* Symbols defined outside but used within current module */
    CST_EXTERN m33_cst_store_isr_registers_content
    CST_EXTERN m33_cst_store_fpu_registers_content
    CST_EXTERN m33_cst_set_cst_interrupt_vector_table
    CST_EXTERN m33_cst_restore_isr_registers_content
    CST_EXTERN m33_cst_restore_fpu_registers_content
    CST_EXTERN m33_cst_test_tail_end
    CST_EXTERN m33_cst_ISR_address
    
    CST_SET(PRESIGNATURE, 0x2B973A73)
    
    /* Values stored in registers to be preserved during CST exception handling:
       - R0 ... expected exception address
       - R2 ... expected exception vector
       - R5 ... Current Signature value
       - R6 ... CONTROL
       - R7 ... PRIMASK
       - R8 ... FAULTMASK
       - R9 ... BASEPRI
       - R10... FPCCR register address    
     */

    /*------------------------------------------------------------------------*/    
    CST_SECTION_EXEC(mcal_text)
    /*------------------------------------------------------------------------*/    
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_SpfpuExcStackTest" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_THUMB2   
    CST_TYPE(M33_Cst_SpfpuExcStackTest, function)
M33_Cst_SpfpuExcStackTest:

    PUSH    {R4-R12,R14}
    MRS     R1,CONTROL  /* Store CONTROL prior first FPU instruction */
    PUSH    {R1}
    /* We will use only S0-S15 FPU registers thus there is no need to store
       other FPU registers on stack */
    
    /*------------------------------------------------------------------------*/
    /* Test - preparation                                                     */
    /*------------------------------------------------------------------------*/
    LDR     R5,=PRESIGNATURE    /* Load PRESIGNATURE to R5  */
    
    /*------------------------------------------------------------------------*/
    /* Store register content                                                 */
    /*------------------------------------------------------------------------*/
    /* !! Don't use R6, R7, R8, R9 for any other purpose until CONTROL, PRIMASK,
          FAULTMASK and BASEPRI are restored !! */
    BL      m33_cst_store_isr_registers_content
    BL      m33_cst_store_fpu_registers_content

    /*------------------------------------------------------------------------*/
    /* Test - start                                                           */
    /*------------------------------------------------------------------------*/    
    /* Store address of the CST-own ISR into the m33_cst_ISR_address pointer */
    ADR.W   R4,m33_cst_spfpu_exc_stack_test_ISR1
    ORR     R4,R4,#1     /* Set bit [0] as this address will be used within 
                            BX command and should indicate the Thumb code */
    
    LDR     R3,=m33_cst_ISR_address
    STR     R4,[R3]

    /* Before vector table is changed to the CST-own one, set R2 to
       to the expected ISR number and R0 to the expected ISR address.
       
       Content of R0 and R2 will allow determining whether interrupt was 
       triggered by CST library or by alien SW.

    !! Don't use R0 and R2 for any other purpose until vector table is changed 
       to a user one. !! */
    MOV     R2,#3       /* R2 indicates that CST library will explicitly  
                           initiate an exception with IRQ number 0x3 */
    LDR     R0,=m33_cst_spfpu_exc_stack_test_ISR_addr1
    BIC     R0,R0,#1    /* Clear bit [0]. Ensures that label is not treated as 
                           Thumb function */
                           
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table /* !! Sets CONTROL=0 !! */
    
    CPSID   i   /* Set PRIMASK -> Escalate exception to HardFault */
    CPSIE   f   /* Clear FAULTMASK */
    
    /* Configure FPCCR for FPU registers stacking */
    LDR     R10,=M33_FPCCR_REG   /* Don't use R10 it contains FPCCR address */
    LDR     R4,[R10]            /* Load FPCCR register content */
    ORR     R4,R4,#(1<<31)      /* Set ASPEN bit */
    BIC     R4,R4,#(1<<30)      /* Clear LSPEN bit -> No Lazy stacking */
    STR     R4,[R10]            /* Write FPCCR register */
    DSB
    
    /* Prepare S0-S15 and FPSCR registers for stacking
       For FPSCR we change only NZCV. Expected value FPSCR_NZCV==0x20000000 */
    MOV     R12,#0x55555555
    /* Running this subroutine will set CONTROL.FPCA==1 because 
       FPCCR.ASPEN==1 */
    BL      m33_cst_spfpu_exc_stack_test_init
    
    /* We must check that CONTROL.FPCA bit is set */
    MRS     R4,CONTROL      
    TST     R4,#(1<<2)          /* Check CONTROL.FPCA==1 */
    BEQ     m33_cst_spfpu_exc_stack_test_error     
    
m33_cst_spfpu_exc_stack_test_ISR_addr1:
    /* See encoding of 64-bit transfers between ARM core and extension 
        registers. If T == 1 the instruction is UNDEFINED. */
    CST_OPCODE_START           /* Used opcode to prepare FPU Undefined */
    CST_OPCODE32_HIGH(0xFC41)  /* VMOV  S0,S1,R1,R1 with T==1 => UNDEFINED */
    CST_OPCODE32_LOW(0x1A10)   /* -> Generate exception !! */
    CST_OPCODE_END
    
    
m33_cst_spfpu_exc_stack_test_ISR_ret_addr1:
    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_fpu_registers_content
    BL      m33_cst_restore_isr_registers_content
    
    /* We need to check S0-S15 and FPSCR were correctly restored from stack */
    BL      m33_cst_spfpu_exc_stack_test_check
    
    /* We need to check that UNDEFINSTR flag was set */
    LDR     R3,=M33_UFSR_REG
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag is set */

    /* We need to clear UNDEFINSTR flag */
    STRH    R4,[R3]         /* Clear UFSR[UNDEFINSTR](rc_w1) flag */
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag was cleared */ 

    /* !! Update signature !! */
    ROR     R3,R5,R2    
    EOR     R5,R3,R5
    
    /* Store address of the CST-own ISR into the m33_cst_ISR_address pointer */
    ADR.W   R4,m33_cst_spfpu_exc_stack_test_ISR2
    ORR     R4,R4,#1     /* Set bit [0] as this address will be used within 
                            BX command and should indicate the Thumb code */
    
    LDR     R3,=m33_cst_ISR_address
    STR     R4,[R3]

    /* Before vector table is changed to the CST-own one, set R2 to
       to the expected ISR number and R0 to the expected ISR address.
       
       Content of R0 and R2 will allow determining whether interrupt was 
       triggered by CST library or by alien SW.

    !! Don't use R0, R2 for any other purpose until vector table is changed 
       to a user one. !! */
    /* No need to write R2 since it was restored from stack !! */
                        /* R2 indicates that CST library will explicitly  
                           initiate an exception with IRQ number 0x3 */
    LDR     R0,=m33_cst_spfpu_exc_stack_test_ISR_addr2
    BIC     R0,R0,#1    /* Clear bit [0]. Ensures that label is not treated as 
                           Thumb function */
                           
                       
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table /* !! Sets CONTROL=0 !! */
    
    CPSID   i   /* Set PRIMASK -> Escalate exception to HardFault */
    CPSIE   f   /* Clear FAULTMASK */
    
    /* Configure FPCCR for FPU registers stacking */
    LDR     R4,[R10]            /* Load FPCCR register content */
    ORR     R4,R4,#(1<<31)      /* Set ASPEN bit */
    ORR     R4,R4,#(1<<30)      /* Set LSPEN bit */
    STR     R4,[R10]            /* Write FPCCR register */
    DSB
    
    /* Prepare S0-S15 and FPSCR registers for stacking
       For FPSCR we change only NZCV. Expected value FPSCR_NZCV==0x80000000 */
    MOV     R12,#0xAAAAAAAA
    /* Running this subroutine will set CONTROL.FPCA==1 because 
       FPCCR.ASPEN==1 */
    BL      m33_cst_spfpu_exc_stack_test_init
    
    /* We will destroy FPCAR by writing expected inverted value */
    ADD     R4,SP,#20           /* Prepare expected FPCAR value */
    MVN     R4,R4               /* Invert bits */
    LDR     R3,=M33_FPCAR_REG
    STR     R4,[R3]             /* Write FPCAR with inverted value */
    
    /* We must check that CONTROL.FPCA bit is set */
    MRS     R4,CONTROL      
    TST     R4,#(1<<2)          /* Check CONTROL.FPCA==1 */
    BEQ     m33_cst_spfpu_exc_stack_test_error     
    
    
m33_cst_spfpu_exc_stack_test_ISR_addr2:
    /* Instruction is unsupported in single precision FPU variant when
       sz==1, this indicates db-precision, see encoding of FPU instructions  */
    CST_OPCODE_START           /* Used opcode to avoid compilation problem  */
    CST_OPCODE32_HIGH(0xEEB6)  /* VMOV.F64 D0,#0.5 with sz==1 => UNDEFINED  */
    CST_OPCODE32_LOW(0x0B00)   /* -> Generate exception !! */
    CST_OPCODE_END

m33_cst_spfpu_exc_stack_test_ISR_ret_addr2:
    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_fpu_registers_content
    BL      m33_cst_restore_isr_registers_content
    
    /* We need to check that UNDEFINSTR flag was set */
    LDR     R3,=M33_UFSR_REG
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag is set */

    /* We need to clear UNDEFINSTR flag */
    STRH    R4,[R3]         /* Clear UFSR[UNDEFINSTR](rc_w1) flag */
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag was cleared */ 

    /* We need to check S0-S15 and FPSCR were correctly restored from stack */
    BL      m33_cst_spfpu_exc_stack_test_check
    
    /* !! Update signature !! */
    ROR     R3,R5,R2    
    EOR     R5,R3,R5
    
    /* Store address of the CST-own ISR into the m33_cst_ISR_address pointer */
    ADR.W   R4,m33_cst_spfpu_exc_stack_test_ISR3
    ORR     R4,R4,#1     /* Set bit [0] as this address will be used within 
                            BX command and should indicate the Thumb code */
    
    LDR     R3,=m33_cst_ISR_address
    STR     R4,[R3]

    /* Before vector table is changed to the CST-own one, set R2 to
       to the expected ISR number and R0 to the expected ISR address.
       
       Content of R0 and R2 will allow determining whether interrupt was 
       triggered by CST library or by alien SW.

    !! Don't use R0, R2 for any other purpose until vector table is changed 
       to a user one. !! */
    /* No need to write R2 since it was restored from stack !! */
                        /* R2 indicates that CST library will explicitly  
                           initiate an exception with IRQ number 0x3 */
    LDR     R0,=m33_cst_spfpu_exc_stack_test_ISR_addr3
    BIC     R0,R0,#1    /* Clear bit [0]. Ensures that label is not treated as 
                           Thumb function */
                           
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table /* !! Sets CONTROL=0 !! */
    
    CPSID   i   /* Set PRIMASK -> Escalate exception to HardFault */
    CPSIE   f   /* Clear FAULTMASK */
    
    /* Configure FPCCR for FPU registers stacking */
    LDR     R4,[R10]            /* Load FPCCR register content */
    ORR     R4,R4,#(1<<31)      /* Set ASPEN bit */
    ORR     R4,R4,#(1<<30)      /* Set LSPEN bit */
    STR     R4,[R10]            /* Write FPCCR register */
    DSB
    
    /* Prepare S0-S15 and FPSCR registers for stacking
       For FPSCR we change only NZCV. Expected value FPSCR_NZCV==0x30000000 */
    MOV     R12,#0xFFFFFFFF
    /* Running this subroutine will set CONTROL.FPCA==1 because 
       FPCCR.ASPEN==1 */
    BL      m33_cst_spfpu_exc_stack_test_init
    
    /* We must check that CONTROL.FPCA bit is set */
    MRS     R4,CONTROL      
    TST     R4,#(1<<2)          /* Check CONTROL.FPCA==1 */
    BEQ     m33_cst_spfpu_exc_stack_test_error
    
m33_cst_spfpu_exc_stack_test_ISR_addr3:
    /* See encoding of 64-bit transfers between ARM core and extension 
        registers. If T==1 the instruction is UNDEFINED. */
    CST_OPCODE_START           /* Used opcode to avoid compilation problem */
    CST_OPCODE32_HIGH(0xFD2D)  /* VPUSH  {S0-S15} with T==1 => UNDEFINED */
    CST_OPCODE32_LOW(0x0A10)   /* -> Generate exception !! */
    CST_OPCODE_END
    
    
m33_cst_spfpu_exc_stack_test_ISR_ret_addr3:
    /* We must check that CONTROL.FPCA is set */
    MRS     R4,CONTROL
    TST     R4,#(1<<2)          /* Check CONTROL.FPCA==1 */
    BEQ     m33_cst_spfpu_exc_stack_test_error
      
    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/     
    BL      m33_cst_restore_fpu_registers_content
    BL      m33_cst_restore_isr_registers_content
    
    /* We need to check that UNDEFINSTR flag was set */
    LDR     R3,=M33_UFSR_REG
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag is set */

    /* We need to clear UNDEFINSTR flag */
    STRH    R4,[R3]         /* Clear UFSR[UNDEFINSTR](rc_w1) flag */
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag was cleared */ 

    /* We need to check S0-S15 and FPSCR weren't destroyd with data from stack*/
    BL      m33_cst_spfpu_exc_stack_test_check
    
    /* !! Update signature !! */
    ROR     R3,R5,R2    
    EOR     R5,R3,R5
    
    /* Store address of the CST-own ISR into the m33_cst_ISR_address pointer */
    ADR.W   R4,m33_cst_spfpu_exc_stack_test_ISR4
    ORR     R4,R4,#1     /* Set bit [0] as this address will be used within 
                            BX command and should indicate the Thumb code */
    
    LDR     R3,=m33_cst_ISR_address
    STR     R4,[R3]

    /* Before vector table is changed to the CST-own one, set R2 to
       to the expected ISR number and R0 to the expected ISR address.
       
       Content of R0 and R2 will allow determining whether interrupt was 
       triggered by CST library or by alien SW.

    !! Don't use R0, R2 for any other purpose until vector table is changed 
       to a user one. !! */
    /* No need to write R2 since it was restored from stack !! */
                        /* R2 indicates that CST library will explicitly  
                           initiate an exception with IRQ number 0x3 */
    LDR     R0,=m33_cst_spfpu_exc_stack_test_ISR_addr4
    BIC     R0,R0,#1    /* Clear bit [0]. Ensures that label is not treated as 
                           Thumb function */
                           
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table /* !! Sets CONTROL=0 !! */
    
    CPSID   i   /* Set PRIMASK -> Escalate exception to HardFault */
    CPSIE   f   /* Clear FAULTMASK */
    
    /* Configure FPCCR not to stack FPU registers */
    LDR     R4,[R10]            /* Load FPCCR register content */
    BIC     R4,R4,#(1<<31)      /* Clear ASPEN bit */
    BIC     R4,R4,#(1<<30)      /* Clear LSPEN bit */
    STR     R4,[R10]            /* Write FPCCR register */
    DSB
    
     /* Prepare S0-S15 and FPSCR registers for stacking
       For FPSCR we change only NZCV. Expected value FPSCR_NZCV==0x60000000 */
    MOV     R12,#0x00000000
    /* Running this subroutine will not set CONTROL.FPCA==1 because 
       FPCCR.ASPEN==0 */
    BL      m33_cst_spfpu_exc_stack_test_init
    
    /* We must check that CONTROL.FPCA was not set because FPCCR.ASPEN==0  */
    MRS     R4,CONTROL      
    TST     R4,#(1<<2)          /* Check CONTROL.FPCA==0 */
    BNE     m33_cst_spfpu_exc_stack_test_error
    
    /* Configure FPCCR for FPU registers stacking */
    LDR     R4,[R10]            /* Load FPCCR register content */
    ORR     R4,R4,#(1<<31)      /* Clear ASPEN bit */
    BIC     R4,R4,#(1<<30)      /* Clear LSPEN bit */
    STR     R4,[R10]            /* Write FPCCR register */
    DSB
    
    
m33_cst_spfpu_exc_stack_test_ISR_addr4:
    /* See encoding of 64-bit transfers between ARM core and extension 
        registers. If T==1 the instruction is UNDEFINED. */
    CST_OPCODE_START           /* Used opcode to avoid compilation problem */
    CST_OPCODE32_HIGH(0xFC52)  /* VMOV  R0,R2,D0 with T==1 => UNDEFINED */
    CST_OPCODE32_LOW(0x0B10)   /* -> Generate exception !! */
    CST_OPCODE_END
    

m33_cst_spfpu_exc_stack_test_ISR_ret_addr4:    
    
    /* We must check that CONTROL.FPCA was cleared */
    MRS     R4,CONTROL
    TST     R4,#(1<<2)          /* Check CONTROL.FPCA==0 */
    BNE     m33_cst_spfpu_exc_stack_test_error

    /* We need to check that UNDEFINSTR flag was set */
    LDR     R3,=M33_UFSR_REG
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag is set */

    /* We need to clear UNDEFINSTR flag */
    STRH    R4,[R3]         /* Clear UFSR[UNDEFINSTR](rc_w1) flag */
    LDRH    R4,[R3]         /* Use halfword access to UFSR */
    AND     R4,R4,#(1<<0)   /* Check UFSR[UNDEFINSTR] flag was cleared */ 
    
    /* !! Update signature !! */
    ROR     R3,R5,R2    
    EOR     R5,R3,R5
 
m33_cst_spfpu_exc_stack_test_error: 
    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/     
    BL      m33_cst_restore_fpu_registers_content
    BL      m33_cst_restore_isr_registers_content
    
    /* We need to check S0-S15 and FPSCR weren't destroyd with data from stack*/
    BL      m33_cst_spfpu_exc_stack_test_check
    
    /* !! Update signature !! */
    ROR     R3,R5,R2    
    EOR     R5,R3,R5
    
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/        
m33_cst_spfpu_exc_stack_test_end:
    /* Test result is returned in R0, according to the conventions */
    MOV     R0,R5
    POP     {R1}        /* Restore CONTROL register */
    MSR     CONTROL,R1
    ISB
    LDR     R5,=m33_cst_test_tail_end
    BX      R5    /* -> Exit test */
    
    

    /*------------------------------------------------------------------------*/
    /* Internal routines                                                      */
    /*------------------------------------------------------------------------*/
    /* Write S0-S15 with value in R12 and update FPSCR */
m33_cst_spfpu_exc_stack_test_init:   
    VMOV    S0,S1,R12,R12
    VMOV    D1,R12,R12
    VMOV    S4,S5,R12,R12
    VMOV    D3,R12,R12
    VMOV    S8,S9,R12,R12
    VMOV    D5,R12,R12
    VMOV    S12,S13,R12,R12
    VMOV    D7,R12,R12
    VCMP.F32    S0,#0.0 /* Change FPSCR_NZCV flags */     
    VMRS    R11,FPSCR   /* R11 contains expected FPSCR value */
    
    BX  LR
    
    /* Check S0-S15 and FPSCR againts expected values in R12 and R11 */    
m33_cst_spfpu_exc_stack_test_check:
    VMOV    R3,R4,D0
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMOV    R3,R4,S2,S3
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMOV    R3,R4,D2
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMOV    R3,R4,S6,S7
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMOV    R3,R4,D4
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMOV    R3,R4,S10,S11
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMOV    R3,R4,D6
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMOV    R3,R4,S14,S15
    EORS    R3,R3,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    EORS    R4,R4,R12
    BNE     m33_cst_spfpu_exc_stack_test_end
    VMRS    R3,FPSCR        /* Check FPSCR */
    EORS    R3,R3,R11
    BNE     m33_cst_spfpu_exc_stack_test_end
    
    BX      LR


   
    /*------------------------------------------------------------------------*/
    /* Test - ISR routines                                                    */
    /*------------------------------------------------------------------------*/
    /* ISR1 - Lazy stacking is disabled CPU creates Extended stack and writes 
              data immediatelly. 
       ISR2 - Lazy stacking is enabled CPU creates Extended stack but it 
              writes data to stack after an FPU instruction.
              (we will test address written in to the FPCAR register)
       ISR3 - Lazy stacking is enabled CPU creates Extended stack but data
              is not written to stack because we do not activate lazy stacking
              here by executing an FPU instruction. 
       ISR4 - Lazy stacking is disabled CPU creates Basic stack because 
              FPCCR.ASPEN==1. We will activate Extended stack in Handler mode. */
              
m33_cst_spfpu_exc_stack_test_ISR4:
    
    /* We need to check CONTROL register */
    MRS     R0,CONTROL
    TST     R0,#(1<<2)  /* Check FPCA Bit[2]==0 -> FP extension not active */
    BNE     m33_cst_spfpu_exc_stack_test_ISR_end
    
    /* We need to check EXC_RETURN value*/
    TST     R14,#(1<<4)  /* Check Bit[4]==1 -> Return with Basic stack */
    BEQ     m33_cst_spfpu_exc_stack_test_ISR_end
    
    /* We will execute FPU instruction to ensure the CONTROL.FPCA==1
       Then we will exit ISR and we will check that CONTROL.FPCA==0 */
    B       m33_cst_spfpu_exc_stack_test_ISR_destroy_regs
              
m33_cst_spfpu_exc_stack_test_ISR3: 

    /* We need to check CONTROL register */
    MRS     R0,CONTROL
    TST     R0,#(1<<2)  /* Check FPCA Bit[2]==0 -> FP extension not active */
    BNE     m33_cst_spfpu_exc_stack_test_ISR_end
    /* We will not execute FPU instruction not to set CONTROL.FPCA==1
       and not to activate Lazy stacking 
       Then we will exit ISR and we will check that CONTROL.FPCA==0 */
    B       m33_cst_spfpu_exc_stack_test_ISR_signature
                    
m33_cst_spfpu_exc_stack_test_ISR2:
    
    /* We need to check that FPCAR register was written */
    LDR     R1,[SP,#0xC]    /* Load FPCAR register address from stack !! */
    LDR     R1,[R1]         /* Load Address stored in FPCAR to R1 */
    /* FPCAR points to the reserved S0 stack location (SP + #0x20) */
    SUB     R0,R1,#0x20
    SUBS    R0,SP,R0
    BNE     m33_cst_spfpu_exc_stack_test_ISR_end
    
m33_cst_spfpu_exc_stack_test_ISR1:

    /* We need to check EXC_RETURN value*/
    TST     R14,#(1<<4)  /* Check Bit[4]==0 -> Return with Extended stack */
    BNE     m33_cst_spfpu_exc_stack_test_ISR_end
    
    /* We need to check CONTROL register */
    MRS     R0,CONTROL
    TST     R0,#(1<<2)  /* Check FPCA Bit[2]==0 -> FP extension not active */
    BNE     m33_cst_spfpu_exc_stack_test_ISR_end
    
m33_cst_spfpu_exc_stack_test_ISR_destroy_regs:    
    /* Destroy FPU registers by writing zeros comming from R0 */
    /* Activate FP extension -> Activate FPU registers stacking in ISR2 */
    VMOV    D0,R0,R0
    VMOV    S2,S3,R0,R0
    VMOV    D2,R0,R0
    VMOV    S6,S7,R0,R0
    VMOV    D4,R0,R0
    VMOV    S10,S11,R0,R0
    VMOV    D6,R0,R0
    VMOV    S14,S15,R0,R0
    VCMP.F32    S0,#0.0     /* Destroy FPSCR -> We destroy only flags !! */
    
    /* We need to check CONTROL register again */
    MRS     R2,CONTROL
    TST     R2,#(1<<2)  /* Check FPCA Bit[2]==1 -> FP extension active */
    BEQ     m33_cst_spfpu_exc_stack_test_ISR_end

m33_cst_spfpu_exc_stack_test_ISR_signature:
    ROR     R3,R5,R2    /* !! Update signature !! */
    EOR     R5,R3,R5

m33_cst_spfpu_exc_stack_test_ISR_end:    
    /* We need to change return address that is stored in stack:
       it points again to undefined FPU instruction */
    LDR     R3,[SP,#24]
    ADD     R3,R3,#4    /* Instruction is 32-bit */
    STR     R3,[SP,#24]
    DSB     /* Ensures that stacked value is updated before exception return */
    
    BX      LR
    
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_FPU_ENABLE */

    CST_FILE_END

