/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
* Checks conversions between floating-point and fixed-point numbers provided by 
*  cm33fpu_sp_cvt.v
*
* Following conversions are tested:
* - floating-point to/from fixed-point for all fixed-point fraction bits
*
* In addition, the following functionality is tested:
* - both (32/16-bit) fix-point formats
* - both (32/16-bit) signed/unsigned fixed-point formats
* - conversion of special floating-point (+-NaN, +-Inf, +-0) values
* - producing the FPSCR[IOC, IDC, IXC] flags
* - flush-to-zero functionality
*
* SP FPU Decoder coverage:
* ------------------------
*   - VCVT.U32.F32 - floating-point to fixed-point(unsigned)
*   - VCVT.F32.U32 - fixed-point(unsigned) to floating-point
*   - VCVT.S32.F32 - floating-point to fixed-point(signed)
*   - VCVT.F32.S32 - fixed-point(signed) to floating-point
*
*   - VCVT.U16.F32 - floating-point to fixed-point(unsigned)
*   - VCVT.F32.U16 - fixed-point(unsigned) to floating-point
*   - VCVT.S16.F32 - floating-point to fixed-point(signed)
*   - VCVT.F32.S16 - fixed-point(signed) to floating-point
*
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FPU_ENABLE==1)

    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_SpfpuConvertTest2

    /* Symbols defined outside but used within current module */
    CST_EXTERN m33_cst_test_tail_fpu_end
    CST_EXTERN m33_cst_write_fpu_regs_from_r0
    CST_EXTERN m33_cst_write_fpu_s14_s31_from_r0
    CST_EXTERN m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    CST_EXTERN m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    CST_EXTERN m33_cst_sum_fpu_s14_s31_to_r1_destr_r2
    
    /* Pre-signature */
    CST_SET(PRESIGNATURE_SEED_0, 0x5118)
    CST_SET(PRESIGNATURE_SEED_1, 0x0275)
    
    /* Values stored in registers:
       - R1  ... accumulated signature
       - R12 ... FPSCR register
    */

   /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
   /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_SpfpuConvertTest2" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_TYPE(M33_Cst_SpfpuConvertTest2, function)
    CST_THUMB2
M33_Cst_SpfpuConvertTest2:

    PUSH    {R4-R12,R14}
    MRS     R1,CONTROL  /* Store CONTROL prior first FPU instruction */
    PUSH    {R1}
    VPUSH   {S16-S31}
    
    /*-----------------------------------------------------------------------*/
    /* Test - preparation                                                    */
    /*-----------------------------------------------------------------------*/
    MOV     R1,#PRESIGNATURE_SEED_0
    MOVT    R1,#PRESIGNATURE_SEED_1
        
    /*-----------------------------------------------------------------------*/
    /* Test - start                                                          */
    /*-----------------------------------------------------------------------*/
    VMRS    R12,FPSCR   /* Store FPSCR register */
    
    
    /*************************************************************************/
    /*  Convert between floating-point and fixed-point                       */
    /*************************************************************************/
    /* Test1:
        1) Write all S0...S31 registers with one dedicated floating-point value 
           - Input floating-point values were selected to stimulate all fraction 
           bits in the fixed-point numbers to "0" and "1".
        2) Convert floating-point to fixed-point for fbits <1,32> or <0,16>
           - Fraction bits will change depending on fbits used.
        3) !! Update Signature !!
        4) Convert back fixed-point numbers to floating-point number for 
           fbits <1,32>.
           - Backward conversions will produce floating-point numbers with 
            limited precision, see example below.
        5) !! Update Signature !! 
        
     Example1:   
     ---------
     S0 =0x3F2AAAAA(F32) -> 0x000000001 (Fixed= 1) -> 0x3F2000000 (F32) 
     S1 =0x3F2AAAAA(F32) -> 0x000000002 (Fixed= 2) -> 0x3F2000000 (F32) 
     S2 =0x3F2AAAAA(F32) -> 0x000000005 (Fixed= 3) -> 0x3F2800000 (F32) 
     S3 =0x3F2AAAAA(F32) -> 0x00000000A (Fixed= 4) -> 0x3F2800000 (F32) 
                               ......                         
     S21=0x3F2AAAAA(F32) -> 0x005555555 (Fixed=22) -> 0x3F2AAAAAA (F32)
     S22=0x3F2AAAAA(F32) -> 0x00AAAAAAA (Fixed=23) -> 0x3F2AAAAAA (F32)
                               ......                                       
     S31=0x3F2AAAAA(F32) -> 0xAAAAAAA00 (Fixed=32) -> 0x3F2AAAAAA (F32)
     
     
     Example2:                   
     ---------   
     S0 =0x3E2AAAAA(F32) -> 0x000000000 (Fixed=1)  -> 0x000000000 (F32) 
     S1 =0x3E2AAAAA(F32) -> 0x000000001 (Fixed=2)  -> 0x3E8000000 (F32) 
     S2 =0x3E2AAAAA(F32) -> 0x000000002 (Fixed=3)  -> 0x3E8000000 (F32) 
     S3 =0x3E2AAAAA(F32) -> 0x000000005 (Fixed=4)  -> 0x3EA000000 (F32) 
                               ......                       
     S22=0x3F2AAAAA(F32) -> 0x005555555 (Fixed=23) -> 0x3E2AAAAAA (F32)
     S23=0x3F2AAAAA(F32) -> 0x00AAAAAAA (Fixed=25) -> 0x3E2AAAAAA (F32)
                               ......                       
     S31=0x3E2AAAAA(F32) -> 0x555555500 (Fixed=32) -> 0x3EAAAAAAA (F32)
    ***************************************************************************/
    MOV     R0,#0
    VMSR    FPSCR,R0    /* Write RMode and clear FPSCR */
    
    /***************************************************************************/
    /* Start with unsigned 32-Bit conversions */
    /***************************************************************************/
    /* Prepare floating-point value ~0.666666 */
    MOV     R0,#(126<<23)       /* (-1)(exp) + 127(bias) */
    MOV     R2,#0x55555555
    ORR     R0,R0,R2,LSR #9     /* 0x3F2AAAAA */
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0  
    /* Convert floating-point to fix-point  */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare floating-point value ~0.333333 */
    MOV     R0,#(125<<23)       /* (-2)(exp) + 127(bias) */
    MOV     R2,#0x55555555                  
    ORR     R0,R0,R2,LSR #9     /* 0x3E2AAAAA */  
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0  
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare floating-point value ~0.002604 */
    MOV     R0,#(118<<23)       /* (-9)(exp) + 127(bias) */
    MOV     R2,#0x55555555
    ORR     R0,R0,R2,LSR #9     /* 0x3B2AAAAA */                 
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0  /* ~0.002604 -> S0...S31 */  
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare floating-point value ~0.003255 */
    MOV     R0,#(118<<23)       /* (-9)(exp) + 127(bias) */
    MOV     R2,#0xAAAAAAAA                  
    ORR     R0,R0,R2,LSR #9     /* 0x3B5555555 */                 
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /***************************************************************************/
    /* Continue with unsigned 16-Bit conversion */
    /***************************************************************************/
    /* Prepare floating point value 0.666656494140625 */
    MOV     R0,#(126<<23)       /* (-1)(exp) + 127(bias) */
    MOV     R2,#0x00002AAA                  
    ORR     R0,R0,R2,LSL #8     /* 0x3F2AAA00 */
    /* Write S14...S31 registers (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0   
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_u16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_u16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    /* Prepare floating point value 0.3333282470703125 */
    MOV     R0,#(125<<23)       /* (-2)(exp) + 127(bias) */
    MOV     R2,#0x00002AAA
    ORR     R0,R0,R2,LSL #8     /* 0x3EAAAA00 */
    /* Write S14...S31 registers (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0   
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_u16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_u16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    /***************************************************************************/
    /* Continue with Signed 32-Bit conversions  */
    /***************************************************************************/
    /* Prepare floating-point value +0.3499999940395355224609375 */
    MOV     R0,#(125<<23)       /* (-1)(exp) + 127(bias) */
    MOV     R2,#0x33333333
    ORR     R0,R0,R2,LSR #8     /* 0x3EB33333 */
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_s32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_s32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare floating-point value -0.3812499940395355224609375 */
    MOV     R0,#(125<<23)       /* (-1)(exp) + 127(bias) */
    MOV     R2,#0xCCCCCCCC
    ORR     R0,R0,R2,LSR #8
    ORR     R0,R0,#(1<<31)      /* 0xBECCCCCC */
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_s32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_s32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /***************************************************************************/
    /* Continue with Signed 16-Bit conversions - Positive number */ 
    /***************************************************************************/
    /* Prepare floating-point +0.4999847412109375 */
    MOV     R0,#(125<<23)       /* (-2)(exp) + 127(bias) */
    MOV     R2,#0x00003FFF                  
    ORR     R0,R0,R2,LSL #9     /* 0x3efffe00 */
    /* Write S14...S31 registers - (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0      /* ~+0.6666 -> S0...S31 */
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_s16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_s16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    /* Prepare floating-point -0.25 */
    MOV     R0,#(125<<23)       /* (-2)(exp) + 127(bias) */
    ORR     R0,R0,#(1<<31)      /* 0xBE800000 */
    /* Write S14...S31 registers - (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0      
    /* Convert floating-point to fix-point */
    BL      m33_cst_spfpu_convert_test2_f32_to_s16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to floating-point */
    BL      m33_cst_spfpu_convert_test2_s16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    
    /*************************************************************************/
    /*  Convert between fixed-point and floating-point                       */
    /*************************************************************************/
    /* Test1:
        1) Write S0...S31 with one dedicated fixed-point value
        2) Convert fixed-point to floating-point for fbits <1,32> or <0,16>
           - The floating-point exponent will change depending on fbits used.
           - The fraction bits are same either 0xAA or 0x55 depending on
             test vector.
        3) !! Update Signature !!
        4) Convert back floating-point to fixed-Point for fbits <1,32> or <0,16>
           - Test that backward conversion returns original fixed-point value
             for all tested fbits.
        5  !! Update Signature !! 
        
    Example1:
    ---------
     S0 =0xAAAAAA00(Fixed= 1) -> 0x4EAAAAAAA(F32) -> 0xAAAAAA00 (Fixed= 1) 
     S1 =0xAAAAAA00(Fixed= 2) -> 0x4E2AAAAAA(F32) -> 0xAAAAAA00 (Fixed= 2) 
     S2 =0xAAAAAA00(Fixed= 3) -> 0x4DAAAAAAA(F32) -> 0xAAAAAA00 (Fixed= 3) 
     S3 =0xAAAAAA00(Fixed= 4) -> 0x42AAAAAAA(F32) -> 0xAAAAAA00 (Fixed= 4) 
                                    ......                               
     S30=0xAAAAAA00(Fixed=31) -> 0x3FAAAAAAA(F32) -> 0xAAAAAA00 (Fixed=31)
     S31=0xAAAAAA00(Fixed=32) -> 0x3F2AAAAAA(F32) -> 0xAAAAAA00 (Fixed=32)
    
    Example2:
    ---------
     S0 =0xD5555500(Fixed= 1) -> 0x4ED555555(F32) -> 0xD5555500 (Fixed= 1) 
     S1 =0xD5555500(Fixed= 2) -> 0x4E5555555(F32) -> 0xD5555500 (Fixed= 2) 
     S2 =0xD5555500(Fixed= 3) -> 0x4DD555555(F32) -> 0xD5555500 (Fixed= 3) 
     S3 =0xD5555500(Fixed= 4) -> 0x4D5555555(F32) -> 0xD5555500 (Fixed= 4) 
                                    ......                               
     S30=0xD5555500(Fixed=31) -> 0x3FD555555(F32) -> 0xD5555500 (Fixed=31)
     S31=0xD5555500(Fixed=32) -> 0x3F5555555(F32) -> 0xD5555500 (Fixed=32) 
    */
    /***************************************************************************/
    /* Start with unsigned 32-Bit conversions */
    /***************************************************************************/
    /* Prepare fixed-point value */
    MOV     R0,#0xAAAAAAAA
    LSL     R0,R0,#8        /* 0xAAAAAA00 */ 
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Convert fixed-point to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare fixed-point value */
    ASR     R0,R0,#1        /* 0xD5555500 */
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0  
    /* Convert fixed-point to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare fixed-point value */
    MOV     R0,#0xAA        /* 0x00000AA */
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0  
    /* Convert fixed-point to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare fixed-point value */
    MOV     R0,#0xD55       /* 0x00000D55 */
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Convert fixed-point to floating-point */
    BL      m33_cst_spfpu_convert_test2_u32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_u32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /***************************************************************************/
    /* Continue with unsigned 16-Bit conversion */ 
    /***************************************************************************/
    /* Prepare fixed-point value */
    MOV     R0,#0xAAAA                         /* 0x0000AAAA */
    /* Write S14...S31 registers - (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0
    /* Convert fixed-point to floating-point */
    BL      m33_cst_spfpu_convert_test2_u16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_u16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    /* Prepare fixed-point value */
    MOV     R0,#0xD555                          /* 0x0000D555 */
    /* Write S14...S31 registers - (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0
    /* Convert fixed-point to floating-point */
    BL      m33_cst_spfpu_convert_test2_u16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_u16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    /***************************************************************************/
    /* Continue with Signed 32-Bit conversions  */ 
    /***************************************************************************/
    /* Prepare fixed-point value */
    MOV     R0,#0xAAAAAAAA
    LSR     R0,R0,#4                        /* 0x0AAAAAAA (positive number) */                      
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0  /* 0x0AAAAAAA -> S0...S31 */
    /* Convert fixed-point to floating point */
    BL      m33_cst_spfpu_convert_test2_s32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_s32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /* Prepare fixed-point value */
    LSR     R0,R0,#1            
    ORR     R0,R0,#(1<<31)                  /* 0x85555555 (negative number) */
    /* Write S0...S31 registers */
    BL      m33_cst_write_fpu_regs_from_r0
    /* Convert fixed-point to floating point */
    BL      m33_cst_spfpu_convert_test2_s32fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_s32fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s0_s31_to_r1_destr_r2
    
    /***************************************************************************/
    /* Continue with Signed 16-Bit conversions  */ 
    /***************************************************************************/
    /* Prepare fixed-point value */
    MOV     R0,#0x00000AAA                      /* (Positive number) */
    /* Write S14...S31 registers - (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0
    /* Convert fixed-point to floating point */
    BL      m33_cst_spfpu_convert_test2_s16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_s16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    /* Prepare fixed-point value */
    MOV     R0,#0x000000D5            
    ORR     R0,R0,#(1<<15)                      /* (Negative number) */
    /* Write S14...S31 registers - (S14 will not be used) */
    BL      m33_cst_write_fpu_s14_s31_from_r0
    /* Convert fixed-point to floating point */    
    BL      m33_cst_spfpu_convert_test2_s16fix_to_f32
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    /* Convert back to fix-point (different exponents) */
    BL      m33_cst_spfpu_convert_test2_f32_to_s16fix
    /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s15_s31_to_r1_destr_r2
    
    /**************************************************************************
     * Test producing the FPSCR[IOC, IDC, IXC] flags 
     * 
     * These flags are cumulative error flags, they can be tested only 
     * for 0->1 transition. Each flag is tested twice for 0->1 transition.
     **************************************************************************/
    MOV     R0,#0
    VMSR    FPSCR,R0    /* Write RMode and clear FPSCR */
    
    /**************************************************************************/
    /* Test FPSCR.IOC bit */
    /**************************************************************************/
    VMOV.F32        S31,#1.0
    VCVT.U32.F32    S31,S31,#32     /* => FPSCR.IOC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            
    EOR     R1,R2,R1,ROR #1         
    
    BIC     R2,R2,#(1<<0)              /* Clear FPSCR.IOC */
    VMSR    FPSCR,R2             
    
    VMOV.F32        S30,#16.0
    VCVT.S16.F32    S30,S30,#11     /* => FPSCR.IOC = 1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            
    EOR     R1,R2,R1,ROR #1
    
    /**************************************************************************/
    /* Test FPSCR.IDC bit */
    /**************************************************************************/
    ORR     R2,R2,#(1<<24)         /* Set Flush-to-zero */
    VMSR    FPSCR,R2            /* Write FPSCR.FZ bit */
    
    VMOV.F32        S29,#0.5    /* S30=0x3F000000 */     
    VCVT.U32.F32    S29,S29,#1  /* Prepare denormal input (S30=0x00000001) */    
    VCVT.U32.F32    S29,S29,#1  /* => FPSCR.IDC=1 (convert F32 denormal) */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            
    EOR     R1,R2,R1,ROR #1     
    
    BIC     R2,R2,#(1<<7)          /* Clear FPSCR.IDC */
    VMSR    FPSCR,R2            
   
    VMOV.F32        S28,#0.5    /* S29=0x3F000000 */
    VCVT.U32.F32    S28,S28,#23 /* Prepare denormal input (S29=0x00400000) */ 
    VCVT.S32.F32    S28,S28,#32 /* => FPSCR.IDC=1 (convert F32 denormal)*/
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            
    EOR     R1,R2,R1,ROR #1     
    
    /**************************************************************************/
    /* Test FPSCR.IXC bit */
    /**************************************************************************/
    VMOV.F32        S27,#0.96875
    VCVT.U32.F32    S27,S27,#4  /* => FPSCR.IXC=1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            
    EOR     R1,R2,R1,ROR #1     
    
    BIC     R2,R2,#(1<<4)          /* Clear FPSCR.IXC */
    VMSR    FPSCR,R2 
    
    VMOV.F32        S26,S31
    VCVT.F32.U32    S31,S31     /* => FPSCR.IXC=1 */
    /* !! Update Signature !! */
    VMRS    R2,FPSCR            
    EOR     R1,R2,R1,ROR #1     
    
    /**************************************************************************
     * Test conversions of 
     * - +Inf,-Inf,
     * - +NaN,-NaN,
     * - +0,-0 
     **************************************************************************/
    MOV     R3,#(255<<23)       /* +Inf */
    ORR     R4,R3,#(1<<31)      /* -Inf */
    ORR     R5,R3,#(1<<22)      /* +qNaN */
    ORR     R6,R4,#(1<<0)       /* -sNaN */
    MOV     R7,#(0<<31)         /* +0   */
    MOV     R8,#(1<<31)         /* -0   */
    
    /* Initialize float registers */
    VMOV    D15,R3,R3           /* +Inf */
    VMOV    S28,S29,R4,R4       /* -Inf */
    VMOV    S26,S27,R3,R3       /* +Inf */
    VMOV    S24,S25,R5,R5       /* +NaN*/
    VMOV    S22,S23,R6,R6       /* -NaN*/
    VMOV    S20,S21,R5,R5       /* +NaN*/
    VMOV    S18,S19,R7,R7       /* +0 */
    VMOV    S16,S17,R8,R8       /* -0 */
    VMOV    S14,S15,R7,R7       /* +0 */
    
    /* Test conversions of +-Inf */
    VCVT.U32.F32    S31,S31,#1  
    VCVT.U16.F32    S30,S30,#1  
    VCVT.S32.F32    S29,S29,#1  
    VCVT.S16.F32    S28,S28,#1  
    VCVT.S32.F32    S27,S27,#1  
    VCVT.S16.F32    S26,S26,#1  
    /* Test conversions of +-NaN */     
    VCVT.U32.F32    S25,S25,#1
    VCVT.U16.F32    S24,S24,#1
    VCVT.S32.F32    S23,S23,#1
    VCVT.S16.F32    S22,S22,#1
    VCVT.S32.F32    S21,S21,#1
    VCVT.S16.F32    S20,S20,#1
    /* Test conversions of +-0 */
    VCVT.U32.F32    S19,S19,#1
    VCVT.U16.F32    S18,S18,#1
    VCVT.S32.F32    S17,S17,#1
    VCVT.S16.F32    S16,S16,#1
    VCVT.S32.F32    S15,S15,#1
    VCVT.S16.F32    S14,S14,#1
    
     /* !! Update Signature !! */
    BL      m33_cst_sum_fpu_s14_s31_to_r1_destr_r2
    
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
m33_cst_spfpu_convert_test2_end:
    VMSR    FPSCR,R12   /* Restore FPSCR register */
    /* Test result is returned in R0, according to the conventions */
    MOV     R0,R1 
    B       m33_cst_test_tail_fpu_end
    
    
    /*------------------------------------------------------------------------*/
    /* Internal test routines                                                 */
    /*------------------------------------------------------------------------*/
m33_cst_spfpu_convert_test2_f32_to_u32fix:
    
    /* Test each fix point position */
    VCVT.U32.F32    S0,S0,#1    
    VCVT.U32.F32    S1,S1,#2    
    VCVT.U32.F32    S2,S2,#3    
    VCVT.U32.F32    S3,S3,#4    
    VCVT.U32.F32    S4,S4,#5    
    VCVT.U32.F32    S5,S5,#6    
    VCVT.U32.F32    S6,S6,#7    
    VCVT.U32.F32    S7,S7,#8    
    VCVT.U32.F32    S8,S8,#9    
    VCVT.U32.F32    S9,S9,#10   
    VCVT.U32.F32    S10,S10,#11 
    VCVT.U32.F32    S11,S11,#12 
    VCVT.U32.F32    S12,S12,#13 
    VCVT.U32.F32    S13,S13,#14 
    VCVT.U32.F32    S14,S14,#15 
    VCVT.U32.F32    S15,S15,#16 
    VCVT.U32.F32    S16,S16,#17 
    VCVT.U32.F32    S17,S17,#18 
    VCVT.U32.F32    S18,S18,#19 
    VCVT.U32.F32    S19,S19,#20 
    VCVT.U32.F32    S20,S20,#21 
    VCVT.U32.F32    S21,S21,#22 
    VCVT.U32.F32    S22,S22,#23 
    VCVT.U32.F32    S23,S23,#24 
    VCVT.U32.F32    S24,S24,#25 
    VCVT.U32.F32    S25,S25,#26 
    VCVT.U32.F32    S26,S26,#27 
    VCVT.U32.F32    S27,S27,#28 
    VCVT.U32.F32    S28,S28,#29 
    VCVT.U32.F32    S29,S29,#30 
    VCVT.U32.F32    S30,S30,#31 
    VCVT.U32.F32    S31,S31,#32 
    
    BX      LR
    
m33_cst_spfpu_convert_test2_u32fix_to_f32:
    
    /* Test each fix point position */
    VCVT.F32.U32    S0,S0,#1   
    VCVT.F32.U32    S1,S1,#2   
    VCVT.F32.U32    S2,S2,#3   
    VCVT.F32.U32    S3,S3,#4   
    VCVT.F32.U32    S4,S4,#5   
    VCVT.F32.U32    S5,S5,#6   
    VCVT.F32.U32    S6,S6,#7   
    VCVT.F32.U32    S7,S7,#8   
    VCVT.F32.U32    S8,S8,#9   
    VCVT.F32.U32    S9,S9,#10  
    VCVT.F32.U32    S10,S10,#11
    VCVT.F32.U32    S11,S11,#12
    VCVT.F32.U32    S12,S12,#13
    VCVT.F32.U32    S13,S13,#14
    VCVT.F32.U32    S14,S14,#15
    VCVT.F32.U32    S15,S15,#16
    VCVT.F32.U32    S16,S16,#17
    VCVT.F32.U32    S17,S17,#18
    VCVT.F32.U32    S18,S18,#19
    VCVT.F32.U32    S19,S19,#20
    VCVT.F32.U32    S20,S20,#21
    VCVT.F32.U32    S21,S21,#22
    VCVT.F32.U32    S22,S22,#23
    VCVT.F32.U32    S23,S23,#24
    VCVT.F32.U32    S24,S24,#25
    VCVT.F32.U32    S25,S25,#26
    VCVT.F32.U32    S26,S26,#27
    VCVT.F32.U32    S27,S27,#28
    VCVT.F32.U32    S28,S28,#29
    VCVT.F32.U32    S29,S29,#30
    VCVT.F32.U32    S30,S30,#31
    VCVT.F32.U32    S31,S31,#32
    
    BX      LR
    
    
 m33_cst_spfpu_convert_test2_f32_to_s32fix:

    /* Test each fix point position */
    VCVT.S32.F32    S0,S0,#1    
    VCVT.S32.F32    S1,S1,#2    
    VCVT.S32.F32    S2,S2,#3    
    VCVT.S32.F32    S3,S3,#4    
    VCVT.S32.F32    S4,S4,#5    
    VCVT.S32.F32    S5,S5,#6    
    VCVT.S32.F32    S6,S6,#7    
    VCVT.S32.F32    S7,S7,#8    
    VCVT.S32.F32    S8,S8,#9    
    VCVT.S32.F32    S9,S9,#10   
    VCVT.S32.F32    S10,S10,#11 
    VCVT.S32.F32    S11,S11,#12 
    VCVT.S32.F32    S12,S12,#13 
    VCVT.S32.F32    S13,S13,#14 
    VCVT.S32.F32    S14,S14,#15 
    VCVT.S32.F32    S15,S15,#16 
    VCVT.S32.F32    S16,S16,#17 
    VCVT.S32.F32    S17,S17,#18 
    VCVT.S32.F32    S18,S18,#19 
    VCVT.S32.F32    S19,S19,#20 
    VCVT.S32.F32    S20,S20,#21 
    VCVT.S32.F32    S21,S21,#22 
    VCVT.S32.F32    S22,S22,#23 
    VCVT.S32.F32    S23,S23,#24 
    VCVT.S32.F32    S24,S24,#25 
    VCVT.S32.F32    S25,S25,#26 
    VCVT.S32.F32    S26,S26,#27 
    VCVT.S32.F32    S27,S27,#28 
    VCVT.S32.F32    S28,S28,#29 
    VCVT.S32.F32    S29,S29,#30 
    VCVT.S32.F32    S30,S30,#31 
    VCVT.S32.F32    S31,S31,#32 
    
    BX      LR
    
 m33_cst_spfpu_convert_test2_s32fix_to_f32:
    
    /* Test each fix point position */
    VCVT.F32.S32    S0,S0,#1   
    VCVT.F32.S32    S1,S1,#2   
    VCVT.F32.S32    S2,S2,#3   
    VCVT.F32.S32    S3,S3,#4   
    VCVT.F32.S32    S4,S4,#5   
    VCVT.F32.S32    S5,S5,#6   
    VCVT.F32.S32    S6,S6,#7   
    VCVT.F32.S32    S7,S7,#8   
    VCVT.F32.S32    S8,S8,#9   
    VCVT.F32.S32    S9,S9,#10  
    VCVT.F32.S32    S10,S10,#11
    VCVT.F32.S32    S11,S11,#12
    VCVT.F32.S32    S12,S12,#13
    VCVT.F32.S32    S13,S13,#14
    VCVT.F32.S32    S14,S14,#15
    VCVT.F32.S32    S15,S15,#16
    VCVT.F32.S32    S16,S16,#17
    VCVT.F32.S32    S17,S17,#18
    VCVT.F32.S32    S18,S18,#19
    VCVT.F32.S32    S19,S19,#20
    VCVT.F32.S32    S20,S20,#21
    VCVT.F32.S32    S21,S21,#22
    VCVT.F32.S32    S22,S22,#23
    VCVT.F32.S32    S23,S23,#24
    VCVT.F32.S32    S24,S24,#25
    VCVT.F32.S32    S25,S25,#26
    VCVT.F32.S32    S26,S26,#27
    VCVT.F32.S32    S27,S27,#28
    VCVT.F32.S32    S28,S28,#29
    VCVT.F32.S32    S29,S29,#30
    VCVT.F32.S32    S30,S30,#31
    VCVT.F32.S32    S31,S31,#32
    
    BX      LR
    
    
m33_cst_spfpu_convert_test2_f32_to_u16fix:
    
    /* Test each fix point position */
    VCVT.U16.F32    S15,S15,#0
    VCVT.U16.F32    S16,S16,#1
    VCVT.U16.F32    S17,S17,#2
    VCVT.U16.F32    S18,S18,#3
    VCVT.U16.F32    S19,S19,#4
    VCVT.U16.F32    S20,S20,#5
    VCVT.U16.F32    S21,S21,#6
    VCVT.U16.F32    S22,S22,#7
    VCVT.U16.F32    S23,S23,#8
    VCVT.U16.F32    S24,S24,#9
    VCVT.U16.F32    S25,S25,#10
    VCVT.U16.F32    S26,S26,#11
    VCVT.U16.F32    S27,S27,#12
    VCVT.U16.F32    S28,S28,#13
    VCVT.U16.F32    S29,S29,#14
    VCVT.U16.F32    S30,S30,#15
    VCVT.U16.F32    S31,S31,#16
    
    BX      LR
    
 
m33_cst_spfpu_convert_test2_u16fix_to_f32:

    /* Test each fix point position */
    VCVT.F32.U16    S15,S15,#0
    VCVT.F32.U16    S16,S16,#1
    VCVT.F32.U16    S17,S17,#2
    VCVT.F32.U16    S18,S18,#3
    VCVT.F32.U16    S19,S19,#4
    VCVT.F32.U16    S20,S20,#5
    VCVT.F32.U16    S21,S21,#6
    VCVT.F32.U16    S22,S22,#7
    VCVT.F32.U16    S23,S23,#8
    VCVT.F32.U16    S24,S24,#9
    VCVT.F32.U16    S25,S25,#10
    VCVT.F32.U16    S26,S26,#11
    VCVT.F32.U16    S27,S27,#12
    VCVT.F32.U16    S28,S28,#13
    VCVT.F32.U16    S29,S29,#14
    VCVT.F32.U16    S30,S30,#15
    VCVT.F32.U16    S31,S31,#16
    
    BX      LR
    
    
m33_cst_spfpu_convert_test2_f32_to_s16fix:
  
    /* Test each fix point position */
    VCVT.S16.F32    S15,S15,#0
    VCVT.S16.F32    S16,S16,#1
    VCVT.S16.F32    S17,S17,#2
    VCVT.S16.F32    S18,S18,#3
    VCVT.S16.F32    S19,S19,#4
    VCVT.S16.F32    S20,S20,#5
    VCVT.S16.F32    S21,S21,#6
    VCVT.S16.F32    S22,S22,#7
    VCVT.S16.F32    S23,S23,#8
    VCVT.S16.F32    S24,S24,#9
    VCVT.S16.F32    S25,S25,#10
    VCVT.S16.F32    S26,S26,#11
    VCVT.S16.F32    S27,S27,#12
    VCVT.S16.F32    S28,S28,#13
    VCVT.S16.F32    S29,S29,#14
    VCVT.S16.F32    S30,S30,#15
    VCVT.S16.F32    S31,S31,#16
    
    BX      LR
    
m33_cst_spfpu_convert_test2_s16fix_to_f32:

    /* Test each fix point position */
    VCVT.F32.S16    S15,S15,#0
    VCVT.F32.S16    S16,S16,#1
    VCVT.F32.S16    S17,S17,#2
    VCVT.F32.S16    S18,S18,#3
    VCVT.F32.S16    S19,S19,#4
    VCVT.F32.S16    S20,S20,#5
    VCVT.F32.S16    S21,S21,#6
    VCVT.F32.S16    S22,S22,#7
    VCVT.F32.S16    S23,S23,#8
    VCVT.F32.S16    S24,S24,#9
    VCVT.F32.S16    S25,S25,#10
    VCVT.F32.S16    S26,S26,#11
    VCVT.F32.S16    S27,S27,#12
    VCVT.F32.S16    S28,S28,#13
    VCVT.F32.S16    S29,S29,#14
    VCVT.F32.S16    S30,S30,#15
    VCVT.F32.S16    S31,S31,#16
    
    BX      LR
    
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_FPU_ENABLE */

    CST_FILE_END

