/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
*
* Covers Load/Store unprivileged functionality:
*  - memory protection
*  - exception calling
*  - instruction decoding
*
* Overall coverage:
* -----------------
* Covers LDRBT, LDRHT, LDRSBT, LDRSHT, LDRT, STRBT, STRHT, STRT instructions
*
* NOTE: Encoding of Load/Store Unprivileged instructions is same as encoding of
*       other Load/Store instructions thus encoding of input/output registers
*       is fully covered by other load/store tests.
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_MPU_ENABLE==1)
    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_MPUTest

    /* Symbols defined outside but used within current module */
    CST_EXTERN CST_RAM_TARGET0
    CST_EXTERN CST_RAM_TARGET1
    CST_EXTERN m33_cst_test_tail_end
    CST_EXTERN m33_cst_store_registers_content
    CST_EXTERN m33_cst_restore_registers_content
    CST_EXTERN m33_cst_set_cst_interrupt_vector_table
    CST_EXTERN m33_cst_ISR_address
    CST_EXTERN VAL1
    CST_EXTERN VAL2
    
    CST_SET(PRESIGNATURE, 0x2CF48A0D)
    
    /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
    /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_MPUTest" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_THUMB2
    CST_TYPE(M33_Cst_MPUTest, function)
M33_Cst_MPUTest:

    PUSH    {R4-R12,R14}
    
    /*------------------------------------------------------------------------*/
    /* Store register content                                                 */
    /*------------------------------------------------------------------------*/
    /* !! Don't use R6, R7, R8, R9 for any other purpose until CONTROL, PRIMASK,
          FAULTMASK and BASEPRI are restored !! */
    BL      m33_cst_store_registers_content


    /*------------------------------------------------------------------------*/
    /* Store address of the CST-own ISR into the m33_cst_ISR_address pointer */
    /*------------------------------------------------------------------------*/
    ADR.W   R4,m33_cst_mpu_hard_fault_ISR
    ORR     R4,R4,#1        /* Set bit [0] as this address will be used within 
                               BX command and should indicate the Thumb code */
    LDR.W   R3,=m33_cst_ISR_address
    STR     R4,[R3]

    /*------------------------------------------------------------------------*/
    /* Test - preparation                                                     */
    /*------------------------------------------------------------------------*/    
    LDR.W   R5,=PRESIGNATURE /* Load pre-signature */
    MOV     R2,#3           /* Set expected exception number - hardfault */
    
    /*------------------------------------------------------------------------*/
    /* Test - start                                                           */
    /*------------------------------------------------------------------------*/
    /* Start testing with CST_RAM_TARGET0 */
    LDR.W   R11, =CST_RAM_TARGET0
    BL      m33_cst_mpu_common

    /* Start testing with CST_RAM_TARGET1 */
    LDR.W   R11, =CST_RAM_TARGET1
    BL      m33_cst_mpu_excs
    
    /*------------------------------------------------------------------------*/
    /* Test - end                                                             */
    /*------------------------------------------------------------------------*/
    /* Test result is returned in R0, according to the conventions */
    MVN     R0, R5
m33_cst_mpu_end:
    B      m33_cst_test_tail_end

    /*------------------------------------------------------------------------*/
    /* Test routine - start                                                   */
    /*------------------------------------------------------------------------*/ 
m33_cst_mpu_common:
    PUSH    {LR}
    
    /* Initialize RAM and R12 */
    LDR.W   R12,=VAL1
    MOV     R10,#0xFFFFFFFF
    STR     R10,[R11]       
    DSB
    /* Call byte test with R12=0x55555555*/
    BL      m33_cst_mpu_byte
    
    /* Half word test with R12=0x55555555 */
    MOV     R10,#0xFFFFFFFF
    STR     R10,[R11]
    DSB
    BL      m33_cst_mpu_half
    
    /* Word test with R12=0x55555555 */
    MOV     R10,#0xFFFFFFFF
    STR     R10,[R11]
    DSB
    BL      m33_cst_mpu_word
    
    POP     {LR}

m33_cst_mpu_excs:
    PUSH    {LR}

    /* Byte test with R12=0xAAAAAAAA */
    LDR.W   R12,=VAL2
    MOV     R10,#0x00000000
    STR     R10,[R11]
    DSB
    BL      m33_cst_mpu_byte
    
    /* Half word test with R12=0xAAAAAAAA */
    MOV     R10,#0x00000000
    STR     R10,[R11]
    DSB
    BL      m33_cst_mpu_half
    
    /* Word test with R12=0xAAAAAAAA */
    MOV     R10,#0x00000000
    STR     R10,[R11]
    DSB
    BL      m33_cst_mpu_word

    POP     {PC}            /* Return from test routine */


    /***************************************************************************
    * BYTE instructions:
    *   - STRBT Encoding T1 (32-bit)
    *   - LDRBT Encoding T1 (32-bit)
    *   - LDRSBT Encoding T1 (32-bit)
    ***************************************************************************/
m33_cst_mpu_byte:
    PUSH    {LR}
    
    LDR     R0,=m33_cst_mpu_strbt
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
         
m33_cst_mpu_strbt:
    STRBT   R12,[R11]
    DSB
    
    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    /* Check if STRBT stored data */
    LDR     R10,[R11]
    EOR     R5,R10,R5,ROR #1 /* Update signature */
    MVN     R10,R10         /* Destroy value in R10 */
    
    LDR     R0,=m33_cst_mpu_ldrbt
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
    
m33_cst_mpu_ldrbt:
    LDRBT   R10,[R11]

    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    EOR     R5,R10,R5,ROR #1 /* Update signature */
    MVN     R10,R10         /* Destroy value in R10 */
    
    LDR     R0,=m33_cst_mpu_ldrsbt
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
    
m33_cst_mpu_ldrsbt:
    LDRSBT  R10,[R11]

    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    /*------------------------------------------------------------------------*/
    /* Test routine - end                                                     */
    /*------------------------------------------------------------------------*/
    EOR     R5,R10,R5,ROR #1 /* Update signature */
    
    POP     {PC}            /* We need to return back */


    /***************************************************************************
    * HALF WORD instructions:
    *   - STRHT Encoding T1 (32-bit)
    *   - LDRHT Encoding T1 (32-bit)
    *   - LDRSHT Encoding T1 (32-bit)
    ***************************************************************************/
m33_cst_mpu_half:
    PUSH    {LR}
    
    LDR     R0,=m33_cst_mpu_strht
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
    
m33_cst_mpu_strht:
    STRHT   R12,[R11]
    DSB
    
    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    /* Check if STRHT stored data */
    LDR     R10,[R11]
    EOR     R5,R10,R5,ROR #1 /* Update signature */
    MVN     R10,R10         /* Destroy value in R10 */
    
    LDR     R0,=m33_cst_mpu_ldrht
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
    
m33_cst_mpu_ldrht:
    LDRHT   R10,[R11]

    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    EOR     R5,R10,R5,ROR #1 /* Update signature */
    MVN     R10,R10         /* Destroy value in R10 */
    
    LDR     R0,=m33_cst_mpu_ldrsht
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
    
m33_cst_mpu_ldrsht:
    LDRSHT  R10,[R11]

    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    /*------------------------------------------------------------------------*/
    /* Test routine - end                                                     */
    /*------------------------------------------------------------------------*/
    EOR     R5,R10,R5,ROR #1 /* Update signature */
    
    POP     {PC}            /* We need to return back */


    /***************************************************************************
    * WORD instructions:
    *   - STRT Encoding T1 (32-bit)
    *   - LDRT Encoding T1 (32-bit)
    ***************************************************************************/
m33_cst_mpu_word:
    PUSH    {LR}
    
    LDR     R0,=m33_cst_mpu_strt
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
    
m33_cst_mpu_strt:
    STRT    R12,[R11]
    DSB
    
    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    /* Check if STRT stored data */
    LDR     R10,[R11]
    EOR     R5,R10,R5,ROR #1 /* Update signature */
    MVN     R10,R10         /* Destroy value in R10 */
    
    LDR     R0,=m33_cst_mpu_ldrt
    BIC     R0,R0,#1        /* Clear bit [0]. Ensures that label is not
                               treated as Thumb function */
    
    /*------------------------------------------------------------------------*/
    /* Set CST interrupt vector table & Disable external interrupts          */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_set_cst_interrupt_vector_table
    
    /* Disable all interrupts except hardfault */
    CPSID   i               /* Set PRIMASK */
    CPSIE   f               /* Clear FAULTMASK */
    
m33_cst_mpu_ldrt:
    LDRT    R10,[R11]

    /*------------------------------------------------------------------------*/
    /* Restore all registers & Enable external interrupts                     */
    /*------------------------------------------------------------------------*/
    BL      m33_cst_restore_registers_content

    /*------------------------------------------------------------------------*/
    /* Test routine - end                                                     */
    /*------------------------------------------------------------------------*/
    EOR     R5,R10,R5,ROR #1 /* Update signature */
    
    POP     {PC}            /* We need to return back */


    /***************************************************************************
    * Interrupt routine
    *    Error flags are cleared and signature is updated
    *    to check that exception was always successfully called.
    ***************************************************************************/
m33_cst_mpu_hard_fault_ISR:
    /* We need to check the DACCVIOL and MMARVALID flag is set */
    LDR     R3,=M33_MMFSR_REG
    LDRB    R2,[R3]         /* Use byte access to MMFSR */
    AND     R2,R2,#0x82     /* Check DACCVIOL and MMARVALID flag is set */
    EOR     R5,R2,R5,ROR #1 /* Update signature */
    
    /* We need to clear the DACCVIOL and MMARVALID flag */
    STRB    R2,[R3]         /* Clear DACCVIOL and MMARVALID flag */
    DSB
    LDRB    R2,[R3]         /* Use byte access to MMFSR */
    AND     R2,R2,#0x82     /* Check DACCVIOL and MMARVALID flag was cleared */
    EOR     R5,R2,R5,ROR #1 /* Update signature */
    
    /* Modify return address to return to next instruction */
    LDR     R3,[SP,#24]
    ADD     R3,R3,#4
    STR     R3,[SP,#24]
    DSB
    
    BX      LR

    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
    numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif  /* CORTST_M33_MPU_ENABLE */
    
    CST_FILE_END

