/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
*   Tests forwarding logic functionality. 
*   - Verifies that results from one instruction can be used immediately 
*     as inputs in next instructions. Values are not taken from register file 
*     but they must be forwarded.
*
* The following cases are tested:
* -------------------------------
*  1)   Forwarding from ALU0,ALU1(EX2) to AGU0,AGU1 
*  2)   Forwarding from ALU0,ALU1(EX1) to AGU0,AGU1 
*  3)   Forwarding from Slot0 to MAC accumulator in Slot1
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FORWARD_ENABLE==1)
    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_ForwardingTest4

    /* Symbols defined outside but used within current module */
    CST_EXTERN CST_RAM_TARGET0
    CST_EXTERN m33_cst_test_tail_end
    

    CST_SET(PRESIGNATURE, 0xA3D18626)

    /* Values stored in registers:
       - R0  ... accumulated signature
       - R12 ... PRIMASK
    */
    /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
    /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_ForwardingTest4" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_THUMB2
    CST_TYPE(M33_Cst_ForwardingTest4, function)
M33_Cst_ForwardingTest4:

    PUSH    {R4-R12,R14}
    
    /*------------------------------------------------------------------------*/
    /* Test - preparation                                                     */
    /*------------------------------------------------------------------------*/
    LDR     R0,=PRESIGNATURE    /* Initialize results registers */
    
    
    /*------------------------------------------------------------------------*/
    /* Test - start                                                           */
    /*------------------------------------------------------------------------*/
    MRS     R12,PRIMASK                 /* Save PRIMASK  */
    
    MOV     R1,#0x00000000
    MSR     APSR_nzcvq,R1               /* Clear flags */
   
    /**************************************************************************/
    /* Forwarding ALU (EX2)-> AGU                                             */
    /**************************************************************************/
    LDR     R2,=CST_RAM_TARGET0
    MOV     R1,#0x55555555  /* First value to be forwarded */
    BL      m33_cst_forwarding_test4_part1
    MOV     R1,#0xAAAAAAAA  /* First value to be forwarded */
    BL      m33_cst_forwarding_test4_part1
    
    /**************************************************************************/
    /* Forwarding ALU (EX1)-> AGU                                             */
    /**************************************************************************/
    MOV     R1,#0x55555555  /* First value to be forwarded */
    BL      m33_cst_forwarding_test4_part2
    MOV     R1,#0xAAAAAAAA  /* First value to be forwarded */
    BL      m33_cst_forwarding_test4_part2
    
    /**************************************************************************/
    /* Forwarding ALU->MAC(C0,C1)                                             */
    /**************************************************************************/
    MOV     R1,#0x55555555  /* First value to be forwarded */
    BL      m33_cst_forwarding_test4_part3
    MOV     R1,#0xAAAAAAAA  /* First value to be forwarded */
    BL      m33_cst_forwarding_test4_part3
    
    
    /*------------------------------------------------------------------------*/
    /* Test routine - End                                                     */
    /*------------------------------------------------------------------------*/    
m33_cst_forwarding_test4_end:
    /* Test result is returned in R0, according to the conventions */ 
    B       m33_cst_test_tail_end
    
    
    /*------------------------------------------------------------------------*/
    /* Test - internal routines                                               */
    /*------------------------------------------------------------------------*/
    /**************************************************************************/
    /* Forwarding ALU (EX2)-> AGU                                             */
    /* 1a) Forwarding ALU0(EX2)->AGU0 (operand A)                             */
    /*     Forwarding ALU0(EX2)->AGU1 (operand A)                             */
    /* 1b) Forwarding ALU1(EX2)->AGU0 (operand A)                             */
    /*     Forwarding ALU1(EX2)->AGU1 (operand A)                             */
    /* 1c) Forwarding ALU0(EX2)->AGU0 (operand B)                             */
    /*     Forwarding ALU0(EX2)->AGU1 (operand B)                             */
    /* 1d) Forwarding ALU1(EX2)->AGU0 (operand B)                             */
    /*     Forwarding ALU1(EX2)->AGU1 (operand B)                             */
    /**************************************************************************/
m33_cst_forwarding_test4_part1:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
    STR     R0,[R2]             /* Store data to memory */
    DSB
m33_cst_forwarding_test4_loop1:
    SUB     R10,R2,R1           /* Destroy address */
    SUB     R1,R1,R11            
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ADD     R4,R1,R11           /* Slot0 */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    LDR     R3,[R4,R10]         /* 1a) Forward Address(operandA) to AGU0 */
    LDR     R4,[R4,R10]         /* 1a) Forward Address(operandA) to AGU1 */
    /*------------------------------------------------------------------------*/
    ISB
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    ADD     R6,R1,R11           /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    LDR     R5,[R6,R10]         /* 1b) Forward Address(operandA) to AGU0 */
    LDR     R6,[R6,R10]         /* 1b) Forward Address(operandA) to AGU1 */
    /*------------------------------------------------------------------------*/
    ISB
    /*------------------------------------------------------------------------*/
    ADD     R8,R1,R11           /* Slot0 */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    LDR     R7,[R10,R8]         /* 1c) Forward Offset(operandB) to AGU0 */
    LDR     R8,[R10,R8]         /* 1c) Forward Offset(operandB) to AGU1 */
    /*------------------------------------------------------------------------*/
    ISB
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    ADD     R1,R1,R11           /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    LDR     R9,[R10,R1]         /* 1d) Forward Offset(operandB) to AGU1 */
    LDR     R10, [R10,R1]       /* 1d) Forward Offset(operandB) to AGU0 */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    BL      m33_cst_forwarding_test4_sum_r3_r10
    
    MOV     R1,R0   /* Prepare new data to be forwarded */
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test4_loop1  
    
    POP     {R15}   /* Exit test subroutine */
    
    /**************************************************************************/
    /* Forwarding ALU (EX1)-> AGU                                             */
    /* 2a) Forwarding ALU0(EX1)->AGU0 (operand A)                             */
    /*     Forwarding ALU0(EX1)->AGU1 (operand A)                             */
    /* 2b) Forwarding ALU1(EX1)->AGU0 (operand A)                             */
    /*     Forwarding ALU1(EX1)->AGU1 (operand A)                             */
    /* 2c) Forwarding ALU0(EX1)->AGU0 (operand B)                             */
    /*     Forwarding ALU0(EX1)->AGU1 (operand B)                             */
    /* 2d) Forwarding ALU1(EX1)->AGU0 (operand B)                             */
    /*     Forwarding ALU1(EX1)->AGU1 (operand B)                             */
    /**************************************************************************/
 m33_cst_forwarding_test4_part2:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
    STR     R0,[R2]             /* Store data to memory */
    DSB
m33_cst_forwarding_test4_loop2:
    SUB     R10,R2,R1           /* Destroy address */
    ROR     R1,R1,#1
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ROR     R4,R1,#31
    NOP
    /* Interlock here to avoid Hazard !! */
    LDR     R3,[R4,R10]         /* 2a) Forward Address(operandA) to AGU0 */
    LDR     R4,[R4,R10]         /* 2a) Forward Address(operandA) to AGU1 */
    /*------------------------------------------------------------------------*/
    ISB
    /*------------------------------------------------------------------------*/
    NOP
    ROR     R6,R1,#31
    /* Interlock here to avoid Hazard !! */
    LDR     R5,[R6,R10]         /* 2b) Forward Address(operandA) to AGU0 */
    LDR     R6,[R6,R10]         /* 2b) Forward Address(operandA) to AGU1 */
    /*------------------------------------------------------------------------*/
    ISB
    /*------------------------------------------------------------------------*/
    ROR     R8,R1,#31
    NOP
    /* Interlock here to avoid Hazard !! */
    LDR     R7,[R10,R8]         /* 2c) Forward Offset(operandB) to AGU0 */
    LDR     R8,[R10,R8]         /* 2c) Forward Offset(operandB) to AGU1 */
    /*------------------------------------------------------------------------*/
    ISB
    /*------------------------------------------------------------------------*/
    NOP
    ROR     R1,R1,#31
    /* Interlock here to avoid Hazard !! */
    LDR     R9,[R10,R1]         /* 2d) Forward Offset(operandB) to AGU0 */
    LDR     R10,[R10,R1]        /* 2d) Forward Offset(operandB) to AGU1 */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    BL      m33_cst_forwarding_test4_sum_r3_r10
    
    MOV     R1,R0   /* Prepare new data to be forwarded */
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test4_loop2  
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* C0,C1 read ports and forwarding                                        */
    /* 3)  We can forward from the end of a data-processing instruction       */ 
    /*     in EX2, channel 0, into the MAC accumulator of a mac in EX2,       */ 
    /*     channel 1                                                          */
    /**************************************************************************/
m33_cst_forwarding_test4_part3:
    MOV     R11,#10
    PUSH    {R14}
 m33_cst_forwarding_test4_loop3:
    SUB     R7,R1,R11,LSL #1
    MOV     R8,#0
    MOV     R9,#0
    SUB     R10,R1,R11,LSL #1
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ADD     R7,R7,R11,LSL #1    /* Slot0 - ALU produces result in EX2 */
    UMLAL   R7,R8,R1,R0         /* Slot1 - Forward to C0 port (R7) */
    ADD     R10,R10,R11,LSL #1  /* Slot0 - Alu produces result in EX2 */
    UMLAL   R9,R10,R1,R0        /* Slot1 - Forward to C1 port (R10) */            
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test4_sum_r7_r10
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0     
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test4_loop3
    
    POP     {R15}
    
    /**************************************************************************/
    /* !! Update Signature !!                                                 */
    /**************************************************************************/
m33_cst_forwarding_test4_sum_r3_r10:
    SBC     R3,R3,R0,ROR #1
    EOR     R0,R0,R3
    SBC     R4,R4,R0,ROR #2
    EOR     R0,R0,R4
    SBC     R5,R5,R0,ROR #3
    EOR     R0,R0,R5
    SBC     R6,R6,R0,ROR #4
    EOR     R0,R0,R6
m33_cst_forwarding_test4_sum_r7_r10:
    SBC     R7,R7,R0,ROR #5
    EOR     R0,R0,R7
    SBC     R8,R8,R0,ROR #6
    EOR     R0,R0,R8
    SBC     R9,R9,R0,ROR #7
    EOR     R0,R0,R9
    SBC     R10,R10,R0,ROR #8
    EOR     R0,R0,R10
    
    BX      LR
    
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif /* CORTST_M33_FORWARD_ENABLE */
    
    CST_FILE_END

