/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/******************************************************************************
* Test summary:
* -------------
*   Tests forwarding logic functionality. 
*   - Verifies that results from one instruction can be used immediately 
*     as inputs in next instructions. Values are not taken from register file 
*     but they must be forwarded.
*
* The following cases are tested:
* -------------------------------
* 1) Forward data from the end of EX2 stage to ALU EX1 stage
*    1a) Forward data from W0 port to ALU EX1 stage    
*    1b) Forward data from W1 port to ALU EX1 stage
*    1c) Forward data from W2 port to ALU EX1 stage
*    1d) Forward data from W3 port to ALU EX1 stage
* 2) Forward data from the end of EX2 stage to MAC EX1 stage
*    2a) Forward data from W0 port to MAC EX1 stage    
*    2b) Forward data from W1 port to MAC EX1 stage
*    2c) Forward data from W2 port to MAC EX1 stage
*    2d) Forward data from W3 port to MAC EX1 stage
* 3) Forward data from the end of EX2 stage to DIV EX1 stage
*    3a) Forward data from W0 port to DIV EX1 stage
*    3b) Forward data from W1 port to DIV EX1 stage
*    3c) Forward data from W2 port to DIV EX1 stage
*    3d) Forward data from W3 port to DIV EX1 stage
* 4) Forward data from the end of EX2 stage to SHIFT EX1 stage
*    4a) Forward data from W0 port to SHIFT EX1 stage
*    4b) Forward data from W1 port to SHIFT EX1 stage
*    4c) Forward data from W2 port to SHIFT EX1 stage
*    4d) Forward data from W3 port to SHIFT EX1 stage
* 5) R13 register forwarding (special case)
*
* For each case above data is forwarded to A0,A1,B0,B1 read ports and each
* forwarding path is stimulated with 20 pseudo-random test vectors.
*
******************************************************************************/

#include "CorTst_Compiler.h"
#include "CorTst_M33_Cfg.h"

#if (CORTST_M33_FORWARD_ENABLE==1)
    /* Compatible with ABI. */
    CST_PRES8
    /* Symbols defined in the current module but to be visible to outside */
    CST_EXPORT M33_Cst_ForwardingTest2

    /* Symbols defined outside but used within current module */
    CST_EXTERN CST_RAM_TARGET0
    CST_EXTERN m33_cst_test_tail_end
   
   
    CST_SET(PRESIGNATURE, 0x9A5629C8)
   
    /* Values stored in registers:
       - R0  ... accumulated signature
       - R2  ... CST_RAM_TARGET0 address
       - R12 ... PRIMASK
    */
    /*------------------------------------------------------------------------*/
    CST_SECTION_EXEC(mcal_text)
    /*------------------------------------------------------------------------*/
    /* The ".type" directive instructs the assembler/linker that the label 
       "M33_Cst_ForwardingTest2" designates a function.
       This would cause setting the least significant bit to '1' within any 
       pointer to this function, causing change to Thumb mode whenever this 
       function is called. */
    CST_THUMB2
    CST_TYPE(M33_Cst_ForwardingTest2, function)
M33_Cst_ForwardingTest2:

    PUSH    {R4-R12,R14}
    
    /*------------------------------------------------------------------------*/
    /* Test - preparation                                                     */
    /*------------------------------------------------------------------------*/
    LDR     R0,=PRESIGNATURE    /* Initialize results registers */
    
    
    /*------------------------------------------------------------------------*/
    /* Test - start                                                           */
    /*------------------------------------------------------------------------*/
    MRS     R12,PRIMASK                 /* Save PRIMASK  */
    
    MOV     R1,#0x00000000
    MSR     APSR_nzcvq,R1               /* Clear flags */
    
    LDR     R2,=CST_RAM_TARGET0        /* Do not use R2 !! */
    
    /**************************************************************************/
    /* 1a) Test forwarding from W0 write port to ALU0,ALU1                    */
    /* 1b) Test forwarding from W1 write port to ALU0,ALU1                    */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part1
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part1
    
    /**************************************************************************/
    /* 1c) Test forwarding from W2 write port to ALU0,ALU1                    */
    /* 1d) Test forwarding from W3 write port to ALU0,ALU1                    */
    /**************************************************************************/
    
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part2
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part2
    
    /**************************************************************************/
    /* 2a) Test forwarding from W0 write port to MAC                          */
    /* 2b) Test forwarding from W1 write port to MAC                          */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part3
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part3
    
    /**************************************************************************/
    /* 2c) Test forwarding from W2 write port to MAC                          */
    /* 2d) Test forwarding from W3 write port to MAC                          */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part4
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part4
    
    /**************************************************************************/
    /* 3a) Test forwarding from W0 write port to DIV                          */
    /* 3b) Test forwarding from W1 write port to DIV                          */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part5
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part5
    
    /**************************************************************************/
    /* 3c) Test forwarding from W2 write port to DIV                          */
    /* 3d) Test forwarding from W3 write port to DIV                          */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part6
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part6
    
    /**************************************************************************/
    /* 4a) Test forwarding from W0 write port to SHIFT                        */
    /* 4b) Test forwarding from W1 write port to SHIFT                        */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part7
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part7
    
    /**************************************************************************/
    /* 4c) Test forwarding from W2 write port to SHIFT                        */
    /* 4d) Test forwarding from W3 write port to SHIFT                        */
    /**************************************************************************/
    MOV     R1,#0x55555555      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part8
    MOV     R1,#0xAAAAAAAA      /* First value to be forwarded !! */
    BL      m33_cst_forwarding_test2_part8
    
    /**************************************************************************/
    /* Test forwarding from R13 register                                      */
    /**************************************************************************/
    MOV     R1,#0x00000001      /* Value to be forwarded (SP+R1) !! */
    BL      m33_cst_forwarding_test2_part9
    MOV     R1,#0x00000002      /* Value to be forwarded (SP+R1) !! */
    BL      m33_cst_forwarding_test2_part9
    MOV     R1,#0x00000003      /* Value to be forwarded (SP+R1) !! */
    BL      m33_cst_forwarding_test2_part9
    
    
    /*------------------------------------------------------------------------*/
    /* Test routine - End                                                     */
    /*------------------------------------------------------------------------*/    
m33_cst_forwarding_test2_end:
    /* Test result is returned in R0, according to the conventions */ 
    B       m33_cst_test_tail_end
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG
    
    /*------------------------------------------------------------------------*/
    /* Test - internal routines                                               */
    /*------------------------------------------------------------------------*/
    /**************************************************************************/
    /* 1a) Forwarding from write ports W0->ALU EX1                            */
    /* 1b) Forwarding from write ports W1->ALU EX1                            */
    /**************************************************************************/
m33_cst_forwarding_test2_part1:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop1:
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ADD     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R3,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    SUB     R4,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    SUB     R10,R1,R11,LSR #4   /* Slot0 -  DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R5,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    ADD     R6,R10,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    ADD     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port */
    /* Interlock here to avoid Hazard !! */
    SUB     R7,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    SUB     R8,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 - NOP */
    SUB     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    ADD     R9,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    ADD     R10,R10,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop1  
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 1c) Forwarding from write port W2->ALU EX1                             */
    /* 1d) Forwarding from write port W3->ALU EX1                             */
    /**************************************************************************/
m33_cst_forwarding_test2_part2:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop2:
    STR     R1,[R2]             /* Prepared data to be loaded */
    DSB 
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R3,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    ADD     R4,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    SUB     R5,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    SUB     R6,R10,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    ADD     R7,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    ADD     R8,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    SUB     R9,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    SUB     R10,R10,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop2
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 2a) Forwarding from write port W0->MAC EX1                             */
    /* 2b) Forwarding from write port W1->MAC EX1                             */
    /**************************************************************************/
m33_cst_forwarding_test2_part3:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop3:
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ADD     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    MUL     R3,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    SUB     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    MUL     R4,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */    
    ADD     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    MUL     R5,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    SUB     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    MUL     R6,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    ADD     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R7,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    SUB     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R8,R10,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */    
    ADD     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R9,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    SUB     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R10,R10,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop3
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 2c) Forwarding from write port W2->MAC EX1                             */
    /* 2d) Forwarding from write port W3->MAC EX1                             */
    /**************************************************************************/
m33_cst_forwarding_test2_part4:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop4:
    STR     R1,[R2]             /* Prepared data to be loaded */
    DSB
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    MUL     R3,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    MUL     R4,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    MUL     R5,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    MUL     R6,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R7,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R8,R10,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R9,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    NOP                         /* Slot0 */
    MUL     R10,R10,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop4
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 3a) Forwarding from write port W0->DIV EX1                             */
    /* 3b) Forwarding from write port W1->DIV EX1                             */
    /**************************************************************************/
m33_cst_forwarding_test2_part5:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop5:
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ADD     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    UDIV    R7,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    SUB     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    UDIV    R8,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */    
    ADD     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    UDIV    R9,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    SUB     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    UDIV    R10,R10,R1          /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r7_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop5
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 3c) Forwarding from write port W2->DIV EX1                             */
    /* 3d) Forwarding from write port W3->DIV EX1                             */
    /* Note: Divider runs always in slot0.                                    */
    /**************************************************************************/
m33_cst_forwarding_test2_part6:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop6:
    STR     R1,[R2]             /* Prepared data to be loaded */
    DSB
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    UDIV    R7,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    UDIV    R8,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    UDIV    R9,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    UDIV    R10,R10,R1          /* Slot0 - Forward data to B0 read port !! */
    NOP                         /* Slot1 */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r7_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop6
    
    POP     {R15}   /* Exit test subroutine */
   
   
    /**************************************************************************/
    /* 4a) Forwarding from write port W0->SHIFT EX1                           */
    /* 4b) Forwarding from write port W1->SHIFT EX1                           */
    /**************************************************************************/
m33_cst_forwarding_test2_part7:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop7:
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    ADD     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */   
    /* Interlock here to avoid Hazard !! */
    ROR     R3,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    ROR     R4,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    SUB     R10,R1,R11,LSR #4   /* Slot0 - DPU produces result to W0 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ROR     R5,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    ROR     R6,R10,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    ADD     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */  
    /* Interlock here to avoid Hazard !! */
    ROR     R7,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    ROR     R8,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    SUB     R10,R1,R11,LSR #4   /* Slot1 - DPU produces result to W1 port  */
    /* Interlock here to avoid Hazard !! */
    ROR     R9,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    ROR     R10,R10,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop7
    
    POP     {R15}   /* Exit test subroutine */
    
     
    /**************************************************************************/
    /* 4c) Forwarding from write port W2->SHIFT EX1                           */
    /* 4d) Forwarding from write port W3->SHIFT EX1                           */
    /**************************************************************************/
m33_cst_forwarding_test2_part8:
    MOV     R11,#10             /* Initialize loop counter */
    PUSH    {R14}
m33_cst_forwarding_test2_loop8:
    STR     R1,[R2]             /* Prepared data to be loaded */
    DSB
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ROR     R3,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    ROR     R4,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    LDR     R10,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ROR     R5,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    ROR     R6,R10,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    ROR     R7,R1,R10           /* Slot0 - Forward data to A0 read port !! */
    ROR     R8,R1,R10           /* Slot1 - Forward data to A1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R10,[R2]            /* Slot1 - DPU produces result to the W3 port */
    /* Interlock here to avoid Hazard !! */
    ROR     R9,R10,R1           /* Slot0 - Forward data to B0 read port !! */
    ROR     R10,R10,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Update signature !! */
    BL      m33_cst_forwarding_test2_sum_r3_r10      
    /* Prepare new random data to be forwarded in next cycle */
    MOV     R1,R0
    SUBS    R11,R11,#1
    BNE     m33_cst_forwarding_test2_loop8
    
    POP     {R15}   /* Exit test subroutine */
    
    
    /**************************************************************************/
    /* 5) R13 forwarding                                                      */
    /* - Forwarding logic must ensure bit0, bit1 masking !!                   */
    /**************************************************************************/
m33_cst_forwarding_test2_part9:
    MOV     R11,R13             /* Save SP register */
    STR     R1,[R2]             /* Prepared data to be loaded */
    DSB
    /*------------------------------------------------------------------------*/
    CST_PREPARE_PIPELINE               
    /*------------------------------------------------------------------------*/
    LDR     R13,[R2]            /* Slot0 - DPU produces result to the W2 port */
    NOP                         /* Slot1 */
    /* Interlock here to avoid Hazard !! */
    ADD     R3,R13,R1          /* Slot0 - Forward data to B0 read port !! */
    ADD     R4,R13,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP                         /* Slot0 */
    LDR     R13,[R2]            /* Slot1 - DPU produces result to the W2 port */
    /* Interlock here to avoid Hazard !! */
    SUB     R5,R13,R1           /* Slot0 - Forward data to B0 read port !! */
    SUB     R6,R13,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    ADD     R13,R1
    NOP
    /* Interlock here to avoid Hazard !! */
    ADD     R7,R13,R1           /* Slot0 - Forward data to B0 read port !! */
    ADD     R8,R13,R1           /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    ISB     /* Flush pipeline */
    /*------------------------------------------------------------------------*/
    NOP
    ADD     R13,R1
    /* Interlock here to avoid Hazard !! */
    SUB     R9,R13,R1           /* Slot0 - Forward data to B0 read port !! */
    SUB     R10,R13,R1          /* Slot1 - Forward data to B1 read port !! */
    /*------------------------------------------------------------------------*/
    MOV     R13,R11
    MSR     PRIMASK,R12
    /*------------------------------------------------------------------------*/
    /* Continue -  Update signature and return !! */
    
    
    /**************************************************************************/
    /* !! Update Signature !!                                                 */
    /**************************************************************************/
m33_cst_forwarding_test2_sum_r3_r10:
    SBC     R3,R3,R0,ROR #1
    EOR     R0,R0,R3
    SBC     R4,R4,R0,ROR #2
    EOR     R0,R0,R4
    SBC     R5,R5,R0,ROR #3
    EOR     R0,R0,R5
    SBC     R6,R6,R0,ROR #4
    EOR     R0,R0,R6
m33_cst_forwarding_test2_sum_r7_r10:
    SBC     R7,R7,R0,ROR #5
    EOR     R0,R0,R7
    SBC     R8,R8,R0,ROR #6
    EOR     R0,R0,R8
    SBC     R9,R9,R0,ROR #7
    EOR     R0,R0,R9
    SBC     R10,R10,R0,ROR #8
    EOR     R0,R0,R10
    
    BX      LR
    
    
    CST_ALIGN_BYTES_4
    /* Marks the current location for dumping psuedoinstruction pools containing
       numeric values for used symbolic names used within LDR instruction. */
    CST_LTORG

#endif /* CORTST_M33_FORWARD_ENABLE */
    
    CST_FILE_END

