/**
* @file    ClkTst_Lld.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : ClkTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/


#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations List
 * PRQA S 4342 Rule 10.5: An expression of 'essentially enum' type is being converted to unsigned type.
 */

/*==================================================================================================
 *                                          INCLUDE FILES
==================================================================================================*/
#include "ClkTst_Lld.h"
#include "pSIP_Scu.h"
/*==================================================================================================
 *                                  SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CLKTST_LLD_VENDOR_ID_C                      (180)
#define CLKTST_LLD_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CLKTST_LLD_AR_RELEASE_MINOR_VERSION_C       (4)
#define CLKTST_LLD_AR_RELEASE_REVISION_VERSION_C    (0)
#define CLKTST_LLD_SW_MAJOR_VERSION_C               (2)
#define CLKTST_LLD_SW_MINOR_VERSION_C               (0)
#define CLKTST_LLD_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
 *                                        FILE VERSION CHECKS
==================================================================================================*/
#if (CLKTST_LLD_VENDOR_ID_C != CLKTST_LLD_VENDOR_ID)
#error "ClkTst_Lld.c and ClkTst_Lld.h have different vendor ids"
#endif

#if ((CLKTST_LLD_AR_RELEASE_MAJOR_VERSION_C != CLKTST_LLD_AR_RELEASE_MAJOR_VERSION) || \
     (CLKTST_LLD_AR_RELEASE_MINOR_VERSION_C != CLKTST_LLD_AR_RELEASE_MINOR_VERSION) || \
     (CLKTST_LLD_AR_RELEASE_REVISION_VERSION_C != CLKTST_LLD_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of ClkTst_Lld.c and ClkTst_Lld.h are different"
#endif

#if ((CLKTST_LLD_SW_MAJOR_VERSION_C != CLKTST_LLD_SW_MAJOR_VERSION) || \
     (CLKTST_LLD_SW_MINOR_VERSION_C != CLKTST_LLD_SW_MINOR_VERSION) || \
     (CLKTST_LLD_SW_PATCH_VERSION_C != CLKTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of ClkTst_Lld.c and ClkTst_Lld.h are different"
#endif

/*==================================================================================================
 *                                         GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                          LOCAL VARIABLES
==================================================================================================*/
#ifndef MAC
#define CLKTST_START_SEC_VAR_INIT_BOOLEAN
#include "ClkTst_MemMap.h"
/**
 * Use a global static array to record which clocks were enabled when injecting errors,
 * and then close the corresponding clocks when clearing the injections.
 */
#if defined(CPU_YTM32B1MC0)
CLKTST_VAR static boolean ClkTst_CmuClkStatus[CLKTST_CMU_CHANNEL_NUMS] = {FALSE, FALSE};
#else
CLKTST_VAR static boolean ClkTst_CmuClkStatus[CLKTST_CMU_CHANNEL_NUMS] = {TRUE, FALSE, FALSE, FALSE};
#endif
#define CLKTST_STOP_SEC_VAR_INIT_BOOLEAN
#include "ClkTst_MemMap.h"

#define CLKTST_START_SEC_CONST_UNSPECIFIED
#include "ClkTst_MemMap.h"
CLKTST_CONST static volatile SCU_Type *const Scu_Base = SCU_BASE_PTRS;
#define CLKTST_STOP_SEC_CONST_UNSPECIFIED
#include "ClkTst_MemMap.h"

#define CLKTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "ClkTst_MemMap.h"
CLKTST_VAR static ClkTst_IqrCallBackType ClkTst_ErrorCallBackFunction;
#define CLKTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "ClkTst_MemMap.h"
#endif

#define CLKTST_START_SEC_CONST_32
#include "ClkTst_MemMap.h"
#if defined(CPU_YTM32B1MC0)
CLKTST_CONST static const uint32 ClkTst_Fre[2] = {FIRC_FREQUENCY, FXOSC_FREQUENCY};
#else
CLKTST_CONST static const uint32 ClkTst_Fre[4] = {SLOWBUS_FREQUENCY, FIRC_FREQUENCY, PLL_FREQUENCY, FXOSC_FREQUENCY};
#endif
#define CLKTST_STOP_SEC_CONST_32
#include "ClkTst_MemMap.h"
/*==================================================================================================
 *                                          LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                           LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
 *                                    LOCAL FUNCTION DECLARATIONS
==================================================================================================*/
#define CLKTST_START_SEC_CODE
#include "ClkTst_MemMap.h"
/*==================================================================================================
 *                                          LOCAL FUNCTIONS
==================================================================================================*/
/**
 * @brief           This function is used to clear the FXOSC_CTRL register's EN flag for FXOSC.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          None
 *
 * @trace           YTSDS_ClkTst_775
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_ClearFXOSC(void)
{
    Scu_Base->FXOSC_CTRL &= ~SCU_FXOSC_CTRL_FXOSC_EN_MASK;
}

#if (STD_ON == CLKTST_SCU_SUPPORT_PLL)
/**
 * @brief           This function is used to clear the PLL_CTRL register's EN flag for PLL.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          None
 *
 * @trace           YTSDS_ClkTst_779
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_ClearPLL(void)
{
    Scu_Base->PLL_CTRL &= ~SCU_PLL_CTRL_PLL_EN_MASK;
}
#endif

/**
 * @brief           This function is used to set the FIRC_CTRL register's EN flag for FIRC.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          void
 *
 * @trace           YTSDS_ClkTst_796
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_SetFIRC(void)
{
    Scu_Base->FIRC_CTRL |= SCU_FIRC_CTRL_FIRC_EN_MASK;
}

/**
 * @brief           This function is used to set the FXOSC_CTRL register to enable FXOSC for ClkTst.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          void
 *
 * @trace           YTSDS_ClkTst_797
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_SetFXOSC(void)
{
#if defined (CPU_YTM32B1MC0)
    Scu_Base->FXOSC_CTRL |= SCU_FXOSC_CTRL_GMSEL(5U) |
                            SCU_FXOSC_CTRL_FXOSC_EN(1U) |
                            SCU_FXOSC_CTRL_FXOSC_MODE(1U);
#else
    Scu_Base->FXOSC_CTRL |= SCU_FXOSC_CTRL_GMSEL(5U) |
                            SCU_FXOSC_CTRL_FXOSC_EN(1U) |
                            SCU_FXOSC_CTRL_FXOSC_MODE(1U) |
                            SCU_FXOSC_CTRL_COMPEN(1U);
#endif
}

#if (STD_ON == CLKTST_SCU_SUPPORT_PLL)
/**
 * @brief           This function is used to set the FXOSC_CTRL register to enable FXOSC for ClkTst.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          void
 *
 * @trace           YTSDS_ClkTst_798
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_SetPLL(void)
{
    Scu_Base->PLL_CTRL |= SCU_PLL_CTRL_FBDIV(9U) |
                          SCU_PLL_CTRL_REFDIV(0U) |
                          SCU_PLL_CTRL_REFCLKSRCSEL(0U) |
                          SCU_PLL_CTRL_PLL_DS_EN(0U) |
                          SCU_PLL_CTRL_PLL_EN(1U);
}
#endif

/**
 * @brief           Enable the FIRC clock.
 * @details         Check if the FIRC is valid by using the SCU.STS register.
 *                  If the FIRC is valid, the function will return directly.
 *                  If the FIRC is invalid, this function will enable the FIRC using a stable configuration,
 *                  and the clock will be disabled later.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          Std_ReturnType
 * @retval          E_OK: Enable the FIRC successful
 * @retval          E_NOT_OK: Enable the FIRC failed
 *
 * @trace           YTSDS_ClkTst_787
 */
CLKTST_FUNC LOCAL_INLINE Std_ReturnType ClkTst_Lld_EnableFIRC(void)
{
    Std_ReturnType Res = E_OK;
    /* Check the clock state */
    volatile uint32 RegSTS = Scu_Base->STS;
    if (0U == (SCU_STS_FIRC_VALID_MASK & RegSTS))
    {
        /* Enable the FIRC clock */
        ClkTst_Lld_SetFIRC();
        /* Wait for FIRC to be ready */
        uint32 WaitTime = 0U;
        while (WaitTime < CLKTST_WAIT_TIMEOUT)
        {
            if (0U != (SCU_STS_FIRC_VALID_MASK & RegSTS))
            {
                break;
            }
            RegSTS = Scu_Base->STS;
            ++WaitTime;
        }
        /* If exit `while` by timeout */
        if (WaitTime >= CLKTST_WAIT_TIMEOUT)
        {
            Res = E_NOT_OK;
        }
        if (E_OK == Res)
        {
            ClkTst_CmuClkStatus[CLKTST_CLK_ID_FIRC] = TRUE;
        }
    }
    return Res;
}

/**
 * @brief           Enable the FXOSC clock.
 * @details         Check if the FXOSC is valid by using the SCU.STS register.
 *                  If the FXOSC is valid, the function will return directly.
 *                  If the FXOSC is invalid, this function will enable the FXOSC using a stable configuration,
 *                  and the clock will be disabled later.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          Std_ReturnType
 * @retval          E_OK: Enable the FXOSC successful
 * @retval          E_NOT_OK: Enable the FXOSC failed
 *
 * @trace           YTSDS_ClkTst_788
 */
CLKTST_FUNC LOCAL_INLINE Std_ReturnType ClkTst_Lld_EnableFXOSC(void)
{
    Std_ReturnType Res = E_OK;
    /* Check the clock state */
    volatile uint32 RegSTS = Scu_Base->STS;
    if (0U == (SCU_STS_FXOSC_VALID_MASK & RegSTS))
    {
        /* Enable the FXOSC clock */
        ClkTst_Lld_SetFXOSC();
        /* Wait for FXOSC to be ready */
        uint32 WaitTime = 0U;
        while (WaitTime < CLKTST_WAIT_TIMEOUT)
        {
            if (0U != (SCU_STS_FXOSC_VALID_MASK & RegSTS))
            {
                break;
            }
            RegSTS = Scu_Base->STS;
            ++WaitTime;
        }
        /* If exit `while` by timeout */
        if (WaitTime >= CLKTST_WAIT_TIMEOUT)
        {
            Res = E_NOT_OK;
        }
        if (E_OK == Res)
        {
            ClkTst_CmuClkStatus[CLKTST_CLK_ID_FXOSC] = TRUE;
        }
    }
    return Res;
}

#if (STD_ON == CLKTST_SCU_SUPPORT_PLL)
/**
 * @brief           Enable the PLL clock.
 * @details         Check if the PLL is valid by using the SCU.STS register.
 *                  If the PLL is valid, the function will return directly.
 *                  If the PLL is invalid, this function will enable the PLL using a stable configuration,
 *                  and the clock will be disabled later.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          Std_ReturnType
 * @retval          E_OK: Enable the PLL successful
 * @retval          E_NOT_OK: Enable the PLL failed
 *
 * @trace           YTSDS_ClkTst_789
 */
CLKTST_FUNC LOCAL_INLINE Std_ReturnType ClkTst_Lld_EnablePLL(void)
{
    Std_ReturnType Res = E_OK;
    /* Check the clock state */
    volatile uint32 RegSTS = Scu_Base->STS;
    if (0U == (SCU_STS_PLL_LOCK_MASK & RegSTS))
    {
        /* Enable the PLL reference clock */
        Res = ClkTst_Lld_EnableFXOSC();
        if (E_OK == Res)
        {
            /* Enable the PLL clock, use the FXOSC as PLL reference clock. */
            ClkTst_Lld_SetPLL();
            /* Wait for PLL to be ready */
            uint32 WaitTime = 0U;
            while (WaitTime < CLKTST_WAIT_TIMEOUT)
            {
                if (0U != (SCU_STS_PLL_LOCK_MASK & RegSTS))
                {
                    break;
                }
                RegSTS = Scu_Base->STS;
                ++WaitTime;
            }
            /* If exit `while` by timeout */
            if (WaitTime >= CLKTST_WAIT_TIMEOUT)
            {
                Res = E_NOT_OK;
            }
        }
        if (E_OK == Res)
        {
            ClkTst_CmuClkStatus[CLKTST_CLK_ID_PLL] = TRUE;
        }
    }
    return Res;
}
#endif

/**
 * @brief           Disable the FIRC clock.
 * @details         Check if the FIRC is valid by using the SCU.STS register.
 *                  If the FIRC is valid, the function will disable the FIRC clock.
 *                  If the FIRC is invalid, this function will return directly.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_783
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_DisableFIRC(void)
{
    if (TRUE == ClkTst_CmuClkStatus[CLKTST_CLK_ID_FIRC])
    {
        /* Check the clock state */
        volatile uint32 RegSTS = Scu_Base->STS;
        if (0U != (SCU_STS_FIRC_VALID_MASK & RegSTS))
        {
            /* Disable the clock */
            Scu_Base->FIRC_CTRL &= ~SCU_FIRC_CTRL_FIRC_EN_MASK;
        }
    }
    ClkTst_CmuClkStatus[CLKTST_CLK_ID_FIRC] = FALSE;
}

/**
 * @brief           Disable the FXOSC clock.
 * @details         Check if the FXOSC is valid by using the SCU.STS register.
 *                  If the FXOSC is valid, the function will disable the FXOSC clock.
 *                  If the FXOSC is invalid, this function will return directly.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_784
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_DisableFXOSC(void)
{
    if (TRUE == ClkTst_CmuClkStatus[CLKTST_CLK_ID_FXOSC])
    {
        /* Check the clock state */
        volatile uint32 RegSTS = Scu_Base->STS;
        if (0U != (SCU_STS_FXOSC_VALID_MASK & RegSTS))
        {
            /* Disable the clock */
            /* Note: Direct clear the FXOSC_CTRL all to zero will make error */
            ClkTst_Lld_ClearFXOSC();
            uint32 WaitTime = 0;
            while (WaitTime < CLKTST_WAIT_TIMEOUT)
            {
                if (0U == (SCU_STS_FXOSC_VALID_MASK & RegSTS))
                {
                    break;
                }
                RegSTS = Scu_Base->STS;
                ++WaitTime;
            }
        }
    }
    ClkTst_CmuClkStatus[CLKTST_CLK_ID_FXOSC] = FALSE;
}

#if (STD_ON == CLKTST_SCU_SUPPORT_PLL)
/**
 * @brief           Disable the PLL clock.
 * @details         Check if the PLL is valid by using the SCU.STS register.
 *                  If the PLL is valid, the function will disable the PLL clock.
 *                  If the PLL is invalid, this function will return directly.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_785
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_DisablePLL(void)
{
    if (TRUE == ClkTst_CmuClkStatus[CLKTST_CLK_ID_PLL])
    {
        /* Check the clock state */
        volatile uint32 RegSTS = Scu_Base->STS;
        if (0U != (SCU_STS_PLL_LOCK_MASK & RegSTS))
        {
            /* Disable the clock */
            ClkTst_Lld_ClearPLL();
            uint32 WaitTime = 0;
            while (WaitTime < CLKTST_WAIT_TIMEOUT)
            {
                if (0U == (SCU_STS_PLL_LOCK_MASK & RegSTS))
                {
                    break;
                }
                RegSTS = Scu_Base->STS;
                ++WaitTime;
            }
            ClkTst_Lld_DisableFXOSC();
        }
    }
    ClkTst_CmuClkStatus[CLKTST_CLK_ID_PLL] = FALSE;
}
#endif

/**
 * @brief           Enable the specified CMU channel.
 * @param           Channel CMU channel name
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          Std_ReturnType
 * @retval          E_OK: Enable the CMU channel successful
 * @retval          E_NOT_OK: Enable the CMU channel failed
 *
 * @trace           YTSDS_ClkTst_786
 */
CLKTST_FUNC LOCAL_INLINE Std_ReturnType ClkTst_Lld_EnableCmuClk(ClkTst_ClockID Channel)
{
    Std_ReturnType Res = E_OK;
    switch (Channel)
    {
#if !defined(CPU_YTM32B1MC0)
        case CLKTST_CLK_ID_SLOWBUS:
            break;
#endif
        case CLKTST_CLK_ID_FIRC:
            Res = ClkTst_Lld_EnableFIRC();
            break;
#if (STD_ON == CLKTST_SCU_SUPPORT_PLL)
        case CLKTST_CLK_ID_PLL:
            Res = ClkTst_Lld_EnablePLL();
            break;
#endif
        case CLKTST_CLK_ID_FXOSC:
            Res = ClkTst_Lld_EnableFXOSC();
            break;
        default:
            Res = E_NOT_OK;
            break;
    }
    return Res;
}

/**
 * @brief           Disable the specified CMU channel.
 * @param           Channel CMU channel name
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_782
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_DisableCmuClk(ClkTst_ClockID Channel)
{
    switch (Channel)
    {
        case CLKTST_CLK_ID_FIRC:
            ClkTst_Lld_DisableFIRC();
            break;
#if (STD_ON == CLKTST_SCU_SUPPORT_PLL)
        case CLKTST_CLK_ID_PLL:
            ClkTst_Lld_DisablePLL();
            break;
#endif
        case CLKTST_CLK_ID_FXOSC:
            ClkTst_Lld_DisableFXOSC();
            break;
        default:
            /* SlowBus clock is enable default */
            break;
    }
}

/**
 * @brief           Clears the raw value of the CMU status register.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_774
 */
CLKTST_FUNC LOCAL_INLINE void ClkTst_Lld_ClearCmuRawVal(void)
{
    Scu_Base->CMUSTS = CLKTST_CMU_RAW_CLEAR_VALUE;
}
/*==================================================================================================
                                         GLOBAL FUNCTIONS
==================================================================================================*/

/**
 * @brief           Inject the OUTRNG_CMPH error to the specified CMU channel.
 * @param           Channel CMU channel name
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_958
 */
CLKTST_FUNC void ClkTst_Lld_InjectOutRngLow(uint8 Channel)
{
    /* Config the CMU Channel register */
    Scu_Base->CMUCMP[Channel].LOW = (ClkTst_Fre[Channel] / SIRC_FREQUENCY) * CLKTST_HIGH_MULTIPLIER;
}

/**
 * @brief           Inject the OUTRNG_CMPL error to the specified CMU channel.
 * @param           Channel CMU channel name
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_957
 */
CLKTST_FUNC void ClkTst_Lld_InjectOutRngHigh(uint8 Channel)
{
    /* Config the CMU Channel register */
    Scu_Base->CMUCMP[Channel].HIGH = (ClkTst_Fre[Channel] / SIRC_FREQUENCY) * CLKTST_LOW_MULTIPLIER;
}

/**
 * @brief           Clear the OUTRNG error injection.
 * @details         Disable all CMU channels and clocks enabled during OUTRNG injection.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_954
 */
CLKTST_FUNC void ClkTst_Lld_ClearOutRng(uint8 Channel)
{
    /* Config the CMU Channel register */
    Scu_Base->CMUCMP[Channel].HIGH = CLKTST_CMU_HIGH_DEFAULT;
    Scu_Base->CMUCMP[Channel].LOW = CLKTST_CMU_LOW_DEFAULT;
}

/**
 * @brief           Enable the specified CMU channel.
 * @param           Channel CMU channel name
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          Std_ReturnType
 * @retval          E_OK: Enable the CMU channel successful
 * @retval          E_NOT_OK: Enable the CMU channel failed
 *
 * @trace           YTSDS_ClkTst_956
 */
CLKTST_FUNC Std_ReturnType ClkTst_Lld_EnableCmu(uint8 Channel)
{
    Std_ReturnType Res = E_OK;
    /* Config the CMU Channel register */
    /*
    *MR12 Rule 10.5 VIOLATION: Here, only uint32 can be used because the clocks have different ids from CPU to CPU, so M3CM Rule-10.5 cannot be followed
    */
    Res = ClkTst_Lld_EnableCmuClk((ClkTst_ClockID) Channel); /*PRQA S 4342*/
    Scu_Base->CMU_CTRL |= (uint32)1U << Channel;
    return Res;
}

/**
 * @brief           Clears OUTRNG error injection.
 * @details         Disables all CMU channels and clocks enabled during OUTRNG injection.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_778
 */
CLKTST_FUNC void ClkTst_Lld_ClearInjectOutRng(uint8 Channel)
{
    /* Disable all CMU channel */
    Scu_Base->CMU_CTRL = 0U;
    ClkTst_Lld_DisableCmuClk((ClkTst_ClockID)Channel); /*PRQA S 4342*/
    Scu_Base->CMUCMP[Channel].HIGH = CLKTST_CMU_HIGH_DEFAULT;
    Scu_Base->CMUCMP[Channel].LOW = CLKTST_CMU_LOW_DEFAULT;
}

/**
 * @brief           Gets the raw value of the CMU status register.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return          ClkTst_CmuRawValue Raw value of the CMU status register.
 *
 * @trace           YTSDS_ClkTst_791
 */
CLKTST_FUNC ClkTst_CmuRawValue ClkTst_Lld_GetCmuRawVal(void)
{
    ClkTst_CmuRawValue RegSTS;
    RegSTS = Scu_Base->CMUSTS;
    return RegSTS;
}

/**
 * @brief           Install the error callback function.
 * @details
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace           YTSDS_ClkTst_795
 */
CLKTST_FUNC void ClkTst_Lld_InstallErrorCallBack(ClkTst_IqrCallBackType ErrorCallBack)
{
    ClkTst_ErrorCallBackFunction = ErrorCallBack;
}

/**
 * @brief           SCU interrupt handler for CMU errors.
 * @details         Handles CMU error interrupts, calls registered callback and clears status.
 * @table           @service_id:       --        \n
 *                  @is_reentrant:     false       \n
 *                  @is_synchronous:   true        \n
 *                  @autosar_api:      false       \n
 *                  @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @trace           YTSDS_ClkTst_796
 */
CLKTST_FUNC void ClkTst_SCUIRQHandler(void)
{
    ClkTst_ErrorCallBackFunction();
    ClkTst_Lld_ClearCmuRawVal();
}
#define CLKTST_STOP_SEC_CODE
#include "ClkTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file ClkTst_Lld.c */

