/**
* @file    ClkTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : ClkTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations List
 * PRQA S 4342 Rule 10.5: An expression of 'essentially enum' type is being converted to unsigned type.
 */

/*==================================================================================================
                                          INCLUDE FILES
==================================================================================================*/
#include "ClkTst.h"
#include "ClkTst_Lld.h"
#include "Platform.h"
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif
#ifdef CLKTST_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif
/*==================================================================================================
 *                                  SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CLKTST_VENDOR_ID_C                      (180)
#define CLKTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CLKTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define CLKTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define CLKTST_SW_MAJOR_VERSION_C               (2)
#define CLKTST_SW_MINOR_VERSION_C               (0)
#define CLKTST_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
 *                                        FILE VERSION CHECKS
==================================================================================================*/
#if (CLKTST_VENDOR_ID_C != CLKTST_VENDOR_ID)
#error "ClkTst.c and ClkTst.h have different vendor ids"
#endif

#if (( CLKTST_AR_RELEASE_MAJOR_VERSION_C != CLKTST_AR_RELEASE_MAJOR_VERSION) || \
     ( CLKTST_AR_RELEASE_MINOR_VERSION_C != CLKTST_AR_RELEASE_MINOR_VERSION) || \
     ( CLKTST_AR_RELEASE_REVISION_VERSION_C != CLKTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of ClkTst.c and ClkTst.h are different"
#endif

#if (( CLKTST_SW_MAJOR_VERSION_C != CLKTST_SW_MAJOR_VERSION) || \
     ( CLKTST_SW_MINOR_VERSION_C != CLKTST_SW_MINOR_VERSION) || \
     ( CLKTST_SW_PATCH_VERSION_C != CLKTST_SW_PATCH_VERSION))
#error "Software Version Numbers of ClkTst.c and ClkTst.h are different"
#endif

#if (CLKTST_VENDOR_ID_C != CLKTST_LLD_VENDOR_ID)
#error "ClkTst.c and ClkTst_Lld.h have different vendor ids"
#endif

#if (( CLKTST_AR_RELEASE_MAJOR_VERSION_C != CLKTST_LLD_AR_RELEASE_MAJOR_VERSION) || \
     ( CLKTST_AR_RELEASE_MINOR_VERSION_C != CLKTST_LLD_AR_RELEASE_MINOR_VERSION) || \
     ( CLKTST_AR_RELEASE_REVISION_VERSION_C != CLKTST_LLD_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of ClkTst_Lld.c and ClkTst.c are different"
#endif

#if (( CLKTST_SW_MAJOR_VERSION_C != CLKTST_LLD_SW_MAJOR_VERSION) || \
     ( CLKTST_SW_MINOR_VERSION_C != CLKTST_LLD_SW_MINOR_VERSION) || \
     ( CLKTST_SW_PATCH_VERSION_C != CLKTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of ClkTst.c and ClkTst_Lld.h are different"
#endif

/*==================================================================================================
 *                                         GLOBAL VARIABLES
==================================================================================================*/
#define CLKTST_START_SEC_CONFIG_DATA_UNSPECIFIED
#include "ClkTst_MemMap.h"
CLKTST_CONST extern const ClkTst_ConfigType ClkTst_Config;
#define CLKTST_STOP_SEC_CONFIG_DATA_UNSPECIFIED
#include "ClkTst_MemMap.h"
/*==================================================================================================
 *                                          LOCAL VARIABLES
==================================================================================================*/
#define CLKTST_START_SEC_VAR_INIT_32
#include "ClkTst_MemMap.h"
/* This ptr point to the ClkTst_Config */
CLKTST_VAR static const ClkTst_ConfigType *ClkTst_ConfigPtr = NULL_PTR;
CLKTST_VAR static volatile ClkTst_FlagType ClkTst_CmuFlag = CLKTST_NOT_TEST;
#define CLKTST_STOP_SEC_VAR_INIT_32
#include "ClkTst_MemMap.h"

#define CLKTST_START_SEC_VAR_INIT_8
#include "ClkTst_MemMap.h"
CLKTST_VAR static uint8 ChannelPtr = 0;
#define CLKTST_STOP_SEC_VAR_INIT_8
#include "ClkTst_MemMap.h"

#define CLKTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "ClkTst_MemMap.h"
CLKTST_VAR static ClkTst_StatusType ClkTst_Status;
#define CLKTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "ClkTst_MemMap.h"

#define CLKTST_START_SEC_VAR_INIT_BOOLEAN
#include "ClkTst_MemMap.h"
/*Trigger the error interrupt flag */
CLKTST_VAR static volatile boolean ClkTst_ErrorFlag = FALSE;
#define CLKTST_STOP_SEC_VAR_INIT_BOOLEAN
#include "ClkTst_MemMap.h"
/*==================================================================================================
 *                                          LOCAL FUNCTIONS
==================================================================================================*/
#define CLKTST_START_SEC_CODE
#include "ClkTst_MemMap.h"

/**
 * @brief            Handles CMU error detection callback.
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_952
 */
CLKTST_FUNC static void ClkTst_ErrorCallBack(void)
{
    ClkTst_CmuRawValue CmuRawVal = ClkTst_Lld_GetCmuRawVal();
    /*External influence causes error join interruption and test failure*/
    if (ClkTst_ErrorFlag != TRUE)
    {
        ClkTst_CmuFlag = CLKTST_TEST_ERROR;
    }
    else if ((CmuRawVal & (1UL << ChannelPtr)) != 0U)
    {
        ClkTst_CmuFlag = CLKTST_TEST_OK;
    }
    else
    {
        /*Do Nothing*/
    }
}

/**
 * @brief            Initializes channel result tracking structures.
 * @details          Sets initial values for channel results and overall result.
 *
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_718
 */
CLKTST_FUNC static void ClkTst_InitChannelResult(void)
{
    ClkTst_Status.ClkTstResult = CLKTST_RESULT_NOT_TESTED;
    for (uint32 Idx = 0U; Idx < CLKTST_CMU_CHANNEL_NUMS; ++Idx)
    {
        /*
         *MR12 Rule 10.5 VIOLATION: Here, only uint32 can be used because the clocks have different ids from CPU to CPU, so M3CM Rule-10.5 cannot be followed
         */
        ClkTst_Status.ClkTstChannelResult[Idx].Channel = (ClkTst_ClockID)Idx; /*PRQA S 4342*/
        ClkTst_Status.ClkTstChannelResult[Idx].CheckState = CLKTST_ITEM_NOT_TESTED;
    }
}

/**
 * @brief            Start the CMU channel Test.
 * @details          Enable the CMU channel and wait for the detection result.
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_948
 */
CLKTST_FUNC static void ClkTst_CmuEnableTest(uint8 Channel)
{
    uint32 Timeout = 0U;
    Std_ReturnType Res = E_OK;
#ifdef CLKTST_ENABLE_USER_MODE_SUPPORT
    Res = (Std_ReturnType)OsIf_Trusted_Call_Return1param(ClkTst_Lld_EnableCmu, Channel);
#else
    Res = ClkTst_Lld_EnableCmu(Channel);
#endif
    if (Res != E_OK)
    {
        ClkTst_Status.ClkTstResult = CLKTST_RESULT_NOT_OK;
        ClkTst_Status.ClkTstChannelResult[Channel].CheckState = CLKTST_ITEM_FAILED;
        if (ClkTst_ConfigPtr->ClkTstFailedNotification != NULL_PTR)
        {
            ClkTst_ConfigPtr->ClkTstFailedNotification(CLKTST_MODULE_ID);
        }
        ClkTst_CmuFlag = CLKTST_NOT_TEST;
    }
    else
    {
        /* Wait for the CMU channel to be enabled,if the error interrupt is not triggered, the CMU starts normally and the test passes*/
        do
        {
            MCAL_QUA_FAULT_INJECTION(CLKTST_LLD_SCU_UNLOCK_TIMEOUT);
            if (ClkTst_CmuFlag == CLKTST_TEST_ERROR)
            {
                ClkTst_Status.ClkTstResult = CLKTST_RESULT_NOT_OK;
                ClkTst_Status.ClkTstChannelResult[Channel].CheckState = CLKTST_ITEM_FAILED;
                if (ClkTst_ConfigPtr->ClkTstFailedNotification != NULL_PTR)
                {
                    ClkTst_ConfigPtr->ClkTstFailedNotification(CLKTST_MODULE_ID);
                }
                ClkTst_CmuFlag = CLKTST_NOT_TEST;
                break;
            }
            ++Timeout;
        }
        while (Timeout < TIMOUT);
    }
}

/**
 * @brief            Inject the OUTRNG low level signal to the CMU channel.
 * @details          Inject the OUTRNG low level signal to the CMU channel and wait for the detection result.
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_950
 */
CLKTST_FUNC static void ClkTst_CmuOutRngLowTest(uint8 Channel)
{
    uint32 Timeout = 0U;
    if (ClkTst_Status.ClkTstChannelResult[Channel].CheckState != CLKTST_ITEM_FAILED)
    {
        ClkTst_ErrorFlag = TRUE;
#ifdef CLKTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(ClkTst_Lld_InjectOutRngLow, Channel);
#else
        ClkTst_Lld_InjectOutRngLow(Channel);
#endif
        /*Wait for the CMU response*/
        do
        {
            MCAL_QUA_FAULT_INJECTION(CLKTST_LLD_SCU_UNLOCK_TIMEOUT);
            if (Timeout >= TIMOUT)
            {
                break;
            }
            ++Timeout;
        }
        while (ClkTst_CmuFlag == CLKTST_NOT_TEST);
        if (ClkTst_CmuFlag == CLKTST_TEST_OK)
        {
            ClkTst_CmuFlag = CLKTST_NOT_TEST;
            ClkTst_ErrorFlag = FALSE;
        }
        else
        {
            ClkTst_Status.ClkTstResult = CLKTST_RESULT_NOT_OK;
            ClkTst_Status.ClkTstChannelResult[Channel].CheckState = CLKTST_ITEM_FAILED;
            if (ClkTst_ConfigPtr->ClkTstFailedNotification != NULL_PTR)
            {
                ClkTst_ConfigPtr->ClkTstFailedNotification(CLKTST_MODULE_ID);
            }
            else
            {
                /*Do Nothing*/
            }
            ClkTst_CmuFlag = CLKTST_NOT_TEST;
        }
#ifdef CLKTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(ClkTst_Lld_ClearOutRng, Channel);
#else
        ClkTst_Lld_ClearOutRng(Channel);
#endif
    }
}

/**
 * @brief            Inject the OUTRNG high level signal to the CMU channel.
 * @details          Inject the OUTRNG high level signal to the CMU channel and wait for the detection result.
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_949
 */
CLKTST_FUNC static void ClkTst_CmuOutRngHighTest(uint8 Channel)
{
    uint32 Timeout = 0U;
    if (ClkTst_Status.ClkTstChannelResult[Channel].CheckState != CLKTST_ITEM_FAILED)
    {
        ClkTst_ErrorFlag = TRUE;
#ifdef CLKTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(ClkTst_Lld_InjectOutRngHigh, Channel);
#else
        ClkTst_Lld_InjectOutRngHigh(Channel);
#endif
        /*Wait for the CMU response*/
        do
        {
            MCAL_QUA_FAULT_INJECTION(CLKTST_LLD_SCU_UNLOCK_TIMEOUT);
            if (Timeout >= TIMOUT)
            {
                break;
            }
            ++Timeout;
        }
        while (ClkTst_CmuFlag == CLKTST_NOT_TEST);
        if (ClkTst_CmuFlag == CLKTST_TEST_OK)
        {
            ClkTst_CmuFlag = CLKTST_NOT_TEST;
            ClkTst_ErrorFlag = FALSE;
        }
        else
        {
            ClkTst_Status.ClkTstResult = CLKTST_RESULT_NOT_OK;
            ClkTst_Status.ClkTstChannelResult[Channel].CheckState = CLKTST_ITEM_FAILED;
            if (ClkTst_ConfigPtr->ClkTstFailedNotification != NULL_PTR)
            {
                ClkTst_ConfigPtr->ClkTstFailedNotification(CLKTST_MODULE_ID);
            }
            ClkTst_CmuFlag = CLKTST_NOT_TEST;
        }
#ifdef CLKTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(ClkTst_Lld_ClearOutRng, Channel);
#else
        ClkTst_Lld_ClearOutRng(Channel);
#endif
    }
}

/**
 * @brief            Start the CMU channel Test.
 * @details          Enable the CMU channel and wait for the detection result.
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_951
 */
CLKTST_FUNC static void ClkTst_CmuTest(uint8 Channel)
{
    ClkTst_CmuFlag = CLKTST_NOT_TEST;
    ClkTst_CmuEnableTest(Channel);
    ClkTst_CmuOutRngHighTest(Channel);
    ClkTst_CmuOutRngLowTest(Channel);
    if (ClkTst_Status.ClkTstChannelResult[Channel].CheckState != CLKTST_ITEM_FAILED)
    {
        ClkTst_Status.ClkTstChannelResult[Channel].CheckState = CLKTST_ITEM_PASS;
    }
#ifdef CLKTST_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(ClkTst_Lld_ClearInjectOutRng, Channel);
#else
    ClkTst_Lld_ClearInjectOutRng(Channel);
#endif
}

/*==================================================================================================
 *                                         GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief            Initializes the Clock Test module and prepares internal state.
 * @details          Verifies module state validity, initializes channel result tracking structures,
 *
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        ConfigPtr          Pointer to configuration data structure
 *
 * @trace YTSDS_ClkTst_717
 */
CLKTST_FUNC void ClkTst_Init(const ClkTst_ConfigType *ConfigPtr)
{
    /* Check the `ClkTst_Status.ClkTstState` whether equal to CLKTST_UNINIT */
    if (CLKTST_UNINIT != ClkTst_Status.ClkTstState)
    {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
        /* The Init function needs to be called when the ClkTst_Status.ClkTstState is CLKTST_UNINIT */
        (void)Det_ReportError(CLKTST_MODULE_ID, CLKTST_INSTANCE_ID,
                              CLKTST_SID_INIT, CLKTST_E_INVALID_STATE);
#endif
    }
    else
    {
#if (CLKTST_PRECOMPILE_SUPPORT == STD_ON)
        ClkTst_ConfigPtr = &ClkTst_Config;
        ClkTst_InitChannelResult();
        ClkTst_Status.ClkTstState = CLKTST_INIT;
        if (NULL_PTR != ConfigPtr)
        {
            /* code */
        }
#else
        if (NULL_PTR == ConfigPtr)
        {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CLKTST_MODULE_ID, CLKTST_INSTANCE_ID,
                                  CLKTST_SID_START, CLKTST_E_PARAM_POINTER);
#endif
            /* The ConfigPtr should not be NULL_PTR */
        }
        else
        {
            ClkTst_ConfigPtr = ConfigPtr;
            ClkTst_InitChannelResult();
            ClkTst_Status.ClkTstState = CLKTST_INIT;
        }
#endif
    }
}

/**
 * @brief            Deinitializes the Clock Test module.
 * @details          Clears configuration pointer, resets state and results, and deinitializes hardware.
 *
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_714
 */
CLKTST_FUNC void ClkTst_DeInit(void)
{
    /* Check the `ClkTst_Status.ClkTstState` whether equal to CLKTST_INIT*/
    if ((CLKTST_INIT != ClkTst_Status.ClkTstState) && (CLKTST_TESTED != ClkTst_Status.ClkTstState))
    {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CLKTST_MODULE_ID, CLKTST_INSTANCE_ID,
                              CLKTST_SID_DEINIT, CLKTST_E_UNINIT);
#endif
        /* The Deinit function needs to be called after the Init function. */
    }
    else
    {
        ClkTst_ConfigPtr = NULL_PTR;
        ClkTst_Status.ClkTstState = CLKTST_UNINIT;
        /* Clear all the test result */
        ClkTst_Status.ClkTstResult = CLKTST_RESULT_NOT_TESTED;
        for (uint32 Idx = 0U; Idx < CLKTST_CMU_CHANNEL_NUMS; ++Idx)
        {
            ClkTst_Status.ClkTstChannelResult[Idx].CheckState = CLKTST_ITEM_NOT_TESTED;
        }
    }
}

/**
 * @brief            Starts the Clock Test.
 * @details          Sets injected errors, waits for detection, clears injected errors, and sets result.
 *
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     true      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 *
 * @trace YTSDS_ClkTst_724
 */
CLKTST_FUNC void ClkTst_Start(void)
{
    Platform_IrqHandlerType OldHandler;
    /* Check the ClkTst_Status.ClkTstState whether equal to CLKTST_INIT */
    if (CLKTST_INIT != ClkTst_Status.ClkTstState)
    {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CLKTST_MODULE_ID, CLKTST_INSTANCE_ID,
                              CLKTST_SID_START, CLKTST_E_UNINIT);
#endif
    }
    else
    {
        ClkTst_Status.ClkTstState = CLKTST_TESTING;
        (void)Platform_InstallIrqHandler(SCU_IRQn, ClkTst_SCUIRQHandler, &OldHandler);
        (void)Platform_SetIrq(SCU_IRQn, TRUE);
        ClkTst_Lld_InstallErrorCallBack(ClkTst_ErrorCallBack);
        for (uint8 Idx = 0; Idx < CLKTST_CMU_CHANNEL_NUMS; ++Idx)
        {
            if (TRUE == ClkTst_ConfigPtr->ClkTstCMUConfig[Idx])
            {
                ChannelPtr = Idx;
                ClkTst_CmuTest(ChannelPtr);
            }
        }
        if (ClkTst_Status.ClkTstResult != CLKTST_RESULT_NOT_OK)
        {
            ClkTst_Status.ClkTstResult = CLKTST_RESULT_OK;
        }
        (void)Platform_SetIrq(SCU_IRQn, FALSE);
        (void)Platform_InstallIrqHandler(SCU_IRQn, OldHandler, &OldHandler);
        ClkTst_Status.ClkTstState = CLKTST_TESTED;
    }
}

/**
 * @brief            Gets the overall result of the Clock Test.
 * @details          Returns the overall result of the Clock Test.
 *
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     true      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           ClkTst_ResultType  Overall result of the Clock Test
 *
 * @trace YTSDS_ClkTst_716
 */
CLKTST_FUNC ClkTst_ResultType ClkTst_GetResult(void)
{
    /* Check the `ClkTst_Status.ClkTstState` whether equal to CLKTST_INIT*/
    if (CLKTST_UNINIT == ClkTst_Status.ClkTstState)
    {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
        /* The GetResult function needs to be called after the Init function. */
        (void)Det_ReportError(CLKTST_MODULE_ID, CLKTST_INSTANCE_ID,
                              CLKTST_SID_GET_RESULT, CLKTST_E_UNINIT);
#endif
    }
    return ClkTst_Status.ClkTstResult;
}

/**
 * @brief            Gets detailed results of the Clock Test.
 * @details          Copies detailed results of each channel to the provided pointer.
 *
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[out]       ResPtr             Pointer to store detailed results
 *
 * @trace YTSDS_ClkTst_715
 */
CLKTST_FUNC void ClkTst_GetDetailsResult(ClkTst_ChannelResultType *ResPtr)
{
    if (CLKTST_UNINIT == ClkTst_Status.ClkTstState)
    {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
        /* The GetResult function needs to be called after the Init function. */
        (void)Det_ReportError(CLKTST_MODULE_ID, CLKTST_INSTANCE_ID,
                              CLKTST_SID_GET_DETAILS_RESULT, CLKTST_E_UNINIT);
#endif
    }
    else if (NULL_PTR == ResPtr)
    {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CLKTST_MODULE_ID, CLKTST_INSTANCE_ID,
                              CLKTST_SID_GET_DETAILS_RESULT, CLKTST_E_PARAM_POINTER);
#endif
    }
    else
    {
        for (uint32 Idx = 0U; Idx < CLKTST_CMU_CHANNEL_NUMS; ++Idx)
        {
            /*
            *MR12 Rule 10.5 VIOLATION: Here, only uint32 can be used because the clocks have different ids from CPU to CPU, so M3CM Rule-10.5 cannot be followed
            */
            ResPtr[Idx].Channel = (ClkTst_ClockID)Idx; /*PRQA S 4342*/
            ResPtr[Idx].CheckState = ClkTst_Status.ClkTstChannelResult[Idx].CheckState;
        }
    }
}

/**
 * @brief       Service returns the version information of this module.
 * @details     This function retrieves the version information of the ClkTst module and stores it in the
 *              provided 'VersionInfo' structure. If the 'VersionInfo' pointer is NULL, it reports an error
 *              using the Det module (if enabled). If the pointer is valid, it populates the structure with
 *              the module's vendor ID, module ID, and software version information (major, minor, and patch versions).
 *              The version information includes: Module Id, Vendor Id, Vendor specific version number.\n
 *              The behavior of this function depends on the following compile-time switches:\n
 *              CLKTST_DEV_ERROR_DETECT: If enabled, errors are reported using the Det module.
 * @param[out]  VersionInfo Pointer to a structure of type 'Std_VersionInfoType' where the
 *                         version information will be stored. This parameter must not be NULL.
 * @table       @service_id:       0x06        \n
 *              @is_reentrant:     true       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @return      void
 * @sw_type     sw_detail
 *
 * @trace YTDT_ClkTst_187
 */
CLKTST_FUNC void ClkTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    /* Validate the pointer to the version info */
    if (NULL_PTR == VersionInfo)
    {
#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)CLKTST_MODULE_ID,
            (uint8)CLKTST_INSTANCE_ID,
            (uint8)CLKTST_SID_GET_VERSION_INFO,
            (uint8)CLKTST_E_PARAM_POINTER
        );
#endif
    }
    else
    {
        VersionInfo->vendorID = (uint16)CLKTST_VENDOR_ID;
        VersionInfo->moduleID = (uint16)CLKTST_MODULE_ID;
        VersionInfo->sw_major_version = (uint8)CLKTST_SW_MAJOR_VERSION;
        VersionInfo->sw_minor_version = (uint8)CLKTST_SW_MINOR_VERSION;
        VersionInfo->sw_patch_version = (uint8)CLKTST_SW_PATCH_VERSION;
    }
}
#define CLKTST_STOP_SEC_CODE
#include "ClkTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file ClkTst.c */

