/*
* @file    ClkTst_Types.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : ClkTst_Types
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef CLKTST_TYPES_H
#define CLKTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "ClkTst_Cfg.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define CLKTST_TYPES_VENDOR_ID                    (180)
#define CLKTST_TYPES_AR_REL_MAJOR_VER              (4)
#define CLKTST_TYPES_AR_REL_MINOR_VER              (4)
#define CLKTST_TYPES_AR_REL_REVISION_VER           (0)
#define CLKTST_TYPES_SW_MAJOR_VER                  (2)
#define CLKTST_TYPES_SW_MINOR_VER                  (0)
#define CLKTST_TYPES_SW_PATCH_VER                  (0)
/*==================================================================================================
 *                                        FILE VERSION CHECKS
==================================================================================================*/
#if (CLKTST_TYPES_VENDOR_ID != CLKTST_VENDOR_ID_CFG)
#error "ClkTst_Types.h and ClkTst_Cfg.h have different vendor ids"
#endif

#if ((CLKTST_TYPES_AR_REL_MAJOR_VER != CLKTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (CLKTST_TYPES_AR_REL_MINOR_VER != CLKTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (CLKTST_TYPES_AR_REL_REVISION_VER != CLKTST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of ClkTst_Types.h and ClkTst_Cfg.h are different"
#endif

#if ((CLKTST_TYPES_SW_MAJOR_VER != CLKTST_SW_MAJOR_VERSION_CFG) || \
     (CLKTST_TYPES_SW_MINOR_VER != CLKTST_SW_MINOR_VERSION_CFG) || \
     (CLKTST_TYPES_SW_PATCH_VER != CLKTST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of ClkTst_Types.h and ClkTst_Cfg.h are different"
#endif

/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                ENUMS
==================================================================================================*/
/**
 * @brief            This enumeration defines the possible state of ClkTst module.
 * @sw_type          sw_arch
 */
typedef enum
{
    CLKTST_UNINIT = 0U,                         /**< ClkTst is not initialized */
    CLKTST_INIT = 1U,                           /**< ClkTst is initialized */
    CLKTST_TESTING = 2U,                        /**< ClkTst is testing */
    CLKTST_TESTED = 3U                          /**< ClkTst is tested */
} ClkTst_StateType;

/**
 * @brief            This enumeration defines the possible test results for the CLKTST module.
 * @sw_type          sw_arch
 */
typedef enum
{
    CLKTST_RESULT_NOT_TESTED    = 0x00U,        /**< ClkTst not executed */
    CLKTST_RESULT_OK            = 0x01U,        /**< ClkTst passed */
    CLKTST_RESULT_NOT_OK        = 0x02U         /**< ClkTst failed */
} ClkTst_ResultType;

/**
 * @brief            This enumeration defines the possible test results for each item.
 * @sw_type          sw_detail
 */
typedef enum
{
    CLKTST_ITEM_NOT_TESTED    = 0x00U,          /**< Items that do not need testing */
    CLKTST_ITEM_PASS          = 0x01U,          /**< Items test passed */
    CLKTST_ITEM_FAILED        = 0x02U           /**< Items test failed */
} ClkTst_ItemResultType;

/**
 * @brief            This enumeration defines the name of the monitored clock in CMU.
 * @sw_type          sw_detail
 */
typedef enum
{
#if defined(CPU_YTM32B1MC0)
    CLKTST_CLK_ID_FIRC     = 0x00U,             /**< FIRC */
    CLKTST_CLK_ID_FXOSC    = 0x01U              /**< FXOSC */
#else
    CLKTST_CLK_ID_SLOWBUS  = 0x00U,             /**< Slow bus */
    CLKTST_CLK_ID_FIRC     = 0x01U,             /**< FIRC */
    CLKTST_CLK_ID_PLL      = 0x02U,             /**< PLL */
    CLKTST_CLK_ID_FXOSC    = 0x03U              /**< FXOSC */
#endif
} ClkTst_ClockID;

/**
 * @brief            This enumeration defines the test flag type.
 * @sw_type          sw_arch
 */
typedef enum
{
    CLKTST_NOT_TEST       = 0x00U,              /**< Not test */
    CLKTST_TEST_ERROR     = 0x01U,              /**< Test Error */
    CLKTST_TEST_OK        = 0x02U               /**< Test OK */
} ClkTst_FlagType;

/*==================================================================================================
 *                                              TYPEDEF
==================================================================================================*/
/**
 * @brief            This type is used to store the result of the ClkTst module.
 * @sw_type          sw_arch
 */
typedef void (*ClkTst_FailedNotificationType)(uint16 ModuleId);

/**
 * @brief            This type is used to store the result of the ClkTst module.
 * @sw_type          sw_detail
 */
typedef void (*ClkTst_IqrCallBackType)(void);

/**
 * @brief            Define the CMU_STS register raw value type.
 * @sw_type          sw_detail
 */
typedef uint32 ClkTst_CmuRawValue;

/**
 * @brief            This structure is used to store the configuration of the ClkTst module.
 * @sw_type          sw_arch
 */
typedef struct
{
    const boolean *ClkTstCMUConfig;                  /**< CMU channel configuration */
    ClkTst_FailedNotificationType ClkTstFailedNotification; /**< Failed notification function */
} ClkTst_ConfigType;

/**
 * @brief            This structure is used to store one CMU channel test result.
 * @sw_type          sw_arch
 */
typedef struct
{
    ClkTst_ClockID Channel;                     /**< Channel name */
    ClkTst_ItemResultType CheckState;           /**< Test item result */
} ClkTst_ChannelResultType;

/**
 * @brief            This structure is used to store the status of the ClkTst module.
 * @sw_type          sw_arch
 */
typedef struct
{
    ClkTst_StateType ClkTstState;              /**< ClkTst state */
    ClkTst_ResultType ClkTstResult;            /**< ClkTst result */
    ClkTst_ChannelResultType ClkTstChannelResult[CLKTST_CMU_CHANNEL_NUMS]; /**< ClkTst channel result */
} ClkTst_StatusType;

#ifdef __cplusplus
}
#endif

#endif /* End of file ClkTst_Types.h */
