/**
* @file    ClkTst_Lld.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : ClkTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef CLKTST_LLD_H
#define CLKTST_LLD_H
/**
 * @file ClkTst_Lld.h
*/
#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                          INCLUDE FILES
==================================================================================================*/
#include "ClkTst_Types.h"

/*==================================================================================================
 *                                  HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define CLKTST_LLD_VENDOR_ID                      (180)
#define CLKTST_LLD_AR_RELEASE_MAJOR_VERSION       (4)
#define CLKTST_LLD_AR_RELEASE_MINOR_VERSION       (4)
#define CLKTST_LLD_AR_RELEASE_REVISION_VERSION    (0)
#define CLKTST_LLD_SW_MAJOR_VERSION               (2)
#define CLKTST_LLD_SW_MINOR_VERSION               (0)
#define CLKTST_LLD_SW_PATCH_VERSION               (0)
/*==================================================================================================
 *                                        FILE VERSION CHECKS
==================================================================================================*/
#if (CLKTST_LLD_VENDOR_ID != CLKTST_TYPES_VENDOR_ID)
#error "ClkTst_Lld.h and ClkTst_Types.h have different vendor ids"
#endif

#if ((CLKTST_LLD_AR_RELEASE_MAJOR_VERSION != CLKTST_TYPES_AR_REL_MAJOR_VER) || \
     (CLKTST_LLD_AR_RELEASE_MINOR_VERSION != CLKTST_TYPES_AR_REL_MINOR_VER) || \
     (CLKTST_LLD_AR_RELEASE_REVISION_VERSION != CLKTST_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of ClkTst_Lld.h and ClkTst_Types.h are different"
#endif

#if ((CLKTST_LLD_SW_MAJOR_VERSION != CLKTST_TYPES_SW_MAJOR_VER) || \
     (CLKTST_LLD_SW_MINOR_VERSION != CLKTST_TYPES_SW_MINOR_VER) || \
     (CLKTST_LLD_SW_PATCH_VERSION != CLKTST_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of ClkTst_Lld.h and ClkTst_Types.h are different"
#endif
/*==================================================================================================
 *                                        DEFINES AND MACROS
==================================================================================================*/
#define CLKTST_CMU_HIGH_DEFAULT        (0xFFFFU)  /* Default CMU high threshold value */
#define CLKTST_CMU_LOW_DEFAULT         (0x0000U)  /* Default CMU low threshold value */
#define CLKTST_CMU_RAW_CLEAR_VALUE     (0xFFU)    /* Default value to clear CMU raw value */
#define CLKTST_HIGH_MULTIPLIER         (280U)     /* Multiplier for high threshold */
#define CLKTST_LOW_MULTIPLIER          (118U)     /* Multiplier for low threshold */
#define CLKTST_WAIT_TIMEOUT            (100000U)  /* Timeout duration for waiting */
/*==================================================================================================
 *                                         GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief Enable the specified CMU channel.
 * @param Channel CMU channel name
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
*  @sw_type         sw_detail
 *
 * @trace YTSDS_ClkTst_956
 */
Std_ReturnType ClkTst_Lld_EnableCmu(uint8 Channel);

/**
 * @brief Inject the OUTRNG_CMPH error to the specified CMU channel.
 * @param Channel CMU channel name
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace YTSDS_ClkTst_958
 */
void ClkTst_Lld_InjectOutRngLow(uint8 Channel);

/**
 * @brief Inject the OUTRNG_CMPL error to the specified CMU channel.
 * @param Channel CMU channel name
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace YTSDS_ClkTst_957
 */
void ClkTst_Lld_InjectOutRngHigh(uint8 Channel);

/**
 * @brief Clear the OUTRNG error injection.
 * @details Disable all CMU channels and clocks enabled during OUTRNG injection.
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace YTSDS_ClkTst_954
 */
void ClkTst_Lld_ClearOutRng(uint8 Channel);

/**
 * @brief Clears OUTRNG error injection.
 * @details Disables all CMU channels and clocks enabled during OUTRNG injection.
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 *
 * @trace YTDT_ClkTst_778
 */
void ClkTst_Lld_ClearInjectOutRng(uint8 Channel);


/**
 * @brief Gets the raw value of the CMU status register.
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return ClkTst_CmuRawValue Raw value of the CMU status register.
 *
 * @trace YTDT_ClkTst_791
 */
ClkTst_CmuRawValue ClkTst_Lld_GetCmuRawVal(void);


/**
 * @brief SCU interrupt handler for CMU errors.
 * @details Handles CMU error interrupts, calls registered callback and clears status.
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @trace YTDT_ClkTst_796
 */
void ClkTst_SCUIRQHandler(void);

/**
 * @brief Registers a callback function for CMU error handling.
 * @details Allows user to set a custom callback for CMU error handling.
 * @table           @service_id:       --        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false       \n
                    @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @param ErrorCallBack Pointer to the callback function.
 *
 * @trace YTDT_ClkTst_797
 */
void ClkTst_Lld_InstallErrorCallBack(ClkTst_IqrCallBackType ErrorCallBack);
#ifdef __cplusplus
}
#endif

#endif /* End of file ClkTst_Lld.h */

