/*
* @file    ClkTst.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : ClkTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef CLKTST_H
#define CLKTST_H
/**
 * @file ClkTst.h
*/
#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                          INCLUDE FILES
==================================================================================================*/
#include "ClkTst_Types.h"
/*==================================================================================================
 *                                  HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define CLKTST_VENDOR_ID                      (180)
#define CLKTST_AR_RELEASE_MAJOR_VERSION       (4)
#define CLKTST_AR_RELEASE_MINOR_VERSION       (4)
#define CLKTST_AR_RELEASE_REVISION_VERSION    (0)
#define CLKTST_SW_MAJOR_VERSION               (2)
#define CLKTST_SW_MINOR_VERSION               (0)
#define CLKTST_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                        FILE VERSION CHECKS
==================================================================================================*/
/**
 * @file     ClkTst.h
 */
#if (CLKTST_VENDOR_ID != CLKTST_TYPES_VENDOR_ID)
#error "ClkTst.h and ClkTst_Types.h have different vendor ids"
#endif

#if ((CLKTST_AR_RELEASE_MAJOR_VERSION != CLKTST_TYPES_AR_REL_MAJOR_VER) || \
     (CLKTST_AR_RELEASE_MINOR_VERSION != CLKTST_TYPES_AR_REL_MINOR_VER) || \
     (CLKTST_AR_RELEASE_REVISION_VERSION != CLKTST_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of ClkTst.h and ClkTst_Types.h are different"
#endif

#if ((CLKTST_SW_MAJOR_VERSION != CLKTST_TYPES_SW_MAJOR_VER) || \
     (CLKTST_SW_MINOR_VERSION != CLKTST_TYPES_SW_MINOR_VER) || \
     (CLKTST_SW_PATCH_VERSION != CLKTST_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of ClkTst.h and ClkTst_Types.h are different"
#endif
/*==================================================================================================
 *                                        DEFINES AND MACROS
==================================================================================================*/
/** @brief CLKTST MODULE INSTANCE ID */
#define CLKTST_INSTANCE_ID    ((uint8)0U)
/** @brief CLKTST MODULE ID */
#define CLKTST_MODULE_ID      ((uint16)514U)

#if (CLKTST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup        ClkTst_DetErrorGroup
 * @brief           ClkTst development errors
 * @sw_type         sw_detail
 * @{
 */
/** @brief Invalid state error */
#define CLKTST_E_INVALID_STATE                 ((uint8)0x10)
/** @brief NULL_PTR Error */
#define CLKTST_E_PARAM_POINTER                 ((uint8)0x11)
/** @brief Enable clock failed error */
#define CLKTST_E_ENABLE_CLK_FAILED             ((uint8)0x12)
/** @brief API called without module initialization */
#define CLKTST_E_UNINIT                        ((uint8)0x13)
/** @brief Initialization failed error */
#define CLKTST_E_INIT_FAILED                   ((uint8)0x14)
/** @} */
#endif

/**
 * @defgroup        ClkTst_ApiGroup
 * @brief           ClkTst service ID
 * @sw_type         sw_detail
 * @{
 */
/** @brief Service ID for ClkTst_Init */
#define CLKTST_SID_INIT                        (0x01U)
/** @brief Service ID for ClkTst_DeInit */
#define CLKTST_SID_DEINIT                      (0x02U)
/** @brief Service ID for ClkTst_Start */
#define CLKTST_SID_START                       (0x03U)
/** @brief Service ID for ClkTst_GetResult */
#define CLKTST_SID_GET_RESULT                  (0x04U)
/** @brief Service ID for ClkTst_GetDetailsResult */
#define CLKTST_SID_GET_DETAILS_RESULT          (0x05U)
/** @brief Service ID for ClkTst_GetVersionInfo */
#define CLKTST_SID_GET_VERSION_INFO            (0x06U)
/** @} */
#define TIMOUT      ((uint32)128*(FIRC_FREQUENCY/SIRC_FREQUENCY))
/*==================================================================================================
 *                                             VARIATES
==================================================================================================*/
/*==================================================================================================
 *                                        EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                         GLOBAL FUNCTIONS
==================================================================================================*/
/**
* @brief            Initializes the Clock Test module and prepares internal state.
* @details          Verifies module state validity, initializes channel result tracking structures,
*
* @table            @service_id:       0x01        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false        \n
                    @memory_map:       .mcal_text  \n
* @sw_type          sw_arch
* @param[in]        ConfigPtr         Pointer to configuration data structure
*/
void ClkTst_Init(const ClkTst_ConfigType *ConfigPtr);

/**
* @brief            Deinitializes the Clock Test module and resets internal state.
* @details          Clears configuration pointers, resets channel result tracking structures,
*
* @table            @service_id:       0x02        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false        \n
                    @memory_map:       .mcal_text  \n
* @sw_type          sw_arch
*/
void ClkTst_DeInit(void);

/**
* @brief            Starts the Clock Test sequence.
* @details          Injects errors, waits for detection, and clears injected errors.
*
* @table            @service_id:       0x03        \n
                    @is_reentrant:     true       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false        \n
                    @memory_map:       .mcal_text  \n
* @sw_type          sw_arch
*/
void ClkTst_Start(void);

/**
* @brief            Returns the result of the Clock Test.
* @details          Provides the overall result of the Clock Test sequence.
*
* @table            @service_id:       0x04        \n
                    @is_reentrant:     true       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false        \n
                    @memory_map:       .mcal_text  \n
* @sw_type          sw_arch
* @return           ClkTst_ResultType
*/
ClkTst_ResultType ClkTst_GetResult(void);

/**
* @brief            Provides detailed results of the Clock Test.
* @details          Fills the provided structure with detailed results for each channel.
*
* @table            @service_id:       0x05        \n
                    @is_reentrant:     false       \n
                    @is_synchronous:   true        \n
                    @autosar_api:      false        \n
                    @memory_map:       .mcal_text  \n
* @sw_type          sw_arch
* @param[out]       ResPtr            Pointer to the structure to store detailed results
*/
void ClkTst_GetDetailsResult(ClkTst_ChannelResultType *ResPtr);

#if (CLKTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @details
 * @table           @service_id:       0x06        \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 *
 * @sw_type          sw_arch
 * @param[out]       versioninfo Pointer to where to store version information of this module
 * @return           void
 *
 */
void ClkTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif /* CLKTST_VERSION_INFO_API == STD_ON */

#ifdef __cplusplus
}
#endif

#endif /* End of file ClkTst.h */

