/**
* @file    SchM_RamTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : SchM_RamTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

/**
*   @file    SchM_RamTst.c
*
*   @addtogroup SCHM_RAMTST_MODULE
*   @{
*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 3432 Rule-20.7: Expressions resulting from the expansion of macro parameters shall be enclosed in parentheses.
 * PRQA S 3335 Rule-1.1-C99: The program shall contain no violations of the standard C syntax and constraints, and shall
 *                           not exceed the implementation's translation limits Suppression Options.
 * PRQA S 3415 Rule-13.5: Right hand operand of '&&' or '||' is an expression with persistent side effects.
 * PRQA S 4434 Rule-10.3: The value of an expression shall not be assigned to an object with a narrower
 *                        essential type or of a different essential type category.
 *
 */

/* MR12 Rule-13.5 VIOLATION: Right hand operand of '&&' or '||' is an expression with persistent side effects.
 * The variables msr_RAMTST_EXCLUSIVE_AREA and reentry_guard_RAMTST_EXCLUSIVE_AREA are volatile but it's write
 * read operations are full tested and verified with no side effects in such usages in the whole file.
 * Thus, couldn't adhere to M3CM Rule-13.5
 */
/* PRQA S 3415 EOF */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "RamTst_Cfg.h"
#include "SchM_RamTst.h"
#include "OsIf.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define SCHM_RAMTST_VENDOR_ID_C                      (180)
#define SCHM_RAMTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define SCHM_RAMTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define SCHM_RAMTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define SCHM_RAMTST_SW_MAJOR_VERSION_C               (2)
#define SCHM_RAMTST_SW_MINOR_VERSION_C               (0)
#define SCHM_RAMTST_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and SCHM_RAMTST header file are of the same vendor */
#if (SCHM_RAMTST_VENDOR_ID_C != SCHM_RAMTST_VENDOR_ID)
#error "SchM_RamTst.c and SchM_RamTst.h have different vendor ids"
#endif

/* Check if source file and SCHM_RAMTST header file are of the same Autosar version */
#if (( SCHM_RAMTST_AR_RELEASE_MAJOR_VERSION_C != SCHM_RAMTST_AR_RELEASE_MAJOR_VERSION) || \
      ( SCHM_RAMTST_AR_RELEASE_MINOR_VERSION_C != SCHM_RAMTST_AR_RELEASE_MINOR_VERSION) || \
      ( SCHM_RAMTST_AR_RELEASE_REVISION_VERSION_C != SCHM_RAMTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of SchM_RamTst.c and SchM_RamTst.h are different"
#endif

/* Check if source file and SCHM_RAMTST header file are of the same Software version */
#if (( SCHM_RAMTST_SW_MAJOR_VERSION_C != SCHM_RAMTST_SW_MAJOR_VERSION) || \
      ( SCHM_RAMTST_SW_MINOR_VERSION_C != SCHM_RAMTST_SW_MINOR_VERSION) || \
      ( SCHM_RAMTST_SW_PATCH_VERSION_C != SCHM_RAMTST_SW_PATCH_VERSION))
#error "Software Version Numbers of SchM_RamTst.c and SchM_RamTst.h are different"
#endif

/* Check if source file and RAMTST Configure header file are of the same vendor */
#if (SCHM_RAMTST_VENDOR_ID_C != RAMTST_VENDOR_ID_CFG)
#error "SchM_RamTst.c and RamTst_Cfg.h have different vendor ids"
#endif

/* Check if source file and RAMTST Configure header file are of the same Autosar version */
#if (( SCHM_RAMTST_AR_RELEASE_MAJOR_VERSION_C != RAMTST_AR_REL_MAJOR_VER_CFG) || \
      ( SCHM_RAMTST_AR_RELEASE_MINOR_VERSION_C != RAMTST_AR_REL_MINOR_VER_CFG) || \
      ( SCHM_RAMTST_AR_RELEASE_REVISION_VERSION_C != RAMTST_AR_REL_REVISION_VER_CFG))
#error "AutoSar Version Numbers of SchM_RamTst.c and RamTst_Cfg.h are different"
#endif

/* Check if source file and RAMTST Configure header file are of the same Software version */
#if (( SCHM_RAMTST_SW_MAJOR_VERSION_C != RAMTST_SW_MAJOR_VER_CFG) || \
      ( SCHM_RAMTST_SW_MINOR_VERSION_C != RAMTST_SW_MINOR_VER_CFG) || \
      ( SCHM_RAMTST_SW_PATCH_VERSION_C != RAMTST_SW_PATCH_VER_CFG))
#error "Software Version Numbers of SchM_RamTst.c and RamTst_Cfg.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
*                                       LOCAL CONSTANTS
==================================================================================================*/
#ifdef MCAL_PLATFORM_ARM
#if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
#define ISR_STATE_MASK     ((uint32)0x000000C0UL)   /**< @brief DAIF bit I and F */
#elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
#define ISR_STATE_MASK     ((uint32)0x00000080UL)   /**< @brief CPSR bit I */
#else
#if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
#define ISR_STATE_MASK     ((uint32)0x000000FFUL)   /**< @brief BASEPRI[7:0] mask */
#else
#define ISR_STATE_MASK     ((uint32)0x00000001UL)   /**< @brief PRIMASK bit 0 */
#endif
#endif
#endif
/*==================================================================================================
*                                       LOCAL MACROS
==================================================================================================*/
#ifdef MCAL_PLATFORM_ARM
#if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
#define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) != (uint32)(ISR_STATE_MASK))
#elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
#define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) != (uint32)(ISR_STATE_MASK))
#else
#define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) == (uint32)0)
#endif
#endif

#if(STD_ON == RAMTST_SCHM_SUPPORT)
/*==================================================================================================
*                                       LOCAL VARIABLES
==================================================================================================*/

#ifdef MCAL_SCHM_LITE

#define RTE_START_SEC_VAR_CLEARED_8_NO_CACHEABLE
#include "Rte_MemMap.h"

RTE_VAR static volatile uint8 SchM_RamTst_Msr;
RTE_VAR static volatile uint8 SchM_RamTst_ReentryGuard;

#define RTE_STOP_SEC_VAR_CLEARED_8_NO_CACHEABLE
#include "Rte_MemMap.h"

#else

#define RTE_START_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "Rte_MemMap.h"

RTE_VAR static volatile uint32 SchM_RamTst_Msr_00;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_00;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_01;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_01;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_02;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_02;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_03;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_03;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_04;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_04;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_05;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_05;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_06;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_06;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_07;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_07;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_08;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_08;
RTE_VAR static volatile uint32 SchM_RamTst_Msr_09;
RTE_VAR static volatile uint32 SchM_RamTst_ReentryGuard_09;

#define RTE_STOP_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "Rte_MemMap.h"

#endif /* MCAL_SCHM_LITE */

/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
#define RTE_START_SEC_CODE
#include "Rte_MemMap.h"

/**
* @brief   This function returns the MSR register value (32 bits).
* @details This function returns the MSR register value (32 bits).
*
* @param[in]     void        No input parameters
* @return        uint32 msr  This function returns the MSR register value (32 bits).
*
* @pre  None
* @post None
*
*/
RTE_FUNC static uint32 SchM_RamTst_ReadMsr(void)
{
#if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
    /** MR12 RULE-20.7 VIOLATION: Expressions resulting from the expansion of macro parameters shall be enclosed in parentheses.
    * In this specific case, the OsIf_Trusted_Call1param() is used to embedded call the LLD driver with
    * CPU mode changed to privileged mode for user mode support, such usage is full-verified and works
    * as expected. thus, couldn't adhere to M3CM Rule-20.7
    *
    * MR12 RULE-10.4 VIOLATION: No function declaration of the __get_BASEPRI(). Implicit declaration inserted: 'extern int %1s();'.
    * In this specific case, the __get_BASEPRI() is used to read the BASEPRI register value. The function is referred from the
    * CMSIS library with static_inline attribute. The CMSIS library is a standard library for ARM Cortex-M processors.
    *
    * MR12 Rule-10.3 VIOLATION: The value of an expression shall not be assigned to an object with a narrower
    * essential type or of a different essential type category.
    * Here the __get_BASEPRI() function returns a uint32 value when define, which is casted to a uint32 type.So, No risk of data loss
    *
    */
    return (uint32)OsIf_Trusted_Call_Return(__get_BASEPRI);  /* PRQA S 3335, 3432, 4434 */
#else
    return __get_PRIMASK();
#endif
}

#define RTE_STOP_SEC_CODE
#include "Rte_MemMap.h"

/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/

#define RTE_START_SEC_CODE
#include "Rte_MemMap.h"

#ifdef MCAL_SCHM_LITE

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA(void)
{
    if (0UL == SchM_RamTst_ReentryGuard)
    {
        SchM_RamTst_Msr = (uint8)SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA(void)
{
    SchM_RamTst_ReentryGuard -= 1U;
    if (ISR_ON(SchM_RamTst_Msr) && (0UL == SchM_RamTst_ReentryGuard))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

#else

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_00(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_00)
    {
        SchM_RamTst_Msr_00 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_00)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_00 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_00(void)
{
    SchM_RamTst_ReentryGuard_00 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_00) && (0UL == SchM_RamTst_ReentryGuard_00))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_01(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_01)
    {
        SchM_RamTst_Msr_01 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_01)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_01 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_01(void)
{
    SchM_RamTst_ReentryGuard_01 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_01) && (0UL == SchM_RamTst_ReentryGuard_01))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_02(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_02)
    {
        SchM_RamTst_Msr_02 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_02)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_02 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_02(void)
{
    SchM_RamTst_ReentryGuard_02 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_02) && (0UL == SchM_RamTst_ReentryGuard_02))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_03(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_03)
    {
        SchM_RamTst_Msr_03 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_03)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_03 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_03(void)
{
    SchM_RamTst_ReentryGuard_03 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_03) && (0UL == SchM_RamTst_ReentryGuard_03))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_04(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_04)
    {
        SchM_RamTst_Msr_04 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_04)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_04 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_04(void)
{
    SchM_RamTst_ReentryGuard_04 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_04) && (0UL == SchM_RamTst_ReentryGuard_04))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_05(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_05)
    {
        SchM_RamTst_Msr_05 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_05)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_05 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_05(void)
{
    SchM_RamTst_ReentryGuard_05 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_05) && (0UL == SchM_RamTst_ReentryGuard_05))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_06(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_06)
    {
        SchM_RamTst_Msr_06 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_06)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_06 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_06(void)
{
    SchM_RamTst_ReentryGuard_06 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_06) && (0UL == SchM_RamTst_ReentryGuard_06))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_07(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_07)
    {
        SchM_RamTst_Msr_07 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_07)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_07 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_07(void)
{
    SchM_RamTst_ReentryGuard_07 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_07) && (0UL == SchM_RamTst_ReentryGuard_07))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_08(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_08)
    {
        SchM_RamTst_Msr_08 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_08)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_08 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_08(void)
{
    SchM_RamTst_ReentryGuard_08 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_08) && (0UL == SchM_RamTst_ReentryGuard_08))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

RTE_FUNC void SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_09(void)
{
    if (0UL == SchM_RamTst_ReentryGuard_09)
    {
        SchM_RamTst_Msr_09 = SchM_RamTst_ReadMsr();  /*read MSR (to store interrupts state)*/
        if (ISR_ON(SchM_RamTst_Msr_09)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
        }
    }
    SchM_RamTst_ReentryGuard_09 += 1U;
}

RTE_FUNC void SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_09(void)
{
    SchM_RamTst_ReentryGuard_09 -= 1U;
    if (ISR_ON(SchM_RamTst_Msr_09) && (0UL == SchM_RamTst_ReentryGuard_09))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
    }
}

#endif /* MCAL_SCHM_LITE */

#define RTE_STOP_SEC_CODE
#include "Rte_MemMap.h"

#endif /* STD_ON == RAMTST_SCHM_SUPPORT */

#ifdef __cplusplus
}
#endif

/** @} */

/* End of file SchM_RamTst.c */

