/**
****************************************************************************************
* @file    : bsp_config.c
* @author  : Samir
* @version : V1.2
* @date    : 2025-4-15
* @brief   : ļ
****************************************************************************************
* @attention
*
*
*
****************************************************************************************
*/
#include "./sys/bsp_sys.h"
#include "./config/bsp_config.h"
#include "flash_driver.h"
#include "./uart/bsp_uart.h"


static __IO uint32_t sul_tick;                      /* ϵͳʱӼ */
uint32_t gul_tick_prio = (1UL << __NVIC_PRIO_BITS); /* ̶ȼ */
x_tick_freq_t tick_freq = x_tick_freq_default;      /* 1KHz */

volatile uint8_t thumb_instr_size = 0;
volatile uint16_t instruction_opcode = 0;
volatile uint32_t stack_pointer_address = 0;
volatile uint32_t instruction_program_counter = 0;
volatile uint32_t *p_instruction_address = 0;

#define TEST_DATA_START_ADDR (0x02019000U)
#define TEST_DATA_END_ADDR   (0x0202FFFFU)
#define TEST_ECC_INJECT_ADDR (0x02019100U)

/**
 * @brief   ջַ
 * @param   addr:ջַ
 * @author  Samir
 * @version 1.0
 * @date    2024-09-18
 */
void sys_msr_msp(uint32_t addr)
{
    __set_MSP(addr); /* ջַ */
}

/**
 * @brief   ϵͳλ
 * @author  Samir
 * @version 1.0
 * @date    2024-09-18
 */
void sys_soft_reset(void)
{
    NVIC_SystemReset();
}

/**
 * @brief   ϵͳʼ ȼΪ4 Ĭʱ1ms
 * @return  0:ɹ, 0:ʧ
 * @author  Samir
 * @version 1.0
 * @date    2024-11-12
 */
x_status_t x_init(void)
{
    /* ȼ */
    NVIC_SetPriorityGrouping(NVIC_PRIORITYGROUP_4);

#ifndef USING_OS_FREERTOS
    /* ʹfreertosʱ ʹsystickΪʱԴ1ms̶ */
    x_init_tick(TICK_INT_PRIORITY);
#endif

    return x_ok;
}

/**
 * @brief   ô˺ȫֱsul_tickӦóʱ
 * @note    ĬʵУñÿ 1ms һ
 * @author  Samir
 * @version 1.0
 * @date    2024-11-12
 */
void x_inc_tick(void)
{
    sul_tick += tick_freq;
}

/**
 * @brief   ṩԺΪλĿ̶ֵ
 * @return  ̶ֵ
 * @author  Samir
 * @version 1.0
 * @date    2024-11-12
 */
uint32_t x_get_tick(void)
{
    return sul_tick;
}

/**
 * @brief úһ̶ȼ
 * @retval ̶ȼ
 */
uint32_t x_get_tick_prio(void)
{
    return gul_tick_prio;
}

/**
 * @brief ؿ̶Ƶʡ
 * @retval ̶Ƶʡ
 */
x_tick_freq_t x_get_tick_freq(void)
{
    return tick_freq;
}

/**
 * @brief   úṩСӳ٣ԺΪλ
 * @param   delay:ָӳʱ䳤ȣλΪ
 * @author  Samir
 * @version 1.0
 * @date    2024-11-13
 */
void x_delay(uint32_t delay)
{
    uint32_t tickstart = x_get_tick();
    uint32_t wait = delay;

    /* ƵԱ̵֤ȴ */
    if (wait < X_MAX_DELAY)
    {
        wait += (uint32_t)(tick_freq);
    }

    while ((x_get_tick() - tickstart) < wait)
    {
    }
}

/**
 * @brief   ʱԴ ĬSysTickʱʱԴ
 * @param   tick_priority:̶ȵжȼ
 * @return  0:ɹ, 0:ʧ
 * @author  Samir
 * @version 1.0
 * @date    2024-11-12
 */
x_status_t x_init_tick(uint32_t tick_priority)
{
    uint32_t prioritygroup = 0x00U;

    /*  SysTick  1ms ʱΪж*/
    if (x_systick_config(SystemCoreClock / (1000U / tick_freq)) > 0U)
    {
        return x_error;
    }

    /*  SysTick IRQ ȼ */
    if (tick_priority < (1UL << __NVIC_PRIO_BITS))
    {
        prioritygroup = NVIC_GetPriorityGrouping();
        NVIC_SetPriority(SysTick_IRQn, NVIC_EncodePriority(prioritygroup, tick_priority, 0U));
        gul_tick_prio = tick_priority;
    }
    else
    {
        return x_error;
    }

    return x_ok;
}

/**
 * @brief   ʼϵͳʱжϣϵͳĶʱ
 * @param   ticks_numb:ָж֮ĵδ
 * @return  0:ɹ, 0:ʧ
 * @author  Samir
 * @version 1.0
 * @date    2024-11-12
 */
uint32_t x_systick_config(uint32_t ticks_numb)
{
    return SysTick_Config(ticks_numb);
}

/**
 * @brief   ʼDWT(M7)DWTCortex-M3/M4/M7ںˣעⲻͬں˵ļĴַܲͬ
 * @author  Samir
 * @version 1.0
 * @date    2025-03-13
 */
void bsp_dwt_init(void)
{
    DEM_CR |= (unsigned int)DEM_CR_TRCENA;    /* ʹDWT */
    DWT_CYCCNT = (unsigned int)0u;            /* DWT CYCCNTĴ0 */
    DWT_CR |= (unsigned int)DWT_CR_CYCCNTENA; /* ʹCortex-M DWT CYCCNTĴ */
}

/**
 * @brief   ΪõײڴRTOSиõļ
 *          һʽӳٺڵײmsӳҪڳʼӰʵʱԡ
 * @param   delay_time:ʱʱ(ms)
 * @author  Samir
 * @version 1.0
 * @date    2025-03-14
 */
void bsp_delay_ms(uint32_t delay_time)
{
    bsp_delay_us(1000 * delay_time);
}

/**
 * @brief   ʱCPUڲDWTʵ֣32λ
 * @param   delay_time:ʱʱ(us)
 * @author  Samir
 * @version 1.0
 * @date    2025-4-15
 * @note    1. Ƶ200MHz£32λ2^32/200000000 = 21.475
 *              ʹñӳٵĻӳ1¡
 *           2. ʵͨ߼ 1Ghz ԣ΢ӳٺʵʵʶ1.5usҵʱ䡣
 *           ݲ
 *           1. MDK5.42Żȼ1, ͬMDKŻȼûӰ졣
 *           2. YTM32B1HA01
 *           3. Է
 *                  bsp_gpio_write_pin(GPIOC, 2, 1);
 *                  bsp_delay_us(1);
 *                  bsp_gpio_write_pin(GPIOC, 2, 0);
 *           -------------------------------------------
 *                               ʵִ
 *           bsp_delay_us(1)          2.591us
 *           bsp_delay_us(2)          3.678us
 *           bsp_delay_us(50)         51.717us
 *           bsp_delay_us(1000)       1001.478us
 *          3. 32λ޷ȡĽٸֵ32λ޷ȻȷĻȡֵ
 *            A,B,C32λ޷
 *            A > B  ôA - B = Cܺ⣬ȫû
 *            A < B  ôA - B = C Cֵ0xFFFFFFFF - B + A + 1һҪرע⣬ڱ
 */
void bsp_delay_us(uint32_t delay_time)
{
    uint32_t t_cnt, t_delay_cnt;
    uint32_t t_start;

    t_start = DWT_CYCCNT; /* սʱļֵ */
    t_cnt = 0;
    t_delay_cnt = delay_time * (SystemCoreClock / 1000000); /* ҪĽ */

    while (t_cnt < t_delay_cnt)
    {
        t_cnt = DWT_CYCCNT - t_start; /* Уһ32λ¼Ȼȷ */
    }
}

/**
 * @brief   ʹCPUڲDWTĽĽʱ
 * @param   delay_time:ʱʱ
 * @author  Samir
 * @version 1.0
 * @date    2025-03-14
 */
void bsp_delay_dwt(uint32_t delay_time)
{
    uint32_t t_cnt, t_delay_cnt;
    uint32_t t_start;

    t_cnt = 0;
    t_delay_cnt = delay_time; /* ҪĽ */
    t_start = DWT_CYCCNT;     /* սʱļֵ */

    while (t_cnt < t_delay_cnt)
    {
        t_cnt = DWT_CYCCNT - t_start; /* Уһ32λ¼Ȼȷ */
    }
}

/**
 * @brief   ȡǰʱ,ʹںDWTʵֵus
 * @return  ǰʱus
 * @author  Samir
 * @version 1.0
 * @date    2025-03-13
 */
uint32_t bsp_dwt_timestamp(void)
{        
  return ((uint32_t)DWT_CYCCNT / (SystemCoreClock / 1000000));
}

/**
 * @brief   EFM޷ָ
 * @author  Samir
 * @version 1.0
 * @date    2025-04-16
 * @note    ECC޷ԶbitԶ
 */
void efm_unrecovery_error_handler(void)
{
    uint32_t error_addr = EFM->ECC_ERR_ADDR;
    LOG_ERROR(15, "Flash ECC Un-recovery Error Handler.\r\n");
    LOG_ERROR(15, "Ecc Error Address(EFM->ECC_ERR_ADDR): 0x%x\r\n", error_addr);

    /* ECC־EFM-> ECC_ERR_ADDRͬʱ */
    EFM->STS = EFM_STS_UNRECOVERR_MASK | EFM_STS_CI_UNRECOVERR_MASK;

    LOG_ERROR(15, "EFM->STS 0x%x\r\n", EFM->STS);

    /* ɸӦóԴݽ磺Ĭֵֵ */
    /* ECCַ޸ */
    // if (error_addr > TEST_DATA_START_ADDR && error_addr < TEST_DATA_END_ADDR)
    // {
    //     FLASH_DRV_EraseSector(0,
    //                           error_addr & ~(FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE - 1),
    //                           FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE);
    //     LOG_ERROR(15, "Erased the sector containing the ECC error address.\r\n");
    // }
    // else
    // {
    //     /* ַ */
    // }
}

/**
 * @brief   EFMɻָ
 * @author  Samir
 * @version 1.0
 * @date    2025-04-16
 * @note    ECCɶԵbitԶ
 */
void efm_recovery_error_handler(void)
{
    uint32_t data[FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE / 4] = { 0 };
    uint32_t error_addr = EFM->ECC_ERR_ADDR;
    uint32_t error_sector_addr = error_addr & ~(FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE - 1);

    /* ע⣺жϳ򡣿ܻӳ١ */
    LOG_ERROR(15, "Flash ECC Recovery Error Handler.\r\n");
    LOG_ERROR(15, "Ecc Error Address(EFM->ECC_ERR_ADDR): 0x%x\r\n", error_addr);

    /* ECC־EFM-> ECC_ERR_ADDRͬʱ */
    EFM->STS = EFM_STS_RECOVERR_MASK | EFM_STS_CI_RECOVERR_MASK;

    /* ضӦó򷽰ȷǷ±̡
       Note±ڼ䣬豸ܶϵ硣ͻдֻECC־Ҳԡ
              TEST_DATA_START_ADDR ֻǲԵַʵʹʱҪݾӦó򷽰ȷ
    */
    if (error_addr > TEST_DATA_START_ADDR && error_addr < TEST_DATA_END_ADDR)
    {
        for (uint32_t i = 0; i < FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE / 4; i++)
        {
            data[i] = *(volatile uint32_t *)(error_sector_addr + i * 4);
        }

        /* ɾ1λECCַ޸ */
        FLASH_DRV_EraseSector(0, error_sector_addr, FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE);
        FLASH_DRV_Program(0, error_sector_addr, FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE, (uint32_t *)&data);
        LOG_ERROR(15, "Erased and reprogrammed the sector containing the ECC error address.\r\n");
    }
    else
    {
        /* ַ */
    }
}

/**
 * @brief   ضFLASH ECCжϴ
 * @author  Samir
 * @version 1.0
 * @date    2025-04-16
 */
void EFM_Ecc_IRQHandler(void)
{
    if (EFM->STS & (EFM_STS_UNRECOVERR_MASK | EFM_STS_CI_UNRECOVERR_MASK))
    {
        LOG_ERROR(15, "2 Flash ECC Un-recovery Error.\r\n");
        efm_unrecovery_error_handler();
    }
    else if (EFM->STS & (EFM_STS_RECOVERR_MASK | EFM_STS_CI_RECOVERR_MASK))
    {
        LOG_ERROR(15, "2 Flash ECC Recovery Error.\r\n");
        efm_recovery_error_handler();
    }
}

/**
 * @brief   ضӲжϴ
 * @author  Samir
 * @version 1.0
 * @date    2025-04-16
 */
void HardFault_Handler(void)
{
    LOG_ERROR(15, "HardFault_Handler\r\n");

    uint8_t temp = 0x30;
    bsp_uart_send(UART_0, &temp, 1);
    /* ȷ쳣ʱʹõĶջָ루MSPPSP䱣浽ȫֱ stack_pointer_address
        Cortex-M У쳣ʹջָ루MSP̶ջָ루PSP
       ͨ EXC_RETURN ĵ 2 λLR Ĵеֵжʹõĸջ 
    */
    __asm volatile("MOVS R0, #4                    \n" /* R0 = 4ڲ EXC_RETURN ĵ2λ */
                   "MOV R1, LR                     \n" /* R1 = LRLR д洢 EXC_RETURN ֵ */
                   "TST R1, R0                     \n" /*  EXC_RETURN[2]жϽ쳣ʱʹõ MSP  PSP */
                   "ITE NE                         \n" /* ִУݲԽѡ֧ */
                   "MRSNE R1, PSP                  \n" /*  EXC_RETURN[2]=1ʹ PSP PSP ֵ R1 */
                   "MRSEQ R1, MSP                  \n" /*  EXC_RETURN[2]=0ʹ MSP MSP ֵ R1 */
                   "LDR R0, =stack_pointer_address \n" /* R0 = stack_pointer_address ĵַ */
                   "STR R1, [R0]                     " /*  R1MSP/PSP ֵ洢 stack_pointer_address */
    );

    /* ڱ׼쳣ջ֡УPCλ stack_pointer_address + 88ֽƫƣ */
    p_instruction_address = ((uint32_t *)stack_pointer_address) + 8; /* stack_pointer_address ָ쳣ջĻַ */
    instruction_opcode = *((uint16_t *)(*p_instruction_address));    /*  ȡõַָ루16 λ */
    instruction_program_counter = *p_instruction_address;            /*  津쳣ַָ */

    /* Thumb-2 ָ 16 λ 32 λָɡͨĸ 5-8 λжָȣ
       0b11101x...0b11110x...0b11111x... ͷĲʾ 32 λָ
    */
    if (((instruction_opcode & 0xE800) == 0xE800) || /* 0b11101x... */
        ((instruction_opcode & 0xF000) == 0xF000) || /* 0b11110x... */
        ((instruction_opcode & 0xF800) == 0xF800))   /* 0b11111x... */
    {
        thumb_instr_size = 4;   /* 32λָ */
    }
    else
    {
        thumb_instr_size = 2;   /* 16λָ */
    }

    /* flash ECC ־ */
    if (EFM->STS & (EFM_STS_UNRECOVERR_MASK | EFM_STS_CI_UNRECOVERR_MASK))
    {
        LOG_ERROR(15, "EFM->STS 0x%x\r\n", EFM->STS);
        /* ״̬߹ϵַĴ */
        SCB->CFSR = SCB_CFSR_BFARVALID_Msk;
        SCB->BFAR = 0x0;

        efm_unrecovery_error_handler();   /* efmɻָ Ӳж */

/*  IAR УPa082 ʾдδʼı˴ָͨʱøþ棬ֱӲջе PC ֵܴ˾档*/
#if defined(__ICCARM__)
        _Pragma("diag_suppress=Pa082")
#endif
            /*  ޸Ķջеķصַǰָ ѭ */
            *p_instruction_address = instruction_program_counter + thumb_instr_size;
#if defined(__ICCARM__)
        _Pragma("diag_default=Pa082")
#endif
    }
    else
    {
        LOG_ERROR(15, "HardFault_Handler sys_soft_reset\r\n");
        /* 쳣 λ */
        sys_soft_reset();
    }
}

/************************************* END OF FILE ************************************/
