/*
 *  Copyright 2020-2023 Yuntu Microelectronics co.,ltd
 *  All rights reserved.
 *
 *  YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 *  used strictly in accordance with the applicable license terms. By expressly
 *  accepting such terms or by downloading, installing, activating and/or otherwise
 *  using the software, you are agreeing that you have read, and that you agree to
 *  comply with and are bound by, such license terms. If you do not agree to be
 *  bound by the applicable license terms, then you may not retain, install,
 *  activate or otherwise use the software. The production use license in
 *  Section 2.3 is expressly granted for this software.
 *
 * @file RSA_example.c
 * @brief This is an example of how to use the RSA encryption and decryption functions with the PSA layer.
 *
 */

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "RSA_example.h"
#include "psa/crypto.h"
#include "common.h"
// #include "printf.h"
#include <string.h>

#define USE_MY_PUBLIC_KEY_TEST 1
/*==================================================================================================
                                        LOCAL VARIABLES
==================================================================================================*/

#if USE_MY_PUBLIC_KEY_TEST


/* The public key with 4096 bits*/
static const uint8_t RSA_public_key_4096_der[] =
{
    0x30,0x82,0x01,0xa2,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
    0x01,0x05,0x00,0x03,0x82,0x01,0x8f,0x00,0x30,0x82,0x01,0x8a,0x02,0x82,0x01,0x81,
    0x00,0xcf,0x9f,0x51,0xd6,0xbf,0x79,0x34,0x5b,0x5c,0xdb,0x61,0xce,0x8a,0x18,0x24,
    0x8f,0xa3,0x35,0x03,0xc1,0x4f,0x6e,0x12,0xf1,0xc3,0xc9,0x15,0x07,0xd0,0xec,0x75,
    0x34,0x9d,0x0e,0xa2,0x74,0x0a,0x74,0x64,0x9e,0x4f,0x98,0x53,0xec,0xd4,0x79,0x8c,
    0xac,0x85,0x11,0xd9,0x4f,0x93,0x06,0x52,0x23,0x7e,0x7c,0x4d,0x4f,0x8a,0x9f,0x1a,
    0x57,0xa0,0x79,0x8c,0x67,0xc7,0x8f,0x5a,0x34,0x54,0xd9,0x89,0x94,0xc5,0x19,0x19,
    0xfc,0x66,0x72,0x21,0xf7,0x3e,0x3e,0x53,0x39,0x4b,0x8c,0xe6,0xd5,0xf6,0xc1,0x21,
    0x38,0xf2,0x32,0x7c,0x43,0x85,0x87,0x9c,0xb6,0x3f,0x8b,0x59,0xcc,0x61,0xeb,0x86,
    0x3a,0xfb,0x5f,0xa1,0xf7,0x38,0x0b,0x79,0xd1,0x81,0x9c,0xcc,0xb9,0x65,0xf9,0xb7,
    0xa1,0xe0,0xef,0x10,0x9b,0xb3,0xe0,0x0e,0xd4,0xe9,0x49,0x0f,0xa3,0xf7,0xeb,0x14,
    0x74,0xca,0x0e,0x6d,0xf0,0xc8,0x39,0xf4,0x73,0x7b,0x5c,0xd0,0xf9,0x90,0xee,0x10,
    0xa6,0x38,0x43,0x3e,0xcf,0x83,0x99,0x63,0x86,0x5a,0x3b,0xec,0x84,0xbe,0x91,0xd1,
    0xdd,0xfe,0x51,0xc8,0x8c,0x9a,0xe5,0xce,0x1a,0xcc,0xc3,0xa8,0x55,0xe2,0xb0,0x59,
    0xb2,0x19,0x11,0xb2,0xb0,0x2b,0xc9,0xed,0xa3,0xda,0x57,0x4d,0xb8,0xb5,0xb7,0x03,
    0x77,0xa1,0x8b,0x39,0x84,0x5c,0x17,0xa4,0x11,0xdf,0x16,0x17,0x3e,0x3b,0xb1,0x9f,
    0x4c,0xd3,0x93,0xcd,0x35,0x74,0xbe,0xe5,0xe4,0xde,0x60,0xc0,0xf6,0xa8,0x0e,0x23,
    0x65,0xf2,0x18,0xad,0xd6,0xc8,0xac,0x20,0xaf,0x95,0xae,0x09,0x17,0xb3,0xb3,0xfe,
    0x31,0x3b,0x15,0x20,0x66,0x08,0xde,0x40,0x23,0x8a,0xbb,0xab,0x96,0xe3,0x75,0x3f,
    0x28,0x67,0x21,0x6a,0x6a,0x25,0xd9,0x69,0x1a,0x5c,0xf0,0x67,0x3f,0x9a,0x58,0x30,
    0xa7,0xf4,0x5d,0xbe,0xd3,0x90,0xbf,0x57,0x9d,0xbf,0x1f,0xab,0xec,0xbe,0xb5,0xd5,
    0x75,0xce,0x8b,0x07,0x62,0x00,0x52,0xf0,0x65,0xe7,0x22,0x85,0x2a,0xbd,0xd2,0x74,
    0xa2,0x9d,0x41,0x46,0xc9,0xd9,0x6b,0x1f,0x35,0x58,0x48,0xf5,0x76,0x4d,0xbb,0x18,
    0xb8,0x8b,0x30,0x80,0x78,0x08,0xb5,0x10,0x0c,0xa2,0x99,0x25,0x5a,0x48,0xe3,0xef,
    0xda,0x51,0x62,0xe2,0x24,0xe6,0xe5,0x11,0x65,0xa1,0x44,0x9d,0xa3,0x8b,0xc7,0x85,
    0x69,0x36,0xff,0xee,0x40,0x5a,0x31,0x09,0x01,0x7f,0xce,0x64,0xab,0xc4,0xc3,0x0f,
    0x1b,0x02,0x03,0x01,0x00,0x01 
};

/* The signature of the data */
static const uint8_t signature_data[384] =
{
    0xac,0xfd,0x93,0xd9,0x71,0xd8,0x3a,0x02,0x5f,0x56,0x07,0xc5,0x31,0x09,0xf2,0x74,
    0xd6,0x13,0xb5,0xdd,0x45,0xfe,0x19,0x4b,0x1d,0xc0,0x8d,0xca,0xc2,0xb3,0x8a,0x9a,
    0xf1,0x47,0xc0,0xa9,0xab,0xb5,0xa4,0x9f,0x11,0x93,0xc8,0x78,0xd7,0xee,0x47,0x3c,
    0x89,0xef,0x3a,0x4e,0xe5,0xc8,0xb2,0x2c,0x04,0x9a,0x30,0xf5,0xfa,0x03,0x12,0xb3,
    0x10,0x15,0xe3,0xbf,0x9e,0xb0,0x06,0x35,0x84,0xdf,0x56,0x3b,0x05,0x46,0xd6,0xd1,
    0x66,0x27,0xa9,0x5b,0xc6,0x13,0x1a,0x9c,0x55,0xbd,0xb0,0x36,0xb3,0x22,0x95,0x47,
    0xa7,0x4e,0xbb,0xee,0xd6,0x64,0xad,0x04,0x9a,0x11,0x31,0xf0,0x96,0x39,0x5f,0x7b,
    0x4a,0x34,0xa8,0x67,0xac,0x5c,0x9d,0x97,0x8f,0xd7,0x3f,0x95,0xe5,0xb4,0xa5,0xe2,
    0x2b,0x99,0xf0,0x94,0x39,0xec,0x5d,0x6b,0x64,0xe8,0x16,0x05,0xbe,0x7e,0x47,0xbb,
    0xad,0x64,0xa4,0xb5,0x4b,0xb6,0x86,0x2e,0x99,0xea,0x0a,0x82,0xd1,0xac,0xda,0xdc,
    0x3a,0xf3,0x9e,0x09,0x5e,0xac,0x08,0x70,0x41,0x30,0x2f,0x80,0xa7,0xd2,0xe6,0xc9,
    0xa3,0x96,0x07,0xfb,0x15,0x36,0xa7,0x17,0x50,0x14,0x2a,0x15,0xfd,0x95,0x79,0xf0,
    0x9f,0x75,0x5f,0xde,0xb1,0x62,0x72,0x7e,0xe8,0x23,0x8b,0xba,0x8b,0x01,0x4f,0xc3,
    0x80,0x4c,0xd0,0xfe,0x63,0x87,0x59,0x6d,0xb1,0xae,0x00,0x9c,0x04,0x6b,0xff,0xd6,
    0xfb,0xb4,0xb6,0x2b,0xbe,0x51,0x51,0x26,0xa7,0xae,0x52,0x08,0x5e,0x40,0x51,0x77,
    0x9f,0x16,0x18,0x9f,0xd2,0x56,0x47,0x58,0xc2,0xc8,0x19,0x32,0x90,0xf0,0x08,0x10,
    0x86,0x5e,0x06,0x73,0x08,0x1a,0xee,0x66,0xfe,0xdb,0xb2,0x31,0xab,0x0c,0x63,0x1c,
    0x74,0x41,0x81,0x2a,0x65,0x8e,0x25,0x6e,0x12,0x50,0x38,0x66,0x57,0x9d,0xf4,0x00,
    0xdf,0x4d,0x12,0x5b,0xb0,0x14,0x5c,0x78,0x89,0xe4,0xe3,0x3d,0xfd,0xa5,0xee,0x5b,
    0x6a,0x3a,0x96,0xc2,0x26,0x1b,0x82,0xbc,0xb7,0xb7,0x99,0xb6,0x9b,0x75,0x28,0x32,
    0x4f,0xeb,0xb7,0xe4,0xde,0xf5,0xa4,0xcb,0x29,0x0f,0x13,0xd0,0xfd,0x22,0x7f,0xef,
    0xe0,0xb4,0x08,0xeb,0x88,0x4b,0x79,0x34,0xf2,0x03,0x06,0xf3,0x08,0xd3,0xc1,0x8e,
    0xe5,0x8c,0x89,0x2e,0x1a,0xce,0x21,0x52,0x6a,0x95,0x51,0xf5,0x85,0x4e,0x4a,0xb3,
    0xe4,0xa6,0xb2,0x01,0x4a,0xe8,0x04,0xef,0xaa,0x97,0x17,0x02,0x10,0x2f,0x18,0xd8
};

/* The message that need to be signed*/
static const uint8_t message[] = "hello world!";


#else

/* The private key with 4096 bits */
#if 0 
static const uint8_t RSA_private_key_4096_der[] =
{
 0x30,
0x82,0x06,0xe3,0x02,0x01,0x00,0x02,0x82,0x01,0x81,0x00,0x9c,0x0b,0x7b,0x60,0xe0,
0x69,0xa7,0x75,0x03,0x91,0xc3,0x83,0x60,0xb7,0x23,0x41,0xe3,0x88,0x29,0x38,0xb3,
0x42,0x99,0x39,0x45,0x4d,0x01,0xb9,0x27,0x3a,0xe5,0x9c,0x62,0x7d,0x32,0x7c,0xda,
0x70,0x70,0xe8,0xeb,0xd3,0x3d,0x56,0xc6,0xb5,0x01,0x17,0x43,0x3e,0x81,0x3a,0x9d,
0x36,0x27,0xed,0x39,0xae,0xa9,0x3d,0x1d,0x80,0x52,0x57,0xb8,0x29,0xf7,0xb3,0x91,
0x62,0x27,0x9e,0xf8,0x1e,0x46,0x86,0x39,0xbe,0x69,0x85,0xe0,0xba,0xae,0xd3,0x83,
0x56,0xa6,0x9e,0xf3,0x3e,0xad,0x78,0x01,0x59,0xc4,0x39,0x36,0x47,0xee,0x67,0x99,
0x4d,0x17,0x63,0x75,0x29,0xd1,0x8f,0xd3,0xcf,0x65,0x06,0x93,0x28,0x71,0xf1,0x35,
0xac,0x8e,0x1c,0x86,0xef,0xf1,0xad,0x1f,0xa4,0xdb,0x3c,0x39,0x94,0x76,0x81,0x57,
0x0e,0x84,0x89,0xba,0x4a,0x62,0xa5,0x9a,0x17,0xcf,0xe7,0xd3,0x6d,0xf3,0x78,0x30,
0xf4,0xc6,0xfb,0x44,0x09,0x76,0xbb,0x7c,0x3d,0x6e,0x48,0x57,0x32,0xdb,0x9c,0x12,
0x70,0x8d,0x61,0xc7,0x9e,0x96,0x29,0x07,0xeb,0x5e,0x1f,0x50,0x13,0x03,0xca,0x36,
0x7d,0x52,0xb4,0xca,0x3c,0x87,0x82,0x4b,0x09,0x07,0xef,0x54,0x0b,0xa4,0xe0,0xfd,
0x64,0x9e,0x96,0xf1,0x9f,0xb4,0x91,0x07,0x0a,0x51,0x23,0x9d,0x16,0x33,0x52,0x79,
0xbd,0x56,0xb4,0x52,0xef,0xcf,0x89,0xf3,0xda,0xf1,0x9c,0x0d,0xb0,0xac,0x11,0xa1,
0x92,0xbe,0xd9,0x35,0x7e,0x4c,0x38,0xdb,0xc7,0xf5,0xbb,0xb1,0x35,0x76,0xc1,0xfa,
0x24,0xb8,0x85,0x8d,0xc6,0x8e,0xc7,0x0e,0x1b,0xce,0x2e,0x20,0xed,0x42,0x27,0xb8,
0x90,0x37,0x7f,0x1e,0xef,0xbc,0xa6,0x7e,0xf9,0x46,0xca,0x9b,0x6d,0x87,0xce,0x01,
0xbb,0x55,0x3f,0xe7,0x4d,0x07,0xb0,0xa3,0xd5,0xec,0xe2,0x89,0x34,0xef,0x34,0xc3,
0x56,0x4c,0xe5,0x24,0x28,0xfa,0xe8,0x6c,0x73,0x1a,0x69,0xf9,0xcd,0xb8,0xb6,0xa7,
0x12,0x7d,0x12,0xc8,0x0f,0x1b,0x1a,0x16,0x78,0x67,0x88,0x49,0x5d,0xf6,0xc6,0x73,
0x77,0x1d,0x9d,0x99,0x32,0xc8,0xc6,0x1a,0x2c,0x78,0x75,0xcd,0x6e,0x95,0x90,0x34,
0x0e,0x81,0xac,0xc9,0xa3,0x9d,0x58,0xf8,0x11,0x26,0x34,0x23,0xb2,0xc8,0x96,0xc6,
0xd4,0x51,0x5d,0x56,0x3e,0x40,0x73,0x25,0xf1,0x28,0x1d,0x75,0x81,0x6f,0x3a,0x58,
0x87,0xd9,0x15,0x36,0x0b,0xb9,0xf3,0x1a,0xe0,0x68,0x85,0x02,0x03,0x01,0x00,0x01,
0x02,0x82,0x01,0x80,0x06,0x0b,0x8e,0x7e,0x7f,0x64,0xe6,0xa8,0x15,0x2a,0xa6,0x73,
0x3f,0x61,0xee,0x39,0xbc,0x57,0x0a,0xb5,0x54,0x48,0xef,0x18,0xe3,0xc6,0xa0,0x29,
0xcc,0x52,0x99,0xca,0x20,0xf0,0x6a,0xee,0x4b,0xc0,0x15,0x2a,0x5f,0x0f,0x8b,0x3f,
0xad,0x22,0x90,0x34,0x08,0x81,0x4e,0xb8,0xc5,0x07,0xb6,0xf3,0xd0,0xa0,0x79,0x22,
0xd8,0xc1,0x10,0xfd,0xca,0xbf,0x1a,0xcd,0x42,0xbc,0x5c,0xb7,0xb0,0xf9,0xeb,0x48,
0x2c,0xcb,0x2a,0xf5,0x95,0x00,0x49,0xfd,0x1c,0x96,0x8d,0xa9,0xea,0x49,0x23,0xfd,
0x05,0x0e,0x98,0xfb,0xad,0x87,0xc4,0x4b,0x82,0x42,0xdc,0x4b,0x8a,0x4d,0x51,0xfd,
0xae,0xdd,0xd7,0x16,0x87,0xac,0x78,0x54,0xaa,0xc3,0x6c,0xc7,0xbf,0x78,0xb8,0x2e,
0x8e,0xc7,0x0c,0xdb,0xaa,0xde,0x7e,0xa0,0xc8,0xfe,0xd3,0x34,0xf5,0x61,0x07,0x73,
0x86,0xe9,0x5c,0xcd,0x37,0x01,0xa4,0xfe,0x15,0x73,0xe2,0x94,0x30,0x06,0x59,0x63,
0x67,0xc8,0x15,0x04,0x17,0xfa,0xec,0xa0,0x65,0x39,0x25,0xaf,0x3c,0x45,0x9b,0xd9,
0x63,0x85,0x5f,0xdb,0x6b,0x93,0xd4,0x8e,0xde,0x01,0x88,0xe3,0xd1,0x6f,0x33,0xc1,
0x99,0xfc,0x57,0x5b,0x67,0x4e,0xa7,0x1a,0x0d,0x34,0xc7,0xfb,0x52,0xd7,0x41,0xe5,
0x22,0xfa,0x4d,0x3c,0xfa,0x9b,0x9a,0xf2,0xff,0x19,0x5c,0x2c,0x1a,0x61,0x32,0x4c,
0xf1,0xba,0x74,0x2c,0x2d,0x15,0x56,0x0d,0x3d,0x24,0xb8,0x47,0xf3,0x09,0x77,0xfc,
0x75,0xcf,0x95,0x53,0xf7,0x43,0xc1,0x0f,0xc0,0x87,0x39,0xec,0x5d,0xcc,0x58,0xa0,
0x73,0xcf,0x87,0xc0,0xdc,0xcd,0x1b,0x2b,0x03,0x37,0x2c,0x51,0x7d,0xc5,0x8d,0xc1,
0x2f,0x5e,0x18,0x72,0xa4,0x27,0x05,0x2f,0xe7,0x06,0xc1,0x6a,0x95,0xdc,0xc6,0xa7,
0x36,0xe4,0x07,0x1f,0x70,0x7c,0x97,0x7b,0x2f,0xf2,0xf4,0x10,0x69,0x0c,0xa6,0xfa,
0x45,0x6d,0xeb,0xdc,0xa9,0xe5,0x15,0xaa,0x09,0x43,0x71,0x3b,0xa8,0x49,0x4d,0x07,
0x3f,0xcb,0x45,0x36,0xb8,0xd7,0x99,0x76,0xf6,0x64,0x7c,0xe1,0xeb,0x9a,0xa0,0x8d,
0xf2,0x84,0x7b,0xd4,0x08,0x47,0x49,0xf2,0x2a,0xc0,0x8f,0x4e,0x73,0x9b,0xf4,0x93,
0xed,0x70,0x10,0x83,0x32,0x5f,0xd7,0x22,0xbc,0xbe,0xef,0x20,0xe7,0x0a,0x17,0x8d,
0x52,0x92,0x41,0xb1,0x9b,0xa9,0xba,0xc1,0x36,0xbf,0xdc,0x9d,0x9d,0x0a,0x7f,0x22,
0x5a,0x3d,0x99,0xd9,0x02,0x81,0xc1,0x00,0xda,0xc1,0x1b,0x3b,0xee,0x5a,0x66,0x3c,
0xe0,0x82,0x12,0x9c,0x8f,0x49,0xd5,0xea,0x20,0x8e,0xfd,0x35,0x09,0x27,0x27,0x6d,
0xf6,0xe6,0xfd,0xb3,0xc3,0xcb,0x96,0xb3,0xf2,0xcb,0x37,0x1f,0x16,0x13,0xe8,0x5c,
0x6f,0x7c,0xbb,0x14,0x15,0xa3,0x51,0x23,0x6a,0xa8,0xc9,0x8b,0xdc,0x73,0xe5,0xbf,
0xd0,0xc3,0x1d,0x5f,0x0b,0x50,0xc6,0x0d,0x5e,0x5b,0xd0,0x31,0xd3,0x5b,0xd5,0x30,
0xc3,0x3a,0xe4,0x2b,0xb4,0x11,0x87,0xf2,0x9e,0xdd,0x92,0xf5,0xe5,0x63,0x2d,0x96,
0x8c,0x98,0xf4,0xb5,0xf6,0x08,0xc4,0xf0,0x5f,0x67,0x93,0x05,0x1e,0xe6,0x30,0x60,
0x2c,0x48,0x0c,0x30,0x5a,0x5c,0x61,0x12,0xd1,0xa5,0xb5,0x17,0x42,0xc0,0xcf,0xa0,
0x25,0x19,0x11,0xac,0x36,0x53,0x56,0xf4,0xe0,0xaa,0x69,0xbe,0x53,0x8b,0x4c,0x04,
0xfd,0x33,0xe1,0xa0,0x4a,0x6a,0x63,0xcd,0x25,0xa0,0xe5,0xdf,0x04,0x4e,0xb5,0xa7,
0x5a,0x3f,0xb1,0x09,0xc4,0x65,0x07,0xcd,0xfb,0x32,0x6c,0x43,0x13,0xb5,0x21,0xc1,
0xd4,0xd6,0xfb,0x29,0xb9,0x65,0xc8,0xdc,0xcd,0xae,0x9f,0x3f,0x94,0xba,0xbc,0x33,
0x75,0x1b,0xf3,0xe2,0x00,0xed,0xc3,0xaf,0x02,0x81,0xc1,0x00,0xb6,0x9d,0x0b,0x38,
0x7c,0xf9,0x6d,0xf6,0x12,0x33,0x2e,0x62,0x7c,0xdf,0x5b,0xdc,0x2d,0x42,0xb0,0xcc,
0xc5,0x0c,0x29,0xed,0xcb,0xd3,0x8e,0x8e,0x81,0x3b,0xe9,0x03,0xfc,0xb7,0xec,0x62,
0xd1,0x63,0x16,0xba,0x5f,0x31,0xba,0x7d,0xe2,0x7b,0xcd,0xaa,0xe1,0x7f,0xf0,0x5e,
0xc2,0x42,0xd9,0x5b,0x38,0x9b,0x18,0x75,0xa8,0x50,0xbb,0xc5,0x26,0x2b,0xa1,0x7f,
0x08,0x46,0x77,0xf9,0x50,0x88,0x31,0x00,0xa0,0xe5,0xbf,0x5a,0x1c,0x58,0xe1,0x1f,
0x63,0x65,0xd8,0x84,0xed,0x10,0x28,0xe4,0xec,0x42,0x46,0xac,0xc3,0x6b,0xe4,0x93,
0xd0,0x74,0x19,0x38,0xf6,0xa6,0xb1,0xa8,0x0d,0xad,0x75,0xca,0xa5,0x08,0x93,0xa7,
0x63,0x23,0xa5,0xf7,0x0b,0x0e,0x98,0xc7,0x4f,0x01,0xce,0x79,0xd6,0x42,0xf8,0x37,
0x09,0x75,0x22,0x7d,0x33,0x8b,0xd9,0x71,0xa4,0x0b,0x1d,0x75,0x21,0x8a,0xbb,0xe4,
0x3f,0xb8,0x26,0x77,0x48,0x4b,0x3b,0xee,0xea,0xd9,0xb3,0x19,0x64,0x5b,0xa3,0xe1,
0xa8,0xf5,0xb3,0x6c,0x02,0x13,0x7f,0x9a,0xa3,0x0e,0xeb,0x55,0x16,0xa0,0x2c,0xdf,
0x9c,0xdc,0x86,0x8f,0xb6,0x20,0x16,0x83,0x0f,0x27,0x00,0x0b,0x02,0x81,0xc0,0x56,
0xe7,0xa8,0x9e,0x22,0x8f,0x1f,0x56,0xc8,0x0a,0x64,0xc2,0x75,0xda,0x6b,0x75,0x5c,
0x01,0xf2,0x6e,0x03,0x78,0xde,0x9d,0x3d,0x1d,0x7f,0x64,0xcf,0x40,0x2e,0x5a,0xc6,
0xf6,0x1c,0xab,0xe8,0xe5,0x77,0xc6,0x61,0x34,0x38,0x4e,0x3e,0x5a,0x67,0x7d,0x9e,
0x7d,0xf7,0xb3,0xec,0x9c,0xce,0x85,0x69,0xb6,0x5a,0x10,0xcc,0x3a,0x06,0xb4,0x80,
0xec,0x10,0xc5,0xfa,0xa5,0xdd,0x08,0x46,0x47,0x3b,0x26,0x71,0x9f,0x15,0x57,0x39,
0xed,0xe4,0xd8,0x73,0x8c,0xf7,0x2b,0x7c,0x27,0xb7,0xf0,0x22,0x12,0x51,0xca,0xfc,
0xdb,0xf4,0xf8,0x6f,0xd2,0x1b,0x1d,0xd9,0xf9,0xb5,0x7d,0xf0,0x28,0x0b,0x18,0xa2,
0xd9,0x26,0x1c,0x5b,0x94,0xa5,0xe7,0x85,0x55,0x74,0x43,0xea,0x41,0x29,0xa6,0x97,
0x2a,0x9c,0x5f,0xd4,0x68,0xde,0x86,0x8c,0xe0,0xf3,0x9d,0xee,0x72,0x5d,0xbb,0xbe,
0x5b,0x31,0x77,0x6c,0x0a,0xa8,0xf8,0x17,0x4e,0x7e,0x52,0xd7,0x07,0x1c,0x16,0xa4,
0xd8,0xdf,0x51,0xb6,0x9e,0x9b,0x0e,0x9a,0x33,0xde,0x0c,0x31,0x6f,0x52,0x61,0x06,
0x05,0x7d,0x96,0x90,0xcb,0xae,0xc0,0xf2,0xa1,0x6c,0x3e,0x39,0x08,0x3c,0x59,0x02,
0x81,0xc0,0x1b,0x80,0x7f,0x46,0x1c,0x9c,0x3c,0x11,0x49,0x20,0x68,0x25,0xf1,0x04,
0x7a,0x83,0x28,0x55,0xd8,0xe6,0x9d,0xee,0x7d,0x0a,0x11,0x1e,0x65,0x63,0xd4,0xd6,
0xbe,0xfb,0x18,0xbe,0x62,0x5d,0x67,0x87,0x72,0xce,0xd2,0xb3,0x5c,0x90,0xc4,0xe1,
0x3c,0x52,0x35,0xef,0xdc,0xb5,0x88,0xb0,0x24,0xa7,0xf6,0x58,0x75,0x50,0x92,0xee,
0xc5,0xb8,0xa5,0x4d,0x00,0xdf,0x6e,0xd1,0x9e,0x13,0xf0,0x2a,0xf2,0xa6,0x53,0x95,
0x06,0x42,0x7d,0xee,0x6c,0xc9,0x1b,0x61,0x25,0x40,0xaf,0xab,0x35,0x5d,0x74,0xaf,
0x6e,0x91,0x6a,0x44,0xff,0x81,0xbb,0x6d,0x20,0xb7,0x51,0x7e,0x3d,0x57,0x3f,0xe8,
0x15,0xcb,0x55,0x1b,0x7b,0x91,0x43,0xff,0x04,0x69,0xb6,0x42,0x96,0x83,0xbd,0xc4,
0x59,0xe2,0x43,0x80,0x86,0x7d,0x53,0x71,0x4e,0xa6,0x68,0x72,0xf1,0x66,0x89,0xfc,
0x1a,0x8d,0x9f,0x65,0x93,0x72,0x86,0x9e,0x27,0x3c,0xe3,0xa8,0x60,0xf8,0x54,0x6a,
0x40,0xec,0xdc,0x0e,0x69,0x19,0xfd,0x26,0x33,0xc5,0x5a,0x72,0xb1,0x4f,0x0a,0xab,
0x9e,0x89,0x8c,0x04,0xc7,0x7d,0x4d,0xd4,0x70,0x77,0x88,0x48,0x40,0xdb,0xb0,0x16,
0x81,0x8d,0x02,0x81,0xc1,0x00,0xad,0x0f,0x94,0x8c,0x15,0x33,0x72,0x52,0xd7,0x1d,
0x6b,0xcd,0x16,0x81,0x57,0xe7,0xf7,0xe9,0x17,0xf9,0x61,0x19,0x58,0x10,0x00,0x45,
0x47,0xcc,0x0f,0x3d,0x9e,0x4d,0x5a,0x99,0x00,0x09,0xd7,0xed,0x6e,0x69,0xee,0x13,
0x33,0x02,0xcb,0x48,0xc2,0xc7,0x0d,0x65,0x84,0xef,0x18,0xb7,0x10,0x6b,0xf0,0xb5,
0x6c,0x75,0xd5,0x2c,0x3b,0xab,0x1f,0x94,0x4b,0xf9,0x6b,0x5f,0x4a,0x32,0xb9,0xa6,
0x5f,0x9e,0x2e,0x56,0x3d,0xb0,0x76,0xce,0xce,0xd1,0x3a,0xf0,0x1c,0x0d,0xa7,0xfd,
0x8f,0xc9,0x36,0x43,0x44,0xb4,0x4b,0x23,0x09,0x69,0xd3,0x3f,0xf6,0x34,0x00,0x38,
0xdb,0x66,0x0c,0xa0,0x98,0x93,0xc1,0x1b,0xa2,0x85,0x48,0x28,0x5f,0x0f,0x17,0xda,
0x0d,0xc8,0x0d,0x33,0xaf,0x30,0x56,0x2e,0xa7,0x30,0xbb,0x76,0x8c,0x0b,0x86,0xdc,
0x1c,0x82,0x91,0x90,0x9c,0x7e,0x78,0xec,0xcf,0x6c,0x06,0x46,0x49,0x3b,0xfc,0x09,
0x3c,0xa7,0x03,0xe7,0x07,0x0c,0xfe,0x68,0x49,0x9b,0xc4,0x73,0xfd,0x5a,0xb1,0x0c,
0xa1,0xc3,0x11,0x53,0x87,0x02,0x8b,0xa9,0x53,0x63,0x72,0xfb,0x54,0xc9,0xd0,0x53,
0x79,0x4f,0xd8,0x69,0x7d,0xf4
};
#endif
/* The public key with 4096 bits*/
static const uint8_t RSA_public_key_4096_der[] =
{
 0x30,0x82,0x01,0x8a,0x02,0x82,0x01,0x81,0x00,0x9c,0x0b,0x7b,0x60,0xe0,0x69,0xa7,0x75,
0x03,0x91,0xc3,0x83,0x60,0xb7,0x23,0x41,0xe3,0x88,0x29,0x38,0xb3,0x42,0x99,0x39,
0x45,0x4d,0x01,0xb9,0x27,0x3a,0xe5,0x9c,0x62,0x7d,0x32,0x7c,0xda,0x70,0x70,0xe8,
0xeb,0xd3,0x3d,0x56,0xc6,0xb5,0x01,0x17,0x43,0x3e,0x81,0x3a,0x9d,0x36,0x27,0xed,
0x39,0xae,0xa9,0x3d,0x1d,0x80,0x52,0x57,0xb8,0x29,0xf7,0xb3,0x91,0x62,0x27,0x9e,
0xf8,0x1e,0x46,0x86,0x39,0xbe,0x69,0x85,0xe0,0xba,0xae,0xd3,0x83,0x56,0xa6,0x9e,
0xf3,0x3e,0xad,0x78,0x01,0x59,0xc4,0x39,0x36,0x47,0xee,0x67,0x99,0x4d,0x17,0x63,
0x75,0x29,0xd1,0x8f,0xd3,0xcf,0x65,0x06,0x93,0x28,0x71,0xf1,0x35,0xac,0x8e,0x1c,
0x86,0xef,0xf1,0xad,0x1f,0xa4,0xdb,0x3c,0x39,0x94,0x76,0x81,0x57,0x0e,0x84,0x89,
0xba,0x4a,0x62,0xa5,0x9a,0x17,0xcf,0xe7,0xd3,0x6d,0xf3,0x78,0x30,0xf4,0xc6,0xfb,
0x44,0x09,0x76,0xbb,0x7c,0x3d,0x6e,0x48,0x57,0x32,0xdb,0x9c,0x12,0x70,0x8d,0x61,
0xc7,0x9e,0x96,0x29,0x07,0xeb,0x5e,0x1f,0x50,0x13,0x03,0xca,0x36,0x7d,0x52,0xb4,
0xca,0x3c,0x87,0x82,0x4b,0x09,0x07,0xef,0x54,0x0b,0xa4,0xe0,0xfd,0x64,0x9e,0x96,
0xf1,0x9f,0xb4,0x91,0x07,0x0a,0x51,0x23,0x9d,0x16,0x33,0x52,0x79,0xbd,0x56,0xb4,
0x52,0xef,0xcf,0x89,0xf3,0xda,0xf1,0x9c,0x0d,0xb0,0xac,0x11,0xa1,0x92,0xbe,0xd9,
0x35,0x7e,0x4c,0x38,0xdb,0xc7,0xf5,0xbb,0xb1,0x35,0x76,0xc1,0xfa,0x24,0xb8,0x85,
0x8d,0xc6,0x8e,0xc7,0x0e,0x1b,0xce,0x2e,0x20,0xed,0x42,0x27,0xb8,0x90,0x37,0x7f,
0x1e,0xef,0xbc,0xa6,0x7e,0xf9,0x46,0xca,0x9b,0x6d,0x87,0xce,0x01,0xbb,0x55,0x3f,
0xe7,0x4d,0x07,0xb0,0xa3,0xd5,0xec,0xe2,0x89,0x34,0xef,0x34,0xc3,0x56,0x4c,0xe5,
0x24,0x28,0xfa,0xe8,0x6c,0x73,0x1a,0x69,0xf9,0xcd,0xb8,0xb6,0xa7,0x12,0x7d,0x12,
0xc8,0x0f,0x1b,0x1a,0x16,0x78,0x67,0x88,0x49,0x5d,0xf6,0xc6,0x73,0x77,0x1d,0x9d,
0x99,0x32,0xc8,0xc6,0x1a,0x2c,0x78,0x75,0xcd,0x6e,0x95,0x90,0x34,0x0e,0x81,0xac,
0xc9,0xa3,0x9d,0x58,0xf8,0x11,0x26,0x34,0x23,0xb2,0xc8,0x96,0xc6,0xd4,0x51,0x5d,
0x56,0x3e,0x40,0x73,0x25,0xf1,0x28,0x1d,0x75,0x81,0x6f,0x3a,0x58,0x87,0xd9,0x15,
0x36,0x0b,0xb9,0xf3,0x1a,0xe0,0x68,0x85,0x02,0x03,0x01,0x00,0x01   
};

/* The signature of the data */
static const uint8_t signature_data[384] =
{
 0x60,
0x41,0xb7,0xaf,0xfd,0xf5,0xa3,0x94,0xde,0x44,0x71,0x75,0x65,0xf3,0xd6,0xb0,0x86,
0xf3,0x86,0xd3,0xce,0xa5,0x90,0xd4,0xa3,0x40,0xdd,0x2a,0xb0,0x58,0x2e,0x10,0x54,
0x4b,0x8e,0x2c,0x2c,0x05,0x8c,0x15,0x60,0xa9,0x03,0x71,0x12,0x27,0x2d,0x29,0x76,
0xab,0xdd,0x67,0xe3,0x07,0xd1,0xd1,0x96,0x1d,0xf0,0xbf,0x53,0xbc,0xb6,0x24,0xb3,
0x6e,0x1c,0xa0,0xb3,0x0a,0x17,0xa0,0xdb,0x1c,0x09,0x22,0x87,0x19,0xbf,0xef,0x78,
0xd6,0x83,0x58,0x38,0x2e,0xea,0xf4,0xb8,0x18,0x61,0x07,0x42,0x88,0x47,0xc2,0xd2,
0x52,0x8d,0xcc,0x76,0x5a,0x86,0x2a,0xaa,0xe1,0x5d,0x56,0x3d,0x0a,0x4b,0x91,0x00,
0x5d,0xf9,0x0e,0x3f,0x62,0x13,0x7e,0xb9,0x90,0x80,0xa1,0x8c,0x24,0x15,0xcb,0x25,
0xa5,0x08,0x02,0xc8,0x07,0xa3,0x6e,0x36,0xae,0x4b,0xd9,0xa2,0xc7,0xea,0xc4,0x17,
0xb6,0x14,0x4a,0x44,0x2a,0x44,0x8b,0x45,0x32,0xe3,0x26,0xcf,0x76,0xd8,0x2a,0xcc,
0x22,0xbb,0x27,0xe2,0x87,0xf6,0x6c,0x99,0xbe,0x1a,0x1a,0xb5,0x5b,0x06,0xa6,0xaf,
0xfd,0x65,0x62,0xf8,0x68,0xad,0x2d,0x95,0xd8,0x8d,0xec,0x94,0x05,0x5f,0xc0,0x87,
0x4d,0x09,0x68,0x38,0x39,0xba,0x7d,0x50,0x3f,0xb6,0x23,0x1b,0x0c,0x9c,0xcc,0x2a,
0xcc,0xc7,0xa9,0x7b,0x06,0xfe,0x4f,0xa5,0x32,0x11,0xed,0x31,0xb1,0xbb,0xd5,0x95,
0xd5,0x77,0x48,0x7c,0x33,0x2c,0x9c,0xf0,0x95,0xca,0xe3,0xa3,0x69,0x49,0x27,0x89,
0x23,0x3e,0x36,0x9f,0x00,0x66,0xd4,0xb6,0xc6,0xf4,0x02,0x58,0x77,0x7a,0xad,0xaa,
0x1f,0x69,0x6a,0x2e,0x46,0x3a,0xdd,0x56,0x26,0xb2,0x1e,0x86,0xb8,0x8a,0xed,0x16,
0xee,0x73,0x20,0xa2,0x52,0x12,0xb0,0xf2,0x58,0x97,0x28,0xb6,0x6e,0x84,0xbb,0xd4,
0x5d,0xef,0xef,0xa6,0x37,0xac,0x93,0xde,0x06,0x8c,0x7d,0x95,0xd5,0x7f,0xd1,0x2a,
0xc3,0x7e,0xb1,0x27,0x4f,0x6f,0x5f,0xc7,0x1b,0x2c,0x42,0x49,0x35,0xb0,0xee,0x90,
0x23,0x89,0xfb,0x84,0xf0,0x16,0xaa,0x96,0xa0,0x15,0x5a,0x27,0xa7,0x0c,0xd1,0x01,
0x8a,0x69,0x80,0xed,0xa6,0x62,0x00,0xc2,0x5f,0x7d,0x0a,0x41,0x82,0x1e,0x18,0x25,
0x69,0x64,0xdd,0x73,0x39,0x1d,0x04,0x73,0x96,0xe3,0x41,0x94,0x86,0x88,0x35,0xc7,
0xf4,0x65,0x51,0x34,0xea,0x35,0x3d,0x01,0xa8,0xa6,0x0a,0x6d,0xad,0xb1,0xaa 
};

/* The message that need to be signed*/
static const uint8_t message[] = "Hello, World!";

#endif

/*==================================================================================================
                                         LOCAL FUNCTION
==================================================================================================*/
/**
 * @brief RSA sign test
 * @details This function tests the RSA sign function.
 *
 * @return  int : The result of the test.
 * @retval  0: Test passed.
 * @retval othe status: Test failed.
 */
 #if 0
static int RSA_sign_test()
{
    psa_status_t status;
    psa_key_attributes_t attributes = PSA_KEY_ATTRIBUTES_INIT;
    psa_key_id_t key_id = 0;
    static uint8_t signature[512] = {0};
    static size_t signature_length;

    //Set Key Attributes for Importing an RSA Private Key
    psa_set_key_usage_flags(&attributes, PSA_KEY_USAGE_SIGN_HASH);
    psa_set_key_algorithm(&attributes, PSA_ALG_RSA_PKCS1V15_SIGN(PSA_ALG_SHA_256));
    psa_set_key_type(&attributes, PSA_KEY_TYPE_RSA_KEY_PAIR);
    psa_set_key_bits(&attributes, 3072);

    /* Import the RSA Private Key*/
    status = psa_import_key(&attributes, RSA_private_key_4096_der, sizeof(RSA_private_key_4096_der), &key_id);
    if (status != PSA_SUCCESS)
    {
        PRINTF("Key import failed! (status = %d)\n\r", status);
        return status;
    }

    /* Sign the message*/
    status = psa_sign_message(key_id, PSA_ALG_RSA_PKCS1V15_SIGN(PSA_ALG_SHA_256), message, sizeof(message), signature, sizeof(signature), &signature_length);
    if (status != PSA_SUCCESS)
    {
        PRINTF("Signature failed! (status = %d)\n\r", status);
        return status;
    }

    if (buf_cmp(signature_data, signature, signature_length) != 0)
    {
        PRINTF("Signature is not correct!\n\r");
        return -1;
    }


    PRINTF("Signature length: %d\n\r", signature_length);
    print_buf("Signature", signature, signature_length);

    /* Destroy the key*/
    psa_destroy_key(key_id);
    return 0;
}
#endif
/**
 * @brief RSA verify test
 * @details This function tests the RSA verify function.
 *
 * @return  int : The result of the test.
 * @retval  0: Test passed.
 * @retval othe status: Test failed.
 */
int rsa_ret = 0;
static int RSA_verify_hash_test()
{
    psa_status_t status;
    psa_key_attributes_t attributes = PSA_KEY_ATTRIBUTES_INIT;
    psa_key_id_t key_id = 0;

    //Set Key Attributes for Importing an RSA Public Key
    psa_set_key_usage_flags(&attributes, PSA_KEY_USAGE_VERIFY_HASH);
    psa_set_key_algorithm(&attributes, PSA_ALG_RSA_PKCS1V15_SIGN(PSA_ALG_SHA_256));
    psa_set_key_type(&attributes, PSA_KEY_TYPE_RSA_PUBLIC_KEY);
    psa_set_key_bits(&attributes, 3072);

    /* Import the RSA Public Key*/
    status = psa_import_key(&attributes, RSA_public_key_4096_der, sizeof(RSA_public_key_4096_der), &key_id);
    if (status != PSA_SUCCESS)
    {
        // PRINTF("Key import failed! (status = %d)\n\r", status);
        rsa_ret = status;
        return status;
    }

		uint8_t hash[32];  // SHA-256 ĹϣֵΪ 32 ֽ
		size_t hash_length;

    #if USE_MY_PUBLIC_KEY_TEST
		status = psa_hash_compute(PSA_ALG_SHA_256, message, strlen(message), hash, sizeof(hash), &hash_length);
    #else
        status = psa_hash_compute(PSA_ALG_SHA_256, message, sizeof(message), hash, sizeof(hash), &hash_length);
    #endif
		if (status != PSA_SUCCESS) {
				//  PRINTF("Hash computation failed! (status = %d)\n\r", status);
                rsa_ret = status;
				return status;
		}

		// ʹùϣֵ֤ǩ
		status = psa_verify_hash(key_id, PSA_ALG_RSA_PKCS1V15_SIGN(PSA_ALG_SHA_256), hash, hash_length, signature_data, sizeof(signature_data));
		if (status != PSA_SUCCESS) {
            rsa_ret = status;
				//  PRINTF("Signature verification failed! (status = %d)\n\r", status);
				return status;
		} else {
            //  PRINTF("PASS~\r\n");
        }

    /* Destroy the key*/
    psa_destroy_key(key_id);
    return 0;
}



/*==================================================================================================
                                          GLOBAL FUNCTION
==================================================================================================*/
#if 0
int RSA_key_generate(uint32_t key_length)
{
    psa_status_t status;
    psa_key_attributes_t attributes = PSA_KEY_ATTRIBUTES_INIT;
    psa_key_id_t key_id = 0;
    size_t output_length;
    uint8_t private_key[4096];
    uint8_t public_key[4096 / 2];

    /* Set the key attributes */
    psa_set_key_usage_flags(&attributes, PSA_KEY_USAGE_SIGN_HASH | PSA_KEY_USAGE_VERIFY_HASH | PSA_KEY_USAGE_EXPORT);
    psa_set_key_algorithm(&attributes, PSA_ALG_RSA_PKCS1V15_SIGN(PSA_ALG_SHA_256));
    psa_set_key_type(&attributes, PSA_KEY_TYPE_RSA_KEY_PAIR);
    psa_set_key_bits(&attributes, key_length);

    /* Generate the key pair*/
    status = psa_generate_key(&attributes, &key_id);
    if (status != PSA_SUCCESS)
    {
        PRINTF("Key generation failed! (status = %d)\n\r", status);
        return status;
    }

    /* Export the public key*/
    status = psa_export_public_key(key_id, public_key, sizeof(public_key), &output_length);
    if (status != PSA_SUCCESS)
    {
        PRINTF("Public key export failed! (status = %d)\n\r", status);
        return status;
    }

    PRINTF("Public key length: %d\n\r", output_length);
    print_buf("Public key", public_key, output_length);

    /* Export the private key*/
    status = psa_export_key(key_id, private_key, sizeof(private_key), &output_length);
    if (status != PSA_SUCCESS)
    {
        PRINTF("Private key export failed! (status = %d)\n\r", status);
        return status;
    }

    PRINTF("Private key length: %d\n\r", output_length);
    print_buf("Private key", private_key, output_length);

    /* Destroy the key*/
    psa_destroy_key(key_id);
    return 0;
}
#endif
/**
 * @brief RSA sign and verify test with 4096 bits key.
 *
 * @return  int : The result of the test.
 * @retval  0: Test pass.
 * @retval -1: Test fail.
 */
int RSA_sign_verify_test()
{
    int status;
#if 0
    PRINTF("\n\rTest RSA Sign\n\r");
    status = RSA_sign_test();
    if (status != 0)
    {
        PRINTF("RSA Sign test failed!\n\r");
        return status;
    }
    else
    {
        PRINTF("RSA Sign test succeeded!\n\r");
    }
#endif
    //  PRINTF("\n\rTest RSA Verify\n\r");
    status = RSA_verify_hash_test();
    if (status != 0)
    {
        //  PRINTF("RSA Verify test failed!\n\r");
        return status;
    }
    else
    {
        //  PRINTF("RSA Verify test succeeded!\n\r");
    }

    return 0;
}

#ifdef __cplusplus
}
#endif

/* End of file RSA_example.c */
