
import fs from 'fs/promises'
import path from 'path'
import { createCipheriv, randomUUID } from 'crypto'
import {DiagRequest,DiagResponse,CRC16_CCIT_ZERO, ServiceName} from 'YT'

let maxLenBlock = 0;
let requestReceived = false;

const appBin=path.join(process.env.PROJECT_ROOT,'fw','app.bin')
const FlashDriverBin=path.join(process.env.PROJECT_ROOT,'fw','FlashDriver.bin')
/*
   modify crc and size
*/
const downloadInit = async function(
    fileName: string,
    downloadReqName: ServiceName,
    crcCheckReqName: ServiceName,
    isApp = false
) {
    const file = await fs.readFile(fileName)
    const diagRequestDownload = new DiagRequest(downloadReqName)
    
    let length = file.length
    if (length % 8 != 0) {
        /* length must 8 bytes align */
        length += (8 - length % 8)
    }

    const rawFileLength = Buffer.alloc(4)
    rawFileLength.writeUInt32BE(length)
    await diagRequestDownload.diagSetParameterRaw('memorySize', rawFileLength)

    const crcDataBuffer = Buffer.alloc(length, 0xFF);
    for (let i = 0; i < file.length; i += 2) {
        // 读取 2 个字节（16 位整数）
        const value = file.readUInt16BE(i); // 读取大端顺序的 16 位整数
        crcDataBuffer.writeUInt16LE(value, i); // 写入小端顺序的 16 位整数
    }
    if (file.length % 2 != 0)
    {
        // 如果 bin 文件没有 2 字节对其，需要特殊处理最后一位。
        crcDataBuffer[file.length] = file[file.length - 1];
    }

    const crcDataSize = length / 2;
    const crcValue = CRC16_CCIT_ZERO(crcDataBuffer);
    
    const diagRouteCrc = new DiagRequest(crcCheckReqName);
    const downloadAddr = diagRequestDownload.diagGetParameterRaw('memoryAddress');
    await diagRouteCrc.diagSetParameterRaw('address', downloadAddr);

    const crcSizeBuffer = Buffer.alloc(4);
    crcSizeBuffer.writeInt32BE(crcDataSize);
    await diagRouteCrc.diagSetParameterRaw('size', crcSizeBuffer);
    
    await diagRouteCrc.diagSetParameterRaw('crcResult', crcValue);
    console.log(`${fileName} crcResult ${crcValue.toString('hex')}`)

    if(isApp){
        //modify app info
        const writeAppInfo=new DiagRequest('Can.writeAppInfo')
        //appLen is little-endian
        const appLenLE=Buffer.alloc(4)
        appLenLE.writeUInt32LE(length)
        await writeAppInfo.diagSetParameterRaw('appLen',appLenLE)
        const crcValueLe=Buffer.alloc(2)
        crcValueLe.writeUInt16LE(crcValue.readUInt16BE(0))
        await writeAppInfo.diagSetParameterRaw('crc',crcValueLe)
        //modify erase size
        const eraseReq=new DiagRequest('Can.RoutineControlEraseFlashMemory')
        const size=Buffer.alloc(4)
        //size must algin with sector size
        const eraseSize=Math.ceil(length/4096)*4096
        size.writeInt32BE(eraseSize)
        await eraseReq.diagSetParameterRaw('size',size)
    }

}

const uploadInit = async function(
    downloadReqName: ServiceName,
    uploadReqName: ServiceName,
) {
    const diagRequestDownload = new DiagRequest(downloadReqName)
    const downloadAddr = diagRequestDownload.diagGetParameterRaw('memoryAddress')
    const downloadSize = diagRequestDownload.diagGetParameterRaw('memorySize')

    const diagRequestUpload = new DiagRequest(uploadReqName)
    await diagRequestUpload.diagSetParameterRaw('memoryAddress', downloadAddr)
    await diagRequestUpload.diagSetParameterRaw('memorySize', downloadSize)
}

const registerFunc = async function(diagName: ServiceName) {
    const diag = new DiagRequest(diagName);
    diag.On('recv', (v)=>{
        const recvData = v.diagGetRaw();
        maxLenBlock = 0;
        if (recvData.length > 2)
        {
            const size = recvData[1] >> 4;
            for(let i = 0; i < size; i++){
                maxLenBlock = maxLenBlock * 256 + recvData[2 + i]
            }
            requestReceived = true
            console.log(`${diagName} maxLenBlock:${maxLenBlock}`)
        }
    })
}

UDS.Init(async ()=>{
    /* =========================================== app ========================================== */
    await downloadInit(
        appBin,
        'Can.RequestDownloadFile',
        'Can.RoutineControlCrcCheck',
        true
    )
    registerFunc('Can.RequestDownloadFile');

    await uploadInit(
        'Can.RequestDownloadFile',
        'Can.RequestUploadFile'
    )
    registerFunc('Can.RequestUploadFile');
    
    /* ====================================== flash driver ====================================== */
    await downloadInit(
        FlashDriverBin,
        'Can.RequestDownloadFlashDriver',
        'Can.RoutineControlFlashDriverCrcCheck'
    )
    registerFunc('Can.RequestDownloadFlashDriver');
    await uploadInit(
        'Can.RequestDownloadFlashDriver',
        'Can.RequestUploadFlashDriver'
    )
    registerFunc('Can.RequestUploadFlashDriver');
    console.log('init Done')
})

let SeedKey: Buffer = Buffer.from([1]);
UDS.On("Can.SecurityAccessRequestSeed.recv", async (val) => {
    // 获取 buffer
    const recvBuffer = val.diagGetRaw();
    if (recvBuffer.length > 2)
    {
        const recvHead = Uint8Array.prototype.slice.call(recvBuffer, 0, 2);
        const recvData = Uint8Array.prototype.slice.call(recvBuffer, 2);
        if ((recvHead[0] == 0x67) && (recvHead[1] == 0x01))
        {
            const iv = "";
            const algorithm = 'aes-128-ecb';
            /* Change your own crypto key here */
            const key = Buffer.from(
                [0x16, 0x15, 0x7e, 0x2b, 0xa6, 0xd2, 0xae, 0x28, 0x88, 0x15, 0xf7, 0xab, 0x3c, 0x4f, 0xcf, 0x09])
            const cipher = createCipheriv(algorithm, key, iv);
            SeedKey = cipher.update(Buffer.from(recvData));
        }
    }
})

UDS.On("Can.SecurityAccessSendKey.preSend", async (val) => {
    if (val.diagGetParameterSize('KEY') != 128)
    {
        await val.diagSetParameterSize('KEY', 128);
    }
    if (SeedKey == undefined)
    {
        console.error('Seed key is undefined!')
        return;
    }
    await val.diagSetParameterRaw('KEY', SeedKey);
})

const getDownloadDiaList = async function(fileName: string) {
    let diagList:DiagRequest[]=[]

    const diagTransferList:DiagRequest[]=[]
    const fileReadSize = maxLenBlock - 1
    let bsc=1;
    if(requestReceived){
        /* split file into blocks, each block is fileReadSize bytes */
        const file = await fs.readFile(fileName)
        const blocks:Buffer[] = []
        for (let i = 0; i < file.length; i += fileReadSize) {
            const endIndex = Math.min(i + fileReadSize, file.length);
            const block = Buffer.concat([
                Buffer.from([0x36, bsc]),
                file.subarray(i, endIndex)
            ]);
            blocks.push(block);
            bsc = (bsc + 1) & 0xFF; // Increment bsc and wrap around at 255
        }
        const lastBlock = blocks[blocks.length - 1];
        console.log(`Last block size: ${lastBlock.length} bytes`);
        const lastBlockDataSize = lastBlock.length - 2; // Subtract header size
        if (lastBlockDataSize % 8 !== 0) {
            const paddingSize = 8 - (lastBlockDataSize % 8);
            const paddedLastBlock = Buffer.alloc(lastBlock.length + paddingSize, 0xFF);
            lastBlock.copy(paddedLastBlock);
            blocks[blocks.length - 1] = paddedLastBlock;
            console.log(`Padded last block from ${lastBlockDataSize} to ${lastBlockDataSize + paddingSize} bytes`);
        }

        /* 0x36 TransferData */
        for(const block of blocks){
            const diagReq=new DiagRequest()
            await diagReq.diagSetRaw(block)
            diagTransferList.push(diagReq)
        }

        /* 0x37 TransferExit */
        const daigTransferExit = new DiagRequest();
        await daigTransferExit.diagSetRaw(Buffer.from([0x37]))

        diagList = [...diagTransferList, daigTransferExit];
    }

    return diagList
}

const getUploadDiaList = async function(
    uploadDiaName: ServiceName
) {
    let recvBuffer = Buffer.alloc(0);
    const recvFunction = function (v: DiagRequest) {
        const buffer = v.diagGetRaw();
        const trimmedBuffer = Uint8Array.prototype.slice.call(buffer, 2);
        recvBuffer = Buffer.concat([recvBuffer, trimmedBuffer]);
    }
    const uploadDiag = new DiagRequest(uploadDiaName)
    
    const memorySizeBuffer = uploadDiag.diagGetParameterRaw('memorySize');
    const memorySize = memorySizeBuffer.readUInt32BE(0); // 将 4 字节 buffer 转换为整数    

    let diagList:DiagRequest[]=[]

    const diagTransferList:DiagRequest[]=[]
    const fileReadSize = maxLenBlock - 1
    let bsc=1;
    if(requestReceived){
        const blocks:Buffer[] = []
        for (let i = 0; i < memorySize; i += fileReadSize) {
            const endIndex = Math.min(i + fileReadSize);
            const block = Buffer.from([0x36, bsc])
            blocks.push(block);
            bsc = (bsc + 1) & 0xFF; // Increment bsc and wrap around at 255
        }

        /* 0x36 TransferData */
        for(const block of blocks){
            const diagReq=new DiagRequest()
            await diagReq.diagSetRaw(block)
            diagReq.On('recv', recvFunction)
            diagTransferList.push(diagReq)
        }

        /* 0x37 TransferExit */
        const daigTransferExit = new DiagRequest();
        await daigTransferExit.diagSetRaw(Buffer.from([0x37]))
        daigTransferExit.On('recv', async ()=>{
            console.log(recvBuffer.toString('hex'));
            await fs.writeFile(path.join(process.env.PROJECT_ROOT,'binFile', 'upload.bin'), recvBuffer)
        })

        diagList = [...diagTransferList, daigTransferExit];
    }
    return diagList
}

UDS.Register("Can.DownloadFile",async (v)=>{
    const diagList:DiagRequest[] = await getDownloadDiaList(appBin);
    return diagList;
})

UDS.Register("Can.DownloadFlashDriver",async (v)=>{
    const diagList:DiagRequest[] = await getDownloadDiaList(FlashDriverBin);
    return diagList;
})

UDS.Register("Can.UploadFile",async (v)=>{
    const diagList:DiagRequest[] = await getUploadDiaList('Can.RequestUploadFile');
    return diagList;
})

UDS.Register("Can.UploadFlashDriver",async (v)=>{
    const diagList:DiagRequest[] = await getUploadDiaList('Can.RequestUploadFlashDriver');
    return diagList;
})

